<?php

namespace Modules\Ads\Controllers\Api;

use Illuminate\Support\Facades\Log;
use Modules\Ads\Controllers\ProductReactService;
use Modules\Ads\Models\Click;
use Modules\Ads\Models\Order;
use Modules\Ads\Models\Product;
use Modules\Ads\Models\ProductAdwordsConversion;
use Modules\Ads\Models\ProductReact;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Modules\Ads\Services\EmailService;


class ApiProductReactService extends ProductReactService
{
    protected $clicks = [];

    public function productLowRate(Request $request)
    {
        set_time_limit(1800);
        ini_set('memory_limit', '6048M');
        $inputs = $request->all();
        if (isset($inputs['clear_cache'])) {
            unset($inputs['clear_cache']);
        }
        ksort($inputs);
        $isReport = $request->input('report');
        $key = 'cache::productLowRatee' . env('APP_LOCALE') . '::' . serialize($inputs);
        $result = cacheGet($key);
        if (!$result || $isReport) {
            $filter = $this->buildQueryFilter($request);
            $conversions = $this->getConversionByProduct($filter);
            $clicks = $this->getClickByProduct(array_merge($filter, [
                'source' => 'adwords'
            ]));
            $mergeData = $this->merge($clicks, $conversions, $inputs);
            $result = $this->filter($mergeData, $inputs);
            if ($request->input('report') == 1) {
                $result = $this->forReport($result, $inputs);
                return response()->json($result);
            }
            \Cache::put($key, $result, 60 * 24 * 30);
        }
        $response = array(
            "status" => 'successful',
            "result" => $result,
        );
        return response()->json($response);
    }

    protected function cronProductLowRate(Request $request) {
        $inputs = $request->all();
        $filter = $this->buildQueryFilter($request);
        $conversions = $this->getConversionByProduct($filter);
        $clicks = $this->getClickByProduct(array_merge($filter, [
            'source' => 'adwords'
        ]));
        $mergeData = $this->merge($clicks, $conversions, $inputs);
    }

    protected function forReport($result, $params) {
        $col = isset($params['reportCol']) ? $params['reportCol'] : 'conversion';
        usort($result, function ($item1, $item2) use ($col){
            if ($item1[$col] == $item2[$col]) return 0;
            return $item1[$col] > $item2[$col] ? -1 : 1;
        });
        $top = isset($params['top']) ? $params['top'] : 30;
        $result = array_slice($result, 0, $top);
        foreach ($result as &$item) {
            $productId = $item['product_id'];
            $pInfo = DB::table('product_info')->where('product_id', $productId)
                ->where('is_default', 1)
                ->first(['sku', 'name']);
            if ($pInfo) {
                $item['sku'] = $pInfo->sku;
                $item['name'] = $pInfo->name;
            }
        }
        return $result;

    }

    protected function filter(&$items, $params = []) {
        $result = [];
        $clickFrom = isset($params['click_from']) ? $params['click_from'] : 0;
        $crTo = isset($params['cr_to']) ? $params['cr_to'] : 0.017;
        foreach ($items as $productId => $item) {
            if ($item['click'] >= $clickFrom  && $item['cr'] <= $crTo) {
                $item['product_id'] = $productId;
                $result[$productId] = $item;
            }
        }
        return $result;

    }

    protected function merge(&$clicks, &$conversions, $params) {
        $result = [];
        $pIds = array_merge(array_keys($clicks), array_keys($conversions));
        $pIds = array_unique($pIds);
        foreach (array_chunk($pIds, 300) as $chunkPIds) {
            $params['pids'] = $chunkPIds;
            $products = $this->getEnableProduct($params);
            foreach ($products as $product) {
                $pId = $product->id;
                $click = isset($clicks[$pId]) ? $clicks[$pId] : 0;
                $conversion = isset($conversions[$pId]) ? $conversions[$pId] : 0;
                $cr = $click > 0 ? $this->ceiling($conversion/$click, 0.001) : 0;
                $item = [
                    'click' => $click,
                    'conversion' => $conversion,
                    'cr' => $cr,
                ];
                $result[$pId] = $item;
            }
        }
        return $result;
    }

    protected function getEnableProduct($params = []) {
        $query = DB::table('product')
            ->where('approve_advertising', 1)
            ->where('is_violation', 0)
            ->where('is_trademark', 0)
            ->where('status', 'ACTIVE');
        if (isset($params['cr_to']) && $params['cr_to'] > 0) {
            //$query->where('sold', '>', 0);
        }
         if (array_key_exists('pids', $params)) {
             $query->whereIn('id', $params['pids']);
         }
        return $query->get(['id', 'name']);
    }

    protected function getClickByProduct($filter) {
        $result = [];
        $fromTime = strtotime($filter['date_gte']);
        $toTime = strtotime($filter['date_lte']);
        while ($fromTime <= $toTime) {
            $items = $this->getClick(array_merge($filter, [
                'date_gte' => date('Y-m-d', $fromTime),
                'date_lte' => date('Y-m-d', $fromTime),
            ]));
            $fromTime += 86400;
            if ($items && count($items)) {
                foreach ($items as $item) {
                    if (!array_key_exists($item->product_id, $result)) {
                        $result[$item->product_id] = 0;
                    }
                    $result[$item->product_id] += $item->clicks;
                }
            }
        }
        return $result;
    }

    protected function getConversionByProduct($filter) {
        $result = [];
        $orders = $this->getOrder($filter);
        if ($orders) {
            foreach ($orders as $item) {
                $pId = $item->product_id;
                if (!array_key_exists($pId, $result)) {
                    $result[$pId] = 0;
                }
                $result[$pId] += $item->conversion;
            }
        }

        return $result;

    }

    protected function getClick($filter)
    {
        $query = Click::buildClickQuery($filter);
        $columns = [
            'product_id', 'url',
            DB::raw('SUM(clicks) as clicks'),
        ];
        $groupBy = [
            'product_id'
        ];
        $query->groupBy($groupBy);
        return $query->get($columns);
    }

    protected function buildQueryFilter($request)
    {
        $days = $request->input('days', 30);
        $from = $request->input('time_from', date('d/m/Y', strtotime("-$days days")));
        $to = $request->input('time_to', date('d/m/Y', time()));
        $fromDate = $this->getDateTimeFilter($from);
        $toDate = $this->getDateTimeFilter($to);
        $toDate->setTime(23, 59, 59);
        $filter = [
            'time_from' => $fromDate->format('Y-m-d H:i:s'),
            'time_to' => $toDate->format('Y-m-d H:i:s'),
            'date_gte' => $fromDate->format('Y-m-d'),
            'date_lte' => $toDate->format('Y-m-d'),
        ];

        return $filter;
    }

    protected function getOrder($filter)
    {
        $columns = [
            'order_item.product_id',
            DB::raw("1 as conversion")
        ];
        $queryFilter = [
            'type' => Order::STATUS_OUT,
            'create_time_gte' => $filter['time_from'],
            'create_time_lte' => $filter['time_to'],
            'payment_status' => 'PAID',
            'join_order_item' => 1,
            'join_order_meta_adwords' => 1,
            'group_by' => ['order_item.product_id', 'order.id'],
        ];

        $query = Order::buildOrderQuery($queryFilter);
        $items = $query->get($columns);
        return $items;
    }

}
