<?php

namespace Modules\Ads\Controllers;

use App\Utils\LogHelper;
use Module;
use Illuminate\Support\Facades\DB;

class BulkApproveProductController extends Controller
{
    public function showScreen()
    {
        $staffs = \DB::table('users')
            ->where('role', 'STAFF')
            ->where('status', 'ACTIVE')
            ->select(['id', 'name'])
            ->get();

        return view('ads::bulk-approve.index', [
            'staffs' => $staffs
        ]);
    }

    public function getProducts(\Illuminate\Http\Request $request)
    {
        set_time_limit(1800);

        $from = $request->input('from', date('Y-m-d') . ' 00:00:00');
        $to = date('Y-m-d', strtotime($from)) . ' 23:59:59';
        
        $staffId = (int) $request->input('staffId');

        // Cache categories since they don't change often
        $categories = cache()->remember('product_categories', 3600, function() {
            return DB::table('category')
                ->select(['id', 'name'])
                ->pluck('name', 'id')
                ->toArray();
        });

        // Build query using query builder for better readability and performance
        $query = DB::table('product')
            ->select([
                'product.id'
            ])
            ->from(DB::raw("sb_product"))
            ->whereNull('product.deleted_at')
            ->where([
                ['product.status', 'ACTIVE'],
                ['product.is_trademark', 0],
                ['product.approve_advertising', 0],
                ['product.price', '>', 0],
                ['product.is_violation', 0],
                ['product.created_at', '>=', $from],
                ['product.created_at', '<=', $to]
            ]);

        if ($staffId) {
            $query->where('product.actor_id', $staffId);
        }

        $products = $query->groupBy('product.id')
            ->get();

        $productCategories = $this->getProductCategories($products->pluck('id')->toArray());

        $products = $products->map(function($product) use ($productCategories) {
            $product->category_id = $productCategories[$product->id] ?? null;
            return $product;
        });

        $categoryReport = [];
        foreach ($products as &$product) {
            if (!isset($categoryReport[$product->category_id])) {
                $categoryReport[$product->category_id] = [
                    'name' => $categories[$product->category_id] ?? 'Unknown Category',
                    'count' => 0
                ];  
            }
            $categoryReport[$product->category_id]['count']++;
        }

        return [
            'status' => 'successful',
            'result' => [
                'report' => $categoryReport,
                'products' => $products
            ]
        ];
    }

    public function getProductCategories($productIds)
    {
        $productIdChunks = array_chunk($productIds, 1000);

        $productCategories = [];
        foreach ($productIdChunks as $chunk) 
        {
            $productCategoriesPart = DB::table('product_n_category')
                ->whereIn('product_id', $chunk)
                ->where('is_parent', 0)
                ->select(['category_id', 'product_id'])
                ->get()
                ->toArray();

            foreach ($productCategoriesPart as $productCategory) {
                $productCategories[$productCategory->product_id] = $productCategory->category_id;
            }
        }

        return $productCategories;
    }   

    public function logPush(\Illuminate\Http\Request $request) 
    {
        // user id
        // filters
        // productIds

        // $filters = $request->input('filters');

        $userId = $request->input('userId');
        $productIds = $request->input('productIds');

        $logArray = [];

        $countProducts = count($productIds);
        for ($i = 0; $i < $countProducts; $i++) {
            $logArray[] = [
                'actor_type' => 'USER',
                'target_type' => 'PRODUCT',
                'event_type' => 'MANUAL_BULK_APPROVE_PRODUCT',
                'target_id' => $productIds[$i],
                'data' => 'userId: ' . $userId . ', productId: ' . $productIds[$i],
                'created_at' => date('Y-m-d H:i:s')
            ];
        }

        $response = [
            'status' => 'successful'
        ];
        LogHelper::saveLogBulk($logArray);

        return $response;
    }
}