<?php

namespace Modules\Ads\Controllers\Cron;

use Carbon\Carbon;
use Module;
use Illuminate\Http\Request;
use Modules\Ads\Controllers\Service\MerchantProductService;
use Modules\Ads\Models\AdPerformanceView;
use Modules\Ads\Models\Product;
use Modules\Ads\Controllers\Controller;
use Modules\Ads\Services\ProductService;
use Modules\Ads\Services\CampaignService;
use Modules\Ads\Models\ApproveAdvertising;
use Modules\Ads\Models\ProductAdvertising;
use Modules\Ads\Services\GoogleAdsService;
use Modules\Localization\Helpers\DBConnect;
use Modules\Ads\Services\ProductInfoService;
use Modules\Ads\Models\ProductApproveAdvertising;
use Modules\Ads\Services\ProductAdvertisingService;
use Modules\Ads\Services\ProductToAdvertisingService;
use Modules\Ads\Services\ProductToAdvertisingAdsService;
use DB;
class AdsController extends Controller
{
    protected $googleAdsService;
    protected $campaignService;
    protected $productAdvertisingService;
    protected $productToAdvertisingService;
    protected $productInfoService;
    protected $productToAdvertisingAdsService;
    protected $productService;
    protected $googleAdsApiToken;
    protected $facebookAdsApiToken;
    protected $merchantProductService;

    public function __construct(GoogleAdsService $googleAdsService,
                                CampaignService $campaignService,
                                ProductAdvertisingService $productAdvertisingService,
                                ProductToAdvertisingService $productToAdvertisingService,
                                ProductInfoService $productInfoService,
                                ProductToAdvertisingAdsService $productToAdvertisingAdsService,
                                ProductService $productService,
                                MerchantProductService $merchantProductService
                                )
    {        
        $this->googleAdsService = $googleAdsService;
        $this->campaignService = $campaignService;
        $this->productAdvertisingService = $productAdvertisingService;
        $this->productToAdvertisingService = $productToAdvertisingService;
        $this->productToAdvertisingAdsService = $productToAdvertisingAdsService;
        $this->productInfoService = $productInfoService;
        $this->productService = $productService;
        $this->googleAdsApiToken = \Config::get('ads::ads.google_ads_api_token');
        $this->merchantProductService = $merchantProductService;
        $this->facebookAdsApiToken = \Config::get('ads::ads.facebook_ads_api_token');
    }

    public function cronCampaign() {
        $accountIds =  \Config::get('ads::ads.account_ids');
        $url = \Config::get('ads::ads.google_ads_api_url') . '/campaign';
        foreach ($accountIds as $accountId) {
            $result = $this->triggerSyncRequest($url.'?customerId=' . $accountId,'GET',[],['token: ' . $this->googleAdsApiToken,'Content-Type: application/json']);
            if (is_array($result['result'])){
                foreach ($result['result'] as $item) {
                    $insertData = [
                        'campaign_id' => (string) $item['id'],
                        'campaign_name' => $item['name'],
                        'campaign_type' => $item['type'],
                        'status' => $item['status'],
                        'account' => $accountId
                    ];
                    $checkExists = $this->campaignService->checkExists(['campaign_id' => $insertData['campaign_id']]);
                    if (!$checkExists) {
                        $this->campaignService->create($insertData);
                    } else {
                        $this->campaignService->updateByField(['campaign_id' => $insertData['campaign_id']], $insertData);
                    }
                }
            }
        }
        
        $response = ['status' => 'successful'];
        return response()->json($response);
    }

    public function cronProductAdvertising() {
        set_time_limit(300);
        $accounts = config('google_ads.accounts', []);
        if (empty($accounts)) {
            throw new \Exception('The module Product Advertising require the config "Google ads account" in /config/google_ads.php');
        }
        foreach ($accounts as $key => $value) {
            $session = $this->googleAdsService->buildSessionAccount($value['customer_id']);
            $campaigns = $this->googleAdsService->getCampaigns($session);
            $campaignIds = [];
            $campaignByIds = [];
            foreach($campaigns as $item) {
                $campaignIds[] = $item['id'];
                $campaignByIds[$item['id']] = $item['status'];
            }
            $adGroups = $this->googleAdsService->getAdgroups($session, $campaignIds);
            $dataSave = [];
            $date = new \DateTime();
            $dateWhere = date('Y-m-d', strtotime('now'));
            $skus = [];
            if (count($adGroups) > 0) {
                foreach($adGroups as $item) {
                    $arrProducts = explode("|", $item['name']);
                    array_shift($arrProducts);
                    if (count($arrProducts) > 0) {
                        foreach($arrProducts as $code) {
                            $arrCode = explode(" ", trim($code));
                            $arrCode = explode("*", $arrCode[0]);
                            $newData = [];
                            $type = $item['type'];
                            if ($item['type'] == 'SEARCH_STANDARD') {
                                $type = 'text_ad';
                            }
                            if ($item['type'] == 'SEARCH_DYNAMIC_ADS') {
                                $type = 'dsa';
                            }
                            if ($item['type'] == 'SHOPPING_PRODUCT_ADS') {
                                $type = 'ggsp';
                            }
                            if ($item['type'] == 'SHOPPING_GOAL_OPTIMIZED_ADS') {
                                $type = 'sgoa';
                            }
                            if ($arrCode[0] != '*') {
                                $newData['sku'] = strtoupper($arrCode[0]);
                                $skus[] = strtoupper($arrCode[0]);
                                $newData['ad_status'] = strtolower($item['status']);
                                $newData['ad_type'] = $type;
                                $newData['date'] = $date;
                                $newData['adgroup_id'] = $item['id'];
                                $newData['adgroup_name'] = $item['name'];
                                $newData['account_key'] = $value['customer_id'];
                                $newData['account_name'] = $value['name'];
                                $newData['campaign_id'] = $item['campaign_id'];
                                $newData['campaign_name'] = $item['campaign_name'];
                                $newData['campaign_status'] = strtolower($campaignByIds[$item['campaign_id']]);
                                if ($newData['sku'] != '') {
                                    $dataSave[] = $newData;
                                }
                            }
                        }
                    }
                }
            }
            $isGroup = \Config::get('ads::ads.is_group');
            if ($isGroup) {
                $this->buildProductAdvertiseById($dataSave, $skus);
            } else {
                $this->buildProductAdvertiseBySku($dataSave);
            }
            
        }
        $response = [
            'status' => 'successful',
            'message' => 'Get data successfully'
        ];
        return response()->json($response);
    }

    private function buildProductAdvertiseById($dataSave, $skus) {
        $productBySkus = $this->productInfoService->getProductInfoBySkus($skus);
        foreach($dataSave as $data) {
            $arrChecks = [
                    'sku' => $data['sku'],
                    'account_key' => $data['account_key'],
                    'ad_type' => $data['ad_type'],
                    'adgroup_id' => $data['adgroup_id'],
                    'campaign_id' => $data['campaign_id'],
                    'date' => $dateWhere
            ];
            if (isset($productBySkus[$data['sku']])) {
                $data['product_id'] = $productBySkus[$data['sku']]->product_id;
                $data['product_sku_id'] = $productBySkus[$data['sku']]->product_sku_id;
            }
            $object = $this->productAdvertisingService->getFirst($arrChecks);
            if ($object) {
                $data['id'] = $object->id;
                $this->productAdvertisingService->update($data);
            } else {
                $this->productAdvertisingService->create($data);
            }
        }
    }

    private function buildProductAdvertiseBySku($dataSave) {
        foreach($dataSave as $data) {
            $arrChecks = [
                'sku' => $data['sku'],
                'account_key' => $data['account_key'],
                'ad_type' => $data['ad_type'],
                'adgroup_id' => $data['adgroup_id'],
                'campaign_id' => $data['campaign_id'],
                'date' => $dateWhere
            ];
            $object = $this->productAdvertisingService->getFirst($arrChecks);
            if ($object) {
                $data['id'] = $object->id;
                $this->productAdvertisingService->update($data);
            } else {
                $this->productAdvertisingService->create($data);
            }
        }
    }

    public function cronProductToAdvertising() {
        set_time_limit(0);
        ini_set('memory_limit', '2048M');
        $response = ['status'=>'successful'];
        $from = date('d-m-Y', strtotime('now'));
        $to = date('d-m-Y', strtotime('now'));
        $dateFrom = $this->convertDate($from);
        $dateTo = $this->convertDate($to);
        $isGroup = \Config::get('ads::ads.is_group');
        if ($isGroup) {
            $this->buildProductToAdvertiseById($dateFrom, $dateTo);
        } else {
            $this->buildProductToAdvertiseBySku($dateFrom, $dateTo);
        }
        return response()->json($response);
    }



    private function buildProductToAdvertiseById($dateFrom, $dateTo) {
        $products = $this->productService->find(['status' => 'ACTIVE']);
        $items = [];
        $ids = [];
        foreach ($products as $product) {
            $items[] = (object) $product->toArray();
            $ids[] = $product->id;
        }
        $adgroups = $this->productAdvertisingService->find(
            ['from' => $dateFrom, 'to' => $dateTo]
        )->toArray();
        $adgroupByKey = [];
        if (!empty($adgroups) && is_array($adgroups)){
            foreach ($adgroups as $item) {
                if (!isset($adgroupByKey[$item['product_id']])) {
                    $adgroupByKey[$item['product_id']]['enabled'] = [];
                    $adgroupByKey[$item['product_id']]['paused'] = [];
                }
                if ($item['ad_status'] == 'enabled' && $item['campaign_status'] == 'enabled') {
                    $adgroupByKey[$item['product_id']]['enabled'][] = $item;
                }
                if (($item['ad_status'] == 'paused' && $item['campaign_status'] == 'paused')
                    || ($item['ad_status'] == 'enabled' && $item['campaign_status'] == 'paused')
                    || ($item['ad_status'] == 'paused' && $item['campaign_status'] == 'enabled')) {
                    $adgroupByKey[$item['product_id']]['paused'][] = $item;
                }
            }
        }
        $domain =  \URL::to('/');
        $this->productToAdvertisingAdsService->truncate();
        $campaignByIds = $this->campaignService->campaignByIds();
        $products = [];
        foreach ($items as $item) {
            $product = [];
            $product['title'] = $item->name;
            $product['sku'] = $item->sku;
            $product['product_id'] = $item->id;
            $product['inventory'] = 0;
            $product['status'] = 'pending';
            $link = $domain . "/" . $item->slug . '-p' . $item->id . '.html';
            $product['link'] = $link;
            $data = [];
            if (isset($adgroupByKey[$item->id])){
                $product['status'] = 'pending';
                if (count($adgroupByKey[$item->id]['enabled']) > 0) {
                    $product['status'] = 'running';
                    foreach($adgroupByKey[$item->id]['enabled'] as $info) {
                        $info = (object) $info;
                        $newInfo = [
                            'campaign_name' => $info->campaign_name,
                            'adgroup_name' => $info->adgroup_name,
                            'ad_type' => $info->ad_type,
                            'account_key' => $info->account_key,
                            'account_name' => $info->account_name,
                            'ad_status' => $info->ad_status,
                            'campaign_status' => $info->campaign_status,
                            'campaign_id' => $info->campaign_id,
                            'campaign_type' => ''
                        ];
                        if (isset($campaignByIds[$info->campaign_id])) {
                            $newInfo['campaign_type'] = $campaignByIds[$info->campaign_id]->campaign_type;
                        }
                        $data[] = $newInfo;
                    }
                }
                if (count($adgroupByKey[$item->id]['paused']) > 0) {
                    foreach($adgroupByKey[$item->id]['paused'] as $info) {
                        $info = (object) $info;
                        $newInfo = [
                            'campaign_name' => $info->campaign_name,
                            'adgroup_name' => $info->adgroup_name,
                            'ad_type' => $info->ad_type,
                            'account_key' => $info->account_key,
                            'account_name' => $info->account_name,
                            'ad_status' => $info->ad_status,
                            'campaign_status' => $info->campaign_status,
                            'campaign_id' => $info->campaign_id,
                            'campaign_type' => ''
                        ];
                        if (isset($campaignByIds[$info->campaign_id])) {
                            $newInfo['campaign_type'] = $campaignByIds[$info->campaign_id]->campaign_type;
                        }
                        $data[] = $newInfo;
                    }
                }
            }
            $product['data'] = json_encode($data);
            $products[] = $product;
            $first = $this->productToAdvertisingService->getFirst(['id' => $product['product_id']]);
            if (!$first) {
                $object = $this->productToAdvertisingService->create($product);
            } else {
                $product['id'] = $first->id;
                $object = $this->productToAdvertisingService->update($product);
            }
            if (!empty($data)) {
                foreach ($data as $ad) {
                    $ad['pta_id'] = $object->id;
                    $this->productToAdvertisingAdsService->create($ad);
                }
            }

        }
        $deleteItems = $this->productToAdvertisingService->find(['!productIds' => $ids, 'columns' => 'id,sku']);
        $deleteIds = [];
        foreach ($deleteItems as $item) {
            $deleteIds[] = $item->id;
        }
        if (!empty($deleteIds)) {
            $this->productToAdvertisingService->delete($deleteIds);
        }
    }


    private function buildProductToAdvertiseBySku($dateFrom, $dateTo) {
        $productBySkus = $this->productInfoService->getProductInfoBySkus();
        $items = [];
        $skus = [];
        foreach ($productBySkus as $key => $value) {
            $items[] = (object) $value->toArray();
            $skus[] = $key;
        }
        $adgroups = $this->productAdvertisingService->find(
            ['from' => $dateFrom, 'to' => $dateTo]
        )->toArray();
        $adgroupByKey = [];
        if (!empty($adgroups) && is_array($adgroups)){
            foreach ($adgroups as $item) {
                if (!isset($adgroupByKey[$item['sku']])) {
                    $adgroupByKey[$item['sku']]['enabled'] = [];
                    $adgroupByKey[$item['sku']]['paused'] = [];
                }
                if ($item['ad_status'] == 'enabled' && $item['campaign_status'] == 'enabled') {
                    $adgroupByKey[$item['sku']]['enabled'][] = $item;
                }
                if (($item['ad_status'] == 'paused' && $item['campaign_status'] == 'paused')
                    || ($item['ad_status'] == 'enabled' && $item['campaign_status'] == 'paused')
                    || ($item['ad_status'] == 'paused' && $item['campaign_status'] == 'enabled')) {
                    $adgroupByKey[$item['sku']]['paused'][] = $item;
                }
            }
        }
        $domain =  \URL::to('/');
        $this->productToAdvertisingAdsService->truncate();
        $campaignByIds = $this->campaignService->campaignByIds();
        $products = [];
        foreach ($items as $item) {
            $product = [];
            $product['title'] = $item->name;
            $product['sku'] = $item->sku;
            $product['inventory'] = 0;
            $product['status'] = 'pending';
            $product['product_id'] = $item->product_id;
            $link = $domain . "/" . $item->slug . '-p' . $item->product_id . '.html';
            if ($item->product_sku_id != null && $item->product_sku_id !=  0) {
                $link .= '?spid=' . $item->product_sku_id;
                $product['product_sku_id'] = $item->product_sku_id;
            }
            $product['link'] = $link;
            $data = [];
            if (isset($adgroupByKey[$item->sku])){
                $product['status'] = 'pending';
                if (count($adgroupByKey[$item->sku]['enabled']) > 0) {
                    $product['status'] = 'running';
                    foreach($adgroupByKey[$item->sku]['enabled'] as $info) {
                        $info = (object) $info;
                        $newInfo = [
                            'campaign_name' => $info->campaign_name,
                            'adgroup_name' => $info->adgroup_name,
                            'ad_type' => $info->ad_type,
                            'account_key' => $info->account_key,
                            'account_name' => $info->account_name,
                            'ad_status' => $info->ad_status,
                            'campaign_status' => $info->campaign_status,
                            'campaign_id' => $info->campaign_id,
                            'campaign_type' => ''
                        ];
                        if (isset($campaignByIds[$info->campaign_id])) {
                            $newInfo['campaign_type'] = $campaignByIds[$info->campaign_id]->campaign_type;
                        }
                        $data[] = $newInfo;
                    }
                }
                if (count($adgroupByKey[$item->sku]['paused']) > 0) {
                    foreach($adgroupByKey[$item->sku]['paused'] as $info) {
                        $info = (object) $info;
                        $newInfo = [
                            'campaign_name' => $info->campaign_name,
                            'adgroup_name' => $info->adgroup_name,
                            'ad_type' => $info->ad_type,
                            'account_key' => $info->account_key,
                            'account_name' => $info->account_name,
                            'ad_status' => $info->ad_status,
                            'campaign_status' => $info->campaign_status,
                            'campaign_id' => $info->campaign_id,
                            'campaign_type' => ''
                        ];
                        if (isset($campaignByIds[$info->campaign_id])) {
                            $newInfo['campaign_type'] = $campaignByIds[$info->campaign_id]->campaign_type;
                        }
                        $data[] = $newInfo;
                    }
                }
            }
            $product['data'] = json_encode($data);
            $products[] = $product;
            $first = $this->productToAdvertisingService->getFirst(['sku' => $product['sku']]);
            if (!$first) {
                $object = $this->productToAdvertisingService->create($product);
            } else {
                $product['id'] = $first->id;
                $object = $this->productToAdvertisingService->update($product);
            }
            if (!empty($data)) {
                foreach ($data as $ad) {
                    $ad['pta_id'] = $object->id;
                    $this->productToAdvertisingAdsService->create($ad);
                }
            }
        }
        $deleteItems = $this->productToAdvertisingService->find(['!skus' => $skus, 'columns' => 'id,sku']);
        $deleteIds = [];
        foreach ($deleteItems as $item) {
            $deleteIds[] = $item->id;
        }
        if (!empty($deleteIds)) {
            $this->productToAdvertisingService->delete($deleteIds);
        }
    }

    private function convertDate($str) {
        $arrDate = explode("-", $str);
        $arrDate = array_reverse($arrDate);
        return implode("-", $arrDate);
    }
    private function normalizeAndHash($value){
        return hash('sha256', strtolower(trim($value)));
    }
    private function buildOfflineUserDataJobOperations($customers){
        $retVal = [];
        foreach ($customers as $item){
            $fullName = explode(' ',$item->full_name);
            $firstName = $fullName[count($fullName) -1];
            unset($fullName[count($fullName) -1]);
            $lastName = implode(' ',$fullName);
            if (empty(trim($lastName))){
                $lastName = $firstName;
            }
            if (
                empty($item->email) ||
                empty($firstName) ||
                empty($lastName) ||
                empty($item->iso) ||
                empty($item->zip_code) ||
                empty($item->phonecode.$item->phone)
            ){
                continue;
            }
            $data = [
                'hashed_email' => $this->normalizeAndHash($item->email),
                'hashed_first_name' => $this->normalizeAndHash($firstName),
                'hashed_last_name' => $this->normalizeAndHash($lastName),
                'country_code' => $item->iso,
                'postal_code' => $item->zip_code,
                'hashed_phone_number' => $this->normalizeAndHash($item->phonecode.$item->phone)
            ];
            $retVal[] = $data;
        }
        return $retVal;
    }
    public function pushCustomerList(){
        $locales = getModuleLocale();
        $customerList = [];
        foreach ($locales as $locale){
            if (!$locale['enable']) {
                continue;
            }
            $connection = DBConnect::connect($locale['locale']);
            $query = $connection->table('order as o')
                ->join(DB::raw('`' . env('DB_DATABASE_CUSTOMER', 'printerval_customer') . '`.`customer` as sb_c'),'c.id','=','o.customer_id')
                ->join('countries as ct','ct.id','=','c.country_id')
                ->where('payment_status','=','PAID')
                ->groupBy('c.id');
            $customers = $query->get(['c.email','c.full_name','c.phone','o.zip_code','ct.iso','ct.phonecode']);
            $data = $this->buildOfflineUserDataJobOperations($customers);
            $customerList = array_merge($customerList,$data);
        }
        $mccAccountId = \Config::get('ads::ads.mcc_account_id');
        $url = \Config::get('ads::ads.google_ads_api_url').'/add-customer-match-list';
        $requestData = [
            'customerId' => $mccAccountId,
            'customers' => $customerList
        ];
        $response = $this->triggerSyncRequest($url,'POST',$requestData,['token: '.$this->googleAdsApiToken,'Content-Type: application/json']);
        return response($response);
    }

    public function updateCtr(Request $request){
        $retVal = ['status' => 'fail'];
        $updatedSkus = [];
        $customerId = $request->get('customerId');
        $campaignId = $request->get('campaignId');
        if (!empty($customerId) && !empty($campaignId)){
            $url = \Config::get('ads::ads.google_ads_api_url').'/shopping-performance-report';
            $requestData = [
                'customerId' => $customerId,
                'campaignId' => $campaignId
            ];
            $url = $url .'?'. http_build_query($requestData);
            $response = $this->triggerSyncRequest($url,'GET',[],['token: ' . $this->googleAdsApiToken,'Content-Type: application/json']);
            $result = isset($response['result'])?$response['result']:[];
            if (empty($result)){
                $retVal['result'] = $response;
            }else{
                ProductApproveAdvertising::where('ctr','>',0)->update(['ctr' => 0]);
                foreach ($result as $sku => $ctr){
                    $pa = ProductApproveAdvertising::where('code',$sku)->first();
                    if (!empty($pa)){
                        $pa->ctr = round($ctr,2);
                        $pa->save();
                        $updatedSkus[] = $sku;
                    }
                }
                $retVal['result'] = $updatedSkus;
                $retVal['count'] = count($updatedSkus);
                $retVal['status'] = 'successful';
            }
        }else{
            $retVal['message'] = 'customerId, campaignId required';
        }
        return response()->json($retVal);
    }

    public function storeCost(Request $request){
        set_time_limit(180);
       $retVal = [];
       $accountIds =  \Config::get('ads::ads.account_ids');
       $url = \Config::get('ads::ads.google_ads_api_url') . '/campaign/cost';
       $urlStoreCost = \Config::get('ads::ads.store_cost_service');
       $urlAccount = \Config::get('ads::ads.google_ads_api_url') . '/account';
       $params = [
           'columns' => ["campaign.id","campaign.name","segments.hour","segments.date","metrics.cost_micros","metrics.clicks"]
       ];
       if ($request->has('fromDate')){
           $params['fromDate'] = $request->input('fromDate');
       }
       if ($request->has('toDate')){
            $params['toDate'] = $request->input('toDate');
       }
        $dayAgo = 7;
        if ($request->has('dayAgo')){
            $dayAgo = $request->input('dayAgo');
        }
        $input = [
            'accountIds' => $accountIds,
            'url' => $url,
            'urlStoreCost' => $urlStoreCost,
            'urlAccount' => $urlAccount
        ];
        if (isset($params['fromDate']) || isset($params['toDate'])){
            $input['params'] = $params;
            $retVal[] = $this->sendAdsCost($input);
        }else{
            for ($i = -1; $i<= $dayAgo; $i++){
                if ($i==-1){
                    $day = '+1 day';
                }else{
                    $day = '-' . $i. ' day';
                }
                $params['fromDate'] = date('Y-m-d',strtotime($day));
                $params['toDate'] = date('Y-m-d',strtotime($day));
                $input['params'] = $params;
                $retVal[] = $this->sendAdsCost($input);
            }
        }
       return response()->json($retVal);
    }

    private function sendAdsCost($input){
        $retVal = [];
        $url = $input['url'];
        $params = $input['params'];
        $urlAccount = $input['urlAccount'];
        $accountIds = $input['accountIds'];
        $urlStoreCost = $input['urlStoreCost'];
        foreach ($accountIds as $accountId) {
            $params['customerId'] = $accountId;
            $result = $this->triggerSyncRequest($url, 'GET', $params, ['token: ' . $this->googleAdsApiToken, 'Content-Type: application/json']);
            $resultAccount = $this->triggerSyncRequest($urlAccount . '?customerId=' . $accountId, 'GET', [], ['token: ' . $this->googleAdsApiToken, 'Content-Type: application/json']);
            $allDataCost = $this->buildDataCost($result);
            $requestData = [
                'account' => $resultAccount['result'],
                'items' => $allDataCost
            ];
            $retVal[$accountId] = $this->triggerSyncRequest($urlStoreCost, 'POST', $requestData, ['Content-Type: application/json']);
            sleep(2);
//            \Log::info("AdsCost: " . $params['fromDate'] . " |  $retVal[$accountId]");
        }
        //           $chunkCost = array_chunk($allDataCost,$chunk);
//           foreach ($chunkCost as $index => $data){
//               $requestData = [
//                   'account' => $resultAccount['result'],
//                   'items' => $data
//               ];
//               $retVal[$index . '|'. $accountId] = $this->triggerSyncRequest($urlStoreCost,'POST',$requestData,['Content-Type: application/json']);
//           }
        return $retVal;
    }

    private function buildDataCost($result){
        $retVal = [];
        $costs = $result['result'];
        if (!empty($costs) && is_array($costs)){
            foreach ($costs as $item){
                $data = [
                    'campaign_id' => $item['campaign.id'],
                    'cost' => $item['metrics.cost_micros']/1000000,
                    'date_report' => $item['segments.date'],
                    'campaign_name' => $item['campaign.name'],
                    'click' => $item['metrics.clicks'],
                    'hour' => $item['segments.hour'],
                ];
                $retVal[] = $data;
            }
        }
        return $retVal;
    }

    public function cronBulkApprove(Request $request)
    {
        set_time_limit(10 * 3600);

        $retVal = [
            'status' => 'fail',
            'message' => '',
        ];
        if ($request->header('token') == 'megaads@123' || $request->get('service_token') == 'megaads@123') {
            if (
                $request->has('product_ids') ||
                ($request->has('from_id') && $request->has('to_id')) ||
                $request->has('update_from') ||
                $request->has('create_from')
            ) {
                $brand = $request->get('brand', 'Printerval');
                $violateKeywords = app('Modules\Ads\Controllers\Cron\ViolationController')->getViolateKeywords();
                $limit = $request->get('limit', 1000);
                $query = \DB::table('product')
                    ->select(['product.id', 'product.name', 'product.sku', 'product.description'])
                    ->where('product.status', 'ACTIVE')
                    ->where('product.is_violation', 0)
                    ->whereNull('product.deleted_at');
                if ($request->get('is_approved', 1)) {
                    $query->where('product.approve_advertising', 1);
                }

                if ($request->has('product_ids')) {
                    $productIds = explode(',', $request->get('product_ids'));
                    foreach (array_chunk($productIds, 300) as $partProductIds) {
                        $cloneQuery = clone $query;
                        $products = $cloneQuery->whereIn('product.id', $partProductIds)
                            ->get();
                        $this->approveProducts($products, $violateKeywords, $brand, $request->has('is_approved') && $request->get('is_approved') == 0);
                    }
                } else if ($request->has('from_id') && $request->has('to_id')) {
                    $query->leftJoin('product_approve_advertising', 'product_approve_advertising.product_id', 'product.id')
                        ->whereNull('product_approve_advertising.product_id');
                    $step = $limit;
                    $fromId = $request->get('from_id');
                    $toId = $request->get('to_id');
                    while($fromId < $toId) {
                        $cloneQuery = clone $query;
                        $tmpToId = ($fromId + $step) > $toId ? $toId : ($fromId + $step);
                        $cloneQuery->where('product.id', '>=', $fromId)
                            ->where('product.id', '<=', $tmpToId);
                        $products = $cloneQuery->get();
                        if (!$products->isEmpty()) {
                            $this->approveProducts($products, $violateKeywords, $brand);
                        }
                        if ($tmpToId >= $toId) {
                            break;   
                        }
                        $fromId += $step + 1;
                    }
                } else if ($request->has('update_from')) {
                    $query->leftJoin('product_approve_advertising', 'product_approve_advertising.product_id', 'product.id')
                        ->whereNull('product_approve_advertising.product_id');
                    $updateTo = $request->get('update_to', date('Y-m-d H:i:s', time()));
                    $query->where('product.updated_at', '>=', $request->get('update_from'));
                    $query->where('product.updated_at', '<=', $updateTo);
                    $query->limit($limit);
                    while(true) {
                        $products = $query->get();
                        if ($products->isEmpty()) {
                            break;
                        }
                        
                        $this->approveProducts($products, $violateKeywords, $brand);
                    }
                } else if ($request->has('create_from')) {
                    $createTo = $request->get('create_to', date('Y-m-d H:i:s', time()));
                    $query->where('product.created_at', '>=', $request->get('create_from'));
                    $query->where('product.created_at', '<=', $createTo);
                    $query->limit($limit);
                    while(true) {
                        $products = $query->get();
                        if ($products->isEmpty()) {
                            break;
                        }
                        
                        $this->approveProducts($products, $violateKeywords, $brand);
                    }
                }

                $retVal['status'] = 'successful';
            } else {
                $retVal['message'] = 'missing product_ids || from_id + to_id || update_from || create_from';
            }
        } else {
            $retVal['message'] = 'token mismatch';
        }

        return $retVal;
    }

    public function cronBulkDisapprove(Request $request) {
        set_time_limit(10 * 3600);

        $retVal = [
            'status' => 'fail',
            'message' => '',
        ];

        if ($request->header('token') == 'megaads@123' || $request->get('service_token') == 'megaads@123') {
            if ($request->has('product_ids')) {
                $productIds = explode(',', $request->get('product_ids'));
                foreach (array_chunk($productIds, 300) as $partProductIds) {
                    \DB::table('product')
                        ->whereIn('product.id', $partProductIds)
                        ->update([
                            'approve_advertising' => 0,
                            'updated_at' => date('Y-m-d H:i:s', time())
                        ]);
                }
                $retVal['status'] = 'successful';
            } else {
                $retVal['message'] = 'missing product_ids';
            }
        } else {
            $retVal['message'] = 'token mismatch';
        }

        return $retVal;
    }

    public function approveProducts($products, $violateKeywords = [], $brand = 'Printerval', $updateProduct = false) {
        $approveAdvertisingData = [];
        $violateProducts = [];
        $productIds = [];
        foreach ($products as $product) {
            $productIds[] = $product->id;
            if ($updateProduct && ApproveAdvertising::where('product_id', $product->id)->exists()) {
                continue;
            }
            $productName = preg_replace("/[\-\.\_\+]+/m", '', $product->name);
            $check = false;
            foreach ($violateKeywords as $keyword) {
                $isViolate = app('Modules\Ads\Controllers\Cron\ViolationController')->isViolate($productName, $keyword);
                if ($isViolate) {
                    $check = true;
                    break;
                }
            }
            if ($check) {
                $violateProducts[] = $product->id;
                continue;
            }
            $approveAdvertisingData[] = [
                'title' => $product->name,
                'description' => $product->description,
                'brand' => $brand,
                'code' => $product->sku,
                'product_id' => $product->id,
                'advertising_status' => 1,
                'created_at' => new \DateTime(),
                'updated_at' => new \DateTime()
            ];
        }

        ApproveAdvertising::insert($approveAdvertisingData);
        if (!empty($violateProducts)) {
            Product::whereIn('id', $violateProducts)
                ->update([
                    'is_violation' => 1,
                    'approve_advertising' => 0
                ]);
        }
        if ($updateProduct) {
            $diff = array_diff($productIds, $violateProducts);
            Product::whereIn('id', $diff)
                ->where('approve_advertising', 0)
                ->update([
                    'approve_advertising' => 1
                ]);
        }
    }

    public function savePerformanceGoogleAds(Request $request){
        set_time_limit(60*60);
        $locale = env('APP_LOCALE');
        $countryCode = 'US';
        if (!empty($locale)){
            $countryCode = strtoupper($locale);
        }
        $retVal = ['status' => 'fail'];
        $updatedSkus = [];
        $input = $request->all();
        $toDate = date('Y-m-d');
        $fromDate = date('Y-m-d',strtotime('-1 day'));
        if (array_key_exists('toDate', $input)){
            $toDate = $input['toDate'];
        }
        if (array_key_exists('countryCode', $input)){
            $countryCode = $input['countryCode'];
        }
        if (array_key_exists('fromDate', $input)){
            $fromDate = $input['fromDate'];
        }
        if (array_key_exists('customerId', $input)){
            $url = \Config::get('ads::ads.google_ads_api_url').'/shopping-performance';
            $requestData = [
                'customerId' => $input['customerId'],
                'columns' => [
                    "segments.date",
                    "segments.product_item_id",
                    "metrics.all_conversions_value",
                    "campaign.name",
                    "campaign.shopping_setting.sales_country",
                    "metrics.clicks",
                    "metrics.cost_micros",
                    "campaign.id"
                ],
                'wheres' => [
                    ['column' => 'metrics.clicks', 'operator' => '>', 'value' => '0', 'boolean' => 'and'],
                    ['column' => 'campaign.shopping_setting.sales_country', 'operator' => '=', 'value' => '"'.$countryCode.'"', 'boolean' => 'and'],
                ],
                "fromDate" => $fromDate,
                "toDate" => $toDate
            ];
            $url = $url .'?'. http_build_query($requestData);
            $response = $this->triggerSyncRequest($url,'GET',$requestData,['token: ' . $this->googleAdsApiToken,'Content-Type: application/json']);
            $result = isset($response['result'])?$response['result']:[];
            if (empty($result)){
                $retVal['result'] = $response;
            }else{
                foreach ($result as $item){
                    if (empty($item['segments.product_item_id'])){
                        continue;
                    }
                    $adPerformanceView = AdPerformanceView::where('campaign_id',$item['campaign.id'])
                        ->where('product_sku',$item['segments.product_item_id'])
                        ->where('date',$item['segments.date'])
                        ->first();
                    if (empty($adPerformanceView)) {
                        $adPerformanceView = new AdPerformanceView();
                        $adPerformanceView->date = $item['segments.date'];
                        $adPerformanceView->product_sku = $item['segments.product_item_id'];
                        $adPerformanceView->campaign_id = strval($item['campaign.id']);
                        $adPerformanceView->sales_country = $item['campaign.shopping_setting.sales_country'];
                        $adPerformanceView->campaign_name = $item['campaign.name'];
                        $adPerformanceView->created_at = date('Y-m-d H:i:s');
                    }
                    $adPerformanceView->updated_at = date('Y-m-d H:i:s');
                    $adPerformanceView->all_conversions_value = $item['metrics.all_conversions_value'];
                    $adPerformanceView->clicks = $item['metrics.clicks'];
                    $adPerformanceView->cost = $item['metrics.cost_micros']/1000000;
                    $adPerformanceView->save();
                    $updatedSkus[] = $adPerformanceView->product_sku;
                    }
                }
                $retVal['result'] = $updatedSkus;
                $retVal['count'] = count($updatedSkus);
                $retVal['status'] = 'successful';
        }else{
            $retVal['message'] = 'customerId required';
        }
        return response()->json($retVal);
    }

    public function cronBuildProductDisapprove()
    {
        set_time_limit(60*30);
        DB::table('product_disapprove_advertising')
            ->delete();
        $end_date = date("Y-m-d");
        $start_date = date('Y-m-d', strtotime("-3 months"));
        $products = DB::table('ad_performance_view')
            ->where('date', '>=', $start_date)
            ->where('date', '<=', $end_date)
            ->select('product_sku', DB::raw('SUM(clicks) AS clicks'), DB::raw('SUM(all_conversions_value) / SUM(cost) AS roas'))
            ->groupBy('product_sku')
            ->having('clicks', '>=', 100)
            ->get();
        $dataInsert = [];
        $listProductSku = [];
        foreach ($products as $product) {
            $listProductSku[] = strtoupper($product->product_sku);
        }
        $productSkus = DB::table('product_sku')
                        ->whereIn('sku', $listProductSku)
                        ->pluck('product_id', 'sku');
        foreach ($products as $product) {
            $productId = null;
            if (isset($productSkus[strtoupper($product->product_sku)])) {
                $productId = $productSkus[strtoupper($product->product_sku)];
            } else if (str_contains($product->product_sku, 'p')) {
                $productId = $this->parseProductSkuToProductId($product->product_sku);
            }
            if ($productId && !key_exists($productId, $dataInsert) && $product->roas < 1) {
                $dataInsert[$productId] = [
                    'product_id' => $productId,
                    'product_sku' => strtoupper($product->product_sku),
                    'roas' => $product->roas,
                    'clicks' => $product->clicks,
                    'created_at' => new \Datetime(),
                    'updated_at' => new \Datetime()
                ];
            }
        }
        DB::table('product_disapprove_advertising')->insert($dataInsert);
        return [
            'result' => count($dataInsert),
        ];
    }

    private function parseProductSkuToProductId($productSku)
    {
        preg_match_all('/\d+/', $productSku, $matches);
        if(isset($matches[0])) {
            return $matches[0][0];
        } else {
            return null;
        }
    }

    public function getPerformaneReport(Request $request){
        $retVal = ['status' => 'fail'];
        $updatedSkus = [];
        $customerId = $request->get('customerId');
        $campaignId = $request->get('campaignId');
        if (!empty($customerId) && !empty($campaignId)){
            $url = \Config::get('ads::ads.google_ads_api_url').'/shopping-performance';
            $requestData = [
                'customerId' => $customerId,
                'campaignId' => $campaignId
            ];
            $url = $url .'?'. http_build_query($requestData);
            $response = $this->triggerSyncRequest($url,'GET',[],['token: ' . $this->googleAdsApiToken,'Content-Type: application/json']);
            $result = isset($response['result'])?$response['result']:[];
            if (empty($result)){
                $retVal['result'] = $response;
            }else{
                ProductApproveAdvertising::where('ctr','>',0)->update(['ctr' => 0]);
                foreach ($result as $sku => $ctr){
                    $pa = ProductApproveAdvertising::where('code',$sku)->first();
                    if (!empty($pa)){
                        $pa->ctr = round($ctr,2);
                        $pa->save();
                        $updatedSkus[] = $sku;
                    }
                }
                $retVal['result'] = $updatedSkus;
                $retVal['count'] = count($updatedSkus);
                $retVal['status'] = 'successful';
            }
        }else{
            $retVal['message'] = 'customerId, campaignId required';
        }
        return response()->json($retVal);
    }

    public function saveProductWarning(Request $request){
        set_time_limit(60*60);
        $retVal = ['status' => 'fail'];
        $input = $request->all();
        $customerId = $request->get('customerId');
        $campaignId = $request->get('campaignId');
        if (!empty($customerId) && !empty($campaignId)){
            $url = \Config::get('ads::ads.google_ads_api_url').'/shopping-performance';
            $requestData = [
                'customerId' => $input['customerId'],
                'campaignId' => $campaignId,
                'columns' => [
                    "metrics.clicks",
                    "metrics.ctr",
                    "metrics.impressions",
                    "metrics.conversions_from_interactions_rate",
                    "segments.product_item_id",
                    "campaign.id"
                ],
                'wheres' => [
                    ['column' => 'metrics.impressions', 'operator' => '>', 'value' => '2000', 'boolean' => 'and'],
                    ['column' => 'metrics.conversions_from_interactions_rate', 'operator' => '<', 'value' => 0.01, 'boolean' => 'and'],
                    ['column' => 'campaign.id', 'operator' => '=', 'value' => $campaignId, 'boolean' => 'and'],
                ],
                "duringDate" => 'LAST_7_DAYS',
            ];
            $url = $url .'?'. http_build_query($requestData);
            $response = $this->triggerSyncRequest($url,'GET',$requestData,['token: ' . $this->googleAdsApiToken,'Content-Type: application/json']);
            $result = isset($response['result'])?$response['result']:[];
            $data = [];
            if (empty($result)){
                $retVal['result'] = $response;
            }else{
                foreach ($result as $item){
                    if (empty($item['segments.product_item_id'])){
                        continue;
                    }
                    $productId = $this->getProductIdFromSku($item['segments.product_item_id']);
                    $product = [
                        'product_id' => $productId,
                        'sku' => $item['segments.product_item_id'],
                        'impression' => $item['metrics.impressions'],
                        'click' => $item['metrics.clicks'],
                        'cr' => $item['metrics.conversions_from_interactions_rate'],
                        'ctr' => $item['metrics.ctr'],
                        'duration' => 'LAST_7_DAYS',
                        'created_at' => date('Y-m-d H:i:s'),
                        'updated_at' => date('Y-m-d H:i:s')
                    ];
                    $data[] = $product;
                }
                \DB::table('ads_product_performance')->insert($data);

            }
            $retVal['count'] = count($data);
            $retVal['status'] = 'successful';
        }else{
            $retVal['message'] = 'customerId required';
        }
        return response()->json($retVal);
    }

    public function saveTiktokCost(Request $request){
        set_time_limit(180);
        $retVal = [];
        $accountIds =  \Config::get('ads::ads.tiktok_account_ids');
        $url = \Config::get('ads::ads.google_ads_api_url') . '/tiktok/campaign/cost';
        $urlStoreCost = \Config::get('ads::ads.bingadsDomain','https://bingads.agoz.me') . '/tiktok/save-cost';
        $params = [
                'advertiser_id' => '7298181815888805890',
                'dimensions' => '["campaign_id","stat_time_hour"]',
                'metrics' => '["clicks","spend","campaign_name"]',
                'report_type' => 'BASIC',
                'data_level' => 'AUCTION_CAMPAIGN',
                'page_size' => '1000',
                'page' => '1'
        ];
        $dayAgo = 7;
        if ($request->has('dayAgo')){
            $dayAgo = $request->input('dayAgo');
        }
        $input = [
            'accountIds' => $accountIds,
            'url' => $url,
            'urlStoreCost' => $urlStoreCost,
        ];
        for ($i = -1; $i<= $dayAgo; $i++){
            if ($i==-1){
                $day = '+1 day';
            }else{
                $day = '-' . $i. ' day';
            }
            $params['start_date'] = date('Y-m-d',strtotime($day));
            $params['end_date'] = date('Y-m-d',strtotime($day));
            $input['params'] = $params;
            $retVal[] = $this->sendTiktokAdsCost($input);
        }
        return response()->json($retVal);
    }

    private function sendTiktokAdsCost($input){
        $retVal = [];
        $url = $input['url'];
        $params = $input['params'];
        $accountIds = $input['accountIds'];
        $urlStoreCost = $input['urlStoreCost'];
        foreach ($accountIds as $accountId) {
            $params['advertiser_id'] = $accountId;
            $result = $this->triggerSyncRequest($url, 'GET', $params, ['token: ' . $this->googleAdsApiToken, 'Content-Type: application/json']);
            $dataRequest = [
                'account_key' => $accountId,
                'response' => $result,
            ];
            $retVal[$accountId][1] = $this->triggerSyncRequest($urlStoreCost, 'POST', $dataRequest, ['Content-Type: application/json']);
            if (isset($result['result']['data']['page_info']['total_page'])){
                $totalPage = $result['result']['data']['page_info']['total_page'];
                for ($i = 2; $i <= $totalPage; $i++){
                    $params['page'] = $i;
                    $result = $this->triggerSyncRequest($url, 'GET', $params, ['token: ' . $this->googleAdsApiToken, 'Content-Type: application/json']);
                    $dataRequest['response'] = $result;
                    $retVal[$accountId][$i] = $this->triggerSyncRequest($urlStoreCost, 'POST', $dataRequest, ['Content-Type: application/json']);
                }
            }
        }
        return $retVal;
    }

    public function saveKeywordToSeller(Request $request)
    {
        // Extend execution time and initialize variables
        set_time_limit(3600);
        $retVal = [];
        $accountIds = \Config::get('ads::ads.account_ids');
        $url = \Config::get('ads::ads.google_ads_api_url') . '/keyword/search-term';
        $postbackUrl = \Config::get('ads::ads.seller_domain', 'https://q.seller.printerval.com') . '/api/keyword/save';
        $sellerApiAuthCode = \Config::get('ads::ads.seller_api_auth_code', 'jlkjajjjsdsasc');
        $dayAgo = $request->input('dayAgo', 1); // Default to 1 if not provided
        $fromRange = $request->input('fromRange', -1); // Default to 1 if not provided

        // Loop through the specified days
        foreach (range($fromRange, $dayAgo) as $i) {
            $dateRange = $this->getDateRange($i);
            $params = $this->buildParams($postbackUrl, $sellerApiAuthCode, $dateRange);

            foreach ($accountIds as $accountId) {
                $params['customerId'] = $accountId;

                // Trigger the sync request and store the response
                $retVal[$accountId][$dateRange['date_from']] = $this->triggerSyncRequest(
                    $url,
                    'GET',
                    $params,
                    ['token: ' . $this->googleAdsApiToken, 'Content-Type: application/json'],
                    10000
                );
                sleep(2);
            }
        }

        return response()->json($retVal);
    }

    private function getDateRange($offset)
    {
        if ($offset == -1) {
            $day = '+1 day';
        } else {
            $day = '-' . $offset . ' day';
        }

        $date = date('Y-m-d', strtotime($day));

        return [
            'date_from' => $date,
            'date_to' => $date,
        ];
    }

    private function buildParams($postbackUrl, $authCode, $dateRange)
    {
        return [
            'date_from' => $dateRange['date_from'],
            'date_to' => $dateRange['date_to'],
            'postbackUrl' => $postbackUrl,
            'postbackParam' => ['auth_key' => $authCode],
        ];
    }

    public function buildSellerOffsiteAds(){
        $serviceUrl = config('ads::offsiteAdsService','https://seller.printerval.com/api/get-all-seller-offsite-ads');
        $response = $this->triggerSyncRequest($serviceUrl,'GET');
        if(!empty($response) && is_array($response) && !empty($response['result'])){
            $sellerData = $response['result'];
            $tokens = [];
            foreach ($sellerData as $seller){
                $tokens[] = $seller['seller_token'];
            }
            if (!empty($tokens)){
                $tokenChunk = array_chunk($tokens,2000);
                foreach ($tokenChunk as $tokenItems){
                    $users = DB::table('users')->whereIn('seller_token',$tokenItems)->get(['id']);
                    if (!empty($users)){
                        foreach ($users as $user){
                            $sellerIds[$user->id] = $user->id;
                        }
                    }
                }
                if (!empty($sellerIds)){
                    $this->merchantProductService->saveOptionByKey('offsiteAdsSellerIds',json_encode($sellerIds));
                }
            }
        }
        return response()->json($sellerIds);
    }

    public function buildLabelVinCustom(){
        set_time_limit(300);
        $lastProductIdVinCustom = $this->merchantProductService->getOptionByKey('lastProductIdVinCustom',1719636317);
        $products = DB::table('product_crawl_code')->where('product_id','>',$lastProductIdVinCustom)->where('site','vincustom')->orderBy('product_id')->get(['product_id']);
        if (!empty($products)){
            $productIds = [];
            foreach ($products as $product){
                $productIds[] = $product->product_id;
            }
            if (!empty($productIds)){
                $lastProductIdVinCustom = end($productIds);
                $productIdsChunk = array_chunk($productIds,1000);
                foreach ($productIdsChunk as $ids){
                    DB::table('product')->whereIn('id',$ids)->update(['barcode' => 'vincustom']);
                }
            }else{
                $lastProductIdVinCustom = DB::table('product_crawl_code')->max('product_id');
            }
            $this->merchantProductService->saveOptionByKey('lastProductIdVinCustom',$lastProductIdVinCustom);

        }
        return response()->json([
            'lastProductIdVinCustom'=>$lastProductIdVinCustom,
            'status'=>'successful',
            ]);
    }

    public function saveFacebookCost(Request $request)
    {
        set_time_limit(180);
        $retVal = [];
        $urlGetFacebookInfo = \Config::get('ads::ads.facebook_ads_api_url') . '/api/facebook/business/list';
        $accountFacebookIds = $this->triggerSyncRequest($urlGetFacebookInfo, 'GET', [], ['Authorization: ' . $this->facebookAdsApiToken, 'Content-Type: application/json']);
        foreach ($accountFacebookIds as $accountFacebookId) {
            $adsAccounts = $this->getAddAccountFacebook($accountFacebookId);
            $url = \Config::get('ads::ads.facebook_ads_api_url') . '/api/facebook/campaign/cost';
            $urlStoreCost = \Config::get('ads::ads.bingadsDomain', 'https://bingads.agoz.me') . '/facebook/save-cost';
            $params = [
                "breakdowns" => "hourly_stats_aggregated_by_advertiser_time_zone",
                "limit" => "200",
                "facebook_id" => $accountFacebookId
            ];
            $dayAgo = 7;
            if ($request->has('dayAgo')) {
                $dayAgo = $request->input('dayAgo');
            }
            $input = [
                'account' => $adsAccounts,
                'url' => $url,
                'urlStoreCost' => $urlStoreCost,
            ];

            $params['start_date'] = Carbon::today()->subDays($dayAgo)->format('Y-m-d');
            $params['end_date'] = Carbon::today()->format('Y-m-d');
            $input['params'] = $params;
            $retVal[] = $this->sendFacebookAdsCost($input);
        }
        return response()->json($retVal);
    }

    private function getAddAccountFacebook($facebookId)
    {
        $params = [
            'facebook_id' => $facebookId,
        ];
        $url = \Config::get('ads::ads.facebook_ads_api_url') . '/api/facebook/campaign/get-list-ads-account';
        $result = $this->triggerSyncRequest($url, 'GET', $params, ['Authorization: ' . $this->facebookAdsApiToken, 'Content-Type: application/json']);
        return $result['data'];
    }

    private function sendFacebookAdsCost($input)
    {
        $retVal = [];
        $url = $input['url'];
        $params = $input['params'];
        $adsAccounts = $input['account'];
        $urlStoreCost = $input['urlStoreCost'];
        foreach ($adsAccounts as $adsAccount) {
            $params['ads_account_id'] = $adsAccount['account_id'];
            $result = $this->triggerSyncRequest($url, 'GET', $params, ['Authorization: ' . $this->facebookAdsApiToken, 'Content-Type: application/json']);
            $result['account'] = $adsAccount;
            $result['start_date'] = $params['start_date'];
            $result['end_date'] = $params['end_date'];
            $retVal[$adsAccount['account_id']] = $this->triggerSyncRequest($urlStoreCost, 'POST', $result, ['Content-Type: application/json']);
        }
        return $retVal;
    }
}
