<?php

namespace Modules\Ads\Controllers\Cron;

use DB;
use Modules\Ads\Controllers\Controller;
use Illuminate\Http\Request;

class AutoApproveProductHasOrder extends Controller
{
    public function autoApprove(Request $request) {
        set_time_limit(3600);
        $query = DB::table('order')
            ->where('payment_status', 'PAID')
            ->where('status', '!=', 'CANCELLED');
        $fromId = $this->getFromId($request, $query);
        $toId = $this->getToId($request, $query);
        if ($fromId == 0 || $toId == 0) {
            return [
                'status' => 'fail',
                'message' => 'from_id and to_id is required'
            ];
        }
        $step = 100;
        $designIds = [];
        while($fromId < $toId) {
            $stepToId = $fromId < $toId ? $fromId + $step : $toId;

            $orderIds = $query->where('id', '>=', $fromId)
                ->where('id', '<=', $stepToId)
                ->get(['id'])
                ->pluck('id')
                ->toArray();
            $designIds = DB::table('order_item')
                ->join('product_n_design', 'product_n_design.product_id', '=', 'order_item.product_id')
                ->join('product', 'product.id', '=', 'order_item.product_id')
                ->leftJoin('merchant_product', 'merchant_product.product_id', '=', 'order_item.product_id')
                ->whereNull('merchant_product.product_id')
                ->whereIn('order_item.order_id', $orderIds)
                ->where('product.is_violation', 0)
                ->where('product.is_trademark', 0)
                ->where('product.status', 'ACTIVE')
                ->pluck('product_n_design.design_id')
                ->toArray();
            $designIds = array_unique($designIds);
            $fromId += $step + 1;
        }
        if (count($designIds)) {
            foreach (array_chunk($designIds, 5000) as $chunk) {
                $this->sendCheckDesignByAiBatch($chunk);
            }
        }

        return [
            'status' => 'successful',
            'result' => $designIds
        ];
    }

    public function sendCheckDesignByAiBatch($designIds) {
        $locale = env('APP_LOCALE', config('localization::module.default_locale', 'us'));
        if (!$locale) {
            $locale = 'us';
        }
        $checkTrademarkUrl = config('trademarks::sa.check_trademark_url', 'https://trademark-checker.printerval.com');
        if ($checkTrademarkUrl) {
            $testTrademarkUrl = $checkTrademarkUrl  . '/check-design-by-ai-batch';
            $this->triggerSyncRequest($testTrademarkUrl, 'POST', [
                'designIds' => $designIds, 
                'locale' => $locale,
                'origin' => url('/')
            ], ['Content-Type: application/json']);
        }
    }

    public function getFromId($request, $query) {
        if ($request->has('from_id')) {
            return $request->get('from_id');
        }
        $cloneQuery = clone $query;
        $from = $cloneQuery
            ->where('created_at', '>=', date('Y-m-d 00:00:00', time() - $request->get('days', 3) * 86400))
            ->where('created_at', '<=', date('Y-m-d 23:59:59', time() - $request->get('days', 3) * 86400))
            ->orderBy('id', 'asc')->first();
        if ($from) {
            return $from->id;
        }

        return 0;
    }

    public function getToId($request, $query) {
        if ($request->has('to_id')) {
            return $request->get('to_id');
        }
        $cloneQuery = clone $query;
        $to = $cloneQuery
            ->where('created_at', '>=', date('Y-m-d 00:00:00', time() - $request->get('days', 3) * 86400))
            ->where('created_at', '<=', date('Y-m-d 23:59:59', time() - $request->get('days', 3) * 86400))
            ->orderBy('id', 'desc')->first();
        if ($to) {
            return $to->id;
        }

        return 0;
    }
}