<?php

namespace Modules\Ads\Controllers\Cron;

use DB;
use Illuminate\Http\Request;
use Modules\Ads\Controllers\ProductReactService;
use Modules\Ads\Models\Click;
use Modules\Ads\Models\Order;
use Modules\Ads\Models\ProductReactBySource;

class ProductReactBySourceController extends ProductReactService
{
    public function buildProductReactBySource(Request $request)
    {
        set_time_limit(3 * 3600);
        $retVal = [
            'status' => 'successful',
            'result' => []
        ];

        $filter = $this->buildCronFilter($request);
        $this->handleInteractData($filter);

        return $retVal;
    }

    public function handleInteractData($filter)
    {
        $source = isset($filter['source']) ? $filter['source'] : 'all';
        $items = $this->getOrder($filter);
        if ($items && count($items) > 0) {
            $this->clicks = $this->buildClick($filter);
            foreach ($items as $item) {
                if (!$item || !$item['product_id']) {
                    continue;
                }
                $key = $this->buildUniqProductKey($item->product_id, $item->product_sku_id, $item->create_date);
                if (isset($this->clicks[$key])) {
                    $this->clicks[$key]['conversion'] = $item->conversion;
                }
            }
            foreach ($this->clicks as $click) {
                $click['create_date'] = $click['date'];
                $this->handleProductReact($click, $source);
            }

        }
    }

    protected function buildCronFilter($request)
    {
        $from = $request->input('time_from', date('d/m/Y',strtotime("-3 days")));
        $to = $request->input('time_to', date('d/m/Y',strtotime("-1 days")));
        $source = $request->input('source', 'all');
        $fromDate = $this->getDateTimeFilter($from);
        $toDate = $this->getDateTimeFilter($to);
        $toDate->setTime(23, 59, 59);
        $filter = [
            'time_from' => $fromDate->format('Y-m-d H:i:s'),
            'time_to' => $toDate->format('Y-m-d H:i:s'),
            'date_gte' => $fromDate->format('Y-m-d'),
            'date_lte' => $toDate->format('Y-m-d'),
            'source' => $source,
        ];

        return $filter;
    }

    protected function getOrder($filter)
    {
        $reportSource = array_key_exists('source', $filter) ? $filter['source'] : '';
        $columns = [
            'order_item.product_id',
            'order_item.product_sku_id',
            DB::raw("DATE_FORMAT(sb_order.created_at, '%Y-%m-%d') as create_date, count(*) as conversion")
        ];
        $queryFilter = [
            'type' => Order::STATUS_OUT,
            'create_time_gte' => $filter['time_from'],
            'create_time_lte' => $filter['time_to'],
            'join_order_item' => 1,
            'group_by' => ['order_item.product_id',  'order_item.product_sku_id']
        ];
        if ($reportSource && $reportSource != 'all') {
            $queryFilter['join_order_meta_source'] = $reportSource;
        }
        $query = Order::buildOrderQuery($queryFilter);
       
        $items = $query->get($columns);
        return $items;
    }

    protected function buildClick($filter) {
        $result = [];
        $clicks = $this->getClick($filter);
        if ($clicks && count($clicks)) {
            foreach ($clicks as $click) {
                $key = $this->buildUniqProductKey($click->product_id, $click->product_sku_id, $click->date);
                $result[$key] = $click;
            }
        }
        return $result;
    }

    protected function buildUniqProductKey($pId, $pSkuId, $date) {
        $key = $pId . '-' . $pSkuId . '-' . $date;

        return $key;
    }

    protected function getClick($filter)
    {
        $query = Click::buildClickQuery($filter);
        $source = isset($filter['source']) ? $filter['source'] : 'all';
        $columns = [
            'product_id', 'product_sku_id', 'url', 'campaign_id', 'campaign_type', 'date',
            DB::raw('SUM(clicks) as clicks'),
            DB::raw("GROUP_CONCAT(source SEPARATOR '||') as sources"),
            DB::raw("GROUP_CONCAT(clicks SEPARATOR '||') as listClick"),
        ];
        $groupBy = [
            'product_id', 'product_sku_id', 'date'
        ];
        if ($source == 'all') {

        } else if ($source == 'adwords') {
            $query->where('source', 'adwords');
        } else if ($source == 'bing ads') {
            $query->where('source', '=', 'bing ads');
        }
        $query->groupBy($groupBy);
        return $query->get($columns);
    }

    protected function buildProductReact($item, $source)
    {
        $click = 0;
        $key = $this->buildUniqProductKey($item->product_id, $item->product_sku_id, $item->create_date);
        if (isset($this->clicks[$key])) {
            $click = $this->clicks[$key]->clicks;
        }
        $data = [
            'product_id' => $item['product_id'],
            'product_sku_id' => $item['product_sku_id'],
            'date' => $item['create_date'],
            'source' => $source
        ];
        if ($click) {
            $data['click'] = $click;
        }
        if (isset($item['conversion'])) {
            $data['conversion'] = $item['conversion'];
        }
        return $data;
    }

    protected function handleProductReact($item, $source)
    {
        $data = $this->buildProductReact($item, $source);
        if (isset($data['click'])) {
            ProductReactBySource::updateOrCreate($data);
        }
    }
}