<?php

namespace Modules\Ads\Controllers;
//use App\Modules\Ads\Models\PlatformFileAds;
use App\Models\Option;
use App\Modules\Ads\Models\PlatformFileAds;
use Carbon\Carbon;
use Google\Cloud\Storage\StorageClient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Redis;
use Modules\Ads\Controllers\Service\DataExportService;
use Modules\Ads\Controllers\Service\FacebookService;

class FacebookController extends Controller
{
    const MAX_SIZE_MERGE = 3.8; //GB
    protected $facebookService;
    protected $dataExportService;
    protected $platformFileAds;
    protected $option;
    protected $locale;

    public function __construct(FacebookService $facebookService, DataExportService $dataExportService,
                                PlatformFileAds $platformFileAds, Option $option
    )
    {
        $this->facebookService = $facebookService;
        $this->dataExportService = $dataExportService;
        $this->locale = !empty(env('APP_LOCALE')) ? env('APP_LOCALE') : 'us';
        $this->platformFileAds = $platformFileAds;
        $this->option = $option;

    }

    public function cronBuildProductAds(Request $request)
    {
        set_time_limit(600);
        $platform = $request->input('platform', 'facebook');
        $orderTag = $request->input('order_tag', 'all_products');
        $filePath = $request->input('file_path', '/feeds/' . $this->locale . '/' . $platform . '/' . $orderTag);
        $overwrite = $request->input('overwrite', null);
        $keyOptionLastCron = 'last_cron_add_all_products_csv_' . $platform . "_" . $orderTag;
        if (!empty($overwrite)) {
            $filePathDeleteDetail = public_path($filePath."/detail");
            if (is_dir($filePathDeleteDetail)) {
                $files = glob($filePathDeleteDetail . '/*');
                foreach ($files as $file) {
                    if (is_file($file)) {
                        @unlink($file);
                    }
                }
            }
            $this->option->updateOrInsert(['key' => $keyOptionLastCron], ['value' => 0]);
//            $this->resetDataSource();
            DB::table('platform_file_ads')
                ->where('platform', $platform)
                ->where('order_tag', $orderTag)
                ->delete();
            DB::table('platform_file_ads_merged')
                ->where('platform', $platform)
                ->where('order_tag', $orderTag)
                ->delete();
        }
        $lastProductAdsId = $this->option::where('key', $keyOptionLastCron)->value('value') ?? 0;
        $maxProductAdsId = DB::table('product_ads')->max('id');

        $batchSizeCron = 10000;
        $ranges = [];
        $maxProductAdsIdCeil = ceil($maxProductAdsId / $batchSizeCron) * $batchSizeCron;
        for ($start = $lastProductAdsId + 1; $start <= $maxProductAdsIdCeil; $start += $batchSizeCron) {
            $end = min($start + $batchSizeCron - 1, $maxProductAdsIdCeil);
            $ranges[] = [$start, $end];
        }
        $domain = $this->getDomain();
        $domainQueue = $this->getDomainQueue();
        foreach ($ranges as $range) {
            list($start, $end) = $range;
            $url = route('ads::facebook-feeds::build-product', [
                'product_id_from' => $start,
                'product_id_to' => $end,
                'file_path' => $filePath,
            ]);
            $urlDetail = str_replace($domain, $domainQueue, $url);
            $this->facebookService->sendRequest($urlDetail, 'POST', [], $request->all());
        }
        $this->option::updateOrCreate(
            ['key' => $keyOptionLastCron],
            ['value' => $maxProductAdsIdCeil]
        );

        return response()->json([
            'status' => 'done',
            'lastProductAdsId' => $lastProductAdsId,
            'maxProductAdsId' => $maxProductAdsId
        ]);
    }

    public function cronUpdateProductAds(Request $request)
    {
        $platform = $request->input('platform', 'facebook');
        $orderTag = $request->input('order_tag', 'all_products');
        $keyOptionLastCron = 'last_cron_add_all_products_csv_' . $platform . "_" . $orderTag;

        $chunkSize = 10000;
        $startProductId = 1;
        $endProductId = $this->option->where(['key' => $keyOptionLastCron])->first()->value ?? 0;
        $processed = 0;
        $domain = $this->getDomain();
        $domainQueue = $this->getDomainQueue();
        for ($i = $startProductId; $i <= $endProductId; $i += $chunkSize) {
            $startId = $i;
            $endId = min($i + $chunkSize - 1, $endProductId);
            $fileName = $orderTag . "_{$startId}_{$endId}.csv.gz";
            $url = route('ads::facebook-feeds::update-product', [
                'product_id_from' => $startId,
                'product_id_to' => $endId,
                'file_name' => $fileName,
            ]);
            $urlDetail = str_replace($domain, $domainQueue, $url);
            try {
                $this->facebookService->sendRequest($urlDetail, 'POST', [], $request->all());
                \Log::info("Pushed file to queue: {$fileName} ({$startId} - {$endId})");
                $processed++;
            } catch (\Exception $e) {
                \Log::error("Failed pushing file {$fileName}: " . $e->getMessage());
            }
        }
        return response()->json([
            'status' => $processed > 0 ? 'done' : 'no_file',
            'files_processed' => $processed
        ]);
    }


    public function cronGroupFilesByIdRange(Request $request)
    {
        $platform = $request->input('platform', 'facebook');
        $orderTag = $request->input('order_tag', 'all_products');
        $chunkSize = 10000;
        $groupKey = 'current_group_' . $platform . '_' . $orderTag;
        $sizeKey = 'current_size_' . $platform . '_' . $orderTag;
        Redis::set($groupKey, 1);
        Redis::set($sizeKey, 0);

        $minId = DB::table('platform_file_ads')
            ->where('platform', $platform)
            ->where('order_tag', $orderTag)
            ->min('id');
        $maxId = DB::table('platform_file_ads')
            ->where('platform', $platform)
            ->where('order_tag', $orderTag)
            ->max('id');
        if ($minId === null || $maxId === null) {
            return response()->json(['status' => 'no_file']);
        }
        $domain = $this->getDomain();
        $domainQueue = $this->getDomainQueue();
        $batches = [];
        for ($start = $minId; $start <= $maxId; $start += $chunkSize) {
            $end = min($start + $chunkSize - 1, $maxId);
            $url = route('ads::facebook-feeds::group-files', [
                'file_id_from' => $start,
                'file_id_to' => $end,
            ]);
            $urlDetail = str_replace($domain, $domainQueue, $url);
            try {
                $this->facebookService->sendRequest($urlDetail, 'POST', [], $request->all());
                \Log::info("Pushed batch: {$start} - {$end}");
            } catch (\Exception $e) {
                \Log::error("Failed pushing batch {$start} - {$end}: " . $e->getMessage());
            }
            $batches[] = ['from' => $start, 'to' => $end];
        }
        return response()->json([
            'status' => 'done',
            'total_batches' => count($batches),
            'batches' => $batches
        ]);
    }


    public function cronMergeByStorage(Request $request)
    {
        $platform = $request->input('platform', 'facebook');
        $orderTag = $request->input('order_tag', 'all_products');
        $result = DB::table('platform_file_ads')
            ->where('platform', $platform)
            ->where('order_tag', $orderTag)
            ->groupBy('group_id')
            ->get();
        foreach ($result as $resultItem) {
            $groupId = $resultItem->group_id;
            if (!empty($groupId)) {
                try {
                    $domain = $this->getDomain();
                    $domainQueue = $this->getDomainQueue();
                    $url = route('ads::facebook-feeds::merge-product', [
                        'group_id' => $groupId,
                    ]);
                    $urlDetail = str_replace($domain, $domainQueue, $url);
                    $this->facebookService->sendRequest($urlDetail, 'POST', [], $request->all());
                    Log::info("Pushed file to queue merge: {$groupId}");
                } catch (\Exception $e) {
                    throw $e;
                }
            }
        }

    }

    public function buildProductAds(Request $request)
    {
        set_time_limit(3600);
        ini_set('memory_limit', '2048M');

        $startId = $request->input('product_id_from', 0);
        $endId = $request->input('product_id_to', 0);
        $platform = $request->input('platform', 'facebook');
        $orderTag = $request->input('order_tag', 'all_products');
        $filePathInput = $request->input('file_path', 'feeds/' . $this->locale . '/' . $platform . '/' . $orderTag);
        $dataSourceName = $orderTag . "_" . $startId . "_" . $endId;
        $filters = $request->all();
        $filters['status'] = 'ACTIVE';
        $productAds = $this->facebookService->getProductAds($filters);
        $totalProductAds = $productAds->count();
        $retVal = [
            'message' => 'No product found in the given range',
            'total_comments' => 0
        ];

        if ($totalProductAds > 0) {
            $dir = public_path($filePathInput . '/detail');
            $dir = str_replace('\\', '/', $dir);
            if (!is_dir($dir)) {
                mkdir($dir, 0777, true);
            } else {
                chmod($dir, 0777);
            }
            $fileName = $orderTag . "_{$startId}_{$endId}.csv";
            $filePath = $dir . '/' . $fileName;
            $filePath = str_replace('\\', '/', $filePath);
            $filePathGCS = $this->getDomain() . "/" . $this->locale . "/" . $platform . "/" . $orderTag . "/" . $fileName;
            $productConvert = [];
            foreach ($productAds as $product) {
                $productConvert[] = $this->facebookService->convertProduct($product);
            }
            $update = $this->facebookService->updateCsvProducts(
                $filePath,
                $productConvert,
                'mpn', false
            );
            $fileSizeBytes = filesize($filePath);
            $fileUrlGCS = uploadToGCS($filePath, $filePathGCS);

            $gzFilePath = $filePath . '.gz';
            $gz = gzopen($gzFilePath, 'wb9');
            $fp = fopen($filePath, 'rb');
            while (!feof($fp)) {
                gzwrite($gz, fread($fp, 1024 * 512));
            }
            fclose($fp);
            gzclose($gz);
            unlink($filePath);
            $gzFileSize = filesize($gzFilePath);
            $this->platformFileAds::create([
                'file_name' => basename($gzFilePath),
                'platform' => 'facebook',
                'order_tag' => $request->input('order_tag', 'all_products'),
                'link_gcs' => $fileUrlGCS['publicUrl'] ?? null,
                'start_id' => $startId,
                'end_id' => $endId,
                'file_size' => $fileSizeBytes,
                'group_id' => null,
            ]);


            $retVal = [
                'message' => 'Build comment CSV completed and gzipped locally',
                'total_product' => $totalProductAds,
                'file_url' => $gzFilePath,
                'file_url_gcs' => $fileUrlGCS ?? null,
                'file_size' => $fileSizeBytes,
                'gz_file_size' => $gzFileSize,
                'group_id' => null
            ];
        }
        return response()->json($retVal);
    }

    public function updateProductAds(Request $request)
    {
        $firstId = $request->input('product_id_from');
        $lastId = $request->input('product_id_to');
        $platform = $request->input('platform', 'facebook');
        $orderTag = $request->input('order_tag', 'all_products');
        $startOfYesterday = Carbon::now()->subHours(24)->startOfDay();
        $endOfYesterday = Carbon::now()->subHours(24)->endOfDay();
        $fileName = $request->input('file_name');
        $fileNameNoGZ = preg_replace('/\.gz$/', '', $fileName);
        $filePathInput = $request->input('file_path', 'feeds/' . $this->locale . '/' . $platform . '/' . $orderTag);
        $filePath = public_path($filePathInput . '/detail/' . $fileName);
        $filePathGCS = $request->input('file_path_gcs', $this->getDomain() . "/" . $this->locale . "/" . $platform . "/" . $orderTag . "/" . $fileNameNoGZ);
        if (empty($firstId) || empty($lastId) || empty($filePath)) {
            Log::error('product_id_from, product_id_to or file path missing for updateMultipleIds');
            $retVal = ['success' => false, 'message' => 'IDs or file path missing'];
        } else {
            $filter = $request->all();
            $filter['updated_at_from'] = $startOfYesterday->toDateTimeString();
            $filter['updated_at_to'] = $endOfYesterday->toDateTimeString();
            $productAds = $this->facebookService->getProductAds($filter);
            $retVal = ['success' => true, 'message' => 'No updated'];
            if ($productAds->isNotEmpty()) {
                $productConvert = [];
                $productConvertDelete = [];
                foreach ($productAds as $product) {
                    if ($product->status == 'PENDING') {
                        $productConvertDelete[] = $this->locale . $product->id;
                    }
                    $productConvert[] = $this->facebookService->convertProduct($product);
                }
                $updateFileProductCsv = $this->facebookService->updateCsvProducts(
                    $filePath,
                    $productConvert,
                    'mpn',
                    true,
                    $productConvertDelete
                );
                if (!empty($updateFileProductCsv['deleted'])) {
                    $data = array_map(function ($id) {
                        return [
                            'product_id' => $id,
                            'platform' => 'facebook',
                            'created_at' => Carbon::now()
                        ];
                    }, $updateFileProductCsv['deleted']);
                    DB::table('deleted_product_in_catalog')->insert($data);
                }
                $csvTempPath = str_replace('.gz', '', $filePath);
                $fpGz = gzopen($filePath, 'r');
                $fpCsv = fopen($csvTempPath, 'w');
                while (!gzeof($fpGz)) {
                    $line = gzgets($fpGz);
                    fwrite($fpCsv, $line);
                }
                gzclose($fpGz);
                fclose($fpCsv);
                $fileUrlGCS = uploadToGCS($csvTempPath, $filePathGCS);
                unlink($csvTempPath);
                $retVal = [
                    'success' => true,
                    'file_url_gcs' => $fileUrlGCS ?? null,
                    'message' => 'Updated successfully',
                    'detail' => $updateFileProductCsv
                ];
            }
        }
        return $retVal;
    }

    public function mergeDataSourceFiles(Request $request)
    {
        $groupId = $request->input('group_id');
        $platform = $request->input('platform', 'facebook');
        $orderTag = $request->input('order_tag', 'all_products');
        $storage = new StorageClient([
            'keyFilePath' => config_path('google-cloud.json'),
        ]);
        $bucket = $storage->bucket('printerval-central');

        $files = DB::table('platform_file_ads')
            ->where('platform', $platform)
            ->where('order_tag', $orderTag)
            ->where('group_id', $groupId)
            ->pluck('link_gcs')
            ->toArray();
        $objectPaths = array_map(function ($url) {
            return str_replace("https://storage.googleapis.com/printerval-central/", '', strtok($url, '?'));
        }, $files);
        array_unshift($objectPaths, $this->getDomain() . '/' . $platform . '/header.csv');
        $round = 1;
        $tempFiles = [];

        while (count($objectPaths) > 1) {
            $tempResults = [];
            foreach (array_chunk($objectPaths, 32) as $i => $chunk) {
                $target = $this->getDomain() . "/" . $this->locale . "/" . $platform . "/" . $orderTag . "/tmp/{$groupId}_round{$round}_{$i}.csv";
                $this->composeFiles($bucket, $chunk, $target);
                $tempResults[] = $target;
                $tempFiles[] = $target;
            }
            $objectPaths = $tempResults;
            $round++;
        }

        $finalFile = $this->getDomain() . "/" . $this->locale . "/" . $platform . "/" . $orderTag . "/merged/{$platform}_{$orderTag}_part_{$groupId}.csv";
        $this->composeFiles($bucket, $objectPaths, $finalFile);

        foreach ($tempFiles as $tempFile) {
            try {
                $bucket->object($tempFile)->delete();
            } catch (\Throwable $e) {
                \Log::warning("Không thể xóa file tạm: {$tempFile} - {$e->getMessage()}");
            }
        }
        $urlGCS = "https://storage.googleapis.com/printerval-central/{$finalFile}";
        DB::table('platform_file_ads_merged')->updateOrInsert(
            [
                'platform' => $platform,
                'order_tag' => $orderTag,
                'group_id' => $groupId
            ],
            [
                'platform' => $platform,
                'order_tag' => $orderTag,
                'group_id' => $groupId,
                'file_path_gcs' => $urlGCS,
                'data_source_id' => null,
            ]
        );

        return $urlGCS;
    }


    public function groupFilesOptimized(Request $request)
    {
        $maxSize = self::MAX_SIZE_MERGE * 1024 * 1024 * 1024;
        $platform = $request->input('platform', 'facebook');
        $orderTag = $request->input('order_tag', 'all_products');
        $fileIdFrom = $request->input('file_id_from', 0);
        $fileIdTo = $request->input('file_id_to', 0);

        $retVal = [
            'status' => 'no_action',
            'last_group' => null,
            'last_id' => null
        ];
        if (!empty($fileIdFrom) && !empty($fileIdTo)) {
            $groupKey = 'current_group_' . $platform . '_' . $orderTag;
            $sizeKey = 'current_size_' . $platform . '_' . $orderTag;

            $currentGroup = Redis::get($groupKey) ?? 1;
            $currentSize = Redis::get($sizeKey) ?? 0;

            $files = DB::table('platform_file_ads')
                ->where('platform', $platform)
                ->where('order_tag', $orderTag)
                ->where('id', '>=', $fileIdFrom)
                ->where('id', '<=', $fileIdTo)
                ->orderBy('id', 'asc')
                ->get(['id', 'file_size']);

            if ($files->isEmpty()) {
                $retVal = [
                    'status' => 'done',
                    'last_group' => $currentGroup,
                    'last_id' => $fileIdFrom
                ];
            } else {
                $groups = [];
                $lastId = $fileIdFrom;

                foreach ($files as $file) {
                    $size = (int)$file->file_size;
                    if ($currentSize + $size > $maxSize && $currentSize > 0) {
                        $currentGroup++;
                        $currentSize = 0;
                    }
                    $groups[$currentGroup][] = $file->id;
                    $currentSize += $size;
                    $lastId = $file->id;
                }

                foreach ($groups as $groupId => $ids) {
                    DB::table('platform_file_ads')
                        ->whereIn('id', $ids)
                        ->update(['group_id' => $groupId]);
                }
                Redis::set($groupKey, $currentGroup);
                Redis::set($sizeKey, $currentSize);
                $retVal = [
                    'status' => 'done',
                    'last_group' => $currentGroup,
                    'last_id' => $lastId
                ];
            }
        }
        return response()->json($retVal);
    }


    private function composeFiles($bucket, $sourceNames, $targetName)
    {
        if (count($sourceNames) === 1) {
            $source = $bucket->object($sourceNames[0]);
            $source->copy($bucket, ['name' => $targetName]);
            return;
        }

        // Nếu có từ 2 file trở lên => compose
        $objects = array_map(function ($n) use ($bucket) {
            return $bucket->object($n);
        }, $sourceNames);

        $bucket->compose($objects, $targetName);
    }

    private function getDomain(): string
    {
        if (class_exists(\Illuminate\Http\Request::class)) {
            try {
                $request = app(\Illuminate\Http\Request::class);
                if ($request->getHost()) {
                    return $request->getHost();
                }
            } catch (\Throwable $e) {
            }
        }

        if (!empty($_SERVER['HTTP_HOST'])) {
            return $_SERVER['HTTP_HOST'];
        }

        if (!empty($_SERVER['SERVER_NAME'])) {
            return $_SERVER['SERVER_NAME'];
        }

        if (!empty($_SERVER['SERVER_ADDR'])) {
            return $_SERVER['SERVER_ADDR'];
        }

        return 'localhost';
    }

    private function getDomainQueue()
    {
        $configDefault = config('ads::queue');
        $domain = $configDefault['DEFAULT'];
        $locale = '';
        if (env('LOCALIZATION') && env('APP_LOCALE') != '') {
            $locale = '/' . env('APP_LOCALE');
        }
        if (!empty($configDefault) && is_array($configDefault)) {
            foreach ($configDefault as $localeQueue => $domainQueue) {
                if ($locale == $localeQueue) {
                    $domain = $domainQueue;
                }
            }
        }
        return $domain;
    }

}