<?php

namespace Modules\Ads\Controllers;

use Google\Service\ShoppingContent\SearchRequest;
use Illuminate\Support\Facades\DB;
use Module;
use Illuminate\Http\Request;
use Modules\Ads\Controllers\Service\CustomerService;
use Modules\Ads\Controllers\Service\MerchantProduct;
use Modules\Ads\Controllers\Service\MerchantProductService;
use Modules\Ads\Controllers\Service\MerchantService;
use Modules\Ads\Models\Category;
use Modules\Ads\Models\Product;
use Modules\Ads\Builders\QueryBuilder;


class MerchantProductContentController extends Controller
{
    const CHANNEL = 'online';
    private $client;
    private $service;
    private $merchantProductService;
    private $merchantService;
    private $languages;
    private $currencyCode;
    private $languageCode;
    private $apparelAccessoriesGMC;
    private $merchantId;
    private $customerService;
    private $scoreTarget;
    private $queue;
    private $merchantServiceConfig;
    public function __construct(MerchantProductService $merchantProductService, MerchantService $merchantService, CustomerService $customerService)
    {
        $this->client = $this->getClient([]);
        $this->service = new \Google_Service_ShoppingContent($this->client);
        $this->merchantProductService = $merchantProductService;
        $this->merchantService = $merchantService;
        $this->languages = \Config::get('ads::clara-ads-general.languages',[]);
        $this->apparelAccessoriesGMC = \Config::get('ads::clara-ads-general.apparelAccessoriesGMC',[]);
        $this->currencyCode = env('CURRENCY_UNIT','USD');
        $this->languageCode = env('APP_LANG','en');
        $this->merchantId = getOption('merchant_id', '398409240');
        $this->customerService = $customerService;
        $this->scoreTarget = $this->merchantProductService->getScoreTarget();
        $this->queue = \Config::get('ads::ads.queue',[]);
        $this->merchantServiceConfig = \Config::get('ads::ads.merchant_service',[]);
    }
    public function cronUpdateProduct(Request $request){
//        ini_set("memory_limit", "3072M");
//        $this->insertByService(file_get_contents("/Users/lapdx/Documents/data.json"),[]);die;
        $input = $request->all();
        $timeLimit = 60*30;
        if (array_key_exists('time_limit',$input)){
            $timeLimit = $input['time_limit'];
        }
        set_time_limit($timeLimit);
        ini_set("memory_limit", "3072M");
        $filter = $this->getDefaultFilter();
        if(array_key_exists('product_id_from',$input)){
            $filter['product_id_from'] = $input['product_id_from'];
        }
        if(!array_key_exists('ignore_updated_at',$input)){
            $filter['updated_at_from'] = date("Y-m-d H:i:s",strtotime('-24 hours'));
        }
        if(array_key_exists('product_id_to',$input)){
            $filter['product_id_to'] = $input['product_id_to'];
        }
        if(array_key_exists('updated_at_from',$input)){
            $filter['updated_at_from'] = $input['updated_at_from'];
        }
        if(array_key_exists('updated_at_to',$input)){
            $filter['updated_at_to'] = $input['updated_at_to'];
        }
        if(array_key_exists('actor_ids',$input)){
            $filter['actor_ids'] = explode(",", $input['actor_ids']);
        }
        if(array_key_exists('sellerIds',$input)){
            $filter['sellerIds'] = explode(",", $input['sellerIds']);
        }
        if(array_key_exists('seller',$input)){
            $filter['seller'] = $input['seller'];
        }
        if(array_key_exists('ignoreSellerIds',$input)){
            $filter['ignoreSellerIds'] = explode(",", $input['ignore_SellerIds']);
        }
        if(array_key_exists('ignore_chunk',$input)){
            $filter['ignore_chunk'] = $input['ignore_chunk'];
        }
        if(array_key_exists('pInfo',$input)){
            $filter['pInfo'] = $input['pInfo'];
        }
        if(array_key_exists('update_expire',$input)){
            $exprieDate = date("Y-m-d H:i:s",strtotime('-25 days'));
            $expireFilter = $filter;
            $expireFilter['updated_at_to'] = $exprieDate;
            $ids = $this->merchantProductService->getProductExprieIds($expireFilter);
            if (empty($ids)){
                $ids = [0];
            }
            $filter['ids'] = $ids;
        }
        if(array_key_exists('sold_from',$input)){
            $filter['sold_from'] = $input['sold_from'];
        }
        if(array_key_exists('ids',$input)){
            $filter['ids'] = explode(',',$input['ids']);
        }
        if(array_key_exists('categoryIds',$input)){
            $filter['categoryIds'] = explode(',',$input['categoryIds']);
        }
        $retVal = $this->updateProduct($filter,$input);
        $count = $retVal['count'];
        $message = $retVal['message'];
        return response()->json(['status' => 'successful', 'result' => $count, 'message' => $message]);

    }

    private function updateProduct($filter,$input){
        $localeData = $this->getLocaleData();
        $prefixUrl = $localeData['prefixUrl'];
        $countryCode = $localeData['countryCode'];
        $query = $this->merchantProductService->buidFilter($filter);
        $countryOfSales = $this->merchantProductService->getCountryOfSales($countryCode);
        if (array_key_exists('countryOfSales', $input)){
            $countryOfSales = explode(',', $input['countryOfSales']);
        }
        $onlySold = $this->merchantProductService->getCountryOfSaleSold($countryCode);
        $count = 0;
        $inputBuildProduct = [
            'prefixUrl' => $prefixUrl,
            'country' => $countryCode,
            'language' => isset($this->languages[$countryCode]) ? $this->languages[$countryCode] : env('APP_LANG'),
            'onlySold' => $onlySold
        ];
        if (isset($input['merchant_id'])){
            $inputBuildProduct['merchant_id'] = $input['merchant_id'];
        }
        if (isset($input['shippingCountryCodes'])){
            $inputBuildProduct['shippingCountryCodes'] = explode(",",$input['shippingCountryCodes']);
        }
        $products = $query->get();
        $data = $this->merchantProductService->buildProduct($products,$inputBuildProduct);
        $count += count($data);
        if (array_key_exists('oldVersion', $input)){
            $message = $this->insert($data, $countryOfSales, $inputBuildProduct);
        }else{
            $message = $this->insertByService($data,['country' => $countryCode]);
        }
//        $message = $this->insertByService($data,['country' => $countryCode]);
        return ['message' => $message, 'count' =>$count];
    }

    private function getDefaultFilter(){
        $filter = [
            'status' => 'ACTIVE',
            'approve_advertising' => 1,
            'advertising_status' => 1,
            'is_trademark' => 0,
            'is_violation' => 0,
            'is_default' => 1,
        ];
        if ($this->scoreTarget > 0){
            $filter['score'] = 1;
        }
        $filter['columns'] = [
            // ORIGIN
            'product.id',
            'product.slug', 'product.brand_id', 'product.image_url',
            'product.description',
            'product.status', 'product.approve_advertising',
            'product.is_trademark',
            'product.gtin',
            'product.is_always_on_ads',
            'product.actor_id',
            'product.created_at',
            'product.sold',
            'product.barcode',
//                'product.price',
//                'product.high_price',
            // SKU
            'product_info.name as sku_name',
            'product_info.variants as sku_specific_info',
            'product_info.brand as sku_brand',
            'product_info.sku as sku_code',
            'product_info.slug as sku_slug',
            'product_info.price',
            'product_info.inventory',
            'product_info.high_price',
            'product_info.category_merchant_id as sku_category_merchant_id',
            'product_info.product_sku_id',
            'product_info.product_sku_id as sku_id',
            'product_info.is_default',
            'product_n_category.category_id as category_id',
            'product.name as title_replacement',
        ];
        return $filter;
    }

    public function cronRemoveProduct(Request $request)
    {
        set_time_limit(60 * 10);
        $prefixUrl = isset($_SERVER['PREFIX_URL']) ? $_SERVER['PREFIX_URL'] : '';
        $isLocalization = env('LOCALIZATION');
        $locale = env('APP_LOCALE');
        $countryCode = \Config::get('ads::clara-ads-general.countryCodeDefault','US');
        if ($prefixUrl == '' && $isLocalization && $locale != '') {
            $prefixUrl = '/' . $locale;
            $countryCode = $locale;
        }
        if ($countryCode == 'uk'){
            $countryCode = 'gb';
        }
        $countryCode = strtoupper($countryCode);
        $input = $request->all();
        $filter = $input;
        if (array_key_exists('skip_ads',$input)){
            if(!array_key_exists('updated_at_from',$input)){
                if(!array_key_exists('ignore_updated_at',$input)){
                    $filter['updated_at_from'] = date("Y-m-d H:i:s",strtotime('-4 hours'));
                }
            }else{
                $filter['updated_at_from'] = $input['updated_at_from'];
            }
//        }elseif(array_key_exists('hard_delete',$input)){
        }else{
            if(array_key_exists('deleted_at_from',$input)){
                $filter['deleted_at_from'] = $input['deleted_at_from'];
            }else{
                $filter['deleted_at_from'] =  date("Y-m-d H:i:s",strtotime('-4 hours'));
            }
        }
        if(array_key_exists('actor_ids',$input)){
            $filter['actor_ids'] = explode(",", $input['actor_ids']);
        }
        if(array_key_exists('ids',$input)){
            $filter['ids'] = explode(",", $input['ids']);
        }
        $query = $this->merchantProductService->buildDeleteQuery($filter);
        $inputData = [
            'prefixUrl' => $prefixUrl,
            'country' => $countryCode,
            'language' => isset($this->languages[$countryCode]) ? $this->languages[$countryCode] : env('APP_LANG'),
        ];
        if (isset($input['merchant_id'])){
            $inputData['merchant_id'] = $input['merchant_id'];
        }
        $productDeletes = [];
        $products = $query->get();
        foreach ($products as $product){
            $productDeletes[] = ['productId' => $product->product_id, 'sku' => $product->sku, 'merchantProductId' => $product->id];
        }
        $data = $this->merchantProductService->buildProductDelete($productDeletes,$inputData);
        $this->deleteByService($data,['method' => 'delete', 'country' => $countryCode]);
        return response()->json(['status' => 'successful', 'result' => count($products)]);
    }

    public function insert($products, $countryOfSales, $input){
        $countryOnlySold = $input['onlySold'];
        try{
            $removeProducts = [];
            foreach ($countryOfSales as $k => $country) {
                $entries = [];
                $skus = [];
                foreach ($products as $index => $product) {
                    if (in_array($country,$countryOnlySold) && $product['sold'] < 1){
                        continue;
                    }
                    if ($k == 0){
                        $removeProducts = array_merge($removeProducts,$this->merchantProductService->saveMerchantProduct($product['id'], $product['product_code']));
                    }
                    $merchantId = $this->buildMerchantId($product, $input);
                    $batchId = $this->makeBatchId($product['id']);
                    $skus[$batchId] = ['id' => $product['id'], 'sku' => $product['product_code']];
                    $product['country'] = $country;
                    $key = $batchId;
                    $entry = new \Google_Service_ShoppingContent_ProductsCustomBatchRequestEntry();
                    $entry->setMethod('insert');
                    $entry->setBatchId($key);
                    $entry->setProduct($this->buildProduct($product));
                    $entry->setMerchantId($merchantId);
                    $entries[] = $entry;
                }
            $this->insertByChunk($entries,$skus,$country);
            }
            $dataDelete = $this->merchantProductService->buildProductDelete($removeProducts, $input);
            $this->delete($dataDelete, $countryOfSales, $input);
            return count($products);
        }catch (\Exception $ex){
            \Log::error("Google Merchant ".$ex->getMessage() . $ex->getTraceAsString());
        }
    }

    private function insertByChunk($allEntries,$skus,$countryCode, $hardChunk = 0){
        $removeIds = [];
        $chunkCount = 10000;
        $chunkCountConfig = config('ads::clara-ads-general.chunkCount');
        if (isset($chunkCountConfig[$countryCode])){
            $chunkCount = $chunkCountConfig[$countryCode];
        }
        $errorMsg = 'SaveMerchantProductError:';
        if (!empty($hardChunk)){
            $chunkCount = $hardChunk;
            $errorMsg = 'UpdatePriceMerchantProductError:';
        }
        $data = array_chunk($allEntries,$chunkCount);
        foreach ($data as $entries){
            $this->client = $this->getClient([]);
            $this->service = new \Google_Service_ShoppingContent($this->client);
            $batchRequest =
                new \Google_Service_ShoppingContent_ProductsCustomBatchRequest();
            $batchRequest->setEntries($entries);
            $batchResponse = $this->service->products->custombatch($batchRequest);
            $productIds = [0];
            foreach ($batchResponse->entries as $entry) {
                $batchId = $entry->getBatchId();
                $sku = $batchId;
                $productId = $batchId;
                $productIds[] = $productId;
                if (isset($skus[$batchId])){
                    $sku = $skus[$batchId]['sku'];
                    $productId = $skus[$batchId]['id'];
                }
                if (!empty($entry->getErrors())) {
                    foreach ($entry->getErrors()->getErrors() as $error) {
                        $message = $error->getMessage();
                        \Log::error("$errorMsg $sku " . $message);
                        if (strpos($message, 'limit exceeded') !== false){
                            $removeIds[] = $productId;
                        }
                    }
                }
            }
            $this->merchantProductService->updateDeleteMerchantProduct($productIds,0);
        }
        if (!empty($removeIds)){
            $this->merchantProductService->clearProductCreateFail($removeIds);
        }
    }
    public function delete($products, $countryOfSales, $input = []){
        die;
        try{
            foreach ($countryOfSales as $k => $country) {
                $entries = [];
                $skus = [];
                foreach ($products as $index => $item) {
                    $merchantId = $this->buildMerchantId($item, $input);
                    $batchId = $this->makeBatchId($item['merchantProductId']);
                    $skus[$batchId] = ['id' => $item['id'], 'sku' => $item['product_code']];
                    $key = $batchId;
                    $item['country'] = $country;
                    $entry = new \Google_Service_ShoppingContent_ProductsCustomBatchRequestEntry();
                    $entry->setMethod('delete');
                    $entry->setBatchId($key);
                    $entry->setProductId($this->buildProductId($item));
                    $entry->setMerchantId($merchantId);
                    $entries[] = $entry;
                }
                $this->client = $this->getClient([]);
                $this->service = new \Google_Service_ShoppingContent($this->client);
                $batchRequest = new \Google_Service_ShoppingContent_ProductsCustomBatchRequest();
                $batchRequest->setEntries($entries);
                $batchResponses = $this->service->products->custombatch($batchRequest);
                $deleteIds = [];
                $deletedData = [];
                foreach ($batchResponses->entries as $entry) {
                    $batchId = $entry->getBatchId();
                    $sku = $batchId;
                    $productId = $batchId;
                    $flag = false;
                    if (isset($skus[$batchId])){
                        $sku = $skus[$batchId]['sku'];
                        $productId = $skus[$batchId]['id'];
                        $deleteIds[] = $productId;
                        $deletedData[] = ['product_id' => $productId, 'sku' => $sku, 'created_at' => date('Y-m-d H:i:s')];
                        $flag = true;
                    }
                    if (!empty($entry->getErrors())) {
                        foreach ($entry->getErrors()->getErrors() as $error) {
                            $messageError = $error->getMessage();
                            \Log::info("DeleteMerchantProductError: $sku " . $messageError);
                            if ($flag && $messageError == 'item not found'){
                                $deleteIds[] = $productId;
                            }
                        }
                    }
                }
                if (!empty($deleteIds)){
                    $this->merchantProductService->removeMerchantProduct($deleteIds);
                }
                if (!empty($deletedData)){
                    $this->merchantProductService->createDeleteMerchantProduct($deletedData);
                }
            }
            return count($products);
        }catch (\Exception $ex){
            \Log::error($ex->getMessage() . $ex->getTraceAsString());
        }
    }
    public function cronRemoveTempProduct(Request $request)
    {
        return "This function inactive";
        $input = $request->all();
        $timeLimit = 60*30;
        if (array_key_exists('time_limit',$input)){
            $timeLimit = $input['time_limit'];
        }
        set_time_limit($timeLimit);
        $prefixUrl = isset($_SERVER['PREFIX_URL']) ? $_SERVER['PREFIX_URL'] : '';
        $isLocalization = env('LOCALIZATION');
        $locale = env('APP_LOCALE');
        $countryCode = \Config::get('ads::clara-ads-general.countryCodeDefault','US');
        if ($prefixUrl == '' && $isLocalization && $locale != '') {
            $prefixUrl = '/' . $locale;
            $countryCode = $locale;
        }
        if ($countryCode == 'uk'){
            $countryCode = 'gb';
        }
        $countryCode = $input['countryCode'];
        $countryCode = strtoupper($countryCode);
        $filter = [

        ];
        if(array_key_exists('product_id_from',$input)){
            $filter['product_id_from'] = $input['product_id_from'];
        }
        if(array_key_exists('product_id_to',$input)){
            $filter['product_id_to'] = $input['product_id_to'];
        }
        if(array_key_exists('ids',$input)){
            $filter['ids'] = explode(",",$input['product_id_to']);
        }
        $query = $this->merchantProductService->buildDeleteTempQuery($filter);
        $countryOfSales = [$countryCode];
        $chunkCount = 10000;
        $query->chunk($chunkCount, function($products) use($prefixUrl,$countryCode, $countryOfSales){
            $input = [
                'prefixUrl' => $prefixUrl,
                'country' => $countryCode,
                'ignoreDatabase' => 'true',
                'language' => isset($this->languages[$countryCode]) ? $this->languages[$countryCode] : env('APP_LANG'),
            ];
            $productDeletes = [];
            foreach ($products as $product){
                $productDeletes[] = ['productId' => $product->product_id, 'sku' => $product->sku, 'merchantProductId' => $product->id];
            }
            $data = $this->merchantProductService->buildProductDelete($productDeletes,$input);
            $this->delete($data, $countryOfSales,$input);
        });
        return response()->json(['status' => 'successful']);
    }
    public function fetchNow(){
        set_time_limit(600);
        try {
            $response = ["status" => "successful"];
            $parameters = ['maxResults' => 200];
            $merchantId = getOption('merchant_id','398409240');
            $names = [];
            do {
                $feeds = $this->service->datafeeds->listDatafeeds(
                    $merchantId, $parameters);
                foreach ($feeds->getResources() as $feed) {
                    $feedId = $feed->getId();
                    $names[] = $feed->getName();
                    $this->service->datafeeds->fetchnow($merchantId,$feedId);
                    sleep(6);
                }
                $parameters['pageToken'] = $feeds->nextPageToken;
            } while (!empty($parameters['pageToken']));
            $response['name'] = $names;

        } catch (Exception $exc) {
            $response = ["status" => "fail", "message" => $exc->getMessage() . $exc->getTraceAsString()];
        }
        return $response;
    }


    private function buildProduct($item){
        $product = new \Google_Service_ShoppingContent_Product();
        $product->setOfferId($item['product_code']);
        $product->setTitle($item['title']);
        $product->setDescription($item['description']);
        $product->setCondition($item['condition']);
        $product->setGoogleProductCategory($item['sku_category_merchant_id']);
        $product->setProductTypes($item['product_type']);
        $product->setLink($item['link']);
        $product->setImageLink($this->encodeImageUrl($item['image_url']));
        $product->setAdditionalImageLinks($item['gallery']);
        $product->setAvailability($item['availability']);
        $product->setExcludedDestinations($item['excluded_destination']);
        $price = new \Google_Service_ShoppingContent_Price();
        $price->setValue($item['price']);
        $price->setCurrency($item['currency']);
        $product->setSalePrice($price);
        $highPrice = new \Google_Service_ShoppingContent_Price();
        $highPrice->setValue($item['high_price']);
        $highPrice->setCurrency($item['currency']);
        $product->setPrice($highPrice);
        $product->setBrand($item['brand']);
        $product->setMpn($item['product_code']);
        if (isset($this->apparelAccessoriesGMC[$item['sku_category_merchant_id']])){
            $product->setColor($item['color']);
            $product->setGender($item['gender']);
            $product->setAgeGroup($item['age']);
            $product->setSizes($item['size']);
        }
        $product->setContentLanguage($item['language']);
        $product->setFeedLabel($item['country']);
        $product->setChannel(self::CHANNEL);
        $product->setCustomLabel0($item['barcode']);
        $product->setMaterial($item['material']);
//        if (!empty($item['custom_label_1'])){
        $product->setCustomLabel1($item['custom_label_1']);
//        }
        $product->setAvailabilityDate(date('Y-m-d',strtotime('+ 30 days')));
//        $product->setSalePriceEffectiveDate(date("Y-m-d", strtotime('-2 days')) . '/' .date("Y-m-d", strtotime('+30 days')));
        $shippingFee = [];
        foreach ($item['shippingFees'] as $shippingItem){
            $shippingPrice = new \Google_Service_ShoppingContent_Price();
            $shippingPrice->setValue($shippingItem['price']);
            $shippingPrice->setCurrency($shippingItem['currency']);
            $shipping = new \Google_Service_ShoppingContent_ProductShipping();
            $shipping->setCountry($shippingItem['country']);
            if (isset($shippingItem['shippingTime'])){
                $shipping->setMinHandlingTime($shippingItem['shippingTime']['min_handling_time']);
                $shipping->setMaxHandlingTime($shippingItem['shippingTime']['max_handling_time']);
                $shipping->setMinTransitTime($shippingItem['shippingTime']['min_transit_time']);
                $shipping->setMaxTransitTime($shippingItem['shippingTime']['max_transit_time']);
//                $shippingPrice->setValue($shippingItem['shippingTime']['shipping_fee']);

            }
            $shipping->setPrice($shippingPrice);
            $shippingFee[] = $shipping;
        }
        $product->setShipping($shippingFee);
        return $product;
    }

    private function buildProductId($item) {
        return sprintf('%s:%s:%s:%s', self::CHANNEL, $item['language'],
            $item['country'], $item['product_code']);
    }

    private function getClient($input) {
//        $authconfig = "megaads",$credentials = 'sheet', $authCode = '4/Ajm2xQHHJJpPgKGl6cq-A7cfarCCU8meTr6xnsXcm1U',$scopes = Google_Service_Sheets::SPREADSHEETS
        $scopes = \Google_Service_ShoppingContent::CONTENT;
        $credentialsPath = app_path('Modules/Ads/AuthConfig/credentials.json');
        $authConfigPath = app_path('Modules/Ads/AuthConfig/auth.json');
        $authCode = '4/Ajm2xQHHJJpPgKGl6cq-A7cfarCCU8meTr6xnsXcm1U';
        $client = new \Google_Client();
        $client->setApplicationName("Content API");
        $client->setScopes($scopes);
        $client->setAuthConfig($authConfigPath);
        $client->setAccessType('offline');

        // Load previously authorized credentials from a file.
        if (file_exists($credentialsPath)) {
            $accessToken = json_decode(file_get_contents($credentialsPath), true);
        } else {
            // Request authorization from the user.
            $authUrl = $client->createAuthUrl();
            printf("Open the following link in your browser:\n%s\n", $authUrl);
            print 'Enter verification code: ';
            $authCode = trim($authCode);

            // Exchange authorization code for an access token.
            $accessToken = $client->fetchAccessTokenWithAuthCode($authCode);

            // Store the credentials to disk.
            if (!file_exists(dirname($credentialsPath))) {
                mkdir(dirname($credentialsPath), 0700, true);
            }
            file_put_contents($credentialsPath, json_encode($accessToken));
            printf("Credentials saved to %s\n", $credentialsPath);
        }
        $client->setAccessToken($accessToken);

        // Refresh the token if it's expired.
        if ($client->isAccessTokenExpired()) {
            $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
            file_put_contents($credentialsPath, json_encode($client->getAccessToken()));
        }
        return $client;
    }
    private function makeBatchId($productId){
        return $productId;
        list($usec, $sec) = explode(' ', microtime());
        $retVal  = $sec + $usec * 1000000;
        return $retVal;
    }

    public function cronInsertProductVariant(Request $request){
        return "This function inactive";
        $input = $request->all();
        $timeLimit = 60*30;
        if (array_key_exists('time_limit',$input)){
            $timeLimit = $input['time_limit'];
        }
        set_time_limit($timeLimit);
        ini_set("memory_limit", "3072M");
        $prefixUrl = isset($_SERVER['PREFIX_URL']) ? $_SERVER['PREFIX_URL'] : '';
        $isLocalization = env('LOCALIZATION');
        $locale = env('APP_LOCALE');
        $countryCode = \Config::get('ads::clara-ads-general.countryCodeDefault','US');
        if ($prefixUrl == '' && $isLocalization && $locale != '') {
            $prefixUrl = '/' . $locale;
            $countryCode = $locale;
        }
        if ($countryCode == 'uk'){
            $countryCode = 'gb';
        }
        $countryCode = strtoupper($countryCode);
        $filter = [
            'status' => 'ACTIVE',
            'approve_advertising' => 1,
            'is_trademark' => 0,
            'is_violation' => 0,
            'category_merchant_id_from' => 1,
        ];
        if(array_key_exists('product_id_from',$input)){
            $filter['product_id_from'] = $input['product_id_from'];
        }
        if(!array_key_exists('ignore_updated_at',$input)){
            $filter['updated_at_from'] = date("Y-m-d H:i:s",strtotime('-4 hours'));
        }
        if(array_key_exists('product_id_to',$input)){
            $filter['product_id_to'] = $input['product_id_to'];
        }
        if(array_key_exists('updated_at_from',$input)){
            $filter['updated_at_from'] = $input['updated_at_from'];
        }
        if(array_key_exists('updated_at_to',$input)){
            $filter['updated_at_to'] = $input['updated_at_to'];
        }
        if(array_key_exists('ids',$input)){
            $filter['ids'] = explode(',',$input['ids']);
        }
        $filter['columns'] = [
            // ORIGIN
            'product.id',
            'product.slug', 'product.name', 'product.image_url',
            'product.slug as sku_slug',
            'product.price',
            'product.sku',
            'product_sku.id as sku_id',
            'product_sku.price as sku_price',
            'product_sku.image_url as sku_image',
            'product_sku.sku as sku_code',
            'product_n_category.category_id',
            'category_google_merchant.category_merchant_id',
        ];
        $query = $this->merchantProductService->buidFilterVariant($filter);
        $countryOfSales = $this->merchantProductService->getCountryOfSales($countryCode);
        if (array_key_exists('countryOfSales', $input)){
            $countryOfSales = explode(',', $input['countryOfSales']);
        }
        $onlySold = $this->merchantProductService->getCountryOfSaleSold($countryCode);
        $chunkCount = 10000;
        $count = 0;
        $query->chunk($chunkCount, function($products) use($prefixUrl,$countryCode, $countryOfSales, &$count, $onlySold){
            $input = [
                'prefixUrl' => $prefixUrl,
                'country' => $countryCode,
                'language' => isset($this->languages[$countryCode]) ? $this->languages[$countryCode] : env('APP_LANG'),
                'onlySold' => $onlySold
            ];
            $data = $this->merchantProductService->buildProductVariant($products,$input);
            $count += count($data);
            $this->insertVariant($data, $countryOfSales, $input);
        });
        return response()->json(['status' => 'successful', 'result' => $count]);

    }
    public function insertVariant($products, $countryOfSales, $input){
        return "This function inactive";
        $countryOnlySold = $input['onlySold'];
        try{
            foreach ($countryOfSales as $k => $country) {
                $entries = [];
                $skus = [];
                foreach ($products as $index => $product) {
                    if (in_array($country,$countryOnlySold) && $product['sold'] < 1){
                        continue;
                    }
                    $batchId = $this->makeBatchId($product['id']);
                    $skus[$batchId] = ['id' => $product['id'], 'sku' => $product['product_code']];
                    $product['country'] = $country;
                    $key = $batchId;
                    $entry = new \Google_Service_ShoppingContent_ProductsCustomBatchRequestEntry();
                    $entry->setMethod('insert');
                    $entry->setBatchId($key);
                    $entry->setProduct($this->buildProductVariant($product));
                    $entry->setMerchantId(getOption('merchant_id', '398409240'));
                    $entry->setUpdateMask();
                    $entries[] = $entry;
                }
                $this->client = $this->getClient([]);
                $this->service = new \Google_Service_ShoppingContent($this->client);
                $batchRequest =
                    new \Google_Service_ShoppingContent_ProductsCustomBatchRequest();
                $batchRequest->setEntries($entries);
                $batchResponse = $this->service->products->custombatch($batchRequest);
                foreach ($batchResponse->entries as $entry) {
                    $batchId = $entry->getBatchId();
                    $sku = $batchId;
                    $productId = $batchId;
                    if (isset($skus[$batchId])){
                        $sku = $skus[$batchId]['sku'];
                        $productId = $skus[$batchId]['id'];
                    }
                    if (!empty($entry->getErrors())) {
                        foreach ($entry->getErrors()->getErrors() as $error) {
                            \Log::error("SaveMerchantProductError: $sku " . $error->getMessage());
                            $this->merchantProductService->changeProductUpdatedAt($productId);
                        }
                    }
                }
            }
            return count($products);
        }catch (\Exception $ex){
            \Log::error($ex->getMessage() . $ex->getTraceAsString());
        }
    }
    private function buildProductVariant($item){
        $product = new \Google_Service_ShoppingContent_Product();
        $product->setOfferId($item['product_code']);
        $product->setTitle($item['title']);
        $product->setDescription($item['description']);
        $product->setCondition($item['condition']);
        $product->setGoogleProductCategory($item['sku_category_merchant_id']);
        $product->setProductTypes($item['product_type']);
        $product->setLink($item['link']);
        $product->setImageLink($item['image_url']);
        $product->setAvailability($item['availability']);
        $product->setExcludedDestinations($item['excluded_destination']);
        $price = new \Google_Service_ShoppingContent_Price();
        $price->setValue($item['price']);
        $price->setCurrency($item['currency']);
        $product->setPrice($price);
        $product->setBrand($item['brand']);
        $product->setMpn($item['product_code']);
        $product->setContentLanguage($item['language']);
        $product->setFeedLabel($item['country']);
        $product->setChannel(self::CHANNEL);
        $product->setAgeGroup('adult');
        $product->setColor('Black/White');
        $product->setSizes('one size fits all');
        $product->setGender('unisex');
        $product->setAvailabilityDate(date('Y-m-d',strtotime('+ 30 days')));
        $shippingFee = [];
        foreach ($item['shippingFees'] as $shippingItem){
            $shippingPrice = new \Google_Service_ShoppingContent_Price();
            $shippingPrice->setValue($shippingItem['price']);
            $shippingPrice->setCurrency($shippingItem['currency']);
            $shipping = new \Google_Service_ShoppingContent_ProductShipping();
            $shipping->setPrice($shippingPrice);
            $shipping->setCountry($shippingItem['country']);
            $shippingFee[] = $shipping;
        }
        $product->setShipping($shippingFee);
        return $product;
    }
    public function createTextFile(Request $request){
        $start = microtime(true);
        $input = $request->all();
        $timeLimit = 60*30;
        if (array_key_exists('time_limit',$input)){
            $timeLimit = $input['time_limit'];
        }
        set_time_limit($timeLimit);
        ini_set("memory_limit", "3072M");
        $prefixUrl = isset($_SERVER['PREFIX_URL']) ? $_SERVER['PREFIX_URL'] : '';
        $isLocalization = env('LOCALIZATION');
        $locale = env('APP_LOCALE');
        $countryCode = \Config::get('ads::clara-ads-general.countryCodeDefault','US');
        if ($prefixUrl == '' && $isLocalization && $locale != '') {
            $prefixUrl = '/' . $locale;
            $countryCode = $locale;
        }
        if ($countryCode == 'uk'){
            $countryCode = 'gb';
        }
        $countryCode = strtoupper($countryCode);
        $filter = [
            'status' => 'ACTIVE',
            'approve_advertising' => 1,
            'advertising_status' => 1,
            'is_trademark' => 0,
            'is_violation' => 0,
            'category_merchant_id_from' => 1,
            'is_default' => 1
        ];
        if(array_key_exists('product_id_from',$input)){
            $filter['product_id_from'] = $input['product_id_from'];
        }
        if(!array_key_exists('ignore_updated_at',$input)){
            $filter['updated_at_from'] = date("Y-m-d H:i:s",strtotime('-4 hours'));
        }
        if(array_key_exists('product_id_to',$input)){
            $filter['product_id_to'] = $input['product_id_to'];
        }
        if(array_key_exists('updated_at_from',$input)){
            $filter['updated_at_from'] = $input['updated_at_from'];
        }
        if(array_key_exists('hour_ago',$input)){
            $filter['updated_at_from'] = date("Y-m-d H:i:s",strtotime('-'.$input['hour_ago'].' hours'));
        }
        if(array_key_exists('updated_at_to',$input)){
            $filter['updated_at_to'] = $input['updated_at_to'];
        }
        if(array_key_exists('actor_ids',$input)){
            $filter['actor_ids'] = explode(",", $input['actor_ids']);
        }
        if(array_key_exists('update_expire',$input)){
            $exprieDate = date("Y-m-d H:i:s",strtotime('-25 days'));
            $expireFilter = $filter;
            $expireFilter['updated_at_to'] = $exprieDate;
            $ids = $this->merchantProductService->getProductExprieIds($expireFilter);
            if (empty($ids)){
                $ids = [0];
            }
            $filter['ids'] = $ids;
        }
        if(array_key_exists('ids',$input)){
            $filter['ids'] = explode(',',$input['ids']);
        }
        $filter['columns'] = [
            // ORIGIN
            'product.id',
            'product.slug', 'product.brand_id', 'product.image_url',
            'product.description',
            'product.status', 'product.approve_advertising',
            'product.is_trademark',
            'product.gtin',
            'product.is_always_on_ads',
            'product.actor_id',
            'product.created_at',
            'product.sold',
            // SKU
            'product_info.name as sku_name',
            'product_info.variants as sku_specific_info',
            'product_info.brand as sku_brand',
            'product_info.sku as sku_code',
            'product_info.slug as sku_slug',
            'product_info.price',
            'product_info.inventory',
            'product_info.high_price',
            'product_info.category_merchant_id as sku_category_merchant_id',
            'product_info.product_sku_id',
            'product_info.is_default',
            'product_n_category.category_id as category_id',
            'product.name as title_replacement',
        ];
        $query = $this->merchantProductService->buidFilter($filter);
        $chunkCount = 200000;
        $count = 0;
        $query->chunk($chunkCount, function($products) use($prefixUrl,$countryCode, $input, &$count, $start, $locale){
            \Log::info("Merchant count: " . $count ." " . (microtime(true) - $start));
            $input['prefixUrl'] = $prefixUrl;
            $input['country'] = $countryCode;
            $data = $this->merchantService->buildProducts($products, $input);
            $count += $data;
            $this->merchantService->store($input, strtolower($countryCode));
        });
        $totalTime = microtime(true) - $start;
        \Log::info("Merchant: " . json_encode($input));
        \Log::info("Merchant process time miliseconds: " . $totalTime);
        return response()->json(['status' => 'successful', 'result' => $count]);
    }
    private function encodeImageUrl($url){
        if (strpos($url,'liveview.printerval.com') !== false){
            $regex = "([^\/]+$)";
            try{
                preg_match($regex,$url,$matches);
            }catch(\Exception $e){
                \Log::error("BuildImageMerchantProductError: $url " . $error->getMessage());
            }
            if(!empty($matches)){
                $path = $matches[0];
                $url = preg_replace($regex,urlencode($path),$url);
            }
        }
        return $url;
    }

    public function getProductNotApprove(){
        set_time_limit(1200);
        try {
            $response = ["status" => "successful"];
            $parameters = ['maxResults' => 1000];
            $merchantId = getOption('merchant_id','398409240');
            $searchRequest = new SearchRequest();
            $searchRequest->setPageSize($parameters['maxResults']);
            $query = new QueryBuilder();
            $threeDayAgo = strtotime('-3 days');
            $query = $query->from('ProductView')
                ->where('product_view.aggregated_destination_status', 'IN','("PENDING","NOT_ELIGIBLE_OR_DISAPPROVED","ELIGIBLE_LIMITED")')
                ->where('product_view.currency_code', '=',"'$this->currencyCode'")
                ->where('product_view.language_code', '=',"'$this->languageCode'")

                ->where('product_view.creation_time', '<',$threeDayAgo)
                ->build([
                    'product_view.id',
                    'product_view.offer_id',
                    'product_view.title',
                    'product_view.creation_time',
                    'product_view.item_issues'
                ]);
//            var_dump($query);die;
            $searchRequest->setQuery($query);
            $this->truncateProductPending();
            $i = 0;
            do {
                $i++;
                if ($i%500 == 0){
                    $this->client = $this->getClient([]);
                    $this->service = new \Google_Service_ShoppingContent($this->client);
                }
                $report = $this->service->reports->search($merchantId,$searchRequest);
                $result = $report->getResults();
                $data = [];
                foreach ($result as $item) {
                    $product = $item->getProductView();
                    $issues = $product->getItemIssues();
                    $note = '';
                    foreach ($issues as $issue) {
                        if (isset($issue->getIssueType()->code)){
                            $issueCode = $issue->getIssueType()->code;
                        }else{
                            $issueCode = $issue->getIssueType()->getCode();
                        }
                        $note .= $issueCode . ' | ';
                    }
                    $productId = $this->getProductIdFromSku($product->getOfferId());
                    $data[] = [
                        'sku' => $product->getOfferId(),
                        'product_id' => $productId,
                        'title' => $product->getTitle(),
                        'note' => substr($note,0,-3),
                        'created_at' => date('Y-m-d H:i:s'),
                        'updated_at' => date('Y-m-d H:i:s')
                    ];
                }
                $this->saveProductPending($data);
                $parameters['pageToken'] = $report->nextPageToken;
                $searchRequest->setPageToken($parameters['pageToken']);
                usleep(600000);
            } while (!empty($parameters['pageToken']));

        } catch (Exception $exc) {
            $response = ["status" => "fail", "message" => $exc->getMessage()];
        }
        return response()->json($response);;
    }

    private function saveProductPending($data){
        DB::table('ads_pending_product')->insert($data);
    }
    private function truncateProductPending(){
        DB::table('ads_pending_product')->where('created_at', '<', date('Y-m-d 00:00:00'))->delete();
    }
    public function cronCheckViolation(){
        set_time_limit(1200);
        $errorCodes = [
            'misinformation_policy_violation' => 1,
            'explosives_policy_violation' => 1,
            'guns_parts_policy_violation' => 1,
            'healthcare_misleading_claims_policy_violation' => 1,
            'healthcare_pdt_policy_violation' => 1,
            'healthcare_unapproved_supplements_policy_violation' => 1,
            'illegal_drugs_policy_violation' => 1,
            'live_animals_policy_violation' => 1,
            'offensive_content_policy_violation' => 1,
            'other_weapons_policy_violation' => 1,
            'personal_hardships_policy_violation' => 1,
            'tobacco_policy_violation' => 1,
            'alcohol_policy_violation' => 1,
            'identity_and_belief_policy_violation' => 1,
            'gambling_policy_violation' => 1,
            'restricted_adult_policy_violation' => 1,
            'sexual_interests_policy_violation' => 1,
            'healthcare_gender_detection_policy_violation' => 1,
            'healthcare_otc_policy_violation' => 1
        ];
        try {
            $response = ["status" => "successful"];
            $data = [];
            $parameters = ['maxResults' => 1000];
            $merchantId = getOption('merchant_id','398409240');
            $searchRequest = new SearchRequest();
            $searchRequest->setPageSize($parameters['maxResults']);
            $query = new QueryBuilder();
            $query = $query->from('ProductView')
                ->where('product_view.aggregated_destination_status', 'IN','("NOT_ELIGIBLE_OR_DISAPPROVED")')
                ->where('product_view.currency_code', '=',"'$this->currencyCode'")
                ->where('product_view.language_code', '=',"'$this->languageCode'")
                ->build([
                    'product_view.id',
                    'product_view.offer_id',
                    'product_view.title',
                    'product_view.creation_time',
                    'product_view.item_issues'
                ]);
            $searchRequest->setQuery($query);
            $i = 0;
            do {
                $i++;
                if ($i%100 == 0){
                    $this->client = $this->getClient([]);
                    $this->service = new \Google_Service_ShoppingContent($this->client);
                }
                $report = $this->service->reports->search($merchantId,$searchRequest);
                $result = $report->getResults();
                $ids = [];
                foreach ($result as $item) {
                    $product = $item->getProductView();
                    $issues = $product->getItemIssues();
                    $isViolation = false;
                    $errorCode = [];
                    foreach ($issues as $issue) {
                        if (isset($issue->getIssueType()->code)){
                            $issueCode = $issue->getIssueType()->code;
                        }else{
                            $issueCode = $issue->getIssueType()->getCode();
                        }
//                        if (in_array($issueCode,$errorCodes)){
                        if (isset($errorCodes[$issueCode])){
                            $isViolation = true;
                            $errorCode[] = $issueCode;
                        }
                    }
                    $productId = $this->getProductIdFromSku($product->getOfferId());
                    if ($isViolation){
                        $this->updateProductViolation($productId,$errorCode);
                        $data[] = $productId;
                        $ids[] = $productId;
                        \Log::error("MerchantViolation: $productId ");
                    }

                }
                if (!empty($ids)){
                    $deleteUrl = route('ads::ads:merchant::delete',['ids' => implode(',', $ids),'skip_ads' => 'on']);
                    $this->triggerAsyncRequest($deleteUrl);
                }
                $parameters['pageToken'] = $report->nextPageToken;
                $searchRequest->setPageToken($parameters['pageToken']);
                usleep(600000);
            } while (!empty($parameters['pageToken']));
            $response['result'] = $data;
            $response['count'] = count($data);
        } catch (Exception $exc) {
            $response = ["status" => "fail", "message" => $exc->getMessage()];
        }
        $deleteUrl = route('ads::ads:merchant::delete',['skip_ads' => 'on','updated_at_from' => date("Y-m-d H:i:s",strtotime('-20 minutes'))]);
        $this->triggerAsyncRequest($deleteUrl);
        return response()->json($response);;
    }
    private function updateProductViolation($productId, $errors = []){
        $affected =  DB::table('product')
            ->where('id', $productId)
            ->update(['is_violation' => 1, 'updated_at' => date('Y-m-d H:i:s')]);
        if ($affected > 0){
            $affectedMeta =  DB::table('product_meta')
                ->where('product_id', $productId)
                ->where('key', 'violation_errors')
                ->update(['value' => json_encode($errors), 'updated_at' => date('Y-m-d H:i:s')]);
            if ($affectedMeta < 1){
                DB::table('product_meta')
                    ->where('product_id', $productId)
                    ->where('key', 'violation_errors')
                    ->insert(['value' => json_encode($errors), 'product_id' => $productId, 'key' => 'violation_errors', 'updated_at' => date('Y-m-d H:i:s'), 'created_at' => date('Y-m-d H:i:s')]);
            }
        }
    }

    public function countProductByCategory(){
        set_time_limit(60*30);
        ini_set("memory_limit", "3072M");
        $result = [];
        $maxProductId = DB::table('merchant_product')->max('product_id');
        $step = 10000;
        $categoryIds = $this->getCategoryLevel2();
        if (!empty($categoryIds)){
            for($i = 0; $i<= $maxProductId+$step; $i+= $step){
                $data = DB::table('merchant_product')
                    ->join('product','product.id','=','merchant_product.product_id')
                    ->join('product_n_category','product.id','=','product_n_category.product_id')
                    ->where('product.status','active')
                    ->where('product.is_violation',0)
                    ->where('product.is_trademark',0)
                    ->where('product.id','>=', $i)
                    ->where('product.id','<=', $i+$step)
                    ->whereIn('product_n_category.category_id',$categoryIds)
                    ->select(DB::raw("sb_product_n_category.category_id,COUNT(*) as numberOfProducts"))
                    ->groupBy('product_n_category.category_id')->get();
                foreach ($data as $item){
                    if (isset($result[$item->category_id])){
                        $result[$item->category_id] += $item->numberOfProducts;
                    }else{
                        $result[$item->category_id] = 0;
                    }
                }
            }
        }
        if (!empty($result)){
            $this->saveToOption('merchant_count_product_category',$result);
        }
        return response()->json(['status' => 'successful','result' => $result]);

    }

    private function getCategoryLevel2(){
        $retVal = [];
        $parentIds = Category::where('is_hidden',0)
            ->where('type',Category::TYPE_PRODUCT)
            ->whereNull('parent_id')->pluck('id')->toArray();
        if (!empty($parentIds)){
            $retVal = Category::where('is_hidden',0)
                ->where('type',Category::TYPE_PRODUCT)
                ->whereIn('parent_id',$parentIds)->pluck('id')->toArray();
        }
        return $retVal;
    }
    private function saveToOption($key,$data, $type = 'json'){
        $isExists = DB::table('option')->where('key', $key)->exists();
        if (!$isExists){
            DB::table('option')->insert(['key' => $key,'value' => json_encode($data),'type' => $type]);
        }else{
            DB::table('option')->where('key',$key)->update(['value' => json_encode($data)]);
        }
    }
    public function cronCacheShippingInfo(Request $request){
        $prefixUrl = isset($_SERVER['PREFIX_URL']) ? $_SERVER['PREFIX_URL'] : '';
        $isLocalization = env('LOCALIZATION');
        $locale = env('APP_LOCALE');
        $countryCode = \Config::get('ads::clara-ads-general.countryCodeDefault','US');
        if ($prefixUrl == '' && $isLocalization && $locale != '') {
            $prefixUrl = '/' . $locale;
            $countryCode = $locale;
        }
        if ($countryCode == 'uk'){
            $countryCode = 'gb';
        }
        $countryCode = strtoupper($countryCode);
        $input = $request->all();
        $input['columns'] = ['product.id','product.price','product_info.product_sku_id as sku_id'];
        $input['product_info'] = 'join';
        $query = $this->merchantProductService->buildDeleteQuery($input);
        $products = $query->get();
        $retVal = [];
        foreach ($products as $product){
            $retVal[] = $this->merchantProductService->cacheShippingInfo($product,$countryCode);
        }
        return response()->json(['status' => 'successful','result' => $retVal]);
    }
    public function cronUpdatePrice(Request $request){
        set_time_limit(60*30);
        ini_set("memory_limit", "2048M");
        $prefixUrl = isset($_SERVER['PREFIX_URL']) ? $_SERVER['PREFIX_URL'] : '';
        $isLocalization = env('LOCALIZATION');
        $locale = env('APP_LOCALE');
        $countryCode = \Config::get('ads::clara-ads-general.countryCodeDefault','US');
        if ($prefixUrl == '' && $isLocalization && $locale != '') {
            $prefixUrl = '/' . $locale;
            $countryCode = $locale;
        }
        if ($countryCode == 'uk'){
            $countryCode = 'gb';
        }
        $countryCode = strtoupper($countryCode);
        $input = $request->all();
        $input['columns'] = ['product.id','product.price','merchant_product.sku','product.high_price'];
        if(!array_key_exists('ignore_updated_at',$input)){
            $input['updated_at_from'] = date("Y-m-d H:i:s",strtotime('-4 hours'));
        }
        if(array_key_exists('ids',$input)){
            $input['ids'] = explode(",", $input['ids']);
        }
        $query = $this->merchantProductService->buildDeleteQuery($input);
        $products = $query->get();
        $countryOfSales = $this->merchantProductService->getCountryOfSales($countryCode);
        if (array_key_exists('countryOfSales', $input)){
            $countryOfSales = explode(',', $input['countryOfSales']);
        }
        $onlySold = $this->merchantProductService->getCountryOfSaleSold($countryCode);
        $count = 0;
        $inputBuildProduct = [
            'prefixUrl' => $prefixUrl,
            'country' => $countryCode,
            'language' => isset($this->languages[$countryCode]) ? $this->languages[$countryCode] : env('APP_LANG'),
            'onlySold' => $onlySold
        ];
        $data = $this->merchantProductService->buildProductPrice($products,$inputBuildProduct);
        $count += count($data);
        if (isset($input['merchant_id'])){
            $inputBuildProduct['merchant_id'] = $input['merchant_id'];
        }
        $message = $this->updatePrice($data, $countryOfSales, $inputBuildProduct);
//        $message = $this->updateByService($data, ['method' => 'update', 'country' => $countryCode,'updateMask' => 'salePrice,price,availabilityDate,expirationDate']);

        return response()->json(['status' => 'successful', 'result' => $count, 'message' => $message]);
    }
    public function updatePrice($products, $countryOfSales, $input){
        try{
            foreach ($countryOfSales as $k => $country) {
                $entries = [];
                $skus = [];
                foreach ($products as $index => $product) {
                    $merchantId = $this->buildMerchantId($product);
                    $batchId = $this->makeBatchId($product['id']);
                    $skus[$batchId] = ['id' => $product['id'], 'sku' => $product['product_code']];
                    $product['country'] = $country;
                    $key = $batchId;
                    $entry = new \Google_Service_ShoppingContent_ProductsCustomBatchRequestEntry();
                    $entry->setMethod('update');
                    $entry->setBatchId($key);
                    $entry->setProductId($this->buildProductId($product));
                    $entry->setProduct($this->buildProductPrice($product));
                    $entry->setMerchantId($merchantId);
                    $entry->setUpdateMask('salePrice,price,availabilityDate,expirationDate');
                    $entries[] = $entry;
                }
                $this->insertByChunk($entries,$skus,$country,10000);
            }
            return count($products);
        }catch (\Exception $ex){
            \Log::error("Google Merchant ".$ex->getMessage() . $ex->getTraceAsString());
        }
    }

    private function buildProductPrice($item){
        $product = new \Google_Service_ShoppingContent_Product();
        $price = new \Google_Service_ShoppingContent_Price();
        $price->setValue($item['high_price']);
        $price->setCurrency($item['currency']);
        $product->setPrice($price);
        $salePrice = new \Google_Service_ShoppingContent_Price();
        $salePrice->setValue($item['price']);
        $salePrice->setCurrency($item['currency']);
        $product->setSalePrice($salePrice);
        $product->setAvailabilityDate(date('Y-m-d',strtotime('+ 30 days')));
        $product->setExpirationDate(date('Y-m-d',strtotime('+ 40 days')));
        return $product;
    }

    private function buildMerchantId($product, $input = []){
        $merchantId = $this->merchantId;
        if (isset($input['merchant_id'])){
            $merchantId = $input['merchant_id'];
        }elseif(isset($product['merchantId'])){
            $merchantId = $product['merchantId'];
        }
        return $merchantId;
    }
    public function exportCustomer(Request $request){
        $input = $request->all();
        $query = DB::connection('mysql_customer')->table('customer')
            ->select(['email','id'])->orderBy('id','desc');
        if (array_key_exists('day_ago',$input)){
            $query->where('created_at','>=',date('Y-m-d H:i:s',strtotime('-'.$input['day_ago'].' days')));
        }
        $option = DB::table('option')->where('key', 'lastCustomerId')->first(['value', 'type']);
        $lastCustomerId = 0;
        if (!empty($option)){
            $lastCustomerId = $option->value;
        }
        $query->where('id','>',$lastCustomerId);
        $data = $query->get();
        $items = [];
        if (array_key_exists('truncate',$input)){
            $items['email'] = ['email'];
        }
        foreach ($data as $customer){
            $items[$customer->email] = [$customer->email];
        }
        if (isset($data[0])){ // if not empty
            $lastCustomerId = $data[0]->id;
        }
        $input['items'] = $items;
        $this->customerService->store($input);
        DB::table('option')->where('key', 'lastCustomerId')->update(['value'=>$lastCustomerId]);
        $localFilePath = public_path('feeds/customer/customer.csv');
        $gcsFilePath = 'customer/customer.csv';
        $object = uploadToGcs($localFilePath, $gcsFilePath);
        return response()->json(['status' => 'successful', 'url' => url('feeds/customer/customer.csv'),'result' => $object]);
    }

    public function removeProductByScore(Request $request)
    {
        set_time_limit(60 * 10);
        ini_set("memory_limit", "2048M");
        $checkInProduct = $this->merchantProductService->getProductCalculate('merchant_product_in',1,true);
        if (empty($checkInProduct)){
            return response()->json(['status' => 'successful', 'message' => "Empty Product In"]);
        }
        $prefixUrl = isset($_SERVER['PREFIX_URL']) ? $_SERVER['PREFIX_URL'] : '';
        $isLocalization = env('LOCALIZATION');
        $locale = env('APP_LOCALE');
        $countryCode = \Config::get('ads::clara-ads-general.countryCodeDefault','US');
        if ($prefixUrl == '' && $isLocalization && $locale != '') {
            $prefixUrl = '/' . $locale;
            $countryCode = $locale;
        }
        if ($countryCode == 'uk'){
            $countryCode = 'gb';
        }
        $countryCode = strtoupper($countryCode);
        $input = $request->all();
        $filter = $input;
        $filter['score'] = $this->scoreTarget + 2;
        $ids = $this->merchantProductService->getProductCalculate('merchant_product_out',10000,false,['order'=>['score','asc']]);
        $filter['ids'] = $ids;
        $filter['sold'] = 0;
        $query = $this->merchantProductService->buildDeleteQueryByScore($filter);
        $countryOfSales = $this->merchantProductService->getCountryOfSales($countryCode);
        if (array_key_exists('countryOfSales', $input)){
            $countryOfSales = explode(',', $input['countryOfSales']);
        }
        $products = $query->get();
        $inputData = [
            'prefixUrl' => $prefixUrl,
            'country' => $countryCode,
            'language' => isset($this->languages[$countryCode]) ? $this->languages[$countryCode] : env('APP_LANG'),
        ];
        $productDeletes = [];
        $deletedIds = [];
        $soldIds = [];
        foreach ($products as $product){
            if ($product->sold > 0){
                $soldIds[] = $product->product_id;
                continue;
            }
            $deletedIds[] = $product->product_id;
            $productDeletes[] = ['productId' => $product->product_id, 'sku' => $product->sku, 'merchantProductId' => $product->id];
        }
        $data = $this->merchantProductService->buildProductDelete($productDeletes,$inputData);
        if (isset($input['merchant_id'])){
            $inputData['merchant_id'] = $input['merchant_id'];
        }
        $this->deleteByService($data,['method' => 'delete', 'country' => $countryCode]);
        $dataChunk = array_chunk($data,10000);
        foreach ($dataChunk as $chunkItem){
            $this->delete($chunkItem, $countryOfSales,$inputData);
        }
        $this->merchantProductService->updateDeleteMerchantProduct($deletedIds);
        $this->merchantProductService->logDeleteMerchantProduct($soldIds);
        $countDeleted = count($deletedIds);
        $insertIds = [];
        if ($countDeleted > 0){
            $insertIds = $this->insertByScore($countDeleted);
        }
        return response()->json(['status' => 'successful', 'message' => $countDeleted, 'insertIds' => $insertIds]);
    }

    public function insertByScore($limit){
        $ids = $this->merchantProductService->getProductCalculate('merchant_product_in',$limit);
        if (!empty($ids)){
            $filter = $this->getDefaultFilter();
            $filter['ids'] = $ids;
            $result =$this->updateProduct($filter,[]);
            \Log::info("insertByScore ".json_encode($result));
        }
        return $ids;
    }

    public function preCalculateInsertProduct(Request $request){
        $filter = $request->all();
        $filter['approve_advertising'] = 1;
        $filter['score'] = $this->scoreTarget;
        $products = $this->merchantProductService->getProductIdsByScore($filter);
        $this->merchantProductService->insertPreCalculate($products,'merchant_product_in');
        return response()->json(['status' => 'successful','message' => count($products)]);
    }

    public function preCalculateDeleteProduct(Request $request){
        $filter = $request->all();
        $filter['score'] = $this->scoreTarget + 1;
        $filter['sold'] = 0;
        $query = $this->merchantProductService->buildDeleteQueryByScore($filter);
        $ids = $query->get()->toArray();
        $this->merchantProductService->insertPreCalculate($ids,'merchant_product_out');
        return response()->json(['status' => 'successful','message' => count($ids)]);
    }

    public function updateReport(Request $request){
        set_time_limit(60*60);
        ini_set("memory_limit", "3072M");
        $retVal = ['status' => 'success'];
        try {
            $filter = $request->all();
            $retVal['data'] = $filter;
            $product = $this->merchantProductService->getProduct(['product_id_from' => $filter['product_id_from']]);
            $filter['approve_advertising'] = 1;
            $filter['score'] = $this->scoreTarget;
            $query = $this->merchantProductService->queryReport($filter);
            $total = $query->count();
            $filter['approve_advertising'] = 0;
            $query = $this->merchantProductService->queryReport($filter);
            $notApprove = $query->count();
            $filter['approve_advertising'] = 1;
            $filter['product_info'] = 1;
            $query = $this->merchantProductService->queryReport($filter);
            $missingProductInfo = $query->count();
            unset($filter['product_info']);
            $filter['approve_advertising'] = 1;
            $filter['merchant_product'] = 1;
            $query = $this->merchantProductService->queryReport($filter);
            $pushed = $query->count();
            $waiting = $total - $missingProductInfo - $pushed;
            $data = [
                'product_first_id' => $filter['product_id_from'],
                'total' => $total,
                'not_approve' => $notApprove,
                'missing_product_info' => $missingProductInfo,
                'waiting' => $waiting,
//                'low_score' => $lowScore,
                'product_created_at' => $product->created_at,
            ];
            $this->merchantProductService->saveReport($data,$filter);
        } catch (Exception $ex) {
            $retVal['status'] = 'fail';
            \Log::error("GMC Report ".$ex->getMessage() . $ex->getTraceAsString());
        }
        return response()->json($retVal);
    }

    public function cronBuildReport(Request $request){
        set_time_limit(60*60);
        ini_set("memory_limit", "3072M");
        $retVal = ['status' => 'successful'];
        $prefixUrl = isset($_SERVER['PREFIX_URL']) ? $_SERVER['PREFIX_URL'] : '';
        $isLocalization = env('LOCALIZATION');
        $locale = env('APP_LOCALE');
        $countryCode = \Config::get('ads::clara-ads-general.countryCodeDefault','US');
        if ($prefixUrl == '' && $isLocalization && $locale != '') {
            $prefixUrl = '/' . $locale;
            $countryCode = $locale;
        }
        if ($countryCode == 'uk'){
            $countryCode = 'gb';
        }
        $countryCode = strtoupper($countryCode);
        $queueDomain = '';
        if (isset($_SERVER['HTTP_HOST'])){
            $domain = $_SERVER['HTTP_HOST'];
            $queueDomain = $domain;
        }
        if (isset($this->queue[$countryCode])){
            $queueDomain = $this->queue[$countryCode];
        }elseif(isset($this->queue['DEFAULT'])){
            $queueDomain = $this->queue['DEFAULT'];
        }
        $filter['created_at_from'] = date('Y-m-d H:i:s',strtotime('-91 days'));
        $product = $this->merchantProductService->getProduct($filter);
        $from = $product->id;
        $filter['max'] = 1;
        $product = $this->merchantProductService->getProduct($filter);
        $to = $product->id;
        $chunk = 10000;
        if ($request->has('chunk')){
            $chunk = $request->get('chunk');
        }
        $max = $to + $chunk;
        for($i = $from; $i<= $max; $i = $i+$chunk){
            $url = route('ads::ads:merchant::updateReport',['product_id_from'=> $i, 'product_id_to' => ($i + $chunk)]);
            if (!empty($domain)){
                $url = str_replace($domain,$queueDomain,$url);
            }
            $this->triggerSyncRequest($url);
        }
        $retVal['data'] = ['fromId' => $from, 'toId' => $max];
        return response()->json($retVal);

    }

    public function insertByService($products, $input){
        $result['status'] = 'fail';
        $result['message'] = 'empty product';
        if (!empty($products)){
            $url = $this->merchantServiceConfig['queue'].'/api/google-merchant?site='.$_SERVER['HTTP_HOST'];
//            $url = 'http://shopot.lapdx/module/ads/merchant/insert-to-queue';
            $params = [
                'products' => utf8_encode(gzencode(json_encode($products),9)),
                'method' => 'insert',
                'country' => $input['country'],
                'postback_url' => route("ads::ads:merchant::updateMerchantData")
            ];
            $queueDomain = '';
            $domain = '';
            if (isset($_SERVER['HTTP_HOST'])){
                $domain = $_SERVER['HTTP_HOST'];
                $queueDomain = $domain;
            }
            if(isset($this->queue['DEFAULT'])){
                $queueDomain = $this->queue['DEFAULT'];
            }
            $params['postback_url'] = str_replace($domain,$queueDomain,$params['postback_url']);
            $result = $this->triggerSyncRequest($url,'POST',$params,$this->getHeaderMerchantService(),200);
        }
        return $result;
    }
    public function updateByService($products, $input){
        $result['status'] = 'fail';
        $result['message'] = 'empty product';
        if (!empty($products)){
            $url = $this->merchantServiceConfig['queue'].'/api/google-merchant?site='.$_SERVER['HTTP_HOST'];
            $params = [
                'products' => utf8_encode(gzencode(json_encode($products),9)),
                'method' => 'update',
                'country' => $input['country'],
                'updateMask' => $input['updateMask'],
                'postback_url' => route("ads::ads:merchant::updateMerchantData")
            ];
            $queueDomain = '';
            $domain = '';
            if (isset($_SERVER['HTTP_HOST'])){
                $domain = $_SERVER['HTTP_HOST'];
                $queueDomain = $domain;
            }
            $params['postback_url'] = str_replace($domain,$queueDomain,$params['postback_url']);
            $result = $this->triggerSyncRequest($url,'POST',$params,$this->getHeaderMerchantService());
        }
        return $result;
    }
    public function deleteByService($products, $input){
        $result['status'] = 'fail';
        $result['message'] = 'empty product';
        if (!empty($products)) {
            $url = $this->merchantServiceConfig['queue'] . '/api/google-merchant?site='.$_SERVER['HTTP_HOST'];
            $params = [
                'products' => utf8_encode(gzencode(json_encode($products),9)),
                'method' => 'delete',
                'country' => $input['country'],
                'postback_url' => route("ads::ads:merchant::updateMerchantData")
            ];
            $queueDomain = '';
            $domain = '';
            if (isset($_SERVER['HTTP_HOST'])){
                $domain = $_SERVER['HTTP_HOST'];
                $queueDomain = $domain;
            }
            $params['postback_url'] = str_replace($domain,$queueDomain,$params['postback_url']);
            $result = $this->triggerSyncRequest($url, 'POST', $params, $this->getHeaderMerchantService());
        }
        return $result;
    }

    public function updateMerchantData(Request $request){
        $localeData = $this->getLocaleData();
        $countryCode = $localeData['countryCode'];
        $inputAll = $request->all();
//        \Log::useDailyFiles(storage_path().'/logs/merchant-service.log');
//        \Log::info("Google: " .json_encode($inputAll));
        $input = $inputAll;
        if (isset($inputAll['result']['data'])){
            $input = $inputAll['result']['data'];
            $method = $input['method'];
            $removeIds = $input['removeIds'];
            $productIds = $input['productIds'];
            $productData = $input['productData'];
            if (!empty($removeIds)){
                $this->merchantProductService->clearProductCreateFail($removeIds);
            }
            switch ($method){
                case 'insert':
                    $removeData = [];
                    foreach ($productData as $product){
                        $removeData = array_merge($removeData,$this->merchantProductService->saveMerchantProduct($product['product_id'], $product['sku']));
                    }
                    $removeData = $this->merchantProductService->buildProductDelete($removeData,['language' => isset($this->languages[$countryCode]) ? $this->languages[$countryCode] : env('APP_LANG'),'country' => $countryCode]);
                    $this->deleteByService($removeData,['method' =>'delete','country' => $countryCode]);
                    $this->merchantProductService->updateDeleteMerchantProduct($productIds,0);
                    break;
                case 'update':

                    break;
                case 'delete':
                    if (!empty($productData)){
                        $this->merchantProductService->createDeleteMerchantProduct($productData);
                    }
                    if (!empty($productIds)){
                        $this->merchantProductService->removeMerchantProduct($productIds);
                    }
                    break;
                default:

            }
        }
        return response()->json($input);
    }

    private function getHeaderMerchantService($length = 0){
        $headers = array(
            'Authorization: Basic '. base64_encode($this->merchantServiceConfig['user'].":".$this->merchantServiceConfig['password']),
            'Content-Type: application/json'

        );
        if (!empty($length)){
            $headers[] = 'Content-Encoding: gzip';
            $headers[] = 'Content-Length: ' . $length;
        }
        return $headers;
    }
    public function insertToQueue(Request $request){
        set_time_limit(60*60*24);
        ini_set("memory_limit", "1024M");
        $filter = $request->all();
        $this->getLocaleData();
        $localeData = $this->getLocaleData();
        $countryCode = $localeData['countryCode'];
        $queueDomain = '';
        $domain = '';
        if (isset($_SERVER['HTTP_HOST'])){
            $domain = $_SERVER['HTTP_HOST'];
            $queueDomain = $domain;
        }
        if (isset($this->queue[$countryCode])){
            $queueDomain = $this->queue[$countryCode];
        }elseif(isset($this->queue['DEFAULT'])){
            $queueDomain = $this->queue['DEFAULT'];
        }
        $product = $this->merchantProductService->getProduct(['max' => 1]);
        $maxId = $product->id;
        $lastId = $maxId - 2000000;
        if (array_key_exists('lastId',$filter)){
            $lastId = $filter['lastId'];
        }
        if (array_key_exists('maxId',$filter)){
            $maxId = $filter['maxId'];
        }
        $routeName = 'ads::ads:merchant::insert';
        if (array_key_exists('routeName',$filter)){
            $routeName = $filter['routeName'];
        }
        $step = 5000;
        if (array_key_exists('step',$filter)){
            $step = $filter['step'];
        }
        $this->sendToQueue([
            'start' => $lastId,
            'step' => $step,
            'max' => $maxId,
            'filter' => $filter,
            'routeName' => $routeName,
            'domain' => $domain,
            'queueDomain' => $queueDomain
        ]);
//        $this->merchantProductService->saveOptionByKey('merchant_last_id',$maxId);
        return response()->json(['lastId' => $lastId,'maxId' => $maxId]);
    }

    public function insertPriority(Request $request){
        set_time_limit(60*60*24);
        ini_set("memory_limit", "2048M");
        $filter = $request->all();
        $this->getLocaleData();
        $localeData = $this->getLocaleData();
        $countryCode = $localeData['countryCode'];
        $queueDomain = '';
        $domain = '';
        if (isset($_SERVER['HTTP_HOST'])){
            $domain = $_SERVER['HTTP_HOST'];
            $queueDomain = $domain;
        }
        if (isset($this->queue[$countryCode])){
            $queueDomain = $this->queue[$countryCode];
        }elseif(isset($this->queue['DEFAULT'])){
            $queueDomain = $this->queue['DEFAULT'];
        }
        if (array_key_exists('labelSales',$filter)){
            $ids = $this->merchantProductService->getProductTotalSale();
            $filter['ids'] = implode(",",array_keys($ids));
            $filter['ignore_updated_at'] = 1;
            $routeName = 'ads::ads:merchant::insert';
            $url = route($routeName,$filter);
            $url = str_replace($domain,$queueDomain,$url);
            $retVal = $this->triggerSyncRequest($url);
            return response()->json($retVal);
        }
        if (array_key_exists('crawl',$filter)){
            $productIds = $this->merchantProductService->getProductFromCrawl();
            $data = array_chunk($productIds,400);
            $retVal = [];
            foreach ($data as $chunkIds){
                $filter = [];
                $filter['ids'] = implode(",",$chunkIds);
                $filter['ignore_updated_at'] = 1;
                $routeName = 'ads::ads:merchant::insert';
                $url = route($routeName,$filter);
                $url = str_replace($domain,$queueDomain,$url);
                $retVal[] = $this->triggerSyncRequest($url);
            }
            return response()->json($retVal);
        }
        $product = $this->merchantProductService->getProductEvent(['max' => 1]);
        $maxId = $product->id;
        $lastId = $this->merchantProductService->getOptionByKey('merchant_event_last_id',0);
        if (array_key_exists('lastId',$filter)){
            $lastId = $filter['lastId'];
        }
        if (array_key_exists('maxId',$filter)){
            $maxId = $filter['maxId'];
        }
        $routeName = 'ads::ads:merchant::insert';
        if (array_key_exists('routeName',$filter)){
            $routeName = $filter['routeName'];
        }
        $this->sendToQueue([
            'event' => 1,
            'start' => $lastId,
            'step' => 600,
            'max' => $maxId,
            'filter' => $filter,
            'routeName' => $routeName,
            'domain' => $domain,
            'queueDomain' => $queueDomain
        ]);
        $this->merchantProductService->saveOptionByKey('merchant_event_last_id',$maxId);
        return response()->json(['lastId' => $lastId,'maxId' => $maxId]);
    }
    public function insertSellerPriority(Request $request){
        set_time_limit(60*60*24);
        ini_set("memory_limit", "2048M");
        $filter = $request->all();
        $this->getLocaleData();
        $localeData = $this->getLocaleData();
        $countryCode = $localeData['countryCode'];
        $queueDomain = '';
        $domain = '';
        if (isset($_SERVER['HTTP_HOST'])){
            $domain = $_SERVER['HTTP_HOST'];
            $queueDomain = $domain;
        }
        if (isset($this->queue[$countryCode])){
            $queueDomain = $this->queue[$countryCode];
        }elseif(isset($this->queue['DEFAULT'])){
            $queueDomain = $this->queue['DEFAULT'];
        }

        $sellerIdConfiguration = $this->merchantProductService->getOptionByKey('offsiteAdsSellerIds','{}');
        $sellerIds = json_decode($sellerIdConfiguration,true);
        if (!empty($sellerIds)) {
            $sellerIds = implode(',', $sellerIds);
            if (array_key_exists('lastId', $filter)) {
                $lastId = $filter['lastId'];
            } else {
                $lastId = 0;
            }
            if (array_key_exists('maxId', $filter)) {
                $maxId = $filter['maxId'];
            } else {
                $maxId = $this->getProductMaxId();
            }
            $routeName = 'ads::ads:merchant::post::insert';
            if (array_key_exists('routeName', $filter)) {
                $routeName = $filter['routeName'];
            }
            $filter['sellerIds'] = $sellerIds;
            $filter['seller'] = 1;
            $step = 10000;
            if (array_key_exists('step',$filter)){
                $step = $filter['step'];
            }
            $this->sendToQueue([
                'event' => 1,
                'start' => $lastId,
                'step' => $step,
                'method' => 'POST',
                'max' => $maxId,
                'filter' => $filter,
                'routeName' => $routeName,
                'domain' => $domain,
                'queueDomain' => $queueDomain
            ]);
        }
        return response()->json(['lastId' => $lastId,'maxId' => $maxId]);
    }

    private function sendToQueue($input){
        $start = $input['start'];
        $step = $input['step'];
        $max = $input['max'];
        $filter = $input['filter'];
        $routeName = $input['routeName'];
        $domain = $input['domain'];
        $queueDomain = $input['queueDomain'];
        if (array_key_exists('event',$input)){
            for($i = $start; $i <= $max+$step; $i = $i+$step){
                $query = $this->merchantProductService->getProductEvent(['id_from' => $i, 'id_to' => $i+$step]);
                $ids = $query->pluck('product_id')->toArray();
                if (empty($ids)){
                    continue;
                }
                $filter['ids'] = implode(",",$ids);
                $url = route($routeName,$filter);
                $url = str_replace($domain,$queueDomain,$url);
                $this->triggerSyncRequest($url);
            }
        }else{
            for($i = $start; $i <= $max+$step; $i = $i+$step){
                $filter['product_id_from'] = $i;
                $filter['product_id_to'] = $i+$step;
                if (!empty($input['method']) && $input['method'] == 'POST'){
                    $url = route($routeName);
                    $url = str_replace($domain,$queueDomain,$url);
                    $this->triggerSyncRequest($url,'POST',$filter);
                }else{
                    $url = route($routeName,$filter);
                    $url = str_replace($domain,$queueDomain,$url);
                    $this->triggerSyncRequest($url);
                }

            }
        }
    }

    public function getListExpired(Request $request){
        $timeLimit = 60*30;
        $input = $request->all();
        if (array_key_exists('time_limit',$input)){
            $timeLimit = $input['time_limit'];
        }
        set_time_limit($timeLimit);
        ini_set("memory_limit", "3072M");
        $localeData = $this->getLocaleData();
        $countryCode = $localeData['countryCode'];
        $parameters = ['maxResults' => 250];
        $pageToken = getOption('statusPageToken','');
        if (!empty($pageToken)){
            $parameters['pageToken'] = $pageToken;
        }


        $merchant_id = getOption('merchant_id','398409240');
        $retVal = [];
        do {
            $this->client = $this->getClient([]);
            $this->service = new \Google_Service_ShoppingContent($this->client);

//            try{
                $response = $this->service->productstatuses->listProductstatuses(
                    $merchant_id, $parameters);

                $data = [];
                $ids = [];
                foreach ($response->getResources() as $product) {
                    $expire = $product->getGoogleExpirationDate();
                    $e =  strtotime($expire);
                    $now = strtotime('+3 days');
                    if ($e >= $now){
                        continue;
                    }
                    $code = $product->getProductId();
                    if (!empty($code)){
                        preg_match('/(\w{2}):(P(\d+).*)/', $code, $matches);
                        if (isset($matches[1])){
                            if ($matches[1] != $countryCode){
                                continue;
                            }
                        }else{
                            continue;
                        }
                        if (isset($matches[3])) {
                            $id = $matches[3];
                            $sku = $matches[2];
                            $ids[] = $id;
                            $retVal[] = $id;
                            $isExists = DB::table('merchant_product')->where('product_id', $id)->exists();
                            if (!$isExists){
                                $data[] = ['product_id' => $id, 'sku' => $sku, 'created_at' => date('Y-m-d H:i:s'),'updated_at' => date('Y-m-d H:i:s')];
                            }
                        }
                    }
                }
                if (count($data) > 0){
                    DB::table('merchant_product')->insert($data);
                }
                if (!empty($ids)){
                    $queueDomain = '';
                    $domain = '';
                    if (isset($_SERVER['HTTP_HOST'])){
                        $domain = $_SERVER['HTTP_HOST'];
                        $queueDomain = $domain;
                    }
                    if (isset($this->queue[$countryCode])){
                        $queueDomain = $this->queue[$countryCode];
                    }elseif(isset($this->queue['DEFAULT'])){
                        $queueDomain = $this->queue['DEFAULT'];
                    }
                    $routeName = 'ads::ads:merchant::insert';
                    $url = route($routeName,['ids'=> implode(',', $ids),'ignore_update_at' => 1,'priority' => 1]);
                    $url = str_replace($domain,$queueDomain,$url);
                    $this->triggerSyncRequest($url);
                    $routeName = 'ads::ads:merchant::delete';
                    $url = route($routeName,['ids'=> implode(',', $ids),'ignore_update_at' => 1, 'skip_ads' => 'on','priority' => 1]);
                    $url = str_replace($domain,$queueDomain,$url);
                    $this->triggerSyncRequest($url);
                }
                $parameters['pageToken'] = $response->nextPageToken;
                $this->merchantProductService->saveOptionByKey('statusPageToken',$parameters['pageToken']);
//            }catch (\Exception $ex){
//                $this->client = $this->getClient([]);
//                $this->service = new \Google_Service_ShoppingContent($this->client);
//                echo $ex->getTraceAsString();
//            }
        } while (!empty($parameters['pageToken']));
        return response()->json($retVal);
    }

    public function hardDelete(Request $request){
        die;
        $timeLimit = 60*30;
        $limit = 600;
        $input = $request->all();
        if (array_key_exists('time_limit',$input)){
            $timeLimit = $input['time_limit'];
        }
        if (array_key_exists('limit',$input)){
            $limit = $input['limit'];
        }
        set_time_limit($timeLimit);
        ini_set("memory_limit", "3072M");
        $fileNames = $input['fileNames'];
        $merchantId = $input['merchantId'];
        $language = $input['language'];
        $country = $input['country'];
        foreach ($fileNames as $fileName){
            $filePath = public_path("merchantUpload/$fileName");
            try{
                $fileHandle = fopen($filePath, "r");
            }catch (\Exception $ex){
                $fileHandle = FALSE;
            }
            if ($fileHandle !== FALSE){
//                $limit = 600;
                //Set up a variable to hold our current position in the file
                $offset = 0;
                while(!feof($fileHandle))
                {
                    $stream = fopen('php://temp', 'w');
                    //Go to where we were when we ended the last batch
                    fseek($fileHandle, $offset);
                    $i = 0;
                    $ids = [];
                    $data = [];
                    while (($currRow = fgetcsv($fileHandle,0, "\t")) !== FALSE)
                    {
                        $i++;
                        $row = $currRow;
                        preg_match('/^P(\d+)/', $row[1], $matches);
                        if (isset($matches[1])) {
                            $id = $matches[1];
//                            $isExists = DB::table('merchant_product')->where('product_id',$id)->exists();
//                            if ($isExists){
//                                $ids[] = $id;
//                            }else{
                                $data[$id] = [
                                    'language' => $language,
                                    'country' => $country,
                                    'product_code' => $row[1],
                                    'id' => $id,
                                    'merchantProductId' => $id,
                                    'merchantId' => $merchantId,
                                ];
//                            }
//                            if (preg_match('/USD/',$row[5])) {
//                                $ids[] = $id;
//                                $insert .= "($id, $row[1], '2024-09-25 11:00:00', '2024-09-25 12:00:00'),\n";
//                                $has = true;
//                            }

                        }

                        //If we hit our limit or are at the end of the file
                        if($i >= $limit)
                        {
                            //Update our current position in the file
                            $offset = ftell($fileHandle);

                            //Break out of the row processing loop
                            break;
                        }
                    }
                    $this->delete($data,[$country],[]);
                    if (!empty($ids)){
                        $deleteUrl =  "curl 'https://printerval.com/module/ads/merchant/delete?ignore_updated_at=1&hard_delete=on&ids=" . implode(',', $ids) . "'";
                        file_put_contents($filePath.'.sh', $deleteUrl, FILE_APPEND);
                        file_put_contents($filePath.'.sh', PHP_EOL, FILE_APPEND);
                        $insertUrl =  "curl 'https://printerval.com/module/ads/merchant/insert?ignore_updated_at=1&ids=" . implode(',', $ids) . "'";
                        file_put_contents($filePath.'.sh', $insertUrl, FILE_APPEND);
                    }
                }
                fclose($fileHandle);

            }
        }

    }
    public function getProductMaxId(){
        $product = $this->merchantProductService->getProduct(['max' => 1]);
        return $product->id;
    }

    public function getProductHighClickPotential(){
        set_time_limit(1200);
        try {
            $potentialType = ['HIGH' => 1, 'MEDIUM' => 2];
            $response = ["status" => "successful"];
            $data = [];
            $parameters = ['maxResults' => 5000];
            $merchantId = getOption('merchant_id','398409240');
            $searchRequest = new SearchRequest();
            $searchRequest->setPageSize($parameters['maxResults']);
            $query = new QueryBuilder();
            $query = $query->from('ProductView')
                ->where('product_view.currency_code', '=',"'$this->currencyCode'")
                ->where('product_view.language_code', '=',"'$this->languageCode'")
                ->orderBy('product_view.click_potential_rank', 'asc')
                ->build([
                    'product_view.id',
                    'product_view.offer_id',
                    'product_view.click_potential_rank',
                    'product_view.click_potential',
                ]);
            $searchRequest->setQuery($query);
            $i = 0;
            $isBreak = false;
            do {
                $i++;
                if ($i%100 == 0){
                    $this->client = $this->getClient([]);
                    $this->service = new \Google_Service_ShoppingContent($this->client);
                }
                $report = $this->service->reports->search($merchantId,$searchRequest);
                $result = $report->getResults();
                $ids = [];
                foreach ($result as $item) {
                    $product = $item->getProductView();
                    $productId = $this->getProductIdFromSku($product->getOfferId());
                    $potential = $product->clickPotential;
                    $ids[] = $productId;
                    if (!isset($potentialType[$potential])){
                        $isBreak = true;
                        break;
                    }
                }
                if (!empty($ids)){
                    DB::table('product')->whereIn('id', $ids)->update(['barcode' => 'high_click_potential']);
                    $chunkedIds = array_chunk($ids, 500);
                    foreach ($chunkedIds as $productIds){
                        $updateUrl = route('ads::ads:merchant::insert',['ids' => implode(',', $productIds),'ignore_updated_at' => '1']);
                        $this->triggerAsyncRequest($updateUrl);
                    }
                }
                $data = array_merge($data,$ids);
                if (count($data) >= 5000 || $isBreak){
                   break;
                }
                $parameters['pageToken'] = $report->nextPageToken;
                $searchRequest->setPageToken($parameters['pageToken']);
//                usleep(600000);
                \Log::error("getProductHighClickPotential page " . $i);
            } while (!empty($parameters['pageToken']));
            $this->merchantProductService->saveOptionByKey('high_click_potential',json_encode($data));
            $response['result'] = $data;
            $response['count'] = count($data);
        } catch (Exception $exc) {
            $response = ["status" => "fail", "message" => $exc->getMessage()];
        }
        return response()->json($response);
    }

    public function lookForProductInfoMissing(Request $request){
        $filter = $request->all();
        $products = $this->merchantProductService->getProductMissingProductInfo($filter);
        $this->merchantProductService->insertProductMissingInfo($products);
        $deletedCount = $this->merchantProductService->deleteProductMissingInfo();
        return response()->json(['status' => 'successful','saved' => count($products), 'deleted' => $deletedCount]);
    }

    public function updateExpiredProduct(Request $request){
        $input = $request->all();
        $timeLimit = 60*30;
        if (array_key_exists('time_limit',$input)){
            $timeLimit = $input['time_limit'];
        }
        set_time_limit($timeLimit);
        ini_set("memory_limit", "3072M");
        $filter = [];
        if(array_key_exists('product_id_from',$input)){
            $filter['product_id_from'] = $input['product_id_from'];
        }
        if(array_key_exists('product_id_to',$input)){
            $filter['product_id_to'] = $input['product_id_to'];
        }
        if(array_key_exists('updated_at_from',$input)){
            $filter['updated_at_from'] = $input['updated_at_from'];
        }
        if(array_key_exists('updated_at_to',$input)){
            $filter['updated_at_to'] = $input['updated_at_to'];
        }
        if(array_key_exists('ids',$input)){
            $filter['ids'] = $input['ids'];
        }
        $products = $this->merchantProductService->getProductExpired($filter);
        $localeData = $this->getLocaleData();
        $countryCode = $localeData['countryCode'];
        $inputBuildProduct = [
            'country' => $countryCode,
            'language' => isset($this->languages[$countryCode]) ? $this->languages[$countryCode] : env('APP_LANG')];
        if (isset($input['merchant_id'])){
            $inputBuildProduct['merchant_id'] = $input['merchant_id'];
        }
        $data = $this->merchantProductService->buildProductExpired($products,$inputBuildProduct);
        $count = count($data);
        $dataChunk = array_chunk($data,10000);
        $message = [];
        foreach ($dataChunk as $chunkItem){
            $message[] = $this->updateByService($chunkItem,['updateMask' => 'availabilityDate','country' => $countryCode]);
        }
        return response()->json(['status' => 'successful', 'result' => $count, 'message' => $message]);

    }



}
