<?php

namespace Modules\Ads\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Ads\Controllers\Controller;
use Modules\Ads\Models\ApproveAdvertising;
use Modules\Ads\Models\Category;
use Modules\Ads\Models\GoogleProductTaxonomy;
use Modules\Ads\Models\Option;
use Modules\Ads\Models\Order;
use Modules\Ads\Models\Product;
use Modules\Ads\Models\ProductMeta;
use Modules\Ads\Models\ProductNCategory;
use Modules\Ads\Models\ProductSku;
use Modules\Ads\Models\Tag;
use Modules\Ads\Models\TagRefer;

class ProductController extends Controller
{
    static $cacheCategory = [];
    public function feed(Request $request)
    {
        set_time_limit(60*120);
        $setting = $request->input('setting', []);

        return $this->getFeed($setting);
    }

    public function getFeed($setting, $productHadOrderSkuIds = [])
    {
        set_time_limit(60*120);
        ini_set('memory_limit','10240M');
        $productsFeed = [];

        // load shop information
        $storeInfo = [];
        foreach (Option::all() as $option) {
            $value = json_decode($option['value']);
            if ($value) {
                $storeInfo[$option['key']] = $value;
            } else {
                $storeInfo[$option['key']] = $option['value'];
            }
        }
        $productsFeed['channel'] = $storeInfo;


        // load google categories data
        if (!empty($setting)) {
            // lấy sản phẩm theo danh mục
            $productIds = [];
            // get raw products
            $products = $this->getRawProducts($setting, $productIds, $productHadOrderSkuIds);

            // group product
            $groups = Category::select(['id', 'name'])
                ->where('type', 'GROUP')
                ->get();
            $productsNGroups = [];
            unset($groups);
            $productsMetaGroup = DB::table('product_meta')
                ->whereNotNull('value')
                ->where('key', 'ads_group')
                ->get(['product_id', 'key', 'value']);
            foreach ($productsMetaGroup as $value) {
                $productsNGroups[$value->product_id] = $this->getGroupCategory($value->value);
            }

            //@todo đang ít nên cho phép build tất ~100 bản ghi
            $groupProductsMetas = [];
            $productsMetas = ProductMeta::whereIn('key', ['ads_option', 'product_react_ads_status'])
                ->get(['product_id', 'key', 'value'])
                ->toArray();
            foreach ($productsMetas as $value) {
                $groupProductsMetas[$value['product_id']][] = $value;
            }

            foreach ($products as &$product) {
                // map product group
                if(isset($productsNGroups[$product->id])) {
                    $product->group = $productsNGroups[$product->id];
                }
                // build sku_specific_info
                if(isset($product->sku_specific_info)) { // Size: M, Color: Black
                    $this->buildSpecificInfo($product);
                }
                // map product meta
                if(isset($groupProductsMetas[$product->id])) {
                    $product->meta = $groupProductsMetas[$product->id];
                }
            }
            $productsFeed['items'] = $products;
        }

        return $productsFeed;
    }

    protected function getAdsGroup($productId) {
        $meta = $this->getProductMetaByKey($productId, 'ads_group');
        $result = null;
        if ($meta && isset($meta->value)) {
            $result =  $this->getGroupCategory($meta->value);
        }
        return $result;
    }

    protected function getProductMetaByKey($productId, $key, $columns = ['value']) {
        return DB::table('product_meta')
            ->where('product_id', $productId)
            ->where('key', $key)
            ->first($columns);
    }

    protected function getGroupCategory($id) {
        if (!array_key_exists($id, self::$cacheCategory)) {
            $category = Category::select(['id', 'name'])
                ->where('type', 'GROUP')
                ->where('id', $id)
                ->first();
            self::$cacheCategory[$id]  = null;
            if ($category) {
                self::$cacheCategory[$id] = [
                    'id' => $category->id,
                    'name' => $category->name,
                ];
            }
        }
        return self::$cacheCategory[$id];


    }

    protected function groupTags() {
        $tags = Tag::select(['id', 'title', 'slug'])->get();
        $tagRefers = TagRefer::where('refer_type', TagRefer::REFER_PRODUCT)
            ->orderBy('id', 'desc')
            ->select(['tag_id', 'refer_id'])
            ->get();

        $tags = $tags->toArray();
        $groupTagRefers = [];
        $arrayTags = [];
        foreach ($tags as $tag) {
            $arrayTags[$tag['id']] = $tag;
        }
        unset($tags);
        foreach ($tagRefers as $value) {
            if (!array_key_exists($value->tag_id, $arrayTags) || array_key_exists($value->refer_id, $groupTagRefers)) {
                continue;
            }
            $groupTagRefers[$value->refer_id] = [];
            $groupTagRefers[$value->refer_id][] = $arrayTags[$value->tag_id];
        }
        return $groupTagRefers;
    }

    private function getRawProducts($setting, $productIds = [], $skuIds = [])
    {
        //@todo product-info only default
        $columns = [
            // ORIGIN
            'product.id', 'product.sku', 'product.name',
            'product.slug', 'product.brand_id', 'product.image_url',
            'product.price',
            'product.description',
            'product.status', 'product.approve_advertising',
            'product.id as product_id',
            'product.is_trademark',
            'product.gtin',
            'product.is_always_on_ads',
            'product.actor_id',
            'product.created_at',
            'product.sold',
            // SKU
            'product_info.name as sku_name',
            'product_info.variants as sku_specific_info',
            'product_info.brand as sku_brand',
            'product_info.sku as sku_code',
            'product_info.image_url as sku_image_url',
            'product_info.slug as sku_slug',
            'product_info.price as sku_price',
            'product_info.inventory',
            'product_info.high_price as sku_high_price',
            'product_info.category_merchant_id as sku_category_merchant_id',
            'product_info.product_sku_id',
            'product_info.is_default',
            'category.name as category_name',
            'category.id as category_id'
        ];

        if (isset($productIds) && $productIds && count($productIds) > 0) {
            $setting['productIds'] = $productIds;
        }
        if($setting['type'] != 'shopping_feed' && $skuIds) {
            $setting['skuIds'] = $skuIds;
        }

        $result = $this->buildProductRawQuery($setting)->get($columns)->all();
        return $result;

    }

    protected function buildProductRawQuery($filter) {
        $productIds = isset($filter['productIds']) ? $filter['productIds'] : [];
        $skuIds = isset($filter['skuIds']) ? $filter['skuIds'] : [];
        $productQuery = DB::table('product');
        if (isset($filter['type']) && $filter['type'] == 'shopping_feed') {
            $productQuery->where('product.approve_advertising', 1);
        }
        if ($productIds && count($productIds) > 0) {
            $productQuery->whereIn('product.id', $productIds);
        }
        if (isset($filter['product_id_from']) && isset($filter['product_id_to'])){
            if(!empty($filter['product_id_from'])){
                $productQuery->where('product.id','>=',$filter['product_id_from']);
            }
            if(!empty($filter['product_id_to'])){
                $productQuery->where('product.id','<=',$filter['product_id_to']);
            }
        }
        $productQuery->where('product.status', 'ACTIVE');
        $productQuery->join('product_info', 'product.id', 'product_info.product_id');
        $productQuery->join('product_n_category', 'product_n_category.product_id', 'product.id');
        $productQuery->where('product_n_category.is_parent', 0);
        $productQuery->join('category', 'product_n_category.category_id', 'category.id');
        if (isset($filter['category_id']) && $filter['category_id'] > 0) {
            $category = Category::where('id', $filter['category_id'])->first();
            $categoryIds = [$category->id];

            if ($category->descendants) {
                foreach ($category->descendants as $descendant) {
                    array_push($categoryIds, $descendant->id);
                }
            }
            $productQuery->whereIn('product_n_category.category_id', $categoryIds);
        }

        if($filter['type'] == 'shopping_feed') {
            $productQuery->where('product_info.is_default', '=', 1);
            $productQuery->where('product.is_violation', 0);

        }

        if (isset($filter['check_paid']) && $filter['check_paid']){
            $productQuery->where('product.sold', '>', 0);
        }

        // check trademark // {"title":[],"description":[],"vector":[]}
        $productQuery->where('product.is_trademark', 0);
        $productQuery->groupBy('product.id');

        return $productQuery;
    }

    private function buildProductNotVariant($filter){
        //@todo cần build cả những sản phẩm không có biến thể
        $query = DB::table('product')
            ->where('product.approve_advertising', 1)
            ->where('product.status', 'ACTIVE')
            ->join('product_info', 'product.id', 'product_info.product_id')
            ->where('product_info.product_sku_id', 0)
            ->join('product_n_category', 'product_n_category.product_id', 'product.id')
            ->where('product_n_category.is_parent', 0)
            ->join('category', 'product_n_category.category_id', 'category.id')
            ->where('product.is_violation', 0)
            ->where('product.is_trademark', 0);
        if (isset($filter['check_paid']) && $filter['check_paid']){
            $query->where('product.sold', '>', 0);
        }
        if (isset($filter['product_id_from']) && isset($filter['product_id_to'])){
            if(!empty($filter['product_id_from'])){
                $query->where('product.id','>=',$filter['product_id_from']);
            }
            if(!empty($filter['product_id_to'])){
                $query->where('product.id','<=',$filter['product_id_to']);
            }
        }
        $query->groupBy('product.id');
        return $query;

    }

    private function buildSpecificInfo(&$product)
    {
        $product->specific_info = [];

        $specs = explode(',', $product->sku_specific_info);
        for ($i = 0; $i < count($specs); $i++) {
            $spec = $specs[$i];
            $specPair = explode(':', $spec);

            if(count($specPair) === 2) {
                for ($j = 0; $j < count($specPair); $j++) {
                    $specPair[$j] = trim($specPair[$j]);
                }

                $specPair = array_values($specPair);
                $product->specific_info[$specPair[0]] = $specPair[1];
            }
        }
    }

    public function getApproveProducts($filter)
    {
        $query = ApproveAdvertising::where('advertising_status', 1);
        if (isset($filter['product_id_from']) && isset($filter['product_id_to'])){
            if(!empty($filter['product_id_from'])){
                $query->where('product_id','>=',$filter['product_id_from']);
            }
            if(!empty($filter['product_id_to'])){
                $query->where('product_id','<=',$filter['product_id_to']);
            }
        }
        $retVal = $query->pluck('title', 'product_id')->toArray();
        return $retVal;
    }

    public function bulidSkuCode()
    {
        $productSkus = ProductSku::all();
        foreach ($productSkus as $item) {
            if (!isset($item->sku) || !$item->sku) {
                ProductSku::where('id', $item->id)
                    ->update(['sku' => initSkuCode($item->product_id)]);
            }
        }

        return [
            count($productSkus)
        ];
    }

    public function getProductByStatusOrder($conditions)
    {
        $productsHadOrder = Order::select([
            'order_item.product_id',
            'order_item.product_sku_id'
        ]);

        foreach ($conditions as $column => $condition) {
            $productsHadOrder->where($column, $condition);
        }

        $productsHadOrder->join('order_item', 'order_item.order_id', '=', 'order.id');
        return $productsHadOrder->get();

    }

    public function buildProductAdvertising(Request $request) {
        $products = Product::join("product_approve_advertising as paa", function($join) {
            $join->on("product.id", "=", "paa.product_id");
            $join->on("product.name", "!=", "paa.title");
        })
            ->select(["product.name", "product.id", "paa.id as paid", "paa.title"])
            ->get();
        $productAdvertisingIds = [];
        foreach ($products as $key => $value) {
            ApproveAdvertising::where("id", $value->paid)->update(["title" => $value->name]);
            $productAdvertisingIds[] = $value->id;
        }
        $reponse = [
            "status" => "successful",
            "result" => $productAdvertisingIds
        ];
        return response()->json($reponse);
    }

    public function buildProductAdvertisingSku(Request $request) {
        set_time_limit(30 * 60);
        ini_set('memory_limit','256M');
        $productAdvertisingIds = [];
        $updatedTimeFrom = $request->get('updatedTimeFrom');
        $fromId = $request->get('fromId');
        $toId = $request->get('toId');
        if (!$updatedTimeFrom && (!$fromId || !$toId) ) {
            return;
        }
        $brand = \DB::table("brand")->where("id", 1)->first();
        $query = \DB::table("product")
            ->leftJoin("product_approve_advertising as paa", function($join) {
                $join->on("product.id", "=", "paa.product_id");
            })
            ->where("product.approve_advertising", 1);
        if ($updatedTimeFrom) {
            $query->where("product.updated_at", '>=', $updatedTimeFrom);
        }
        if ($fromId) {
            $query->where("product.id", '>=', $fromId);
        }
        if ($toId) {
            $query->where("product.id", '<=', $toId);
        }
        $query->orderBy("product.id")
            ->select(["product.name", "product.sku", "product.id", "product.description",
                    "paa.id as paid", "paa.code as pacode", "product.brand_id",
                    "product.approve_advertising"])
            ->chunk(100, function($products) use (&$productAdvertisingIds, $brand) {
                foreach ($products as $key => $value) {
                    $defaultVariant = DB::table('product_sku')
                        ->where("product_id", $value->id)
                        ->where("is_default", 1)
                        ->first(["sku"]);
                    $productBrand = "";
                    if (isset($brand) && isset($brand->name)) {
                        $productBrand = $brand->name;
                    }
                    $sku = $value->sku;
                    if (isset($defaultVariant) && isset($defaultVariant->sku)) {
                        $sku = $defaultVariant->sku;
                    }
                    if (!isset($value->paid)) {
                        ApproveAdvertising::insert([
                            "code" => $sku,
                            "title" => $value->name,
                            "description" => $value->description,
                            "brand" => $productBrand,
                            "product_id" => $value->id,
                            "advertising_status" => $value->approve_advertising,
                            "created_at" => new \Datetime(),
                            "updated_at" => new \Datetime(),
                        ]);
                        $productAdvertisingIds[] = $value->id;
                    } else {
                        ApproveAdvertising::where("id", $value->id)->update([
                            "code" => $sku,
                            "advertising_status" => $value->approve_advertising,
                        ]);
                        $productAdvertisingIds[] = $value->id;
                    }
                }
            });


        $reponse = [
            "status" => "successful",
            "result" => $productAdvertisingIds
        ];
        return response()->json($reponse);
    }

    public function getGoogleProductTaxonomy() {
        $query = DB::table('google_product_taxonomy')
            ->select(['product_taxonomy_id', 'product_taxonomy_name']);

        $productTaxonomy = $query->get();

        $productTaxonomyMapId = [];
        foreach ($productTaxonomy as $taxonomy) {
            $productTaxonomyMapId[$taxonomy->product_taxonomy_id] = $taxonomy->product_taxonomy_name;
        }

        return $productTaxonomyMapId;
    }

    public function getProducts($filter, $columns, $callBack = null){
        ini_set('memory_limit','1024M');
        set_time_limit(60*120);
        $productQuery = DB::table('product')
            ->join('product_approve_advertising','product.id','=','product_approve_advertising.product_id')
            ->where('product.status', 'ACTIVE')
            ->where('product.is_trademark', 0)
            ->where('product.is_violation', 0);
        if (isset($filter['check_paid']) && $filter['check_paid']){
            $productQuery->where('product.sold', '>', 0);
        }
        if (isset($filter['product_id_from']) && isset($filter['product_id_to'])){
            if(!empty($filter['product_id_from'])){
                $productQuery->where('product.id','>=',$filter['product_id_from']);
            }
            if(!empty($filter['product_id_to'])){
                $productQuery->where('product.id','<=',$filter['product_id_to']);
            }
        }
        if (isset($filter['updated_from'])){
            $productQuery->where('product.updated_at','>=',$filter['updated_from']);
        }
        if (isset($filter['updated_to'])){
            $productQuery->where('product.updated_at','<=',$filter['updated_to']);
        }
        if (isset($filter['joinCategory']) && $filter['joinCategory']){
            $productQuery->join('product_n_category','product.id','=','product_n_category.product_id')
                ->join('category','category.id','=','product_n_category.category_id')
                ->where('product_n_category.is_parent', 0);
        }
        $productQuery->groupBy('product.id');
        $productQuery->select($columns);
        $productQuery->orderBy('product.updated_at');
        if (!empty($callBack)){
            $productQuery->chunk(5000, function ($products) use ($callBack) {
                return $callBack($products);
            });
        }else{
            return $productQuery->get();
        }


    }

    public function getProductMeta(){
        $groupProductsMetas = [];
        $productsMetas = ProductMeta::whereIn('key', ['ads_option', 'product_react_ads_status'])
            ->get(['product_id', 'key', 'value'])->toArray();
        foreach ($productsMetas as $value) {
            $groupProductsMetas[$value['product_id']][] = $value;
        }
        return $groupProductsMetas;
    }

    public function filterProduct($filter, $columns){
        $query =  DB::table('product')->where('product.status', 'ACTIVE');
        $query->where('is_trademark',0);
        $query->where('is_violation',0);
        if (array_key_exists('ids',$filter)){
            $query->whereIn('id',$filter['ids']);
        }
        if (array_key_exists('id_from',$filter)){
            $query->where('id','>=',$filter['id_from']);
        }
        if (array_key_exists('id_to',$filter)){
            $query->where('id','<=',$filter['id_to']);
        }


        $retVal = $query->get($columns);
        return $retVal;
    }

    public function getProductIdsByTags(){
        $tagIds = explode(',',getOption('eventTagIds','0'));
        return DB::table('product')->join('tag_refer','tag_refer.refer_id','=','product.id')
            ->where('tag_refer.refer_type', 'PRODUCT')
            ->whereIn('tag_refer.tag_id', $tagIds)
            ->where('product.status', 'ACTIVE')
            ->where('product.is_violation',0)
            ->where('product.is_trademark',0)
            ->groupBy('product.id')
            ->pluck('product.id')->toArray();
    }

    public function getProductHasAnOrder($filter){
        $query =  DB::table('order_item');
        if (array_key_exists('product_id_from',$filter)){
            $query->where('product_id','>=',$filter['product_id_from']);
        }
        if (array_key_exists('product_id_to',$filter)){
            $query->where('product_id','<=',$filter['product_id_to']);
        }
        $query->where('created_at','>=',date('Y-m-d', strtotime("-30 days")));
        $query->groupBy('product_id');
        $retVal = $query->pluck('product_id')->toArray();
        return $retVal;
    }
    public function getSoldProductIdsClone($filter){
        $query =  DB::table('product_meta');
        if (array_key_exists('product_id_from',$filter)){
            $query->where('product_id','>=',$filter['product_id_from']);
        }
        if (array_key_exists('product_id_to',$filter)){
            $query->where('product_id','<=',$filter['product_id_to']);
        }
        $query->where('key','product_has_order');
        $query->groupBy('product_id');
        $retVal = $query->pluck('product_id')->toArray();
        return $retVal;
    }
    public function getStopAdsProductIds($filter){
        $query =  DB::table('product_disapprove_advertising');
        if (array_key_exists('product_id_from',$filter)){
            $query->where('product_id','>=',$filter['product_id_from']);
        }
        if (array_key_exists('product_id_to',$filter)){
            $query->where('product_id','<=',$filter['product_id_to']);
        }
        $query->groupBy('product_id');
        $retVal = $query->pluck('product_id')->toArray();
        return $retVal;
    }


    public function cancelApprovalProductByCategory(Request $request){
        set_time_limit(60*30);
        ini_set("memory_limit", "3072M");
        $retVal = [];
        $categoryId = $request->get('categoryId');
        $targetNumber = $request->get('targetNumber',0);
        $retVal['categoryId'] =$categoryId;
        $retVal['targetNumber'] =$targetNumber;
        $count = 0;
        if (!empty($categoryId) && !empty($targetNumber)){
            $maxProductId = DB::table('merchant_product')->max('product_id');
            $step = 100000;
            $isBreak = false;
            for($i = 0; $i<= $maxProductId+$step; $i+= $step){
                $products = DB::table('merchant_product')
                    ->join('product','product.id','=','merchant_product.product_id')
                    ->join('product_n_category','product.id','=','product_n_category.product_id')
                    ->where('product.status','active')
                    ->where('product.is_violation',0)
                    ->where('product.is_trademark',0)
                    ->where('product.approve_advertising',1)
                    ->where('product.sold','<',1)
                    ->where('product.view_count','<',1)
                    ->where('product.id','>=', $i)
                    ->where('product.id','<=', $i+$step)
                    ->where('product_n_category.category_id',$categoryId)
                    ->orderBy('product.id','asc')
                    ->get(['product.id','merchant_product.sku']);
                if ($products->count() > 0){
                    foreach ($products as $product){
                        $this->cancelApproval($product,$categoryId);
                        $count ++;
                        if ($count >= $targetNumber){
                            $isBreak = true;
                            break;
                        }
                    }
                }
                if ($isBreak){
                    break;
                }
            }
        }
        $retVal['count'] = $count;
        \Log::info("Google Merchant Cancel Approval". json_encode($retVal));
        return response()->json(['status' => 'successful', 'result' => $retVal]);
    }

    public function limitProductGMC(Request $request){
        $retVal = [];
        $limitConfig = getOptionByDomain('merchant_limit_product_category');
        if (!empty($limitConfig)){
            $categoryConfig = getOptionByDomain('merchant_count_product_category');
            if (is_object($categoryConfig)){
                $categoryConfig = get_object_vars($categoryConfig);
            }
            if (is_object($limitConfig)){
                $limitConfig = get_object_vars($limitConfig);
            }

            foreach ($categoryConfig as $categoryId => $number){
                $limitNumber = isset($limitConfig[$categoryId])?$limitConfig[$categoryId]:$number;
                $targetNumber = $number - $limitNumber;
                if ($targetNumber <= 0){
                    continue;
                }
                $url = route('ads::product::cancel-approval-product-by-category',['categoryId' => $categoryId, 'targetNumber' =>$targetNumber]);
                $this->triggerAsyncRequest($url);
                $retVal[$categoryId] = $targetNumber;
            }
        }
        return response()->json(['status' => 'successful', 'result' => $retVal]);
    }

    private function logProductUpdated($product, $categoryId, $createdAt){
        $insertData = [];
            $insertData[] = [
                'category_id' => $categoryId,
                'product_id' => $product->id,
                'sku' => $product->sku,
                'created_at' => $createdAt
            ];
        DB::table('ads_log_cancel_approval')->insert($insertData);
    }

    private function cancelApproval($product,$categoryId){
        $now = date('Y-m-d H:i:s');
        DB::table('product')->where('id',$product->id)->update(['approve_advertising'=> 0, 'updated_at' => $now]);
        $this->logProductUpdated($product,$categoryId,$now);
    }
    public function unapproveProductGMC(Request $request){
        $createdAtFrom = date('Y-m-d 00:00:00',strtotime('-366 days'));
        $createdAtTo = date('Y-m-d 23:59:59',strtotime('-366 days'));
        $productIds = DB::table('product')->join('merchant_product','product.id','=','merchant_product.product_id')
            ->where('product.status','active')
            ->where('product.is_trademark',0)
            ->where('product.is_violation',0)
            ->where('product.approve_advertising',1)
            ->where('product.sold',0)
            ->where('product.view_count','<=',200)
            ->where('product.created_at','>=',$createdAtFrom)
            ->where('product.created_at','<=',$createdAtTo)
            ->pluck('product.id')->toArray();
            DB::table('product')->whereIn('id',$productIds)->update(['approve_advertising'=> 0]);
            $chunkedArray = array_chunk($productIds, 700);
            foreach ($chunkedArray as $ids){
                $url = route('ads::ads:merchant::delete',['ignore_updated_at' => 1, 'ids' =>implode(',',$ids)]);
                $this->triggerAsyncRequest($url);
            }

        return response()->json(['status' => 'successful', 'result' => $productIds]);
    }

    /**
     * create merchant product state by day
     */
    public function createMerchantProductState()
    {
        $startTime = date('Y-m-d H:i:s', strtotime('today'));
        $allProductCount = DB::table('merchant_product')
            ->where('is_delete', 0)
            ->count();
        $removedProductCount = DB::table('merchant_product_delete')
            ->where('created_at', '>', $startTime)
            ->count();
        $newProductCount = DB::table('merchant_product')
            ->where('created_at', '>', $startTime)
            ->count();
        $updatedProductCount = DB::table('merchant_product')
            ->where('updated_at', '>', $startTime)
            ->count();
        $date = date('Y-m-d', strtotime('today'));
        $record = DB::table('merchant_product_state')->where('date', $date)->first();
        if ($record) {
            DB::table('merchant_product_state')
                ->where('date', $date)
                ->update(
                    [
                        'all_product' => $allProductCount,
                        'removed_product' => $removedProductCount,
                        'new_product' => $newProductCount,
                        'updated_product' => $updatedProductCount,
                    ]
                );
            
        } else {
            DB::table('merchant_product_state')->insert(
                [
                    'date' => $date,
                    'all_product' => $allProductCount,
                    'removed_product' => $removedProductCount,
                    'new_product' => $newProductCount,
                    'updated_product' => $updatedProductCount,
                ]
            );
        }
            
        return response()->json(['status' => 'successful']);
    }

    /**
     * get merchant product state
     */
    public function getMerchantProductStates(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');
        if (empty($startDate)) {
            $startDate = date('Y-m-d', strtotime("today"));
        }
        if (empty($endDate)) {
            $endDate = date('Y-m-d', strtotime("-7 day"));
        }
        $result = DB::table('merchant_product_state')
            ->where('date', '>=', $startDate)
            ->where('date', '<=', $endDate)
            ->orderBy('date')
            ->get();

        $startTime = date('Y-m-d', strtotime("-90 day"));
        $notPushedProducts = DB::table('merchant_product_not_push')
            ->where('product_created_at', '>', $startTime)
            ->selectRaw('sum(total) as total, sum(not_approve) as not_approve, sum(missing_product_info) as missing_product_info, sum(waiting) as waiting')
            ->get();
        $result = $this->buildDataSets($result, $notPushedProducts);

        $summaryProduct = DB::table('summary_product_date')
            ->join('users', 'summary_product_date.actor_id', '=', 'users.id')
            ->whereBetween('summary_product_date.date', [$startDate, $endDate])
            ->select(
                DB::raw('DATE(sb_summary_product_date.date) as date'),
                DB::raw('SUM(CASE WHEN sb_users.role = "SELLER" THEN count_products ELSE 0 END) as total_seller'),
                DB::raw('SUM(CASE WHEN sb_users.role != "SELLER" THEN count_products ELSE 0 END) as total_non_seller')
            )
            ->groupBy('summary_product_date.date')
            ->orderBy('summary_product_date.date', 'asc')
            ->get();
    
        return response()->json(['status' => 'successful', 'result' => $result, 'summaryProduct' => $summaryProduct]);
    }

    /**
     * gmc feed
     */
    public function gmcFeed(Request $request)
    {
        return view('ads::chart.report');
    }

    /**
     * build data set
     * 
     * @param Collection $data
     * @param Collection $notPushedProducts
     * @return Array
     */
    private function buildDataSets($data, $notPushedProducts)
    {
        $productOnFeed = [];
        $removedProduct = [];
        $newProduct = [];
        $updatedProduct = [];
        $labels = [];

        foreach ($data as $item) {
            $labels[] = $item->date;
            $productOnFeed[] = $item->all_product;
            $removedProduct[] = $item->removed_product;
            $newProduct[] = $item->new_product;
            $updatedProduct[] = $item->updated_product;
        }

        return [
            'categories' => $labels,
            'series' => [
                [
                    'name' => 'On feed',
                    'data' => $productOnFeed,
                ],
                [
                    'name' => 'Removed',
                    'data' => $removedProduct,
                ],
                [
                    'name' => 'New',
                    'data' => $newProduct,
                ],
                [
                    'name' => 'Updated',
                    'data' => $updatedProduct,
                ],
            ],
            'not_pushed_series' => [
                [
                    'name' => 'Approved',
                    'data' => [intval($notPushedProducts[0]->total)]
                ],
                [
                    'name' => 'Not approved',
                    'data' => [intval($notPushedProducts[0]->not_approve)]
                ],
                [
                    'name' => 'Missing info',
                    'data' => [intval($notPushedProducts[0]->missing_product_info)]
                ],
                [
                    'name' => 'Waiting',
                    'data' => [intval($notPushedProducts[0]->waiting)]
                ],
            ]
        ];
    }
}
