<?php

namespace Modules\Ads\Controllers;

use App\Utils\SkuHelper;
use Modules\Ads\Models\Click;
use Modules\Ads\Models\Option;
use Modules\Ads\Models\Order;
use Modules\Ads\Models\Product;
use Modules\Ads\Models\ProductInfo;
use Modules\Ads\Models\ProductReact;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;


class ProductReactService extends Controller
{
    static $globalAdsConfig = null;
    static $globalAdsProductConfig = [];
    static $globalAdsProductStatus = [];

    public function find(Request $request) {
        $filter = $this->buildFilter($request);
        $columns = $filter['columns'];
        $query  = ProductReact::buildProductReactQuery($filter);
        $queryCount = clone $query;
        $queryCount->limit(100000);
        $queryCount->select($columns);
        $count = DB::table(DB::raw("(" . $queryCount->toSql() . ") AS x"))
            ->mergeBindings($queryCount->getQuery())
            ->count();


        $items = $this->getItems($query, $filter);
        $this->decorItems($items, array_merge($filter, [
            'advertising' => 1
        ]));
        $pageCount = $this->recordsCountToPagesCount($count, $filter['page_size']);
        $response = array(
            "status" => 'successful',
            'pagesCount' => $pageCount,
            'pageId' => $filter['page_id'],
            "items" => $items,
            "count" => $count,
        );
        return response()->json($response);
    }

    public function exportProductReact(Request $request) {
        $filter = $this->buildFilter($request);
        $filter['page_size'] = -1;
        $query  = ProductReact::buildProductReactQuery($filter);
        $items = $this->getItems($query, $filter);
        $this->decorItems($items, array_merge($filter, [
            'advertising' => 1
        ]));
        $fileName = 'Bao_cao_tuong_tac_san_pham-' ;
        if (isset($filter['order_date_from']) && isset($filter['order_date_to'])) {
            $fileName = 'Bao_cao_tuong_tac_san_pham'
                . '-' . $filter['order_date_from']->format('Y-m-d')
                . '-' .  $filter['order_date_to']->format('Y-m-d');
        }
        $this->exportProductReactToExcel($items, $fileName);
    }

    protected function exportProductReactToExcel($items, $fileName = '') {
        $fileName = $fileName ? $fileName : 'Bao_cao_tuong_tac_san_pham-' . date("Y-m-d") . time();
        /** Build excel file to export */
        $objPHPExcel = new \PHPExcel();
        /** Setup properties for file excel */
        $objPHPExcel->getProperties()
            ->setCreator("Megaads Technical")
            ->setLastModifiedBy("Megaads Technical")
            ->setTitle("Report Product");
        $objPHPExcel->setActiveSheetIndex(0);
        $objPHPExcel->getActiveSheet()->SetCellValue('A1', "STT");
        $objPHPExcel->getActiveSheet()->SetCellValue('B1', "Mã sản phẩm");
        $objPHPExcel->getActiveSheet()->SetCellValue('C1', "Tên sản phẩm");
        $objPHPExcel->getActiveSheet()->SetCellValue('D1', "Click Tổng");
        $objPHPExcel->getActiveSheet()->SetCellValue('E1', "Click Adwords");
        $objPHPExcel->getActiveSheet()->SetCellValue('F1', "Click Organic");
        $objPHPExcel->getActiveSheet()->SetCellValue('G1', "Click Khác");
        $objPHPExcel->getActiveSheet()->SetCellValue('H1', "Đơn hàng");
        $objPHPExcel->getActiveSheet()->SetCellValue('I1', "Tỉ lệ chuyển đổi");
        $objPHPExcel->getActiveSheet()->SetCellValue('J1', "Tồn kho");
        $objPHPExcel->getActiveSheet()->SetCellValue('K1', "Trạng thái feed");

        $i = 1;
        foreach ($items as $item) {
            $i++;
            $advertisingStatus =  $item['advertising_status'] ? 'On' : 'Off';
            $objPHPExcel->getActiveSheet()->SetCellValue('A' . $i, $i - 1);
            $objPHPExcel->getActiveSheet()->SetCellValue('B' . $i, $item['sku']);
            $objPHPExcel->getActiveSheet()->SetCellValue('C' . $i, $item['name']);
            $objPHPExcel->getActiveSheet()->SetCellValue('D' . $i, $item['total_click']);
            $objPHPExcel->getActiveSheet()->SetCellValue('E' . $i, $item['adwords_click']);
            $objPHPExcel->getActiveSheet()->SetCellValue('F' . $i, $item['organic_click']);
            $objPHPExcel->getActiveSheet()->SetCellValue('G' . $i, $item['other_click']);
            $objPHPExcel->getActiveSheet()->SetCellValue('H' . $i, $item['total_conversion']);
            $objPHPExcel->getActiveSheet()->SetCellValue('I' . $i, $item['rate']);
            $objPHPExcel->getActiveSheet()->SetCellValue('J' . $i, $item['inventory']);
            $objPHPExcel->getActiveSheet()->SetCellValue('K' . $i, $advertisingStatus);
        }

        $writer = \PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel2007');
        ob_end_clean();
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment;filename="' . $fileName . '.xlsx"');
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
    }

    protected function decorItems(&$items, $filter = []) {
        if ($items && count($items) > 0) {
            foreach ($items as &$item) {
                $this->decorProductInfo($item, $filter);
            }
        }
    }

    private function decorAdvertising(&$item) {
        $productId = is_array($item) ? $item['product_id'] : $item->product_id;
        $globalAdsConfig = $this->getGlobalAdsConfig();
        $productAdsConfig = $this->getProductAdsConfig($productId);
        $productAdsStatus = $this->getProductAdsStatus($productId);
        $item['ads_config'] = $globalAdsConfig;
        $item['ads_is_config'] = false;
        if ($productAdsConfig) {
            $item['ads_config'] = $productAdsConfig;
            $item['ads_is_config'] = true;

        }
        $item['advertising_status'] = $productAdsStatus ? $productAdsStatus['status'] : true;
    }



    private function decorProductInfo(&$item, $filter = []) {
        $productId = is_array($item) ? $item['product_id'] : $item->product_id;
        $columns = [
            'name', 'slug', 'image_url', 'sku'
        ];
        if (!isset($item->name)) {
            $product = Product::where('id', $productId)->first($columns);
            if ($product) {
                $product->slug =  (!empty($product->slug) ? $product->slug : "san-pham") . "-p" . $productId;
            }
            foreach ($columns as $column) {
                $val = $product && $product->{$column} ? $product->{$column} : '';
                if (is_array($item)) {
                    $item[$column] = $val;
                } else {
                    $item{$column} = $val;
                }
            }
        }
    }

    protected function getItems($query, $filter)
    {
        $pageId = $filter["page_id"] + 1;
        $pageSize = $filter["page_size"];
        if (array_key_exists('order', $filter)) {
            $order = $filter['order'];
            $query->orderBy($order['column'], $order['direction']);
            if(strtolower($order['direction']) == 'asc') {
                $query->orderBy('product_react.product_id', 'asc');
            }
        }
        if ($pageSize > 0) {
            $query->forPage(($filter["page_id"] + 1), $filter["page_size"]);
        }
        return $query->get($filter['columns']);
    }

    protected function recordsCountToPagesCount($recordsCount, $pageSize) {
        $retVal = (int) ($recordsCount / $pageSize);
        if ($recordsCount % $pageSize > 0) {
            $retVal++;
        }
        //return
        return $retVal;
    }

    protected function buildFilter($request){
        $retVal = [];
        if ($request->input('search')){
            $retVal['search'] = $request->input('search');
        }
        if ($request->input('orderDateFrom')){
            $retVal['order_date_from'] = $this->getDateTimeFilter($request->input('orderDateFrom'));
        }
        if ($request->input('orderDateTo')){
            $toDateTime = $this->getDateTimeFilter($request->input('orderDateTo'));
            $toDateTime->setTime("23", "59", "59");
            $retVal['order_date_to'] = $toDateTime;
        }
        $retVal['order_from'] = $request->input('orderFrom', 0);
        $retVal['click_from'] = $request->input('clickFrom', 0);
        $retVal['adwords_click_from'] = $request->input('adwordsClickFrom', 0);
        $retVal['organic_click_from'] = $request->input('organicClickFrom', 0);
        $retVal['bingads_click_from'] = $request->input('bingadsClickFrom', 0);
        $retVal['inventory_from'] = $request->input('inventoryFrom', 0);
        $retVal['page_size'] = $request->input('pageSize', 100);
        //$retVal['page_size'] = 100000;
        $retVal['page_id'] = $request->input('pageId', 0);
        $retVal['only_product'] = 1;
        $retVal['group'] = 'product_react.product_id';

        $retVal['columns'] = [
            'product_react.product_id', 'product_react.product_sku_id',
            DB::raw('SUM(sb_product_react.conversion) as total_conversion, 
            SUM(sb_product_react.click) as total_click, 
            SUM(sb_product_react.conversion)/SUM(sb_product_react.click) * 100  as rate,
            SUM(sb_product_react.conversion)/SUM(sb_product_react.click) * 100  as product_rate,
            SUM(sb_product_react.adwords_click) as adwords_click,
            SUM(sb_product_react.organic_click) as organic_click,
            SUM(sb_product_react.bingads_click) as bingads_click,
            SUM(sb_product_react.other_click) as other_click')
        ];
        $orderBy = $request->input('order', 'rate,desc');
        $orderInfo = explode(',', $orderBy);
        if ($orderInfo) {
            if (count($orderInfo) == 1) {
                $orderInfo[1] = 'desc';
            }
            $retVal['order'] = [
                'column' => $orderInfo[0],
                'direction' => $orderInfo[1],
            ];
        }

        return $retVal;
    }

    protected function getDateTimeFilter($dateTime, $format = "d/m/Y") {
        $retval = FALSE;
        if (is_object($dateTime)) {
            $retval = clone $dateTime;
        }
        if ($dateTime == NULL || $dateTime == "") {
            $dateTime = "1/1/1900";
        }
        if (is_string($dateTime)) {
            $retval = \DateTime::createFromFormat($format, $dateTime);
        }
        $retval->setTime("00", "00", "00");

        return $retval;
    }


    public function findProductLowRate(Request $request){
        set_time_limit(300);
        ini_set('memory_limit', '-1');
        $filter = $this->buildFilter($request);
        $items = $this->buildProductLowRate($filter);
        $pageCount = $this->recordsCountToPagesCount(count($items), $filter['page_size']);
        $retVal = array_slice($items, $filter['page_id'] * $filter['page_size'], $filter['page_size']);
        $this->decorItems($retVal, $filter);
        $response = array(
            "status" => 'successful',
            'pagesCount' => $pageCount,
            'pageId' => $filter['page_id'],
            "items" => $retVal,
        );
        return response()->json($response);
    }

    public function buildProductLowRate($filter) {
        $conversions = $this->getAdwordsConversion($filter);
        $productIds = [];
        if ($conversions) {
            foreach ($conversions as $item) {
                $productIds[$item->product_id] = $item->product_id;
            }
            $productIds = array_values($productIds);
            $filter['productIds'] = $productIds;

        }
        //@todo optimize truyền product id
        $products = $this->getProduct($filter);
        $productsAdwordsConversion = $this->calculateProductAdwordsConversion($products, $conversions);
        $rateOfCategories = $this->calculateRateOfCategories($productsAdwordsConversion);
        $items = $this->compareRate($rateOfCategories, $productsAdwordsConversion);

        //@todo sort
        if (array_key_exists('order', $filter)) {
            if($filter['order']['direction'] == 'SORT_DESC'){
                $sortFlag = SORT_DESC;
            }
            if($filter['order']['direction'] == 'SORT_ASC'){
                $sortFlag = SORT_ASC;
            }
            $sortByColumn = array_column($items, $filter['order']['column']);
            array_multisort($sortByColumn, $sortFlag, $items);
        }
        return $items;
    }

    public function exportProductLowRate(Request $request)
    {
        $filter = $this->buildFilter($request);
        $items = $this->buildProductLowRate($filter);
        $this->decorItems($items);
        $fileName = '';
        if (isset($filter['order_date_from']) && isset($filter['order_date_to'])) {
            $fileName = 'Bao_cao_tuong_tac_danh_muc'
                . '-' . $filter['order_date_from']->format('Y-m-d')
                . '-' . $filter['order_date_to']->format('Y-m-d');
        }
        $this->exportProductLowRateToExcel($items, $filter, $fileName);
    }

    protected function exportProductLowRateToExcel($items, $filter, $fileName = '')
    {
        $fileName = $fileName ? $fileName : 'Bao_cao_tuong_tac_danh_muc-' . time();
        /** Build excel file to export */
        $objPHPExcel = new \PHPExcel();
        /** Setup properties for file excel */
        $objPHPExcel->getProperties()
            ->setCreator("Megaads Technical")
            ->setLastModifiedBy("Megaads Technical")
            ->setTitle("Report Category Rate");

        $objPHPExcel->setActiveSheetIndex(0);
        $objPHPExcel->getActiveSheet()->SetCellValue('A1', "STT");
        $objPHPExcel->getActiveSheet()->SetCellValue('B1', "Mã sản phẩm");
        $objPHPExcel->getActiveSheet()->SetCellValue('C1', "Tên sản phẩm");
        $objPHPExcel->getActiveSheet()->SetCellValue('D1', "Click");
        $objPHPExcel->getActiveSheet()->SetCellValue('E1', "Danh mục");
        $objPHPExcel->getActiveSheet()->SetCellValue('F1', "Chuyển đổi sản phẩm");
        $objPHPExcel->getActiveSheet()->SetCellValue('G1', "Chuyển đổi danh mục");
        $objPHPExcel->getActiveSheet()->SetCellValue('H1', "Đơn hàng");
        $objPHPExcel->getActiveSheet()->SetCellValue('I1', "Tồn kho");
        $i = 1;
        foreach ($items as $item) {
            $columns = [
                'name', 'slug', 'image_url', 'sku'
            ];
            $productSkuId = $item['product_sku_id'];
            $productId = $item['product_id'];
            $isOnlyProduct = array_key_exists('only_product', $filter) && $filter['only_product'];
            if ($productSkuId && $productSkuId > 0 && !$isOnlyProduct) {
                $productInfo = ProductInfo::buildQuery([
                    'product_id' => $productId,
                    'product_sku_id' => $productSkuId,
                ])->first($columns);
            } else {
                $productInfo = Product::where('id', $productId)->first([
                    'name', 'slug', 'image_url', 'sku'
                ]);
            }
            
            foreach ($columns as $column) {
                $item[$column] = isset($productInfo->{$column}) ? $productInfo->{$column} : '';
            }
            $i++;
            $objPHPExcel->getActiveSheet()->SetCellValue('A' . $i, $i - 1);
            $objPHPExcel->getActiveSheet()->SetCellValue('B' . $i, $item['sku']);
            $objPHPExcel->getActiveSheet()->SetCellValue('C' . $i, $item['name']);
            $objPHPExcel->getActiveSheet()->SetCellValue('D' . $i, $item['total_click']);
            $objPHPExcel->getActiveSheet()->SetCellValue('E' . $i, $item['category_title']);
            $objPHPExcel->getActiveSheet()->SetCellValue('F' . $i, $item['product_rate']);
            $objPHPExcel->getActiveSheet()->SetCellValue('G' . $i, $item['category_rate']);
            $objPHPExcel->getActiveSheet()->SetCellValue('H' . $i, $item['total_conversion']);
            $objPHPExcel->getActiveSheet()->SetCellValue('I' . $i, $item['inventory']);
        }

        $writer = \PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel2007');
        ob_end_clean();
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment;filename="' . $fileName . '.xlsx"');
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
    }

    public function compareRate($rateOfCategories, $productsAdwordsConversion){
        $dataProducts = [];
        foreach($rateOfCategories as $category){
            if ($category['click'] == 0) {
                continue;
            }
            $categoryRate = $this->ceiling(($category['conversion'] / $category['click']) * 100, 0.01);
            foreach($category['products'] as $key => $product){
                if (!isset($productsAdwordsConversion[$key])) {
                    continue;
                }
                $productRate = $productsAdwordsConversion[$key]['rate'] * 100;
                $productRate = $this->ceiling($productRate, 0.01);
                if($productRate <= $categoryRate){
                    $dataProducts[] = [
                        'code' => $product['sku'],
                        'sku' => $product['sku'],
                        'product_rate' => $productRate,
                        'category_rate' => $categoryRate,
                        'category_title' => $product['category_title'],
                        'inventory' => $productsAdwordsConversion[$key]['inventory'],
                        'total_conversion' => $productsAdwordsConversion[$key]['conversion'],
                        'total_click' => $productsAdwordsConversion[$key]['click'],
                        'product_id' => $product['product_id'],
                        'product_sku_id' => $product['product_sku_id'],
                    ];
                }
            }
        }
        return $dataProducts;
    }


    public function calculateRateOfCategories($productsAdwordsConversion){
        $rateCategories = [];
        foreach($productsAdwordsConversion as $product){
            if (!array_key_exists('category_id', $product)) {
                continue;
            }
            $categoryId = $product['category_id'];
            if(!array_key_exists($categoryId, $rateCategories)){
                $rateCategories[$categoryId]['conversion'] = $product['conversion'];
                $rateCategories[$categoryId]['click'] = $product['click'];
            }else {
                $rateCategories[$categoryId]['conversion'] += $product['conversion'];
                $rateCategories[$categoryId]['click'] += $product['click'];
            }
            $productKey = $this->buildKeyProduct($product['product_id'], $product['product_sku_id']);
            $rateCategories[$categoryId]['products'][$productKey] = [
                'category_title' => $product['category_title'],
                'conversion' => $product['conversion'],
                'click' => $product['click'],
                'product_id' => $product['product_id'],
                'product_sku_id' => $product['product_sku_id'],
                'sku' => $product['sku'],
            ];
        }
        return $rateCategories;
    }

    protected function calculateProductAdwordsConversion($products, $conversions) {
        $result = [];

        foreach ($conversions as $conversionItem) {
            $pSkuId = isset($conversionItem->product_sku_id) ? $conversionItem->product_sku_id : '';
            $key = $this->buildKeyProduct($conversionItem->product_id, $pSkuId);
            if (!array_key_exists($key, $result)) {
                $result[$key] = [
                    'rate' => 0,
                    'conversion' => 0,
                    'click' => 0,
                    'depth' => -1,
                    'product_id' => $conversionItem->product_id,
                    'product_sku_id' => $pSkuId,
                ];
            }
            $result[$key]['conversion'] += $conversionItem->total_conversion;
            $result[$key]['click'] += $conversionItem->total_click;
            if ($result[$key]['click'] > 0) {
                $result[$key]['rate'] = $result[$key]['conversion']/$result[$key]['click'];
            }
        }
        foreach ($products as $product) {
            $pSkuId = isset($product->product_sku_id) ? $product->product_sku_id : '';
            $key = $this->buildKeyProduct($product->product_id, $pSkuId);
            if (array_key_exists($key, $result) && $result[$key]['depth'] < $product->depth){
                $result[$key] = array_merge($result[$key], [
                    "product_id" => $product->product_id,
                    "product_sku_id" => $pSkuId,
                    "category_id" => $product->category_id,
                    "category_title" => $product->category_name,
                    "sku" => isset($product->product_sku_sku) ? $product->product_sku_sku : $product->sku,
                    "inventory" => isset($product->product_sku_inventory) ? $product->product_sku_inventory : $product->product_inventory,
                    "depth" => $product->depth,
                ]);
            }
        }
        return $result;
    }

    protected function buildKeyProduct($productId, $pSkuId) {
        return $productId . '-' . $pSkuId;
    }

    protected function getProduct($filter = []) {
        $columns = [
            'product.inventory as product_inventory',
            'product.sku', 'product.id as product_id',
            'pnc.category_id as category_id',
            'c.depth', 'c.name as category_name',
        ];
        $query = DB::table('product')
            ->join('product_n_category as pnc', 'pnc.product_id', '=', 'product.id')
            ->join('category as c', 'c.id', '=', 'pnc.category_id');
        if (array_key_exists('productIds', $filter)) {
            $query->whereIn('product.id', $filter['productIds']);
        }
        $isOnlyProduct = array_key_exists('only_product', $filter) && $filter['only_product'];
        if (!$isOnlyProduct) {
            $columns = array_merge($columns, [
                'product_sku.id as product_sku_id', 'product_sku.sku as product_sku_sku',
                'product_sku.inventory as product_sku_inventory',
            ]);
            $query->leftJoin('product_sku', 'product_sku.product_id', '=', 'product.id');
        }
        ProductReact::decorQuery($query, $filter);
        return $query->get($columns);

    }

    protected function getAdwordsConversion($filter) {
        $columns = [
            'product_react.product_id',
            DB::raw('SUM(sb_product_react.conversion) as total_conversion, 
            SUM(sb_product_react.click) as total_click, 
            SUM(sb_product_react.conversion)/SUM(sb_product_react.click) * 100  as rate')
        ];
        $isOnlyProduct = array_key_exists('only_product', $filter) && $filter['only_product'];

        if (!$isOnlyProduct) {
            $columns[] = 'product_react.product_sku_id';
        }
        return ProductReact::buildProductReactQuery($filter)->get($columns);
    }

    public function ceiling($number, $significance = 1)
    {
        return (is_numeric($number) && is_numeric($significance)) ? (ceil($number / $significance) * $significance) : false;
    }

    private function getGlobalAdsConfig() {
        if (self::$globalAdsConfig == null) {
            self::$globalAdsConfig = ProductReactConfigService::getGlobalConfig();
        }
        return self::$globalAdsConfig;
    }

    private function getProductAdsConfig($productId) {
        if (!array_key_exists($productId, self::$globalAdsProductConfig)) {
            self::$globalAdsProductConfig[$productId] = ProductReactConfigService::getProductMeta($productId);
        }
        return array_key_exists($productId, self::$globalAdsProductConfig) ? self::$globalAdsProductConfig[$productId] : [];
    }

    private function getProductAdsStatus($productId) {
        if (!array_key_exists($productId, self::$globalAdsProductStatus)) {
            self::$globalAdsProductStatus[$productId] = ProductReactConfigService::getProductMeta($productId, 'status');
        }
        return array_key_exists($productId, self::$globalAdsProductStatus) ? self::$globalAdsProductStatus[$productId] : [];
    }






}
