<?php

namespace Modules\Ads\Controllers;

use App\Utils\SkuTemplateHelper;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Ads\Controllers\Controller;
use Modules\Ads\Models\ProductNCategory;
use Modules\Ads\Models\ProductReport;
use Modules\Ads\Models\Product;
use Modules\Ads\Models\ProductInfo;
use Modules\Ads\Services\ProductReportService;
use PHPExcel;
use PHPExcel_IOFactory;


class ProductReportController extends Controller
{
    protected $pageSize = 40;

    public function __construct()
    {
    }


    public function index(Request $request)
    {
        return view('ads::report-device.index');
    }

    public function statistic(Request $request)
    {
        return view('ads::report-product.index');
    }

    public function statisticData(Request $request)
    {
        $reportService = new ProductReportService();
        $filter = $reportService->buildFilter($request);
        $data = $reportService->find($filter);
        return response()->json([
            'status' => 'successful',
            'data' => $data,
        ]);
    }

    public function exportStatisticData(Request $request)
    {
        $reportService = new ProductReportService();
        $filter = $reportService->buildFilter($request);
        $data = $reportService->find($filter);
        if (!$data) {
            return response()->json([
                'status' => 'successful',
                'data' => 'Empty data',
            ]);
        }
        $colInfo = $reportService->excelColumn($data);
        $fileName = 'Bao_cao_doanh_thu_san_pham-' . date("Y-m-d") . time();
        /** Build excel file to export */
        $objPHPExcel = new \PHPExcel();
        /** Setup properties for file excel */
        $objPHPExcel->getProperties()
            ->setCreator("Megaads Technical")
            ->setLastModifiedBy("Megaads Technical")
            ->setTitle("Report Product");
        $objPHPExcel->setActiveSheetIndex(0);

        foreach ($colInfo as $item) {
            if (!isset($item['col_index'])) {
                continue;
            }
            $objPHPExcel->getActiveSheet()->SetCellValue($item['col_index'] . '1', $item['name']);
        }
        foreach ($data as $i => $item) {
            $item = (array) $item;
            foreach ($item as $col => $value) {
                $objPHPExcel->getActiveSheet()->SetCellValue($colInfo[$col]['col_index'] . ($i + 2), $value);
            }
        }
        $writer = \PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel2007');
        ob_end_clean();
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment;filename="' . $fileName . '.xlsx"');
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
    }

    public function find(Request $request)
    {
        $filter = $this->buildFilter($request);
        $response = $this->getData($filter);
        return response()->json($response);
    }

    private function getData($filter)
    {
        $productIds = [];
        if (isset($filter['orders']) && !empty($filter['orders'])) {
            if (isset($filter['keyword']) && !empty($filter['keyword'])) {
                $products = Product::where(function ($query) use ($filter) {
                    $query->orWhere('sku', 'LIKE', '%' . $filter['keyword'] . '%')
                        ->orWhere('name', 'LIKE', '%' . $filter['keyword'] . '%');
                })->get();
                $productIds = [-1];
                if (!empty($products)) {
                    foreach ($products as $item) {
                        $productIds[] = $item->id;
                    }
                }
                $filter['productIds'] = $productIds;
            }
            $result = $this->getReport($filter);
            if (count($result) > 0) {
                foreach ($result as $item) {
                    $productIds[] = $item->product_id;
                }
                $productByIds = $this->buildProductByIds($productIds);
                $this->decorItemReport($result, $productByIds);
            } else {
                unset($filter['orders']);
                $result = $this->getResult($filter);
            }
        } else {
            $result = $this->getResult($filter);
        }
        $productIds = [];
        foreach ($result as $item) {
            $productIds[] = $item->id;
        }
        $filterStast = $filter;
        $filterStast['productIds'] = $productIds;
        unset($filterStast['page_size']);
        unset($filterStast['page_id']);
        if (isset($filter['is_export']) && $filter['is_export']) {
            $data = $this->getDataExport($filterStast);
            return $data;
        } else {
            $statsByProductIds = $this->getReportByProductIds($filterStast);
            $this->decorItems($result, $statsByProductIds);
        }
        $pageId = $filter['page_id'];
        unset($filter['page_size']);
        unset($filter['page_id']);
        $filter['metric'] = 'count';
        $itemCount = $this->getResult($filter);
        $pagesCount = $this->recordsCountToPagesCount($itemCount, $this->pageSize);
        $response = array(
            "status" => 'successful',
            "result" => $result,
            'pagesCount' => $pagesCount,
            'pageId' => $pageId
        );
        return $response;
    }

    private function getDataExport($filter)
    {
        $items = $this->getReport($filter);
        $products = $this->buildProductByIds($filter['productIds']);
        $retVal = [];
        foreach ($items as $item) {
            if (isset($products[$item->product_id])) {
                $item->name = $products[$item->product_id]->name;
                $item->sku = $products[$item->product_id]->sku;
                $item->url = $url = \URL::to('/') . '/' . $products[$item->product_id]->slug . '.p' . $item->product_id . '.html';
            }
            $retVal[] = $item;
        }
        return $retVal;
    }

    private function getReportByProductIds($filter)
    {
        $stasts = $this->getReport($filter);
        $retVal = [];
        foreach ($stasts as $item) {
            $retVal[$item->product_id] = $item;
        }
        return $retVal;
    }

    private function decorItems(&$items, $statsByProductIds)
    {
        foreach ($items as $item) {
            $item->device = '';
            if (isset($statsByProductIds[$item->id])) {
                $item->clicks = (int)$statsByProductIds[$item->id]->clicks;
                $item->sales = (int)$statsByProductIds[$item->id]->sales;
                $item->cvr = round($statsByProductIds[$item->id]->cvr, 2);
                $item->device = $statsByProductIds[$item->id]->device;
                $item->url = \URL::to('/') . '/' . $item->slug . '-p' . $item->id . '.html';
            } else {
                $item->clicks = 0;
                $item->sales = 0;
                $item->cvr = 0;
            }

        }
    }

    private function decorItemReport(&$items, $productByIds)
    {
        foreach ($items as $item) {
            if (isset($productByIds[$item->product_id])) {
                $item->id = $item->product_id;
                $item->sku = $productByIds[$item->product_id]->sku;
                $item->name = $productByIds[$item->product_id]->name;
                $item->slug = $productByIds[$item->product_id]->slug;
                $item->image_url = $productByIds[$item->product_id]->image_url;
            }
        }
    }

    private function buildProductByIds($productIds)
    {
        $query = Product::query();
        if (!empty($productIds)) {
            $query->whereIn('id', $productIds);
        }
        $items = $query->get(['id', 'name', 'slug', 'sku']);
        $retVal = [];
        foreach ($items as $item) {
            $retVal[$item->id] = $item;
        }
        return $retVal;
    }

    private function getResult($filter)
    {
        $query = Product::query();
        if (array_key_exists('skus', $filter) && is_array($filter["skus"])) {
            $query->whereIn("sku", $filter['skus']);
        }
        if (array_key_exists('productIds', $filter) && !empty($filter['productIds'])) {
            $query->whereIn('id', $filter['productIds']);
        }

        if (isset($filter['metric']) && $filter['metric'] == 'count') {
            return $query->count();
        } else {
            if (isset($filter['orders']) && count($filter['orders']) > 0) {
                foreach ($filter['orders'] as $key => $value) {
                    $query->orderBy($key, $value);
                }
            } else {
                $query->orderBy('created_at', 'ASC');
            }
            if (isset($filter['page_size']) && isset($filter['page_id'])) {
                $query->forPage($filter['page_id'] + 1, $filter['page_size']);
            }
            return $query->get(['id', 'name', 'sku', 'slug']);
        }
    }

    private function getReport($filter)
    {
        $query = ProductReport::query();

        if (isset($filter['productIds']) && !empty($filter['productIds'])) {
            $query->whereIn('product_id', $filter['productIds']);
        }
        if (isset($filter['device']) && $filter['device'] != '') {
            $query->where('device', '=', $filter['device']);
        } else {
            $query->where('device', '!=', 'other');
        }
        if (isset($filter['from']) && $filter['from'] != '') {
            $query->where('date', '>=', $filter['from']);
        }
        if (isset($filter['to']) && $filter['to']) {
            $query->where('date', '<=', $filter['to']);
        }
        if (isset($filter['is_export']) && $filter['is_export']) {
            $query->groupBy('product_id', 'device')
                ->select("product_id", "device",
                    \DB::raw("sum(click) as clicks"),
                    \DB::raw("sum(sale) as sales"),
                    \DB::raw("(sum(sale) / sum(click)) as cvr")
                );
        } else {
            $query->groupBy('product_id')
                ->select("product_id",
                    \DB::raw("sum(click) as clicks"),
                    \DB::raw("sum(sale) as sales"),
                    \DB::raw("(sum(sale) / sum(click)) as cvr")
                );
        }

        if (isset($filter['orders']) && count($filter['orders']) > 0) {
            foreach ($filter['orders'] as $key => $value) {
                $query->orderBy($key, $value);
            }
        }
        if (isset($filter['page_size']) && isset($filter['page_id'])) {
            $query->forPage($filter['page_id'] + 1, $filter['page_size']);
        }
        return $query->get();

    }

    private function buildFilter($request)
    {
        $params = ['keyword', 'device', 'is_export'];
        $retVal = [];
        foreach ($params as $param) {
            if ($request->has($param)) {
                $retVal[$param] = $request->input($param);
            }
        }
        if (isset($retVal['keyword'])) {
            $items = Product::where('sku', '=', $retVal['keyword'])
                ->orWhere('name', 'LIKE', '%' . $retVal["keyword"] . '%')
                ->get(['id']);
            $items = $items->toArray();
            $productIds = [];
            if (!empty($items)) {
                foreach ($items as $item) {
                    if (!in_array($item['id'], $productIds)) {
                        $productIds[] = $item['id'];
                    }
                }
            } else {
                $productIds[] = -1;
            }
            $retVal['productIds'] = $productIds;
        }
        $currentDate = new \DateTime();
        $fromDate = $request->has('dateFrom') ? \DateTime::createFromFormat('d/m/Y', $request->input('dateFrom')) : $currentDate;
        $toDate = $request->has('dateTo') ? \DateTime::createFromFormat('d/m/Y', $request->input('dateTo')) : $currentDate;

        $retVal['from'] = $fromDate->setTime(0, 0, 0);
        $retVal['to'] = $toDate->setTime(23, 59, 59);
        $sort = $request->has('sort') ? $request->input('sort') : '';
        if ($sort != '') {
            $retVal['sort'] = $sort;
            $arrOrders = explode('-', $retVal['sort']);
            $retVal['orders'] = [$arrOrders[0] => $arrOrders[1]];
        }
        $isExport = (isset($retVal['is_export']) && $retVal['is_export']) ? $retVal['is_export'] : null;
        if ($isExport != null && $isExport) {
            $retVal['page_size'] = 4000;
        } else {
            $retVal['page_size'] = $this->pageSize;
        }
        $retVal['page_id'] = $request->has('page_id') ? $request->input('page_id') : 0;
        return $retVal;
    }


    private function recordsCountToPagesCount($recordsCount, $pageSize)
    {
        $retVal = (int)($recordsCount / $pageSize);
        if ($recordsCount % $pageSize > 0) {
            $retVal++;
        }
        //return
        return $retVal;
    }

    public function export(Request $request)
    {
        set_time_limit(0);
        ini_set('memory_limit', '2048M');
        $filters = $this->buildFilter($request);
        $products = $this->getData($filters, $request);
        $this->exportToExcel($products);
    }

    private function exportToExcel($items)
    {
        /** Build excel file to export */
        $objPHPExcel = new PHPExcel();
        /** Setup properties for file excel */
        $objPHPExcel->getProperties()
            ->setCreator("Temporaris")
            ->setLastModifiedBy("Temporaris")
            ->setTitle("Template Relevé des heures intérimaires")
            ->setSubject("Template excel")
            ->setDescription("Template excel permettant la création d'un ou plusieurs relevés d'heures")
            ->setKeywords("Template excel");

        $objPHPExcel->setActiveSheetIndex(0);
        $objPHPExcel->getActiveSheet()->SetCellValue('A1', "STT");
        $objPHPExcel->getActiveSheet()->SetCellValue('B1', "Mã sản phẩm");
        $objPHPExcel->getActiveSheet()->SetCellValue('C1', "Tên sản phẩm");
        $objPHPExcel->getActiveSheet()->SetCellValue('D1', "Link");
        $objPHPExcel->getActiveSheet()->SetCellValue('E1', "Click");
        $objPHPExcel->getActiveSheet()->SetCellValue('F1', "Sale");
        $objPHPExcel->getActiveSheet()->SetCellValue('G1', "Cvr");
        $objPHPExcel->getActiveSheet()->SetCellValue('H1', "Thiết bị");
        $objPHPExcel->getActiveSheet()->getStyle("A1:K1")->getFont()->setBold(true);
        $objPHPExcel->getActiveSheet()->getStyle("A1:K1")->getFont()->setSize(12);

        $i = 1;
        $j = 0;
        foreach ($items as $item) {
            $i++;
            $j++;
            $objPHPExcel->getActiveSheet()->SetCellValue('A' . $i, $j);
            if (isset($item->sku)) {
                $objPHPExcel->getActiveSheet()->SetCellValue('B' . $i, $item->sku);
            }
            if (isset($item->name)) {
                $objPHPExcel->getActiveSheet()->SetCellValue('C' . $i, $item->name);
            }
            if (isset($item->url)) {
                $objPHPExcel->getActiveSheet()->SetCellValue('D' . $i, $item->url);
            }
            if (isset($item->clicks)) {
                $objPHPExcel->getActiveSheet()->SetCellValue('E' . $i, $item->clicks);
            }
            if (isset($item->sales)) {
                $objPHPExcel->getActiveSheet()->SetCellValue('F' . $i, $item->sales);
            }
            if (isset($item->cvr)) {
                $objPHPExcel->getActiveSheet()->SetCellValue('G' . $i, $item->cvr);
            }
            if (isset($item->device)) {
                $objPHPExcel->getActiveSheet()->SetCellValue('H' . $i, $item->device);
            }
        }

        $writer = PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel2007');
        ob_end_clean();
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment;filename="Bao_cao_thiet_bi_' . date("Y-m-d") . '_' . time() . '.xlsx"');
        header('Cache-Control: max-age=0');
        $writer->save('php://output');
    }


    public function buildProductReport(Request $request) {
        set_time_limit(1200);
        $reportService = new ProductReportService();
        $filter = [
            'dateFrom' => date('Y-m-d 00:00:00', time() - $request->input('from', 1) * 86400),
            'dateTo' => date('Y-m-d 23:59:59', time() - $request->input('to', 0) * 86400)
        ];
        $items = $this->getItems($filter);
        $count = 0;
        foreach ($items as $item) {
            if ($item && !$reportService->exists($item)) {
                $data = $this->buildData($item);
                $reportService->store($data);
                $count++;
            }
        }
        return response()->json([
            'status' => 'successful',
            'data' => $count,
        ]);

    }

    protected function buildData($item) {
        $isTemplate = SkuTemplateHelper::isExists($item->product_id);
        $table = $isTemplate ? 'product_template_sku_value' : 'product_sku_value';
        $options = $this->getOptions($table, $item->product_sku_id);
        if (!$options && $table == 'product_template_sku_value') {
            $options = $this->getOptions('product_sku_value', $item->product_sku_id);
        }
        $result = (array) $item;
        return array_merge($result, [
            'color_variant_id' => isset($options[ProductReportService::VARIANT_COLOR]) ? $options[ProductReportService::VARIANT_COLOR] : 0,
            'size_variant_id' => isset($options[ProductReportService::VARIANT_SIZE]) ? $options[ProductReportService::VARIANT_SIZE] : 0,
            'type_variant_id' => isset($options[ProductReportService::VARIANT_TYPE]) ? $options[ProductReportService::VARIANT_TYPE] : 0,
            'style_variant_id' => isset($options[ProductReportService::VARIANT_STYLE]) ? $options[ProductReportService::VARIANT_STYLE] : 0,
            'category_id' => ProductNCategory::getCategory($item->product_id),
            'created_at' => new \DateTime(),
            'updated_at' => new \DateTime(),
        ]);
    }

    protected function getItems($filter) {
        return DB::table('order as o')
            ->join('order_item as oi', 'oi.order_id', '=', 'o.id')
            ->where('o.payment_status', 'PAID')
            ->where('o.created_at', '>=', $filter['dateFrom'])
            ->where('o.created_at', '<=', $filter['dateTo'])
            ->get([
                'oi.id as order_item_id', 'oi.order_id', 'oi.product_id',
                'oi.product_sku_id', 'oi.quantity', 'oi.price'
            ]);
    }

    protected function getOptions($table, $skuId) {
        return DB::table($table)->where('sku_id', $skuId)
            ->limit(10)
            ->pluck('variant_option_id', 'variant_id')
            ->toArray();
    }

}
