<?php

namespace Modules\Ads\Controllers;

use Illuminate\Http\Request;
use Modules\Ads\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Modules\Ads\Models\Order;

class ProductSoldController extends Controller
{
    public function __construct()
    {
    }

    public function index()
    {
        return view('ads::product-sold.index');
    }

    public function getSoldProductOffAds(Request $request) {
        // Build filter from request
        $filter = $this->buildFilter($request);

        // Extract pagination parameters
        $pageId = $filter['page_id'];
        $pageSize = $filter['page_size'];
        
        // Unset pagination parameters for total count query
        unset($filter['page_id']);
        unset($filter['page_size']);

        // Get total count for pagination
        $filter['metric'] = 'count';
        $total = $this->getResult($filter);
        $pagesCount = $this->recordsCountToPagesCount($total, $pageSize);

        // Get paginated result
        $filter['page_id'] = $pageId;
        $filter['page_size'] = $pageSize;
        $filter['metric'] = 'data';
        $result = $this->getResult($filter);

        // Prepare response
        $response = array(
            "status" => 'successful',
            'pagesCount' => $pagesCount,
            'pageId' => $pageId,
            "items" => $result,
        );
        return response()->json($response);
    }

    protected function buildFilter($request){
        $retVal = [];
        $retVal['page_id'] = $request->input('pageId', 0);
        $retVal['page_size'] = $request->input('pageSize', 100);
        $fromDateTime = $this->getDateTimeFilter($request->input('orderDateFrom'));
        $fromDateTime->setTime("00", "00", "00");
        $toDateTime = $this->getDateTimeFilter($request->input('orderDateTo'));
        $toDateTime->setTime("23", "59", "59");
        $retVal["date_from"] = $fromDateTime;
        $retVal["date_to"] = $toDateTime;
        $retVal['columns'] = ["product.id", "name", "slug", "sold"];
        if ($request->has('keyword')) {
            $retVal["keyword"] = $request->input('keyword');
        }
        if ($request->has('status')) {
            $retVal["status"] = $request->input('status');
        }
        return $retVal;
    }

    protected function getDateTimeFilter($dateTime, $format = "d/m/Y") {
        $retval = FALSE;
        if (is_object($dateTime)) {
            $retval = clone $dateTime;
        }
        if ($dateTime == NULL || $dateTime == "") {
            $dateTime = "1/1/1900";
        }
        if (is_string($dateTime)) {
            $retval = \DateTime::createFromFormat($format, $dateTime);
        }
        $retval->setTime("00", "00", "00");

        return $retval;
    }

    protected function getResult($filter) {
        // Initialize the query
        $productIds = $this->getProductIdsToGetOrderItem($filter);
        $dataGet = ['product.id', "name", "sold", "slug"];
        $query = DB::table('product')
            ->leftJoin('merchant_product', 'product.id', '=', 'merchant_product.product_id')
            ->whereNull('merchant_product.id')
            ->where("product.status", "ACTIVE")
            ->where("product.is_violation", 0)
            ->where("product.is_trademark", 0)
            ->orderBy("product.sold", 'desc')
            ->whereIn('product.id', $productIds)
            ->groupBy('product.id');
        if (isset($filter['keyword']) && $filter['keyword']) {
            $query->where("product.name", 'LIKE', "%{$filter['keyword']}%");
        }
        if (isset($filter['status']) && $filter['status'] != -1) {
            $dataGet[] = 'product_approve_log.actor_email';
            $query->join('product_approve_log', 'product.id', '=', 'product_approve_log.product_id')
            ->where('approve', $filter['status']);
        }
        $query->select($dataGet);
        // Check if we need to return the count
        if (isset($filter['metric']) && $filter['metric'] == 'count') {
            return $query->get()->count();
        } else {
            // Apply pagination if needed
            if (isset($filter['page_size']) && isset($filter['page_id'])) {
                $query->forPage($filter['page_id'] + 1, $filter['page_size']);
            }
            return $query->get();
        }
    }

    protected function recordsCountToPagesCount($recordsCount, $pageSize)
    {
        return (int)ceil($recordsCount / $pageSize);
    }

    private function getProductIdsToGetOrderItem($filter)
    {
        $ids = Order::query()->join('order_item', 'order.id', '=', 'order_item.order_id')
            ->whereBetween('order.created_at', [$filter["date_from"], $filter["date_to"]])
            ->pluck('order_item.product_id')
            ->toArray();

        return array_unique($ids);
    }
}
