<?php

namespace Modules\Ads\Controllers;

use DOMDocument;
use DOMElement;
use DateTime;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReviewController extends Controller
{
    const KEY = 'review_feed_id';

    /**
     * build review feed
     */
    public function buildReviewFeed(Request $request)
    {
        set_time_limit(3600 * 5);
        ini_set('memory_limit', '1G');
        
        $result = [];
        $soldFrom = $request->input('sold_from', -1);
        $maxSize = 1024 * 1024 * 1024 * 4 - 1024;
        $locale = !empty(env('APP_LOCALE')) ? env('APP_LOCALE') : 'us';
        if (!file_exists("feeds/$locale/review")) {
            mkdir("feeds/$locale/review", 0777, true);
        }

        $cur = DB::table('option')->where('key', self::KEY)->first();
        $curId = 0;
        $curFileId = 0;
        if (!$cur) {
            DB::table('option')->insert([
                'key' => self::KEY,
                'value' => json_encode([
                    'review_id' => 0,
                    'review_file_id' => 0,
                ])
            ]);
        } else {
            $value = json_decode($cur->value, true);
            $curId = $value['review_id'];
            $curFileId = $value['review_file_id'];
        }

        $query = \DB::table('comment as c')
            ->join('product as p', 'c.target_id', 'p.id')
            ->join('comment_n_user as cnu', 'c.id', 'cnu.comment_id')
            ->join('users as u', 'cnu.user_id', 'u.id')
            ->where('c.status', 'ACTIVE')
            ->where('p.is_violation', 0)
            ->where('p.is_trademark', 0)
            ->where('p.status', 'ACTIVE')
            ->where('c.target_type', 'PRODUCT');
        if ($soldFrom >= 0) {
            $query->where('p.sold', '>=', $soldFrom);
        }
        $maxId = $query->orderByDesc('c.id')->first(['c.id'])->id;
        if ($curId >= $maxId) {
            return response()->json([
                'status' => 'successful'
            ]);
        }

        $domain = $request->getHost();
        $gzFilePath = "feeds/$locale/review/reviews_{$domain}_$curFileId.xml.gz";
        $doc = $this->createXml();
        $xmlString = $doc->saveXML();
        $head = str_replace("a</reviews></feed>", '', $xmlString);
        if (!file_exists($gzFilePath)) {
            touch($gzFilePath, 0777, true);
            $gz = gzopen($gzFilePath, 'a9');
            gzwrite($gz, $head);
            gzclose($gz);
        }

        for ($i = $curId + 1; $i <= $maxId; $i += 10000) {
            $records = $this->getData($i, $soldFrom);
            if ($records->isEmpty()) {
                continue;
            }
            $xmlChunk = '';
            foreach ($records as $recordItem) {
                if (strlen($recordItem->title) <= 7) {
                    continue;
                }
                $review = $this->createReview($doc, $recordItem);
                $reviewXml = $doc->saveXML($review);
                $xmlChunk .= $reviewXml;
            }
            clearstatcache();
            if (filesize($gzFilePath) + strlen(gzencode($xmlChunk, 9)) < $maxSize) {
                $gz = gzopen($gzFilePath, 'a9');
                gzwrite($gz, $xmlChunk);
                gzclose($gz);
                DB::table('option')->where('key', self::KEY)->update([
                    'value' => json_encode([
                        'review_id' => $recordItem->review_id,
                        'review_file_id' => $curFileId,
                    ])
                ]);
            } else {
                $gz = gzopen($gzFilePath, 'a9');
                gzwrite($gz, '</reviews></feed>');
                gzclose($gz);
                $curFileId++;
                DB::table('option')->where('key', self::KEY)->update([
                    'value' => json_encode([
                        'review_id' => $recordItem->review_id,
                        'review_file_id' => $curFileId,
                    ])
                ]);
                
                $result[] = $gzFilePath;
                $gzFilePath = "feeds/$locale/review/reviews_{$domain}_$curFileId.xml.gz";
                $gz = gzopen($gzFilePath, 'a9');
                gzwrite($gz, $head);
                gzwrite($gz, $xmlChunk);
                gzclose($gz);
            }
        }

        $gz = gzopen($gzFilePath, 'a9');
        gzwrite($gz, '</reviews></feed>');
        gzclose($gz);
        $curFileId++;
        DB::table('option')->where('key', self::KEY)->update([
            'value' => json_encode([
                'review_id' => $recordItem->review_id,
                'review_file_id' => $curFileId,
            ])
        ]);
        $result[] = $gzFilePath;

        return response()->json([
            'status' => 'successful',
            'result' => $result
        ]);
    }

    private function createXml()
    {
        $doc = new DOMDocument('1.0', 'UTF-8');
        $feed = $doc->createElement('feed');
        $feed->setAttribute('xmlns:vc', 'http://www.w3.org/2007/XMLSchema-versioning');
        $feed->setAttribute('xmlns:xsi', 'http://www.w3.org/2001/XMLSchema-instance');
        $feed->setAttribute('xsi:noNamespaceSchemaLocation', 'http://www.google.com/shopping/reviews/schema/product/2.3/product_reviews.xsd');
        $doc->appendChild($feed);
        $this->createMetaData($doc, $feed);
        $reviews = $doc->createElement('reviews', 'a');
        $feed->appendChild($reviews);
        return $doc;
    }

    private function createMetaData(DOMDocument $doc, DOMElement $feed)
    {
        $version = $doc->createElement('version', 2.3);
        $feed->appendChild($version);

        $aggregator = $doc->createElement('aggregator');
        $feed->appendChild($aggregator);
        $name = $doc->createElement('name', env('APP_NAME'));
        $aggregator->appendChild($name);

        $publisher = $doc->createElement('publisher');
        $feed->appendChild($publisher);
        $name = $doc->createElement('name', env('APP_NAME'));
        $publisher->appendChild($name);
    }

    private function getData($offset, $soldFrom)
    {
        $query = \DB::table('comment as c')
            ->join('product as p', 'c.target_id', 'p.id')
            ->join('comment_n_user as cnu', 'c.id', 'cnu.comment_id')
            ->join('users as u', 'cnu.user_id', 'u.id')
            ->where('c.id', '>=', $offset)
            ->where('c.id', '<', $offset + 10000)
            ->where('c.status', 'ACTIVE')
            ->where('p.is_violation', 0)
            ->where('p.is_trademark', 0)
            ->where('p.status', 'ACTIVE')
            ->where('c.target_type', 'PRODUCT');
        if ($soldFrom >= 0) {
            $query->where('p.sold', '>=', $soldFrom);
        }
        
        return $query->get([
            'c.id as review_id',
            'u.name as reviewer_name',
            'c.created_at as review_timestamp',
            'c.title as title',
            'c.content as content',
            'c.rating as rating',
            'p.name as product_name',
            'p.slug as product_slug',
            'p.id as pid',
            'p.sku as sku'
        ]);
    }

    private function createReview(DOMDocument $doc, $recordItem) : DOMElement
    {
        $locale = env('APP_LOCALE');
        $review = $doc->createElement('review');

        $reviewId = $doc->createElement('review_id', $recordItem->review_id);
        $review->appendChild($reviewId);
        
        $reviewer = $doc->createElement('reviewer');
        $review->appendChild($reviewer);

        $name = $doc->createElement('name', htmlspecialchars($recordItem->reviewer_name));
        $reviewer->appendChild($name);

        $timeStampValue = date(DateTime::ISO8601, strtotime($recordItem->review_timestamp));
        $timeStamp = $doc->createElement('review_timestamp', $timeStampValue);
        $review->appendChild($timeStamp);

        $title = $doc->createElement('title', htmlspecialchars($recordItem->title));
        $review->appendChild($title);

        $content = $doc->createElement('content', htmlspecialchars($recordItem->content));
        $review->appendChild($content);

        $appUrl = env('APP_URL');
        $url = empty($locale) 
            ? "$appUrl/$recordItem->product_slug-p$recordItem->pid#reviewsItem-$recordItem->review_id-Box"
            : "$appUrl/$locale/$recordItem->product_slug-p$recordItem->pid#reviewsItem-$recordItem->review_id-Box";
        $reviewUrl = $doc->createElement('review_url', $url);
        $reviewUrl->setAttribute('type', 'group');
        $review->appendChild($reviewUrl);

        $ratings = $doc->createElement('ratings');
        $review->appendChild($ratings);
        
        $overall = $doc->createElement('overall', $recordItem->rating);
        $overall->setAttribute('min', 1);
        $overall->setAttribute('max', 5);
        $ratings->appendChild($overall);
        
        $products = $doc->createElement('products');
        $review->appendChild($products);
        
        $this->createProduct($doc, $products, $recordItem);

        return $review;
    }
    
    private function createProduct(DOMDocument $doc, DOMElement $products, $recordItem)
    {
        $product = $doc->createElement('product');
        $products->appendChild($product);

        $productIds = $doc->createElement('product_ids');
        $product->appendChild($productIds);

        $gtins = $doc->createElement('gtins');
        $productIds->appendChild($gtins);

        $gtin = $doc->createElement('gtin', '-');
        $gtins->appendChild($gtin);

        $mpns = $doc->createElement('mpns');
        $productIds->appendChild($mpns);

        $mpn = $doc->createElement('mpn', '-');
        $mpns->appendChild($mpn);

        $skus = $doc->createElement('skus');
        $productIds->appendChild($skus);

        $sku = $doc->createElement('sku', htmlspecialchars($recordItem->sku));
        $skus->appendChild($sku);

        $brands = $doc->createElement('brands');
        $productIds->appendChild($brands);

        $brand = $doc->createElement('brand', '-');
        $brands->appendChild($brand);

        $asins = $doc->createElement('asins');
        $productIds->appendChild($asins);

        $asin = $doc->createElement('asin', '-');
        $asins->appendChild($asin);

        $productName = $doc->createElement('product_name', htmlspecialchars($recordItem->product_name));
        $product->appendChild($productName);
        
        $appUrl = env('APP_URL');
        $url = empty($locale) 
        ? "$appUrl/$recordItem->product_slug-p$recordItem->pid"
        : "$appUrl/$locale/$recordItem->product_slug-p$recordItem->pid";

        $productUrl = $doc->createElement('product_url', htmlspecialchars($url));
        $product->appendChild($productUrl);
    }

    private function deleteDirectory($dir) {
        $items = scandir($dir);
        foreach ($items as $item) {
            if ($item == '.' || $item == '..') {
                continue;
            }
    
            $path = $dir . DIRECTORY_SEPARATOR . $item;
            unlink($path);
        }
    
        return rmdir($dir);
    }
}
