<?php

namespace Modules\Ads\Controllers\Service;

use App\Modules\Ads\Controllers\Service\BaseService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Modules\Ads\Controllers\Service\Exception;
use Modules\Ads\Controllers\Service\MerchantProductService;
use Modules\Ads\Models\Category;
use Modules\Ads\Models\Country;
use Modules\Ads\Models\MerchantProduct;
use Modules\Ads\Models\ProductSku;
use Module;
use Cache;
use Modules\Ads\Services\ProductInfoGalleryService;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Config;
class ProductAdsService extends BaseService
{
    private $merchantProductService;

    public function __construct(MerchantProductService $merchantProductService)
    {
        $this->merchantProductService = $merchantProductService;
    }

    public function buildProductFilter($filter){
        $productQuery = DB::table('product');
        $productQuery->join('product_info', 'product.id', 'product_info.product_id');
        if(array_key_exists('status', $filter)){
            $productQuery->where('product.status', $filter['status']);
        }
        if(array_key_exists('approve_advertising', $filter)){
            $productQuery->where('product.approve_advertising', $filter['approve_advertising']);
        }
        if(array_key_exists('product_id_from', $filter)){
            $productQuery->where('product.id','>=', $filter['product_id_from']);
        }
        if(array_key_exists('product_id_to', $filter)){
            $productQuery->where('product.id','<=', $filter['product_id_to']);
        }
        if(array_key_exists('is_trademark', $filter)){
            $productQuery->where('product.is_trademark', $filter['is_trademark']);
        }
        if(array_key_exists('is_default', $filter)){
            $productQuery->where('product_info.is_default', $filter['is_default']);
        }
        if(array_key_exists('is_violation', $filter)){
            $productQuery->where('product.is_violation', $filter['is_violation']);
        }
        if(array_key_exists('sold_from', $filter)){
            $productQuery->where('product.sold', '>=', $filter['sold_from']);
        }
        if(array_key_exists('updated_at_from', $filter)){
            $productQuery->where('product.updated_at', '>=', $filter['updated_at_from']);
        }
        if(array_key_exists('updated_at_to', $filter)){
            $productQuery->where('product.updated_at','<=', $filter['updated_at_to']);
        }
        if(array_key_exists('ids', $filter)){
            if (!is_array($filter['ids'])){
                $filter['ids'] = explode(',',$filter['ids']);
            }
            $productQuery->whereIn('product.id', $filter['ids']);
        }
        if(array_key_exists('columns', $filter)){
            $productQuery->select($filter['columns']);
        }
        return $productQuery;
    }

    public function buildProduct($products, $input = []){
        $retVal = [];
        foreach ($products as $product){

            $paramLink = [
                'slug' => $product->slug,
                'id' => $product->id,
            ];
            if (!empty($product->sku_id)){
                $paramLink['spid'] = $product->sku_id;
            }

            $item = [
                'id' => $product->id,
                'sku' => $product->sku_code,
                'title' => $product->name,
                'link' => route('product', $paramLink),
                'brand' => $product->sku_brand,
                'google_product_category' => $product->sku_category_merchant_id,
                'price' => $product->price,
                'high_price' => $product->high_price,
                'barcode' => $product->barcode,
                'sold' => $product->sold,
                'status' => 'ACTIVE',
                'sku_id' => $product->sku_id,
            ];
            $this->merchantProductService->buildSpecificInfo($product);
            $seller = $this->getSellerInfo($product->id);
            if (!empty($seller)){
                $item['seller_id'] = $seller->id;
                $item['is_owner'] = 0;
                if (!empty($seller->seller_token) && $seller->role == 'SELLER'){
                    $item['is_owner'] = 1;
                }
            }
            $this->setCategory($item);
            $this->setDescription($item, $product);
            $this->setTitle($item, $product);
            $this->setScore($item);
            $this->setImage($item,$product);
            $this->setImages($item,$product);
            $this->setLabel1($item,$product);
            $this->setLabel0($item,$product);
            $this->setLabel2($item,$product);
            $this->setLabel3($item);
            $this->setLabel4($item,$product);
            $this->setShipping($item,$product,$input);
            $this->merchantProductService->buildProductInfo($product, $item);
            $retVal[$item['id']] = $item;
        }
        return $retVal;
    }

    private function getSellerInfo($productId){
        $user = DB::table('product_n_user')
            ->join('users','users.id','=','product_n_user.user_id')
            ->where('product_n_user.product_id', '=', $productId)
            ->first(['users.id','users.seller_token','users.email','role']);
        return $user;
    }

    private function setCategory(&$product)
    {
        $category = DB::table('product_n_category')
            ->where('product_n_category.product_id', '=', $product['id'])
            ->where('product_n_category.is_parent', '=', 0)
            ->first(['product_n_category.category_id']);
        if (!empty($category)){
            $product['category_id'] = $category->category_id;
            $product['product_type'] = Category::getBreadcrumbById($category->category_id);
        }

    }

    /**
     * @param array $filter
     * @param string|null $connectionName Tên của connection CSDL
     * @return \Illuminate\Database\Query\Builder
     */
    public function buildProductAdsFilter($filter, $connectionName = null)
    {
        $dbConnection = null;

        // === Yêu cầu: Kiểm tra connection ===
        // 1. Kiểm tra xem $connectionName có được cung cấp VÀ có tồn tại trong config không
        if ($connectionName && Config::has("database.connections.$connectionName")) {
            // Nếu có, dùng connection đó
            $dbConnection = DB::connection($connectionName);
        } else {
            // 2. Nếu không (hoặc $connectionName là null), dùng connection mặc định (hiện tại)
            $dbConnection = DB::connection();
        }
        // === Kết thúc kiểm tra ===

        // Bắt đầu query từ connection đã được chọn
        $query = $dbConnection->table('product_ads');

        // Các điều kiện filter
        if (array_key_exists('product_id_from', $filter)) {
            $query->where('product_ads.id', '>=', $filter['product_id_from']);
        }
        if (array_key_exists('product_id_to', $filter)) {
            $query->where('product_ads.id', '<=', $filter['product_id_to']);
        }
        if (array_key_exists('ids', $filter)) {
            if (!is_array($filter['ids'])){
                $filter['ids'] = explode(',',$filter['ids']);
            }
            $query->whereIn('product_ads.id', $filter['ids']);
        }
        if (array_key_exists('seller_ids', $filter)) {
            if (!is_array($filter['seller_ids'])){
                $filter['seller_ids'] = explode(',',$filter['seller_ids']);
            }
            $query->whereIn('product_ads.seller_id', $filter['seller_ids']);
        }
        if (array_key_exists('status', $filter)) {
            $query->where('product_ads.status', $filter['status']);
        }
        if (array_key_exists('sold_from', $filter)) {
            $query->where('product_ads.sold','>=', $filter['sold_from']);
        }
        if (array_key_exists('created_at_from', $filter)) {
            $query->where('product_ads.created_at','>=', $filter['created_at_from']);
        }
        if (array_key_exists('created_at_to', $filter)) {
            $query->where('product_ads.created_at','<=', $filter['created_at_to']);
        }
        if (array_key_exists('updated_at_from', $filter)) {
            $query->where('product_ads.updated_at','>=', $filter['updated_at_from']);
        }
        if (array_key_exists('updated_at_to', $filter)) {
            $query->where('product_ads.updated_at','<=', $filter['updated_at_to']);
        }
        if (array_key_exists('orderBy', $filter)) {
            $orderCol = explode('|',$filter['orderBy']);
            $orderBy = isset($orderCol[1]) ? $orderCol[1] : 'asc';
            $query->orderBy('product_ads.'.$orderCol[0], $orderBy);
        }

        return $query;
    }
    /**
     * Chèn hoặc cập nhật hàng loạt sản phẩm, hỗ trợ CSDL động và chỉ return 1 lần.
     *
     * @param array $products Mảng chứa dữ liệu các sản phẩm
     * @param string|null $connectionName Tên connection trong config/database.php
     * @return array Trả về một mảng chứa các lỗi.
     */
    public function bulkInsertOrUpdateProducts(array $products, $connectionName = null)
    {
        // *** THAY ĐỔI 1: Khởi tạo mảng báo cáo ***
        $report = [
            'inserted' => 0,
            'updated'  => 0,
            'failed'   => 0,
            'errors'   => [],
        ];

        $tableName = 'product_ads';
        $validColumns = [];
        $canProceed = false;
        $dbConnection = null;

        try {
            if ($connectionName && Config::has("database.connections.$connectionName")) {
                $dbConnection = DB::connection($connectionName);
            } else {
                $dbConnection = DB::connection();
            }

            $validColumns = $dbConnection->getSchemaBuilder()->getColumnListing($tableName);
            $canProceed = true;

        } catch (\Exception $e) {
            Log::error("Lỗi khi kết nối hoặc lấy cấu trúc bảng $tableName (Connection: " . ($connectionName ?: 'default') . "): " . $e->getMessage());

            // *** THAY ĐỔI 2: Ghi lỗi vào báo cáo ***
            $report['failed'] = count($products); // Lỗi CSDL, coi như tất cả đều lỗi
            $report['errors'][] = [
                'error_message' => 'Lỗi cấu hình CSDL hoặc không tìm thấy bảng.',
                'detail' => $e->getMessage()
            ];
        }

        if ($canProceed) {
            $now = Carbon::now()->toDateTimeString();

            foreach ($products as $index => $productData) {

                try {
                    $filteredData = array_intersect_key($productData, array_flip($validColumns));

                    if (empty($filteredData)) {
                        continue;
                    }

                    $id = isset($filteredData['id']) ? $filteredData['id'] : null;

                    if ($id && $dbConnection->table($tableName)->where('id', $id)->exists()) {

                        // === LOGIC UPDATE ===
                        $filteredData['updated_at'] = $now;
                        unset($filteredData['created_at']);

                        $dbConnection->table($tableName)->where('id', $id)->update($filteredData);

                        // *** THAY ĐỔI 3: Đếm số lượng update ***
                        $report['updated']++;

                    } else {

                        // === LOGIC INSERT ===
                        $filteredData['created_at'] = $now;
                        $filteredData['updated_at'] = $now;

                        if(is_null($id)) {
                            unset($filteredData['id']);
                        }

                        $dbConnection->table($tableName)->insert($filteredData);

                        // *** THAY ĐỔI 4: Đếm số lượng insert ***
                        $report['inserted']++;
                    }

                } catch (\Exception $e) {
                    Log::warning("Lỗi khi xử lý sản phẩm (index: $index, Connection: {$dbConnection->getName()}): " . $e->getMessage(), $productData);

                    // *** THAY ĐỔI 5: Đếm lỗi và ghi chi tiết lỗi ***
                    $report['failed']++;
                    $report['errors'][] = [
                        'index' => $index,
                        'input_id' => isset($productData['id']) ? $productData['id'] : null,
                        'product_sku' => isset($productData['sku']) ? $productData['sku'] : 'N/A',
                        'error' => $e->getMessage()
                    ];
                }
            }
        }

        // === Yêu cầu 1: Chỉ return 1 lần tại cuối function ===
        return $report; // *** THAY ĐỔI 6: Trả về toàn bộ báo cáo ***
    }

    private function replaceDomain($imageUrl){
        $appUrl = env('APP_URL','https://printerval.com');
        $domain = parse_url($appUrl, PHP_URL_HOST);
        $trans = [
            "assets.$domain" => 'asset.prtvstatic.com',
            "cdn.$domain/image" => 'liveview.prtvstatic.com/image',
            "cdn.$domain/sticker" => 'sticker.prtvstatic.com/sticker',
            "cdn.$domain/unsafe" => 'liveview.prtvstatic.com/unsafe',
            "liveview.$domain" => 'liveview.prtvstatic.com',
            "sticker-liveview.$domain" => 'sticker.prtvstatic.com'
        ];
        return strtr($imageUrl,$trans);
    }

    public function setDescription(&$item, $product){
        $item['description'] = $this->merchantProductService->getDescription($product,$item['title']);
    }

    public function setTitle(&$item, $product){
        $item['title'] = $this->merchantProductService->getProductAdsTitle($product,$item['title']);
    }


    private function setLabel0(&$item, $product){
        $label = $this->merchantProductService->getLabel0($product);
        if (empty($label)){
            $item['custom_label_0'] = !empty($item['barcode']) ? $item['barcode'] : $this->merchantProductService->getLabelNoDedicatedCampaign($product,['label1' => $item['custom_label_1']]);
        }
    }
    private function setLabel1(&$item, $product){
        $item['custom_label_1'] = $this->merchantProductService->getLabel1($product);
    }
    private function setLabel2(&$item, $product){
        $item['custom_label_2'] = $this->merchantProductService->getLabel2($product);
    }
    private function setLabel3(&$item){
        $item['custom_label_3'] = $item['sold'] > 0 ? 'ordered':'';
    }
    private function setLabel4(&$item, $product){
        $item['custom_label_4'] = $this->merchantProductService->getLabel4($product);
    }

    private function setImage(&$item, $product){
        $item['image_link'] = $this->merchantProductService->buildImage($product);
    }

    private function setImages(&$item, $product){
        $images = $this->merchantProductService->getProductGallery($product);
        $item['images'] = json_encode($images);
    }

    private function setScore(&$item){
        $scoreProduct = DB::table('score_product')->where('id', $item['id'])->first(['score']);
        if (!empty($scoreProduct)){
            $item['score'] = $scoreProduct->score;
        }
    }

    private function setShipping(&$item,$product, $input){
        $shippingData = [];
        $currency = $input['currency'];
        $countryCode = $input['country'];
        $shippingCountryCodes = [];
        if (isset($input['shippingCountryCodes'])){
            $shippingCountryCodes = $input['shippingCountryCodes'];
        }
        $countryCodes = $this->merchantProductService->getShippingCoutries($countryCode, $product, $shippingCountryCodes);
        $stripeFee = doubleval(config("cart::stripe.include_fee", 0))/100;
        foreach ($countryCodes as $cCode){
            $shippingItem = [];
            $country = MerchantProductService::getCountryByCode($cCode);
            if (!empty($country)){
                $shippingItem['country'] = $cCode;
                $shippingInfo = $this->merchantProductService->getShippingTime($product, $country->id);
                if (!empty($shippingInfo)){
                    $shippingItem['shippingTime'] = $shippingInfo;
                    $fee = doubleval($shippingInfo['shipping_fee']) + round((doubleval($product->price) + doubleval($shippingInfo['shipping_fee'])) * $stripeFee,2);
                }else{
                    continue;
                }
                $shippingItem['price'] = $fee;
                $shippingItem['currency'] = $currency;
                $shippingDataItem = [
                    $cCode,
                    $shippingInfo['min_handling_time'],
                    $shippingInfo['max_handling_time'],
                    $shippingInfo['min_transit_time'],
                    $shippingInfo['max_transit_time'],
                    $shippingInfo['shipping_fee'],
                    $shippingItem['price']
                ];
                $shippingData[] = implode(":",$shippingDataItem);
            }
        }
        $item['shipping'] = json_encode($shippingData);
    }

    /**
     * Cập nhật status của `sb_product_ads` sang 'PENDING'
     * dựa trên điều kiện của bảng `sb_product`.
     *
     * @param array $filter Filter theo (product_id_from, product_id_to, updated_at_from, updated_at_to) của bảng sb_product
     * @param string|null $connectionName Tên connection CSDL
     * @return array Mảng báo cáo kết quả
     */
    public function updateAdsStatus(array $filter, $connectionName = null)
    {
        $dbConnection = null;
        $tableNameAds = 'product_ads';
        $tableNameProduct = 'product';

        try {
            // 1. Xác định CSDL để kết nối
            if ($connectionName && Config::has("database.connections.$connectionName")) {
                $dbConnection = DB::connection($connectionName);
            } else {
                $dbConnection = DB::connection(); // Dùng connection mặc định
            }

            // 2. Bắt đầu xây dựng query UPDATE...JOIN
            // Dùng alias (bí danh) "pa" cho product_ads và "p" cho product
            $query = $dbConnection->table("{$tableNameAds} AS pa")
                ->join("{$tableNameProduct} AS p", 'pa.id', '=', 'p.id');

            // 3. Áp dụng các điều kiện Trigger (sản phẩm vi phạm)
            // Dùng closure để nhóm các điều kiện OR
            $query->where(function ($q) {
                $q->where('p.status', '!=', 'ACTIVE')
                    ->orWhere('p.approve_advertising', '=', 0)
                    ->orWhereNotNull('p.deleted_at')
                    ->orWhere('p.is_trademark', '=', 1)
                    ->orWhere('p.is_violation', '=', 1);
            });

            // 4. Áp dụng các điều kiện Filter (để giới hạn phạm vi update)
            // Filter này áp dụng cho bảng `sb_product` (alias "p")
            if (array_key_exists('product_id_from', $filter)) {
                $query->where('p.id', '>=', $filter['product_id_from']);
            }
            if (array_key_exists('product_id_to', $filter)) {
                $query->where('p.id', '<=', $filter['product_id_to']);
            }
            if (array_key_exists('updated_at_from', $filter)) {
                $query->where('p.updated_at', '>=', $filter['updated_at_from']);
            }
            if (array_key_exists('updated_at_to', $filter)) {
                $query->where('p.updated_at', '<=', $filter['updated_at_to']);
            }
            $query->where('p.sold',0);
            // 5. Tối ưu: Chỉ update những hàng có status KHÁC 'PENDING'
            $query->where('pa.status', '!=', 'PENDING');
            $products = $query->get(['pa.id']);
            $affectedRows = 0;
            foreach ($products as $product){
                $affectedRows += $dbConnection->table($tableNameAds)
                    ->where('id', $product->id)
                    ->update([
                        'status' => 'PENDING',
                        'updated_at' => Carbon::now()->toDateTimeString()
                    ]);
            }
            $retVal = [
                'status' => 'successful',
                'affected_rows' => $affectedRows
            ];

        } catch (\Exception $e) {
            // 7. Xử lý nếu có lỗi
            Log::error("Lỗi khi update status product_ads: " . $e->getMessage());
            $retVal = [
                'status' => 'fail',
                'error' => $e->getMessage()
            ];
        }
        return $retVal;
    }

    /**
     * Đồng bộ (update) 'sold' VÀ 'custom_label_3'
     * từ bảng 'sb_product' sang 'sb_product_ads'.
     *
     * @param array $filter Filter theo (product_id_from, product_id_to, updated_at_from, updated_at_to) của bảng sb_product
     * @param string|null $connectionName Tên connection CSDL
     * @return array Mảng báo cáo kết quả
     */
    public function updateSold(array $filter, $connectionName = null)
    {
        $dbConnection = null;
        $tableNameAds = 'product_ads';
        $tableNameProduct = 'product';

        try {
            // 1. Xác định CSDL để kết nối
            if ($connectionName && Config::has("database.connections.$connectionName")) {
                $dbConnection = DB::connection($connectionName);
            } else {
                $dbConnection = DB::connection(); // Dùng connection mặc định
            }

            // 2. Bắt đầu xây dựng query UPDATE...JOIN
            $query = $dbConnection->table("{$tableNameProduct} AS p")
                ->where('p.sold','>',0);

            if (array_key_exists('product_id_from', $filter)) {
                $query->where('p.id', '>=', $filter['product_id_from']);
            }
            if (array_key_exists('product_id_to', $filter)) {
                $query->where('p.id', '<=', $filter['product_id_to']);
            }
            if (array_key_exists('updated_at_from', $filter)) {
                $query->where('p.updated_at', '>=', $filter['updated_at_from']);
            }
            if (array_key_exists('updated_at_to', $filter)) {
                $query->where('p.updated_at', '<=', $filter['updated_at_to']);
            }
            $products = $query->get(['p.id','p.sold']);
            $affectedRows = 0;
            foreach ($products as $product){
                $affectedRows += $dbConnection->table($tableNameAds)
                    ->where('id', $product->id)
                    ->update([
                        'sold' => $product->sold,
                        // *** ĐÃ THÊM LOGIC custom_label_3 TẠI ĐÂY ***
                        'custom_label_3' => $product->sold > 0 ? 'ordered' : '',
                        'updated_at' => Carbon::now()->toDateTimeString()
                    ]);
            }
            $retVal =  [
                'status'       => 'successful',
                'affected_rows' => $affectedRows
            ];

        } catch (\Exception $e) {
            // 6. Xử lý nếu có lỗi
            \Log::error("Lỗi khi update 'sold' & 'label_3' cho product_ads: " . $e->getMessage());
            $retVal =  [
                'status' => 'fail',
                'error'   => $e->getMessage()
            ];
        }
        return $retVal;
    }
    /**
     * Lấy ID lớn nhất từ bảng sb_product_ads.
     *
     * @param string|null $connectionName Tên connection CSDL
     * @return int ID lớn nhất, hoặc 0 nếu có lỗi hoặc không có dữ liệu.
     */
    public function getMaxProductAdId($connectionName = null)
    {
        $dbConnection = null;
        $tableName = 'product_ads';

        try {
            // 1. Xác định CSDL để kết nối
            if ($connectionName && Config::has("database.connections.$connectionName")) {
                $dbConnection = DB::connection($connectionName);
            } else {
                $dbConnection = DB::connection(); // Dùng connection mặc định
            }

            // 2. Lấy max ID
            // Hàm max() sẽ trả về NULL nếu bảng rỗng,
            // nên chúng ta dùng (int) để ép kiểu về 0
            $maxId = (int) $dbConnection->table($tableName)->max('id');

            return $maxId;

        } catch (\Exception $e) {
            // 3. Xử lý nếu có lỗi
            Log::error("Lỗi khi lấy max(id) từ bảng $tableName: " . $e->getMessage());
            return 0; // Trả về 0 nếu có lỗi
        }
    }
    public function getMaxTableId($tableName, $column = 'id', $connectionName = null)
    {
        $dbConnection = null;
        try {
            // 1. Xác định CSDL để kết nối
            if ($connectionName && Config::has("database.connections.$connectionName")) {
                $dbConnection = DB::connection($connectionName);
            } else {
                $dbConnection = DB::connection(); // Dùng connection mặc định
            }

            // 2. Lấy max ID
            // Hàm max() sẽ trả về NULL nếu bảng rỗng,
            // nên chúng ta dùng (int) để ép kiểu về 0
            $maxId = (int) $dbConnection->table($tableName)->max($column);

            return $maxId;

        } catch (\Exception $e) {
            // 3. Xử lý nếu có lỗi
            Log::error("Lỗi khi lấy max(id) từ bảng $tableName: " . $e->getMessage());
            return 0; // Trả về 0 nếu có lỗi
        }
    }
}
