<?php
namespace Modules\Ads\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Modules\Ads\Models\CategoryMeta;
use Modules\Ads\Models\ProductNCategory;

class Category extends Model {
    protected $table = "category";
    protected $guarded = ['id'];

    const TYPE_PRODUCT = 'PRODUCT';
    const TYPE_POST = 'POST';

    protected $fillable = [
        'name', 'type', 'is_hidden', 'image_url', 'big_image_url', 'description', 'slug', 'sorder', 'parent_id', 'is_display_home_page', 'created_at', 'full_url'
    ];

    protected $appends = [

    ];

    public function meta() {
        return $this->hasMany('App\Models\CategoryMeta', 'category_id', 'id');
    }

    public function parents() {
        return $this->belongsTo('\App\Models\Category', 'parent_id');
    }

    public function getLftName()
    {
        return '_lft';
    }

    public function getRgtName()
    {
        return '_rgt';
    }

    public function getParentIdName()
    {
        return 'parent_id';
    }

    public function products()
    {
        return $this->belongsToMany(Product::class, 'product_n_category')->withPivot(['sorder'])->withTimestamps();
    }

    public function posts() {
        return $this->hasMany(Post::class, 'category_id', 'id');
    }

    public function getDescendantsAttribute()
    {
        $descendants = $this->where('_lft', '>', $this->_lft)
            ->where('_rgt', '<', $this->_rgt)
            ->get();

        foreach ($descendants as $descendant) {
            $descendant->append('depth');
        }
        
        return $descendants;
    }

    public function getDepthAttribute()
    {    
        $depth = $this->where('_lft', '<', $this->_lft)
        ->where('_rgt', '>', $this->_rgt)->count();
        return $depth + 1;
    }


    protected static function boot()
    {
        parent::boot();
        self::created(function($model){
            self::updateToCategoryGoogleMerchant($model->id);

        });
        self::updated(function($model){
            self::updateToCategoryGoogleMerchant($model->id);
        });
        self::deleted(function($model){
            self::updateToCategoryGoogleMerchant($model->id,'delete');
        });
        static::addGlobalScope('sort', function (Builder $builder) {
            $builder->orderBy('category.sorder', 'desc')
                    ->orderBy('category.id', 'asc');
        });


    }

    public function categoryMetas() {
        return $this->hasMany(CategoryMeta::class, 'category_id');
    }

    public function scopeGetChildCategories($query, array $ids) {
        return $query->whereIn('parent_id', $ids)->get();
    }

    private static function updateToCategoryGoogleMerchant ($categoryId, $type = 'update') {
        $category = Category::where('id', '=', $categoryId)->where('type', '=', Category::TYPE_PRODUCT)->first();
        if ($type == 'update' && !empty($category->id)) {
            $merchant = CategoryGoogleMerchant::where('category_id', '=', $categoryId)->first();
            $treeTitle = '';
            $breadcrumbs = json_decode($category->breadcrumb);
            if (!empty($breadcrumbs)) {
                foreach ($breadcrumbs as $item) {
                    $treeTitle .= $item->name . '/';
                }
                $treeTitle = substr($treeTitle, 0, -1);
            }
            if (!empty($merchant->id)) {
                $merchant->name = $treeTitle;
                $merchant->save();
            } else {
                $obj = new categoryGoogleMerchant();
                $obj->category_id = $categoryId;
                $obj->name = $treeTitle;
                $obj->save();
            }
        } else if ($type == 'delete') {
            CategoryGoogleMerchant::where('category_id', '=', $categoryId)->delete();
        }
    }

    public function categoryGoogleMerchant() {
        return $this->hasOne(CategoryGoogleMerchant::class, 'category_id', 'id');
    }

    static $cache = [];
    public static function getBreadcrumbById($categoryId) {
        if (!array_key_exists($categoryId, self::$cache)) {
            $category = Category::where('id', $categoryId)->first(['breadcrumb']);
            $retVal = '';
            if (!empty($category->breadcrumb)){
                try{
                    $breadcrumbs = json_decode($category->breadcrumb,true);
                    foreach ($breadcrumbs as $item){
                        $retVal .= $item['name'] . ' > ';
                    }
                    $retVal = substr($retVal,0,-3);
                }catch (\Exception $e){

                }
            }
            self::$cache[$categoryId] = $retVal;
        }
        return self::$cache[$categoryId];
    }
}
