<?php
namespace Modules\Ads\Models;
use Illuminate\Database\Eloquent\Model;

use Illuminate\Support\Facades\DB;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletes;

require_once base_path() . '/app/Modules/Ads/Helpers/helper.php';

class Product extends Model
{
    use SoftDeletes;

    protected $table = 'product';
    protected $guarded = ['id'];

    protected $fillable = [
        'sku',
        'barcode',
        'name',
        'slug',
        'image_url',
        'price',
        'high_price',
        'add_shipping_fee',
        'weight',
        'status',
        'description',
        'content',
        'note',
        'inventory',
        'brand_id',
        'status_out_stock',
        'pod_parent_id',
        "approve_advertising",
        'created_at',
        'updated_at',
        'gtin'
    ];
    protected $variantDefault = null;

    protected $appends = [
        'url', 'editUrl', 'variant_default', 'display_price', 'display_high_price',
        'sale_percent', 'display_drop_price', 'brand', 'attributes'
    ];

    const DAY_OF_NEW = 10;

    public static $withoutAppends = false;

    public function galleries()
    {
        return $this->hasMany(ProductGallery::class, 'product_id', 'id')->where('type', 'PRODUCT');
    }

    public function meta()
    {
        return $this->hasMany(ProductMeta::class, 'product_id', 'id');
    }

    public function productSkus()
    {
        return $this->hasMany(ProductSku::class, 'product_id', 'id');
    }

    public function approveAdvertisings()
    {
        return $this->hasMany(ApproveAdvertising::class, 'product_id', 'id');
    }

    public function categories()
    {
        return $this->belongsToMany(Category::class, 'product_n_category')->withPivot(['sorder'])->withTimestamps();
    }

    public function brand()
    {
        return $this->belongsTo(Brand::class);
    }

    public function getUrlAttribute () {
        $retval = getPrefixUrl();
        return $retval . "/" . (!empty($this->slug) ? $this->slug : "san-pham") . "-p" . $this->id . ".html";
    }

    public function getEditUrlAttribute () {
        return "/admin/products/" . $this->id;
    }

    public function getImageUrlAttribute($value){
        return ($value) ? $value : 'https://s4.shopbay.vn/files/2/system/product-image-placeholder.png';
    }

    public function getVariantDefaultAttribute() {
        return $this->getDefautVariant();
    }

    public function getDisplayPriceAttribute() {
        if ($this->price > 0) {
            $this->price = doubleval($this->price);
            return formatPrice($this->price);
        } else {
            return translate('Liên hệ');
        }
    }

    public function getDisplayHighPriceAttribute() {
        if ($this->high_price > 0) {
            $this->high_price = doubleval($this->high_price);
            return formatPrice($this->high_price);
        }
        return;
    }

    private function getDefautVariant() {
        $variants = ProductSku::where('product_id', '=', $this->id)
                                ->where("is_default", 1)
                                ->get(['id', 'sku', 'price', 'high_price', 'image_url']);
        $retVal = [];
        if (count($variants) > 0) {
            $retVal = $variants[0];
        }
        return $retVal;
    }

    public function getBrandAttribute(){
        return Brand::where('id', '=', $this->brand_id)->first();
    }

    public function orders()
    {
        return $this->belongsToMany(Order::class, 'order_item')->withPivot(['quantity', 'price'])->withTimestamps();
    }

    public function getSalePercentAttribute () {
        if ($this->high_price && $this->price && $this->high_price > 0) {
            $salePercent = round(($this->high_price - $this->price) / $this->high_price * 100);
            if ($salePercent > 0) {
                return "-$salePercent%";
            }
        }

        return null;
    }

    public function getDisplayDropPriceAttribute () {
        return formatPrice($this->high_price - $this->price);
    }

    protected static function boot()
    {
        parent::boot();

        if (isset($_GET['q']) && $_GET['q']) {
            static::addGlobalScope('search', function (Builder $builder) {
                $keyword = $_GET['q'];
                $keyword = str_slug($keyword);
                $keywordSlug = $keyword;
                $keyword = str_replace('-', ' ', $keyword);
                $keyword = trim($keyword);
                if ($keyword) {
                    $builder->whereRaw("(match (`slug`) against ('$keyword') or slug like '%$keywordSlug%' or sku like '%$keyword%')");
                    // End tìm kiếm
                    $tableName = $builder->getModel()->getTable();
                    // $builder->select($tableName . ".*");
                    $builder->addSelect(DB::raw("match (`slug`) against ('$keyword') as lien_quan"));
                    $builder->orderBy("lien_quan", "desc");
                }
            });
        } else {
            if (isset($_GET['order_by']) && $_GET['order_by'] && $_GET['order_by'] == 'sale') {
                static::addGlobalScope('sort', function (Builder $builder) {
                    $builder->where('price', '>', 0);
                    $builder->where('high_price', '>', 0);
                    $builder->select("*");
                    $builder->addSelect(DB::raw('price/high_price as percent'));
                    $builder->orderBy('percent', 'ASC');
                });
            } else {
                static::addGlobalScope('sort', function (Builder $builder) {
                    $builder->orderBy('product.id', 'desc');
                });
            }

        }

        static::updating(function($product) {
            $oldData = Product::where("id", $product->id)->select(["name", "brand_id", "description"])->first();
            $data = [];
            if ($product->name != $oldData->name) {
                $data["title"] = $product->name;
            }
            if ($product->description != $oldData->description) {
                $data["description"] = $product->description;
            }
            if ($product->brand_id != $oldData->brand_id) {
                $brand = Brand::where("id", $product->brand_id)->first();
                $data["brand"] = $brand->name;
            }
            if (count($data) > 0) {
                $data["updated_at"] = new \DateTime();
                $approveAdvertising = ApproveAdvertising::where("product_id", $product->id)->first();
                if ($approveAdvertising) {
                    $approveAdvertising->update($data);
                }
            }
        });

        static::deleted(function($product) {
            $product->galleries()->delete();
            $product->approveAdvertising()->delete();
        });
    }

    public function filterValues()
    {
        return $this->belongsToMany(FilterValue::class, 'filter_value_n_product');
    }

    public function getAttributesAttribute () {
        $retval = [];
        $attributes = DB::table('product_meta')->where('product_id', $this->id)->get();
        foreach ($attributes as $key => $value) {
            $retval[$value->key] = $value->value;
            if($value->value){
                if ($value->value[0] == '[' || $value->value[0] == '{') {
                    $valueJson =  json_decode($value->value);
                    if (json_last_error() === JSON_ERROR_NONE) {
                        $retval[$value->key] = $valueJson;
                    }
                }
            }
        }
        return $retval;
    }

    public function productsSku(){
        return $this->hasMany(ProductSku::class, 'product_id');
    }

    public function productSku($id){
        return $this->productsSku()->where('id','=', $id);
    }

    public function comments() {
        return $this->morphMany(Comment::class, 'target');
    }

    /**
     * Check if $withoutAppends is enabled.
     *
     * @return array
     */
    protected function getArrayableAppends()
    {
        if(self::$withoutAppends){
            return [];
        }
        return parent::getArrayableAppends();
    }

    public function getTagsAttribute() {
        $result = [];
        $tagRefIds = TagRefer::where('refer_id', $this->id)
            ->where('refer_type', TagRefer::REFER_PRODUCT)
            ->get(['tag_id'])->pluck('tag_id');

        if ($tagRefIds) {
            $result = Tag::whereIn('id', $tagRefIds)->get();
        }
        return $result;
    }
}
