<?php

namespace Modules\Ads\Services;


use App\Models\User;
use Illuminate\Support\Facades\DB;
use Modules\Ads\Models\Category;
use Modules\Ads\Models\Product;


class ProductReportService
{
    static $cache = [];
    const VARIANT_SIZE = 1;
    const VARIANT_COLOR = 2;
    const VARIANT_TYPE = 5;
    const VARIANT_STYLE = 7;
    const ATTRS = [
        'size' => self::VARIANT_SIZE,
        'color' => self::VARIANT_COLOR,
        'type' => self::VARIANT_TYPE,
        'style' => self::VARIANT_STYLE,
    ];

    public function find($filter = [])
    {
        $query = $this->buildQuery($filter);
        $items = $query->get($this->statisticColumns($filter));
        $this->decorAttrName($items, $filter);
        return $items;
    }

    protected function decorAttrName($items, $filter) {
        foreach ($items as $item) {
            $item->category_name = $this->getAttrName($item->category_id, 'category');
            if ($filter['category'] == 1) {
                $addAttrCategory = $this->addAttrCategory($item->category_id);
                $item->user_find_distribute = $addAttrCategory['user_find_distribute'];
                $item->category_url = $addAttrCategory['url'];
                $item->category_created_at = date('d-m-Y H:i:s', strtotime($addAttrCategory['created_at']));
            }
            foreach (self::ATTRS as $key => $variantId) {
                if (array_key_exists($key, $filter) && $filter[$key]) {
                    $item->{$key . "_name"} = $this->getAttrName($item->{$key . '_variant_id'}, 'product_variant_option');
                }
            }
        }
    }

    private function addAttrCategory($categoryId)
    {
        $data = [];
        $category = Category::query()->find($categoryId);
        $data['created_at'] = $category->created_at ?? '';
        $data['url'] = $this->buildUrlCategory($category->full_url ?? '');
        $shopper = '';
        if (!empty($category->user_find_distributor)) {
            $user = User::query()->find($category->user_find_distributor);
            $shopper = $user ? $user->name : '';
        }
        $data['user_find_distribute'] = $shopper;

        return $data;
    }

    private function buildUrlCategory($slug)
    {
        $baseDomain = env('APP_URL', 'https://printerval.com');
        $locale = env('APP_LOCALE', 'us');
        if ($locale != '') {
            return $baseDomain . '/' . $locale . $slug;
        } else {
            return $baseDomain . $slug;
        }
    }

    protected function getAttrName($id, $table) {
        $key = $table . $id;
        if (!array_key_exists($key, self::$cache)) {
           $option = DB::table($table)
               ->where('id', $id)
               ->first(['name']);
           self::$cache[$key] = isset($option->name) ? $option->name : '-';
        }
        return self::$cache[$key];
    }



    private function handleDuplicateName($items) {
        $result = [];

        foreach ($items as $item) {
            if (!isset($result[$item->category_name])) {
                $result[$item->category_name] = $item;
            } else {
                $result[$item->category_name]->total_amount += $item->total_amount;
                $result[$item->category_name]->total_product += $item->total_product;
                $result[$item->category_name]->total_order += $item->total_order;
            }
        }

        return array_values($result);
    }

    protected function buildIoTotal(&$items, $filter) {
        $result = [];
        $filter['sort'] = [];
        $filter['isGroupBy'] = false;
        $query = $this->buildQuery($filter);
        $attrs = $this->attrs($filter);
        $columns = $this->defaultColumns($filter);
        $columns[] = 'oi.order_id';
        $ios = $query->get($columns);
        $tmp = [];
        foreach ($items as $item) {
            $key = $this->keyFromObj($item, $attrs);
            $result[$key] = $item;
        }
        foreach ($ios as $io) {
            $key = $this->keyFromObj($io, $attrs);
            if (!isset($tmp[$key][$io->order_id])) {
                $tmp[$key][$io->order_id] = 0;
            };
            $tmp[$key][$io->order_id]++;
        }
        foreach ($tmp as $key => $listIo) {
            if (isset($result[$key])) {
                $result[$key]->total_order = count($listIo);
            }
            //$result[$key]->tmp = array_sum($listIo);
        }
        return array_values($result);

    }

    protected function keyFromObj($item, $attrs) {
        $result = 'empty';
        $attrsValue = [];
        if ($attrs) {
            foreach ($attrs as $attr) {
                $attrsValue[] = $item->{$attr};
            }
        }
        if ($attrsValue) {
            $result = implode('-', $attrsValue);
        }
        return $result;
    }

    protected function attrs($filter) {
        $columns = [];
        foreach (self::ATTRS as $key => $variantId) {
            if (array_key_exists($key, $filter) && $filter[$key]) {
                $columns[] = "{$key}_name";
            }
        }
        if (array_key_exists('category', $filter) && $filter['category']) {
            $columns[] = 'category_name';
        }
        if (array_key_exists('tag', $filter) && $filter['tag']) {
            $columns[] = 'tag_name';
        }
        return $columns;
    }

    protected function statisticColumns($filter) {
        $columns = $this->defaultColumns($filter);
        $columns[] = DB::raw('sum(sb_ps.price * sb_ps.quantity) as total_amount');
        $columns[] = DB::raw('sum(sb_ps.quantity) as total_product');
        $columns[] = DB::raw('count(sb_ps.order_id) as total_order');
        return $columns;
    }

    protected function defaultColumns($filter) {
        $columns = ['product_id', 'color_variant_id', 'size_variant_id', 'type_variant_id', 'style_variant_id', 'category_id'];
        return $columns;
    }


    public function buildQuery($filter)
    {
        $isGroupBy = !array_key_exists('isGroupBy', $filter) || $filter['isGroupBy'];
        $query = DB::table('order as o')
            ->join('report_product_statistic as ps', 'ps.order_id', '=', 'o.id')
            ->where('o.payment_status', 'PAID');
        foreach (self::ATTRS as $key => $variantId) {
            if (array_key_exists($key, $filter) && $filter[$key]) {
                if ($isGroupBy) {
                    $query->groupBy("ps.{$key}_variant_id");
                }
            }
        }
        if (array_key_exists('category', $filter) && $filter['category'] && $isGroupBy) {
            $query->groupBy("ps.category_id");
        }

//        if (array_key_exists('tag', $filter) && $filter['tag']) {
//            $query->join('tag_refer as tr', function ($join) {
//                $join->on('tr.refer_id', '=', 'oi.product_id')
//                    ->where('tr.refer_type', 'PRODUCT');
//            })
//                ->join('tag as t', 't.id', '=', 'tr.tag_id');
//            if ($isGroupBy) {
//                $query->groupBy("t.id");
//            }
//        }

        if (array_key_exists('dateFrom', $filter)) {
            $query->where('o.created_at', '>=', $filter['dateFrom']);
        }

        if (array_key_exists('dateTo', $filter)) {
            $query->where('o.created_at', '<=', $filter['dateTo']);
        }
        if (array_key_exists('sort', $filter) && count($filter['sort']) == 2) {
            $query->orderBy($filter['sort'][0], $filter['sort'][1]);
        }

        return $query;


    }


    public function buildOrderQuery($filter)
    {
        $query = DB::table('order as o')
            ->join('order_item as oi', 'oi.order_id', '=', 'o.id')
            ->where('o.payment_status', 'PAID')
            ->where('o.created_at', '>=', $filter['dateFrom'])
            ->where('o.created_at', '<=', $filter['dateTo']);
        return $query;
    }

    public function buildFilter($request)
    {
        $result = $request->only([
            'category', 'color', 'style',
            'tag', 'size', 'type',
        ]);
        $currentDate = new \DateTime();
        $fromDate = $request->has('orderDateFrom') ? \DateTime::createFromFormat('d/m/Y', $request->input('orderDateFrom')) : $currentDate;
        $toDate = $request->has('orderDateTo') ? \DateTime::createFromFormat('d/m/Y', $request->input('orderDateTo')) : $currentDate;

        $result['dateFrom'] = $fromDate->format('Y-m-d 00:00:00');
        $result['dateTo'] = $toDate->format('Y-m-d 23:59:59');
        if ($request->input('sort')) {
            $result['sort'] = explode(',', $request->input('sort'));
        }
        return $result;
    }

    public function excelColumn($data) {
        $colInfo = [
            'total_amount' => [
                'name' => 'Tổng tiền'
            ],
            'total_order' => [
                'name' => 'Tổng đơn hàng'
            ],
            'total_product' => [
                'name' => 'Số lượng sản phẩm'
            ],
            'category_name' => [
                'name' => 'Category'
            ],
            'tag_name' => [
                'name' => 'Tag'
            ],
            'size_name' => [
                'name' => 'Size'
            ],
            'color_name' => [
                'name' => 'Color'
            ],
            'type_name' => [
                'name' => 'Type'
            ],
            'style_name' => [
                'name' => 'Style'
            ],
        ];
        $cols = ['A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I'];
        $fistItem = (array) $data[0];
        foreach (array_keys($fistItem) as $i => $col) {
            $colInfo[$col]['col_index'] = $cols[$i];
        }
        return $colInfo;
    }

    public function store($item) {
        DB::table('report_product_statistic')->insert($item);
    }

    public function exists($item) {
        return DB::table('report_product_statistic')
            ->where('order_item_id', $item->order_item_id)
            ->where('product_id', $item->product_id)
            ->where('product_sku_id', $item->product_sku_id)
            ->exists();
    }

}