<?php

namespace Modules\Ads\Services;

use Carbon\Carbon;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Facades\DB;
use Modules\Ads\Models\AdsProductPerformance;
use Modules\Ads\Models\Product;


class ProductWarningService
{
    protected $perPage = 40;
    protected $model;
    const WEEKLY = 'LAST_7_DAYS';
    const PRODUCT_CRAWLED = ',product-crawled';
    const CRAWLED = 1;
    const NOT_CRAWLED = 0;

    public function __construct() {
        $this->model = new AdsProductPerformance();
    }

    public function find($dataFilter) {
        $dataForPaginate = $this->getListData($dataFilter);
        $paginate = $this->customizePaginate($dataFilter['page_id'] + 1, $dataForPaginate);
        
        $metaData = [
            'page_count' => $paginate->lastPage(),
            'page_id' => $paginate->currentPage() - 1,
        ];
        
        return ['data' => $paginate, 'meta_data' => $metaData];
    }

    public function getListData($dataFilter) {
       
        $query = $this->model
        ->select([
                'ads_product_performance.*',
                'product.name',
                'product.note',
                'product.sold',
                DB::raw('sb_product.slug as slug_product'),
                DB::raw('(sb_product.sold / sb_ads_product_performance.impression) as sold_impression')
            ])
        ->join('product','product.id', 'ads_product_performance.product_id');

        $query = $this->buildFilter($query, $dataFilter);

        $listProduct = $query->get();
        $listProduct = $this->handleProductLink($listProduct);

        return $listProduct;
    }

    protected function buildFilter($query, $dataFilter) {
        if($dataFilter['sort']) {
            $dataSort = explode('-', $dataFilter['sort']);
            $field = $dataSort[0];
            $sort = $dataSort[1];
        }

        if(isset($field) && isset($sort)) {
            $query = $query->orderBy($field, $sort);
        }

        if($dataFilter['start_date']) {
            $query->where('ads_product_performance.created_at', '>=', new Carbon( str_replace('/','-',$dataFilter['start_date'])));
        }
        if($dataFilter['end_date']) {
            $query->where('ads_product_performance.created_at', '<=', new Carbon( str_replace('/','-',$dataFilter['end_date'])));
        }
        if($dataFilter['name']) {
            $query->where('product.name', 'LIKE', '%' . $dataFilter['name'] . '%');
        }
        if($dataFilter['sku']) {
            $query->where('ads_product_performance.sku', 'LIKE', '%' . $dataFilter['sku'] . '%');
        }

        if($dataFilter['crawl_status'] === self::CRAWLED) {
            $query->where('product.note', 'LIKE', '%' . self::PRODUCT_CRAWLED . '%');
        }
        if($dataFilter['crawl_status'] === self::NOT_CRAWLED) {
            $query->where('product.note', 'NOT LIKE', '%' . self::PRODUCT_CRAWLED . '%');
        }
        
        return $query->orderBy('impression', 'desc');
    }

    protected function handleProductLink($dataList) {
        foreach ($dataList as $item) {
            $item->percent_cr = number_format($item->cr * 100, 2) . '%';
            $item->percent_ctr = number_format($item->ctr * 100, 2) . '%';
            $item->percent_sold_impression = number_format($item->sold_impression * 100, 2) . '%';
            $item->image_url = route('product', ['slug' => $item->slug_product, 'id' => $item->product_id]);
        }

        return $dataList;
    }

    protected function customizePaginate(int $pageId, $data)
    {
        return new LengthAwarePaginator(
            $data->forPage($pageId, $this->perPage),
            $data->count(),
            $this->perPage,
            $pageId
        );
    }

    public function updateCrawlStatus($dataInput) {
        $record = Product::where('id', $dataInput['product_id'])->orWhere('sku', $dataInput['sku'])->first();

        if($record) {
            $data = [
                'note' => $this->handleDataCrawl($record)
            ];

            return $record->update($data);
        }
    }

    public function handleDataCrawl($record) {
        if(str_contains($record->note, self::PRODUCT_CRAWLED)) {
            $record->note = str_replace(self::PRODUCT_CRAWLED, '', $record->note);
        } else {
            $record->note .= self::PRODUCT_CRAWLED;
        }

        return $record->note;
    }
}