<?php

namespace Modules\Authorization\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use Modules\Authorization\Models\User;
use Illuminate\Support\Facades\Validator;
use Modules\Authorization\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Modules\Authorization\Models\RoleNUser;
use Modules\Authorization\Services\AuthorityService;

class UserController extends Controller
{
    public function index(Request $request) {
        return view('authorization::user');
    }

    public function findUsers(Request $request) {
        $query = User::select('id', 'name', 'email', 'status', 'slug')
            ->whereIn('role', ['STAFF', 'ADMIN'])
            ->with('roles');
        if ($request->get('search')) {
            $search = $request->get('search');
            $query->where(function ($where) use ($search) {
                $where->where('name', 'like', '%' . $search . '%')
                    ->orWhere('email', 'like', '%' . $search . '%');
            });
        }

        if ($request->get('role')) {
            $roleIds = [$request->get('role')];
            $userIds = RoleNUser::userFromRole($roleIds);
            if (!$userIds) {
                $userIds = [];
            }
            $query->whereIn('id', $userIds);
        }

        $query->orderBy('created_at', 'desc');

        return $this->buildPagination($request->all(), $query);
    }

    public function store(Request $request) {
        $retVal = [
            'status' => 'successful',
            'result' => null,
            'message' => ''
        ];
        $validator = Validator::make($request->all(), [
            'name' => 'required|string:max:255',
            'slug' => 'required|string:max:255|unique:users,email,' . $id,
            'email' => 'required|email|max:255|unique:users,email',
            'status' => 'required',
            'password' => 'required|string|max:255',
        ]);
        if ($validator->fails()) {
            $retVal['status'] = 'fail';
            $retVal['message'] = $validator->errors()->first();
        } else {
            $data = $request->only('name', 'email', 'status', 'slug');
            $data['password'] = Hash::make($request->password);
            $data['token'] = $data['password'];

            if($data['name']) {
                $data['slug'] = $slug = str_slug($data['name']);
                $exists = true;
                while ($exists) {
                    $exists = User::where('slug', $slug)->exists();
                    if($exists)
                        $slug = $data['slug'] . '-' . uniqid();
                }
                $data['slug'] = $slug;
            }
            
            $user = User::create($data);
            if ($request->roles && gettype($request->roles) == 'array') {
                $roles = $request->roles;
            } else {
                $roles = [];
            }
            $user->roles()->sync($roles);
        }

        return $retVal;
    }

    public function update(Request $request, $id) {
        $retVal = [
            'status' => 'successful',
            'result' => null,
            'message' => ''
        ];
        $parameters = Route::current()->parameters();
        $id = isset($parameters['user']) ? $parameters['user'] : $id;
        $validator = Validator::make($request->all(), [
            'name' => 'required|string:max:255',
            'slug' => 'required|string:max:255|unique:users,email,' . $id,
            'email' => 'required|email|max:255|unique:users,email,' . $id,
            'status' => 'required',
        ]);
        if (!$this->validateManagerPermission($id)) {
            $retVal['status'] = 'fail';
            $retVal['message'] = 'PERMISSION DENIED';
        } else if ($validator->fails()) {
            $retVal['status'] = 'fail';
            $retVal['message'] = $validator->errors()->first();
        } else {
            $data = $request->only('name', 'email', 'status', 'slug');
            if (isset($request->password)) {
                $data['password'] = Hash::make($request->password);
            }
            $user = User::find($id);
            if (!$user) {
                $retVal['status'] = 'fail';
                $retVal['message'] = __('User not found');
            } else {
                $user->fill($data);
                $user->save();
            }
            $roles = $this->getRoles($user, $request->roles);
            $user->roles()->sync($roles);
        }

        return $retVal;
    }

    protected function getRoles($user, $roles) {
        if (!$roles || gettype($roles) != 'array') {
            $roles = [];
        }
        if (AuthorityService::isManager() && !AuthorityService::isAdmin()) {
            $manager = AuthorityService::getUser();
            $rolesAuthUser =  RoleNUser::roleFromUser($manager->id);
            $roleUser = RoleNUser::roleFromUser($user->id);
            $diffIds = array_diff($roleUser, $rolesAuthUser);
            $roles = array_merge($roles, $diffIds);
            $roles = array_unique($roles);
        }
      
        return $roles;

    }

    public function destroy(Request $request, $id) {
        $retVal = [
            'status' => 'successful',
            'result' => null,
            'message' => ''
        ];

        $user = User::find($id);
        if (!$this->validateManagerPermission($id)) {
            $retVal['status'] = 'fail';
            $retVal['message'] = 'PERMISSION DENIED';
        } else if (!$user) {
            $retVal['status'] = 'fail';
            $retVal['message'] = __('User not found');
        } else {
            $user->delete();
            RoleNUser::where('user_id', $id)->delete();
        }

        return $retVal;
    }

    public function validateManagerPermission($userId) {
        $result = true;
        if (AuthorityService::isManager() && AuthorityService::isAdmin($userId)) {
            $result = false;
        }
        if (!AuthorityService::isAdmin() && AuthorityService::isAdmin($userId)) {
            $result = false;
        }
        return $result;
    }

    public function getToken(Request $request) {
        $mail = $request->input('email');
        $password = $request->input('password');
        $user = User::where('email', '=', $mail)->first();
        $response = [
            'status' => 'fail',
            'result' => null,
        ];
        if ($user && Hash::check($password, $user->password)) {
            $response['status'] = 'successful';
            $response['result'] = $user->token;
        }
        return $response;

    }
}
