<?php

namespace Modules\Authorization\Services;

use Modules\Authorization\Models\Permission;
use Modules\Authorization\Models\Role;
use Modules\Authorization\Models\RoleNPermission;
use Modules\Authorization\Models\RoleNUser;
use Modules\Authorization\Models\User;

class AuthorityService
{

    const CACHE_PREFIX = 'permission_';
    const ROLE_ADMIN = 'SuperUser';
    const ROLE_MANAGER = 'Manager';
    public static function hasPermission($permissionName, $method = '*', $useCache = false)
    {
        $permissionName = self::buildPermissionName($permissionName);
        if ($useCache) {
            $hasCache = self::hasCachePermission($permissionName, $method);
            if ($hasCache) {
                return self::getCachePermission($permissionName, $method);
            }
        }
        $permissionIds = Permission::getPermission($permissionName, $method);
        $result = !$permissionIds;
        $user = self::getUser();
        if ($user && isset($user->id) && $permissionIds) {
            $roleIds = RoleNPermission::roleFromPermission($permissionIds);
            $result = RoleNUser::existsRole($user->id, $roleIds);
            self::cachePermission($permissionName, $result, $method);
        }
        return $result;
    }

    protected static function buildPermissionName($permissionName) {
        $locale = env('APP_LOCALE', '');
        if ($locale && $permissionName && starts_with($permissionName, $locale)) {
            $permissionName = str_replace($locale . '/', '', $permissionName);
        }
        return $permissionName;
    }

    public static function hasRole($roleName)
    {
        $hasCache = self::hasCachePermission($roleName);
        if ($hasCache) {
            return self::getCachePermission($roleName);
        }
        $user = self::getUser();
        $role = Role::getByName($roleName);
        $result = !($role && isset($role->id));
        if ($user && isset($user->id) && $role && isset($role->id)) {
            $result = RoleNUser::existsRole($user->id, [$role->id]);
        }
        self::cachePermission($roleName, $result);

        return $result;
    }
    public static function hasRoleById($roleName, $userId) {
        $result = false;
        $role = Role::getByName($roleName);
        if ( $role && isset($role->id))  {
            $result = RoleNUser::existsRole($userId, [$role->id]);;
        }
        return $result;
    }


    public static function getCachePermission($permissionName, $method = '*')
    {
        $key = self::getKey($permissionName, $method);
        return request()->session()->get($key);
    }

    public static function hasCachePermission($permissionName, $method = '*')
    {
        $key = self::getKey($permissionName, $method);
        return request()->session()->exists($key);
    }

    public static function cachePermission($permissionName, $result, $method = '*')
    {
        $key = self::getKey($permissionName, $method);
        session([$key => $result]);
    }


    public static function getKey($permissionName, $method)
    {
        $locale = env('APP_LOCALE', '');
        $key = $locale . '_' .self::CACHE_PREFIX . $permissionName . $method;
        $key = str_slug($key);
        return $key;
    }

    public static function getUser()
    {
        $retVal = null;
        $user = auth()->user();

        if ($user && (!isset($user->status) || (isset($user->status) && $user->status == 'ACTIVE'))) {
            $retVal = $user;
        }
        return $retVal;
    }

    public static function isAdmin($userId = null) {
        if ($userId == null) {
            return self::hasRole(self::ROLE_ADMIN);
        } else {
            return self::hasRoleById(self::ROLE_ADMIN, $userId);
        }

    }
    public static function isManager() {
        return self::hasRole(self::ROLE_MANAGER);
    }

}
