<?php

namespace Modules\AutoPrice\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\AutoPrice\Controllers\Controller;
use Module;
use Modules\AutoPrice\Models\ApLog;
use Modules\AutoPrice\Models\Country;
use Modules\AutoPrice\Models\Product;
use Modules\AutoPrice\Models\ProductNCategory;
use Modules\AutoPrice\Models\ProductReact;
use Modules\AutoPrice\Models\ProductSku;
use Modules\AutoPrice\Models\ProductSkuValue;

class HomeController extends Controller
{
    public function __construct()
    {        
        Module::onView("content", function() {
            return "This is content view from AutoPrice Module HomeController";
        }, 5);
    }
    public function index(Request $request)
    {
        set_time_limit(3600);
        $data = [];
//        $categoryIds = config('auto-price::sa.category_ids', [-1]);
        $productIds = config('auto-price::sa.product_ids', [-1]);
        foreach (array_chunk($productIds, 1000) as $partProductIds) {
            foreach ($partProductIds as $partProductId) {
                $data[$partProductId] = [];
            }
            $productSkus = ProductSku::whereIn('product_id', $partProductIds)
                ->get(['id', 'product_id']);
            foreach ($productSkus as $item) {
                if (isset($data[$item->product_id])) {
                    $data[$item->product_id][] = $item->id;
                } else {
                    $data[$item->product_id] = [$item->id];
                }
            }
        }
        $listProfit = [];
        foreach ($data as $productId => $skuIds) {
            $filter = [];
            $lastLog = ApLog::where('product_id', '=', $productId)
                ->orderBy('date', 'desc')
                ->first();
            if ($lastLog) {
                $filter['from_date'] = $lastLog->date;
            }
            $dataProfit = $this->getProfit($productId, $filter);
            if ($dataProfit['click'] >= 20) {
                $listProfit[$productId] = $dataProfit;
                if ($skuIds) {
                    foreach ($skuIds as $skuId) {
                        $this->handleAutoPrice([
                            'product_id' => $productId,
                            'sku_id' => $skuId,
                            'click' => $dataProfit['click'],
                            'profit' => $dataProfit['profit'],
                        ]);
                    }
                } else {
                    $this->handleAutoPrice([
                        'product_id' => $productId,
                        'sku_id' => null,
                        'click' => $dataProfit['click'],
                        'profit' => $dataProfit['profit'],
                    ]);
                }
            }
        }
        return $listProfit;
    }

    private function getCr ($productId, $filter = []) {
        return ProductReact::getCRProduct($productId, $filter);
    }

    private function getProfit ($productId, $filter = []) {
        return ProductReact::getProfitProduct($productId, $filter);
    }

    private function handleAutoPrice ($item) {
        $obj = null;
        if ($item['sku_id']) {
            $obj = ProductSku::find($item['sku_id']);
        } else {
            $obj = Product::find($item['product_id']);
        }
        if ($obj) {
            $config = config('auto-price::sa');
            $apLog = new ApLog();
            $queryLastLog = ApLog::where('product_id', '=', $item['product_id']);
            if ($item['sku_id']) {
                $queryLastLog->where('sku_id', '=', $item['sku_id']);
            }
            $lastLog = $queryLastLog->orderBy('date', 'desc')
                ->first();
            if ($lastLog && $lastLog->price == $obj->price) {
                $minPrice = $lastLog->origin_price * (100 - $config['amplitude_percent']) / 100;
                $minPrice = round($minPrice, 2);

                if ($lastLog->profit > 0 && $item['profit'] > $lastLog->profit * 110 / 100) {
                    $newPrice = $obj->price + ($lastLog->origin_price * $config['each_percent'] / 100);
                    $newPrice = round($newPrice, 2);
                } else if ($item['profit'] < $lastLog->profit * 90 / 100 || ($item['profit'] == $lastLog->profit && $lastLog->profit == 0)) {
                    $newPrice = $obj->price - ($lastLog->origin_price * $config['each_percent'] * 0.5 / 100);
                    $newPrice = round($newPrice, 2);
                    $newPrice = max($newPrice, $minPrice);
                } else {
                    $newPrice = $obj->price;
                }

                $apLog->product_id = $item['product_id'];
                $apLog->date = date('Y-m-d');
                $apLog->price = $newPrice;
                $apLog->old_price = $obj->price;
                $apLog->origin_price = $lastLog->origin_price;
                $apLog->sku_id = $item['sku_id'];
                $apLog->sku = $item['sku_id'] ? $obj->sku : null;
                $apLog->click = $item['click'];
                $apLog->profit = $item['profit'];
                $apLog->save();
                if ($newPrice != $obj->price) {
                    $obj->price = $newPrice;
                    $obj->save();
                    if ($item['sku_id']) {
                        $product = Product::find($item['product_id']);
                        $product->updated_at = date('Y-m-d H:i:s');
                        $product->save();
                    }
                }
            } else {
                $apLog->product_id = $item['product_id'];
                $apLog->date = date('Y-m-d');
                $apLog->price = $obj->price;
                $apLog->old_price = $obj->price;
                $apLog->origin_price = $obj->price;
                $apLog->sku_id = $item['sku_id'];
                $apLog->sku = $item['sku_id'] ? $obj->sku : null;
                $apLog->click = $item['click'];
                $apLog->profit = $item['profit'];
                $apLog->save();
            }
        }
    }

    public function getStopAdsProduct (Request $request) {
        $date = $request->input('date', date('Y-m-d'));
        $data = ApLog::where('date', '=', $date)
            ->where('is_low', '=', 1)
            ->get(['product_id', 'sku_id', 'sku']);
        return response()->json([
            'status' => 'successful',
            'data' => $data
        ]);
    }

    public function report (Request $request) {
        $pageId = $request->input('page_id', 0);
        $pageSize = $request->input('page_size', 40);
        $fromDate = $request->input('date_from');
        $toDate = $request->input('date_to');
        $search = $request->input('search');
        $fromClick = $request->input('click_from');
        $toClick = $request->input('click_to');
        $fromProfit = $request->input('profit_from');
        $toProfit = $request->input('profit_to');
        $isStopAds = $request->input('is_stop_ads');
        $query = DB::table('ap_log');
        if ($fromDate) {
            $dt = new \DateTime();
            $dt->setTimestamp($fromDate);
            $query->where('date', '>=', $dt->format('Y-m-d'));
        }
        if ($toDate) {
            $dt = new \DateTime();
            $dt->setTimestamp($toDate);
            $query->where('date', '<=', $dt->format('Y-m-d'));
        }
        if ($search) {
            $productIds = Product::where('sku', '=', $search)
                ->orWhere('name', 'like', '%' . $search . '%')
                ->limit(1000)
                ->pluck('id')->toArray();
            if (count($productIds) == 0) {
                $productIds = [-1];
            }
            $query->whereIn('product_id', $productIds);
        }
        if ($fromClick) {
            $query->where('click', '>=', intval($fromClick));
        }
        if ($toClick) {
            $query->where('click', '<=', intval($toClick));
        }
        if ($fromProfit) {
            $query->where('profit', '>=', floatval($fromProfit));
        }
        if ($toProfit) {
            $query->where('profit', '<=', floatval($toProfit));
        }
        if ($isStopAds) {
            $query->where('is_low', '=', $isStopAds);
        }
        $query->groupBy('product_id');
        $query->groupBy('date');
        $query->orderBy('date', 'desc');
        $query->orderBy('product_id', 'asc');
        $query->select([
            'product_id',
            'click',
            'date',
            'profit',
            'price as new_price',
            'old_price as price',
            'is_low as is_stop_ads'
        ]);
        $totalCount = DB::table(DB::raw("({$query->toSql()}) as sub"))
            ->mergeBindings($query)
            ->count();
        $pageCount = $this->recordsCountToPagesCount($totalCount, $pageSize);
        $data = $query->forPage($pageId + 1, $pageSize)->get();
        foreach ($data as $item) {
            $product = Product::find($item->product_id);
            $item->name = $product ? $product->name : '';
            $item->sku = $product ? $product->sku : '';
            $item->slug = $product ? $product->slug : '';
        }
        return response()->json([
            'status' => 'successful',
            'data' => $data,
            'meta' => [
                'page_count' => $pageCount,
                'page_id' => intval($pageId),
                'page_size' => $pageSize,
                'off_set' => $pageId * $pageSize,
                'total_count' => $totalCount,
                'has_next' => $pageId + 1 >= $pageCount ? false : true
            ]
        ]);
    }

    public function autoPriceReport()
    {
        return view('auto-price::report/index');
    }

    public function rebuildLog () {
        set_time_limit(0);
        $logs = ApLog::all();
        foreach ($logs as $log) {
            $pr = ProductReact::getCRProduct($log->product_id, ['date' => $log->date]);
            $log->click = $pr['click'];
            $log->conversion = $pr['conversion'];
            $log->save();
        }
        return response()->json([
            'status' => 'successful'
        ]);
    }

    public function getResult ($date) {
        $productIds = ApLog::where('date', '=', $date)
            ->whereRaw('price != old_price')
            ->groupBy('product_id')
            ->pluck('product_id')
            ->toArray();
        $data = [
            'previous' => [
                'profit' => 0,
                'quantity' => 0,
            ],
            'after' => [
                'profit' => 0,
                'quantity' => 0,
            ],
            'product_ids' => []
        ];
        if ($productIds) {
            $data['product_ids'] = $productIds;
            $previousItems = DB::table('order_item as oi')
                ->leftJoin('order as o', 'o.id', '=', 'oi.order_id')
                ->whereIn('oi.product_id', $productIds)
                ->where('o.created_at', '<=', date('Y-m-d 18:00:00', strtotime($date)))
                ->where('o.created_at', '>', date('Y-m-d 18:00:00', strtotime('-1 day', strtotime($date))))
                ->get(['oi.product_id', 'oi.product_sku_id', 'oi.quantity', 'o.country_id', 'oi.price'])
                ->toArray();
            foreach ($previousItems as $item) {
                $item = (array) $item;
                $profit = $this->getProfitByItem($item);
                $data['previous']['profit'] += $profit;
                $data['previous']['quantity'] += $item['quantity'];
            }


            $afterItems = DB::table('order_item as oi')
                ->leftJoin('order as o', 'o.id', '=', 'oi.order_id')
                ->whereIn('oi.product_id', $productIds)
                ->where('o.created_at', '<=', date('Y-m-d 18:00:00', strtotime('+1 day', strtotime($date))))
                ->where('o.created_at', '>', date('Y-m-d 18:00:00', strtotime($date)))
                ->get(['oi.product_id', 'oi.product_sku_id', 'oi.quantity', 'o.country_id', 'oi.price'])
                ->toArray();
            foreach ($afterItems as $item) {
                $item = (array) $item;
                $profit = $this->getProfitByItem($item);
                $data['after']['profit'] += $profit;
                $data['after']['quantity'] += $item['quantity'];
            }
        }
        return $data;
    }

    private function getProfitByItem ($item) {
        $id = $item['product_id'];
        $column = 'product_id';
        if (!empty($item['product_sku_id'])){
            $id = $item['product_sku_id'];
            $column = 'product_sku_id';
        }
        $cost = Product::getProductCost($id, $column, $item['country_id']);
        if(empty($cost)){
            $cost = $item['price'] * 0.6;
        }
        $countryCode = 'US';
        if(!empty($item['country_id'])){
            $country = Country::find($item['country_id']);
            if ($country) {
                $countryCode = $country->iso;
            }
        }
        $tax = Product::getTaxPercent($countryCode);
        $totalCost = $cost * $tax;
        return $item['quantity'] * ($item['price'] - $totalCost);
    }

    public function getSummaryResult (Request $request) {
        $from = $request->input('from_date', date('Y-m-d', strtotime('-1 day')));
        $to = $request->input('to_date', date('Y-m-d'));
        $result = [
            'status' => 'successful',
            'items' => [],
            'summary' => [
                'quantity' => 0,
                'profit' => 0,
            ],
        ];
        for ($i = 0; (strtotime($from) + 86400 * $i) <= strtotime($to); $i++) {
            $date = date('Y-m-d', strtotime($from) + 86400 * $i);
            $data = $this->getResult($date);
            $result['items'][$date] = $data;
            $result['summary']['quantity'] += $data['after']['quantity'] - $data['previous']['quantity'];
            $result['summary']['profit'] += $data['after']['profit'] - $data['previous']['profit'];
        }

        return response()->json($result);
    }
}
