<?php

namespace Modules\AutoPrice\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Modules\AutoPrice\Models\Country;
use Modules\AutoPrice\Models\Order;
use Modules\AutoPrice\Controllers\Controller;
use Modules\AutoPrice\Models\Product;
use Modules\AutoPrice\Models\ProductReact;
use Modules\AutoPrice\Services\ClickerService;

class ProductReactController extends Controller
{
    const SOURCES = ['adwords', 'bing ads'];

    public function find(Request $request) {
        $filter['date_gte'] = $request->input('date_from', date('Y-m-d', time()));
        $filter['date_lte'] = $request->input('date_to', date('Y-m-d', time()));
        if ($request->input('product_id')) {
            $filter['product_id'] = $request->input('product_id');
        }
        $items = ProductReact::buildProductReactQuery($filter)->get();
        $response = [
            'status' => 'successful',
            'result' => $items,
        ];
        return response()->json($response);

    }

    public function cronProductReact(Request $request)
    {
        set_time_limit(0);
        $from = $request->input('from', -1);
        $to = $request->input('to', 0);
        $result = [];
        for ($i = $from; $i < $to; $i++) {
            $timeFrom = time() + ($i * 86400);
            $datetimeFrom = date('Y-m-d 18:00:00', $timeFrom);
            $datetimeTo = date('Y-m-d 18:00:00', $timeFrom + 86400);
            $filter = [
                'from_time' => $datetimeFrom,
                'to_time' => $datetimeTo,
            ];
            $items = $this->handleProductReact($filter);
            $result[$datetimeTo] = count($items);
            Log::info(__METHOD__, $items);
        }
        $listCr = [];
        if ($request->input('is_build_log')) {
            $listCr =  app()->call('\Modules\AutoPrice\Controllers\HomeController@index', ['request' => $request]);
        }

        $response = [
            'status' => 'successful',
            'result' => $result,
            'listCr' => $listCr
        ];
        return response()->json($response);
    }

    public function reportOrder(Request $request) {
        $from = $request->input('from', -2);
        $to = $request->input('to', 0);
        $productIds = $this->getProductReport($from + 1);
        //get ProductIds
        $result = $this->initResult($productIds, $from);

        for ($i = $from; $i < $to; $i++) {
            $timeFrom = time() + ($i * 86400);
            $date = date('Y-m-d', $timeFrom + 86400);
            $datetimeFrom = date('Y-m-d 18:00:00', $timeFrom);
            $datetimeTo = date('Y-m-d 18:00:00', $timeFrom + 86400);
            $filter = [
                'from_time' => $datetimeFrom,
                'to_time' => $datetimeTo,
                'product_ids' => $productIds,
            ];
            $orders = $this->getConversion($filter);
            $this->groupOrder($result, $orders, $date);
        }
        return response()->json(array_values($result));
    }

    protected function initResult($productIds, $from) {
        $time = time() + ($from * 86400) + 86400;
        $date = date('Y-m-d', $time);
        $nextDate = date('Y-m-d', $time + 86400);
        $result = [];
        foreach ($productIds as $pId) {
            if (!array_key_exists($pId, $result)) {
                $product = DB::table('product')
                    ->where('id', $pId)
                    ->first(['id', 'name', 'slug', 'sku']);
                $history = $this->getPriceFromHistory($pId, $date);
                $price = isset($history->price) ? $history->price : 0;
                $oldPrice = isset($history->old_price) ? $history->old_price : '';
                $priceStatus = $price >  $oldPrice ? 'up' : 'down';
                $result[$pId] = [
                    'id' => $pId,
                    'name' => isset($product->name) ? $product->name : '',
                    'sku' => isset($product->sku) ? $product->sku : '',
                    'url' => $this->buildProductUrl($product, $pId),
                    'price_status' => $priceStatus,
                    'Price ' . $date => $oldPrice,
                    'Price ' . $nextDate => $price,
                ];
            }
        }
        return $result;
    }

    protected function buildProductUrl($product, $pId) {
        $locale = env('APP_LANG', '');
        $url =  url('/' . $locale . '/san-pham'  . '-p' . $pId);
        if ($product) {
            $url =  url('/' . $locale . '/' . $product->slug . '-p' . $product->id);
        }
        return $url;
    }

    protected function groupOrder(&$result, $orders, $date) {
        foreach ($orders as $order) {
            $pId = $order['product_id'];
            $result[$pId][$date] = $order['conversion'];
        }
        foreach ($result as &$item) {
            if (!array_key_exists($date, $item)) {
                $item[$date] = 0;
            }
        }
    }

    protected function getPriceFromHistory($pId, $date) {
        $skuIdDefault = DB::table('product_sku')
            ->where('product_id', $pId)
            ->where('is_default', 1)
             ->first(['id']);
        $result = null;
        if ($skuIdDefault) {
            $result = $this->getProductLog($skuIdDefault->id, $date);
        }
        if (!$result) {
            $result = $this->getProductLog(null, $date);
        }
        return $result;

    }

    protected function getProductLog($skuId, $date) {
        $query = DB::table('ap_log')
            ->where('date', $date);
        if ($skuId) {
            $query->where('sku_id', $skuId);
        }
        return $query->first(['id', 'price', 'old_price']);
    }


    protected function getProductReport($from) {
        $timeFrom = time() + ($from * 86400);
        $date = date('Y-m-d', $timeFrom);
        return DB::table('ap_log')->where('date', $date)
            ->groupBy('product_id')
            ->pluck('product_id')->toArray();
    }


    public function handleProductReact($filter) {
        $clicks = $this->getClick($filter);
        $orders = $this->getConversion($filter);
        $items = $this->mergeProductReact($clicks, $orders, $filter);
        if ($items) {
            foreach ($items as $item) {
                $this->insertProductReact($item);
            }
        }
        return $items;

    }


    public function getConversion($filter) {
        $columns = [
            'order_item.product_id',
            DB::raw("count(*) as conversion"),
            'order.id as order_id',
            'order_item.product_sku_id',
            'order.country_id',
            'order_item.quantity',
            'order_item.price'
        ];
        $result = [];
        $queryFilter = [
            'type' => Order::STATUS_OUT,
            'payment_status' => 'PAID',
            'create_time_gte' => $filter['from_time'],
            'create_time_lte' => $filter['to_time'],
            'join_order_item' => 1,
            'group_by' => ['order_item.product_id', 'order_item.order_id']
        ];
        if (array_key_exists('product_ids', $filter)) {
            $queryFilter['product_ids'] = $filter['product_ids'];
        }
        $queryFilter['sources'] = self::SOURCES;
        $query = Order::buildOrderQuery($queryFilter);
        $items = $query->get($columns)->toArray();
        if ($items) {
            foreach ($items as $item) {
                $pId = $item['product_id'];
                if (!array_key_exists($pId, $result)) {
                    $result[$pId] = [
                        'product_id' => $pId,
                        'conversion' => 0,
                        'profit' => 0
                    ];
                }
                $result[$pId]['conversion']++;
                $id = $item['product_id'];
                $column = 'product_id';
                if (!empty($item['product_sku_id'])){
                    $id = $item['product_sku_id'];
                    $column = 'product_sku_id';
                }
                $cost = Product::getProductCost($id, $column, $item['country_id']);
                if(empty($cost)){
                    $cost = $item['price'] * 0.6;
                }
                $countryCode = 'US';
                if(!empty($item['country_id'])){
                    $country = Country::find($item['country_id']);
                    if ($country) {
                        $countryCode = $country->iso;
                    }
                }
                $tax = Product::getTaxPercent($countryCode);
                $totalCost = $cost * $tax;
                $result[$pId]['profit'] += $item['quantity'] * ($item['price'] - $totalCost);
            }
            $result = array_values($result);
        }
        return $result;

    }

    public function getClick($filter = []) {
        //@todo env
        $site = 'printerval.com';
        $countryCode = env('APP_LOCALE');
        if ($countryCode) {
            $site .= '/' . $countryCode;
        }
        $filter['sources'] = implode(',', self::SOURCES);
        $filter['site'] = $site;
        $filter['page_size'] = 10000;
        $clickerService = new ClickerService();
        $items = $clickerService->getClickData($filter);
        return $items;
    }


    public function insertProductReact($item) {
        ProductReact::where('product_id', $item['product_id'])
            ->where('date', $item['date'])->delete();
        ProductReact::create($item);
    }


    public function mergeProductReact($clicks, $orders, $filter) {
        $date = explode(' ', $filter['to_time'])[0];
        $types = [
            'click' => $clicks,
            'conversion' => $orders,
            'profit' => $orders
        ];
        $result = [];
        foreach ($types as $type => $items) {
            foreach ($items as $item) {
                $pId = $item['product_id'];
                if (!isset($result[$pId])) {
                    $result[$pId] = $this->defaultItem($pId, $date);
                }
                $result[$pId][$type] += $item[$type];
            }
        }
        return $result;

    }

    protected function defaultItem($pId, $date) {
        return [
            'date' => $date,
            'product_id' => $pId,
            'click' => 0,
            'conversion' => 0,
            'profit' => 0,
            'created_at' => new \DateTime(),
        ];
    }




}
