<?php

namespace Modules\AutoPrice\Models;

use Illuminate\Support\Facades\DB;
use Modules\AutoPrice\Models\Traits\BaseTrait;
use Illuminate\Database\Eloquent\Model;

class ProductReact extends Model
{
    use BaseTrait;
    public $timestamps = false;
    protected $table = "ap_product_react";
    protected $fillable = [
        'product_id', 'click', 'profit',
        'conversion',  'date', 'created_at'
    ];
    static $cache = [];
    static $operationFilter = [
        'eq' => [
            'product_id', 'date'
        ],
        //>
        'gt' => [
        ],
        // >=
        'gte' => [
            [
                'filter' => 'date_gte',
                'column' => 'date',
            ]
        ],
        //<
        'lt' => [],
        // <=
        'lte' => [
            [
                'filter' => 'date_lte',
                'column' => 'date',
            ]
        ],
        //!=
        'ne' => [
        ],
        //not in
        'nin' => [
        ],
        //in
        'in' => [
        ],
        'like' => []
    ];

    public static function buildProductReactQuery($filter) {
        $query = self::buildQuery($filter);
        return $query;
    }

    public static function getCRProduct($pId, $filter = []) {
        $maxDate = 31;
        $result = [
            'cr' => -1,
            'click' => 0,
            'conversion' => 0
        ];
        $startTime = time();
        if (!empty($filter['date'])) {
            $startTime = strtotime($filter['date']);
        }
        if (array_key_exists('from_date', $filter)) {
            $date = $filter['from_date'];
            $item = self::getCRByDate($pId, $date);
            if (isset($item->click) && $item->click >= 30) {
                $result = self::buildConversion($pId, $item, $date);
            }
        } else {
            for ($i = 0; $i < $maxDate; $i += 5) {
                $time = $startTime - ($i * 86400);
                $date = date('Y-m-d',  $time);
                $item = self::getCRByDate($pId, $date);
                if (isset($item->click) && $item->click >= 30) {
                    $result = self::buildConversion($pId, $item, $date);
                    break;
                }
            }
        }
        return $result;

    }

    private static function buildConversion($pId, $item, $date) {
        $result['cr'] = $item->conversion/$item->click;
        self::log($pId, [
            'date' => $date,
            'item' => $item,
        ]);
        $result['click'] = $item->click;
        $result['conversion'] = $item->conversion;
        return $result;
    }

    protected static function log($pId, $data) {
        DB::table('log')->insert([
            'target_type' => 'cr-product',
            'event_type' => 'cr-product',
            'target_id' => $pId,
            'data' => json_encode($data),
            'created_at' => new \DateTime()
        ]);
    }

    protected static function getCRByDate($pId, $fromDate) {
        return self::buildProductReactQuery([
            'product_id' => $pId,
            'date_gte' => $fromDate,
        ])->first([
            DB::raw('sum(click) as click, sum(conversion) as conversion')
        ]);

    }

    public static function getProfitProduct($pId, $filter = []) {
        $maxDate = 31;
        $result = [
            'click' => 0,
            'profit' => 0,
        ];
        $startTime = time();
        if (!empty($filter['date'])) {
            $startTime = strtotime($filter['date']);
        }
        if (array_key_exists('from_date', $filter)) {
            $date = $filter['from_date'];
            $item = self::getProfitByDate($pId, $date);
            if (isset($item->click) && $item->click >= 20) {
                $result['click'] = $item->click;
                $result['profit'] = $item->profit;
            }
        } else {
            for ($i = 0; $i < $maxDate; $i++) {
                $time = $startTime - ($i * 86400);
                $date = date('Y-m-d',  $time);
                $item = self::getProfitByDate($pId, $date);
                if (isset($item->click) && $item->click >= 20) {
                    $result['click'] = $item->click;
                    $result['profit'] = $item->profit;
                    break;
                }
            }
        }
        if (!$result['profit']) {
            $result['profit'] = 0;
        }
        return $result;

    }

    protected static function getProfitByDate ($pId, $fromDate) {
        return self::buildProductReactQuery([
            'product_id' => $pId,
            'date_gte' => $fromDate,
        ])->first([
            DB::raw('sum(click) as click, sum(profit) as profit')
        ]);
    }
}
