<?php

namespace Modules\AutoPrice\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Modules\AutoPrice\Controllers\Controller;

class ClickerService
{

    public function getClickData($filter) {
        //get paginator
        $paginator = $this->getPaginator($filter);
        //loop
        $data = [];
        $i = 1;
        $pageCount = $paginator['pageCount'];
        $filterFromId = $paginator['fromId'];
        $pageSize = $paginator['pageSize'];
        while ($i <= $pageCount) {
            $fromId = $filterFromId + (($i - 1) * $pageSize);
            $toId = $filterFromId + ($i * $pageSize);
            if ($toId > $paginator['toId']) {
                $toId = $paginator['toId'];
            }
            $clickerParams = array_merge($filter, [
                'from_id' => $fromId,
                'to_id' => $toId,
                'columns' => 'id,url,ip,create_time,source',
            ]);
            $items = $this->clickerRequest($clickerParams);
            $itemsData = $this->parseItems($items);
            if ($itemsData) {
                $data = array_merge($data, $itemsData);
            }
            $i++;
        }
        $result = $this->groupItems($data);
        return $result;
    }

    protected function groupItems($items) {
        $result = [];
        foreach ($items as $item) {
            $pId = $item['product_id'];
            if (!array_key_exists($pId, $result)) {
                $result[$pId] = [
                    'click' => 0,
                    'product_id' => $pId,
                ];
            }
            $result[$pId]['click']++;
        }
        return $result;
    }

    protected function parseItems($items) {
        $result = [];
        if ($items) {
            foreach ($items as $item) {
                $itemData = $this->parseItem($item);
                if ($itemData) {
                    $result[] = $itemData;
                }
            }
        }
        return $result;
    }

    protected function parseItem($item) {
        $urlParser = parse_url($item['url']);
        $result = [];
        $productId = null;
        preg_match('/.*-p([0-9]+)\.html$/i', $urlParser['path'], $match);
        if ($match) {
            $productId = isset($match[1]) ? $match[1] : '';
        }
        if (!$match) {
            preg_match('/.*-p([0-9]+)/i', $urlParser["path"], $match);
            $productId = isset($match[1]) ? $match[1] : '';
        }
        if ($productId) {
            $dateExplode = explode(' ', $item['create_time']);
            $result = [
                'product_id' => $productId,
                'date' => $dateExplode[0]
            ];
        }
        return $result;
    }

    protected function getKeywordFromUrl($url) {
        $separator = $this->trackingClickerParams . '=';
        $urlExplode = explode($separator, $url);
        $result = null;
        if ($urlExplode && isset($urlExplode[1]) && $urlExplode[1]) {
            $searchTrackingExplode =  explode('&', $urlExplode[1]);
            $result = $searchTrackingExplode[0];
        }
        return $result;
    }

    protected function getPaginator($filter) {
        $pageSize = $filter['page_size'];
        $filter['page_size'] = 1;
        $filter['order_by'] = 'create_time,asc';
        $filter['columns'] = 'id';

        $firstItem = $this->clickerRequest($filter);
        $filter['order_by'] = 'create_time,desc';
        $lastItem = $this->clickerRequest($filter);
        $fromId = $firstItem && isset($firstItem[0]['id']) ? $firstItem[0]['id'] : -1;
        $toId = $lastItem && isset($lastItem[0]['id']) ? $lastItem[0]['id'] : -1;
        return [
            'fromId' => $fromId,
            'toId' => $toId,
            'pageCount' => ceil(($toId - $fromId)/$pageSize),
            'pageSize' => $pageSize,
        ];
    }

    protected function clickerRequest($params = [], $path = '/service/get-data-tracking') {
        $username = config('app.clickerAuthBasic.username');
        $password = config('app.clickerAuthBasic.password');
        $clickerUrl = config("app.clickerDomain");
        if ($username == null || $password == null) {
            throw new \Exception('The module require "clickerAuthBasic.username" and "clickerAuthBasic.password" in /config/app.php');
        }
        $headers = array(
            'Content-Type:application/json',
            'Authorization: Basic '. base64_encode($username . ":" . $password)
        );
        $apiUrl = $clickerUrl . $path;
        if ($params) {
            $query = http_build_query($params);
            $apiUrl = $apiUrl . '?' . $query;
        }
        $response = $this->triggerSyncRequest($apiUrl, 'GET', [], $headers);

        return $response && array_key_exists('data', $response) ? $response['data'] : [];
    }

    protected function triggerSyncRequest($url, $method = 'GET', $params = [], $headers = []) {
        $ch = curl_init();
        $timeout = 200;
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        if ($headers) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }
        if ($method != 'GET') {
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($params));
        }
        curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
        $data = curl_exec($ch);
        curl_close($ch);
        return json_decode($data, true);
    }
}
