<?php

namespace Modules\BoughtTogether\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Modules\BoughtTogether\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Module;
use Modules\BoughtTogether\Models\OrderItem;
use Modules\BoughtTogether\Models\Product;
use Modules\BoughtTogether\Models\ProductBoughtTogether;
use Modules\BoughtTogether\Models\ProductMeta;
use App\Helpers\ApiClient;
use Modules\BoughtTogether\Models\Category;
use Modules\BoughtTogether\Models\ProductSku;
use Modules\BoughtTogether\Models\Tag;
use Modules\BoughtTogether\Models\TagRefer;
use Modules\BoughtTogether\Models\TagStatistic;

class HomeController extends Controller
{
    static $tagData = [];
    static $cache = [];
    static $hit =  0;
    static $miss =  0;
    const COLOR_VARIANT_ID = 2;
    protected $bestSeller = [];

    public function getBoughtTogetherProductsNoAjax($params) {
        $productId = $params['product_id'];
        $productMeta = ProductMeta::where('product_id', '=', $productId)
            ->where('key', '=', 'bought_together_products')
            ->first();
        $isCustomize = \DB::table('product_custom')->where('product_id', $productId)->exists();
        if ($productMeta && !$isCustomize && is_array(json_decode($productMeta->value, true))) {
            $hasBTGProducts = Product::where("status", "ACTIVE")
                ->whereIn("id", array_slice(json_decode($productMeta->value, true), 0, 8))
                ->exists();
            if ($hasBTGProducts) {
                return view('bought-together::home.no-ajax.bought-together-products');
            }
        }
    }

    public function getBoughtTogetherProducts($params) {
        $productId = $params['product_id'];
        $showNewIcon = isset($params['showNewIcon']) ? $params['showNewIcon'] : false;
        $productMeta = ProductMeta::where('product_id', '=', $productId)
            ->where('key', '=', 'bought_together_products')
            ->first();
        $isCustomize = \DB::table('product_custom')->where('product_id', $productId)->exists();
        if ($productMeta && !$isCustomize && is_array(json_decode($productMeta->value, true))) {
            $products = Product::where("status", "ACTIVE")
                ->whereIn("id", array_slice(json_decode($productMeta->value, true), 0, 8))
                ->get(['id', 'sku', 'name', 'slug', 'status', 'image_url', 'price', 'high_price', 'created_at']);
            if (count($products) > 0) {
                priceChangeForceDecor($products, 'recommendation');
                return view('bought-together::home.bought-together-products', [
                    'products' => $products,
                    'showNewIcon' => $showNewIcon
                ]);
            }

        }    
    }

    public function generate(Request $request)
    {
        ini_set('memory_limit', '2048M');
        set_time_limit(4 * 3600);
        $sendToQ = $request->input('send_to_queue', 0);
        $this->bestSeller = $this->getBestSeller();
        $this->logGenerate('start', [
            'input' => $request->all(),
        ]);
        $pIdsFromView = $this->generateProductFromView();
        $this->updateBoughtTogetherFromList($pIdsFromView);
        $listProductBought = $this->getProductFromOrder(date('Y-m-d 00:00:00', strtotime('-2 days')));
        $this->updateBoughtTogetherFromList($listProductBought);
        $resData = [
            'status' => 'successful',
            'tag' => [
                'hit' => self::$hit,
                'miss' => self::$miss,
            ],
            'activeProductIds' => 0,
            'listProductBought' => $listProductBought ? count($listProductBought) : 0,
            'countViewProduct' => count($pIdsFromView),
        ];
        $this->generateNewProduct($sendToQ);
        $this->logGenerate('end', [
            'response' => $resData,
            'input' => $request->all()
        ]);
        return response()->json($resData);
    }



    public function generateNewProduct($sendToQ = false) {
        $minId = $this->getStartId();
        $maxId = DB::table('product')->max('id');
        $chunkSize = 50000;
        for ($fromId = $minId; $fromId < $maxId; $fromId += $chunkSize) {
            $toId = $fromId + $chunkSize;
            if ($sendToQ) {
                $url ='https://restq.printerval.com/bought-together/generate-by-id?from_id=' . $fromId . '&to_id=' . $toId . '&locale=' . env('APP_LOCALE', 'us');
                $this->triggerAsyncRequest($url);
            } else {
                $this->updateBoughtTogetherByRange($fromId, $toId);
            }
            $this->setStartId($toId);
        }

    }

    public function generateById(Request $request) {
        ini_set('memory_limit', '1048M');
        set_time_limit(4 * 3600);
        $startTime = time();
        $fromId = $request->input('from_id');
        $toId = $request->input('to_id');
        $this->logGenerate('generateById-start', [
            'input' => $request->all(),
        ]);
        $this->updateBoughtTogetherByRange($fromId, $toId);
        $this->logGenerate('generateById-end', [
            'input' => $request->all(),
            'runtime' => time() - $startTime
        ]);
        return [
            'status' => 'successful',
        ];
    }

    protected function updateBoughtTogetherByRange($fromId, $toId) {
        if ($fromId && $toId) {
            for($id = $fromId; $id < $toId; $id += 1000) {
                $ids = range($id, $id + 1000);
                $this->updateBoughtTogetherFromList($ids);
            }
        }
    }

    protected function getProductFromOrder($fromTime) {
        return DB::table('order')
            ->join('order_item', 'order.id', '=', 'order_item.order_id')
             ->where('order.created_at', '>=', $fromTime)
            ->where('order.payment_status', '=', 'PAID')
            ->pluck('product_id', 'product_id')
            ->toArray();
    }

    protected function maxHandleProduct() {
        $maxId = DB::table('product')->max('id');
        $result = ceil($maxId/60);
        if ($result > 1000000) {
            $result = 1000000;
        }
        return $result;
    }


    protected function generateProductFromView() {
        $pIds = DB::table('user_viewed')
            ->where('created_at', '>', date('Y-m-d H:i:s', time() - 88000))
            ->groupBy('target')
            ->pluck('target')
            ->toArray();
        return $pIds;
    }

    protected function getNewProduct() {
        return DB::table('product')
            ->where('created_at', '>', date('Y-m-d H:i:s', time() - 88000))
            ->pluck('id')
            ->toArray();
    }




    protected function getStartId() {
        $result = 0;
        $item =  DB::table('option')
            ->where('key', $this->getKey())
            ->first();
        if (isset($item->id)) {
            $data = json_decode($item->value);
            $result = isset($data->id) ? $data->id : 0;
        }
        if (!$result) {
            $data = DB::table('product')
                ->where('created_at', '>', date('Y-m-d H:i:s', time() - 88000))
                ->orderBy('created_at', 'asc')
                ->first(['id']);
            $result = $data->id;
        }
        return $result;
    }

    protected function setStartId($id) {
        $maxId = DB::table('product')->max('id');
        if ($id > $maxId) {
            $id = $maxId;
        }
        $data = [
            'id' => $id,
            'date' => date('Y-m-d H:i:s', time())
        ];
        $this->setOptionBT(json_encode($data));
    }

    protected function getOption() {
        return DB::table('option')
            ->where('key', $this->getKey())
            ->first();
    }

    protected function setOptionBT($data) {
        $item = DB::table('option')
            ->where('key', $this->getKey())
            ->first();
        if (isset($item->id)) {
            DB::table('option')
                ->where('id', $item->id)
                ->update([
                    'value' => $data
                ]);
        } else {
            DB::table('option')->insert([
                    'value' => $data,
                    'key' => $this->getKey(),
                ]);
        }
    }

    protected function getKey() {
        return 'BoughtTogether::generatev2';
    }

    protected function updateBoughtTogetherFromList($pIds) {
        if (!$pIds) {
            return;
        }
        foreach (array_chunk($pIds, 200) as $chunkPids) {
            $this->updateBoughtTogether($chunkPids);
        }

    }

    protected function updateBoughtTogether($pIds) {
        $pNCategory = $this->getProductNCategory($pIds);
        foreach ($pIds as $productId) {
            $boughtTogetherProductMeta = DB::table('product_meta')
                ->where('product_id', '=', $productId)
                ->where('key', '=', 'bought_together_products')
                ->first(['id']);
            $categoryId = isset($pNCategory[$productId]) ? $pNCategory[$productId] : 0;
            $this->updateBoughtTogetherProduct($productId, $boughtTogetherProductMeta, $categoryId);
        }
    }

    protected function getProductNCategory($pIds) {
        return DB::table('product_n_category')
            ->whereIn('product_id', $pIds)
            ->where('is_parent', 0)
            ->pluck('category_id', 'product_id')
            ->toArray();
    }

    protected function getProductIds($fromId, $toId) {
        return  DB::table('product')
            ->where('status', '=', 'active')
            ->whereNull('deleted_at')
            ->where('id', '>=', $fromId)
            ->where('id', '<', $toId)
            ->pluck('id')
            ->toArray();
    }

    protected function logGenerate($type, $data) {
        DB::table('log')->insert([
            'target_type' => $type,
            'event_type' => 'log-bought-together-generate',
            'data' => json_encode($data),
            'created_at' => new \DateTime()
        ]);
    }

    private function updateBoughtTogetherProduct($productId, $boughtTogetherProductMeta, $categoryId = null) {
        $retval = [];
        $productSameTagStr = '';
        // Select bought together product ids
        $boughtTogetherIds = ProductBoughtTogether::getByProduct($productId);
        if ($boughtTogetherIds) {
            $this->addBoughtTogether($boughtTogetherIds, $retval, [$productId]);
        }

        if (count($retval) < 5) {
            $tagIds = Tag::getTags($productId);
            $productTopIds = TagStatistic::getTopProduct($tagIds, $categoryId);
            $this->addBoughtTogether($productTopIds, $retval, [$productId]);
        }
        if (count($retval) < 5) {
            shuffle($this->bestSeller);
            $this->addBoughtTogether($this->bestSeller, $retval, [$productId]);
        }
        // Save
        $now = new \DateTime();
        if (count($retval) > 0) {
            if ($boughtTogetherProductMeta) {
                DB::table('product_meta')
                    ->where('id', '=', $boughtTogetherProductMeta->id)
                    ->update([
                        'updated_at' => $now,
                        'value' => json_encode($retval),
                    ]);
            } else {
                try {
                    ProductMeta::create([
                        'product_id' => $productId,
                        'key' => 'bought_together_products',
                        'value' => json_encode($retval),
                        'created_at' => $now,
                        'updated_at' => $now,
                    ]);
                } catch (\Exception $e) {

                }

            }
        }
        return true;
    }

    protected function addBoughtTogether($items, &$result, $excludeId = [], $limit = 10, $column = 'id') {
        if (!$items) {
            return;
        }
        foreach ($items as $item) {
            $value = is_numeric($item) ? $item : $item->{$column};
            if(!in_array($value, $result)
                && count($result) < $limit
                && !in_array($value, $excludeId)) {
                $result[] = $value;
            }
        }
    }

    protected function getBestSeller() {

        $result = [];
        $items = DB::table('product')
            ->orderBy('sold', 'desc')
            ->limit(30)
            ->get(['id', 'deleted_at', 'status']);
        foreach ($items as $item) {
            if (!$item->deleted_at && $item->status == 'ACTIVE') {
                $result[] = $item;
            }
        }
        return array_slice($result, 0, 20);
    }


    protected function getProductsSameTag($productId) {
        $result = [];
        $tags = DB::table('tag_refer')
            ->where('refer_id', $productId)
            ->where('refer_type', 'PRODUCT')
            ->pluck('tag_id')
            ->toArray();
        if ($tags) {
            foreach ($tags as $tagId) {
                $productIds = $this->getProductsFromTag($tagId);
                if ($productIds) {
                    $result = array_merge($result, $productIds);
                }
            }
            $result = array_unique($result);
        }
        return $result;
    }

    protected function getProductsFromTag($tagId) {
        if (array_key_exists($tagId, self::$tagData)) {
            self::$hit++;
            return self::$tagData[$tagId];
        }
        self::$miss++;
        $result = DB::table('tag_refer as r')
            ->where('r.tag_id', $tagId)
            ->where('r.refer_type', 'PRODUCT')
            ->limit(300)
            ->pluck('r.refer_id')
            ->toArray();
        self::$tagData[$tagId] = $result;
        return $result;

    }

    public function findProduct(Request $request) {
        $response = [
            "status" => "fail",
        ];
        $productId = $request->input('product_id');
        if ($productId) {
            $result = ApiClient::buildCustomRequest('product/' . $productId, 'GET', [], []);
            $items = [$result["result"]];
            priceChangeForceDecor($items, 'recommendation');
            $data = isset($items[0]) ? $items[0] : $result["result"];
            $isShirt = false;
            $category = [];
            $categories = DB::table("product_n_category")
                ->join("category", "category.id", "=", "product_n_category.category_id")
                ->where("product_id", $productId)
                ->select(["category.id", "category.slug", "is_parent", "is_valid_print_back", "sell_design"])
                ->get();
            foreach ($categories as $cate) {
                if ($cate->id == 6) {
                    $isShirt = true;
                }
                if (isset($cate->is_parent) && $cate->is_parent == 0) {
                    $category = (array) $cate;
                }
            }
            $validPrintBack = false;
            if (isset($category["is_valid_print_back"]) && $category["is_valid_print_back"] == 1) {
                $validPrintBack = true;
            }
            $isMultipleDesign = false;
            $isCustomDesign = \DB::table('product_custom')->where('product_id', $productId)->exists();
            $isDoubleSided = false;
            foreach ($data["attributes"] as $key => $item) {
                if ($key == "multiple_design" && $item == 1) {
                    $isMultipleDesign = true;
                }
                if ($key == "is_double_sided" && $item == 1) {
                    $isDoubleSided = true;
                }
            }
            if (!$isShirt || $isMultipleDesign || $isCustomDesign || $isDoubleSided) {
                $validPrintBack = false;
            }
            $data["is_valid_print_back"] = $validPrintBack;
            if (isset( $data['variant_default'])) {
                $variantDefault = $data['variant_default'];
                priceChangeDecor($variantDefault, [
                    'pIdCol' => 'product_id'
                ]);
                $data['variant_default'] = $variantDefault;
            }
            $response = [
                "status" => "successful",
                "result" => $data,
            ];
        }
        return response()->json($response);
    }

    public function buildBoughtTogetherQuery($productIds) {
        return Product::where("product.status", "ACTIVE")
            ->whereIn("product.id", array_slice($productIds, 0, 8));
    }
    public function findBoughtTogetherProductsNew(Request $request) {
        $productId = $request->input('product_id');
        $limit = $request->input('limit', 3);
        if (!$productId) {
            return;
        }
        $products = [];
        $bgProductIds = $this->getBoughtTogetherProductIds($productId);
        $count = count($bgProductIds);
        if (count($bgProductIds) > 0) {
            $products = DB::table('product')
                 ->where("product.status", "ACTIVE")
                ->whereIn("product.id", array_slice($bgProductIds, 0, $limit))
                ->join('product_n_category', function($join) {
                        $join->on('product_n_category.product_id', '=', 'product.id')
                            ->where('product_n_category.is_parent', 0);
                 })
                ->leftJoin('score_product', 'product.id', '=', 'score_product.id')
                ->leftJoin('product_n_template', 'product.id', '=', 'product_n_template.product_id')
                ->orderBy("score", 'desc')
                ->groupBy('product.id')
                ->limit($limit)
                ->get([
                    'product.id', 'score_product.score', 'product.sku', 'product.name', 'product.slug',
                    'product.status', 'product.image_url', 'product.price', 'product.high_price',
                    'product.created_at', 'product_n_template.template_id', 'product_n_category.category_id'
                ]);
            if (count($products) > 0) {
                priceChangeForceDecor($products, 'recommendation');
                $attrs = $this->groupAttrs($products);
                $categories = $this->getCategories(array_merge([6], $attrs['category_ids']));
                $productCustomList = $this->groupProductCustom($attrs['product_ids']);
                $meteList = $this->groupProductMeta($attrs['product_ids']);
                $clothingCategory = isset($categories[6]) ? $categories[6] : null;
                foreach ($products as &$product) {
                    $product->variant_default = $this->getDefaultSku($product);
                    $category = isset($categories[$product->category_id]) ? $categories[$product->category_id] : null;
                    $isShirt = $this->isShirt($clothingCategory, $category);
                    $isCustomDesign = isset($productCustomList[$product->id]) ? $productCustomList[$product->id] : false;
                    $product->is_custom_design = $isCustomDesign;
                    $product->category_ids = [$product->category_id];
                    $product->categoriesIds = [$product->category_id];
                    $product->attributes = [];
                    $product->attributes['print_locations'] =  isset($meteList[$product->id]['print_locations']) ? json_decode($meteList[$product->id]['print_locations'], true) : null;;
                    $product->attributes['is_double_sided'] = isset($meteList[$product->id]['is_double_sided']) ? $meteList[$product->id]['is_double_sided'] : null;
                    $product->attributes['multiple_design'] = isset($meteList[$product->id]['multiple_design']) ? $meteList[$product->id]['multiple_design'] : null;
                    $product->attributes['is_custom_design'] = $isCustomDesign ? 1 : 0;
                    if ($product->template_id && isset($product->variant_default[0]->print_locations)) {
                        $product->attributes['print_locations'] = $product->variant_default[0]->print_locations;
                    }
                    $validPrintBack = false;
                    if (!$isShirt ||  $product->attributes['multiple_design'] || $isCustomDesign ||  $product->attributes['is_double_sided']) {
                        $validPrintBack = false;
                    } else if (isset($category->is_valid_print_back) && $category->is_valid_print_back == 1) {
                        $validPrintBack = true;
                    }
                    $product->is_valid_print_back = $validPrintBack;
                    if (isset($product->variant_default[0]->id)) {
                        priceChangeDecor($product->variant_default, [
                            'pIdCol' => 'product_id'
                        ]);
                    }
                }

            }
        }
        return response()->json([
            "status" => "successful",
            "result" => $products,
            'count' => $count,
            'version' => 'v2',
        ]);
    }

    public function findBoughtTogetherProducts(Request $request) {
        $referer = request()->headers->get('referer');
        if ($referer && strpos($referer, 'test_btg=1') !== false) {
            return $this->findBoughtTogetherProductsNew($request);
        }
        if ($request->has('debugg') &&  $request->input('debugg') == 'btg') {
            return $this->findBoughtTogetherProductsNew($request);
        }
        $productId = $request->input('product_id');
        $limit = $request->input('limit', 3);
        $productMeta = ProductMeta::where('product_id', '=', $productId)
            ->where('key', '=', 'bought_together_products')
            ->first();
        $products = [];
        $count = 0;
        if ($productMeta && is_array(json_decode($productMeta->value, true))) {
            $bgProductIds = json_decode($productMeta->value, true);
            $boughtTogetherManual = ProductMeta::where('product_id', '=', $productId)
                ->where('key', '=', 'manual_bought_together_products')
                ->first();
            if ($boughtTogetherManual && is_array(json_decode($boughtTogetherManual->value, true))) {
                $boughtTogetherManualProducts = json_decode($boughtTogetherManual->value, true);
                $bgProductIds = array_merge($boughtTogetherManualProducts, $bgProductIds);
            }
            if(
                getOption('product.enable_remove_unwantend_box')
                || in_array($productId, explode(',', getOption('product.product_ids_remove_unwantend_box', '')))
            ) {
                $pnc = DB::table('product_n_category')
                    ->where('product_id', $productId)
                    ->where('is_parent', '0')
                    ->orderBy('category_id', 'asc')
                    ->first();
                if ($pnc) {
                    $categoryId = $pnc->category_id;
                    $bgProductIds = DB::table('product_n_category')
                        ->whereIn('product_id', $bgProductIds)
                        ->where('category_id', $categoryId)
                        ->get(['product_id'])
                        ->pluck('product_id')
                        ->toArray();
                } else {
                    $bgProductIds = [];
                }
            }
            if (count($bgProductIds) > 0) {
                $count = $this->buildBoughtTogetherQuery($bgProductIds)
                    ->count();
                $products = $this->buildBoughtTogetherQuery($bgProductIds)->with(["variantDefault", "categories"])
                    ->leftJoin('score_product', 'product.id', '=', 'score_product.id')
                    ->orderBy("score", 'desc')
                    ->orderByRaw("FIELD(sb_product.id, " . implode(',', $bgProductIds) . ")")
                    ->limit($limit)
                    ->get(['product.id', 'score_product.score', 'product.sku', 'product.name', 'product.slug', 'product.status', 'product.image_url', 'product.price', 'product.high_price', 'product.created_at']);

                if (count($products) > 0) {
                    $products = $products->toArray();
                    priceChangeForceDecor($products, 'recommendation');
                    foreach ($products as &$product) {
                        $isShirt = false;
                        $category = [];
                        $product["category_ids"] = [];
                        $product["categoriesIds"] = [];
                        if (isset($product["categories"])) {
                            foreach ($product["categories"] as $cate) {
                                if ($cate["id"] == 6) {
                                    $isShirt = true;
                                }
                                if (isset($cate["pivot"]) && isset($cate["pivot"]["is_parent"]) && $cate["pivot"]["is_parent"] == 0) {
                                    $category = $cate;
                                    $product["category_ids"][] = $cate["id"];
                                    $product["categoriesIds"][] = $cate["id"];
                                }
                            }
                        }
                        $validPrintBack = false;
                        if (isset($category["is_valid_print_back"]) && $category["is_valid_print_back"] == 1) {
                            $validPrintBack = true;
                        }
                        $isMultipleDesign = false;
                        $isCustomDesign = \DB::table('product_custom')->where('product_id', $product['id'])->exists();
                        $isDoubleSided = false;
                        $product["is_custom_design"] = $isCustomDesign;
                        $productPrintLocations = null;
                        $productPrintLocationsMeta = \DB::table('product_meta')->where('product_id', $product['id'])->where('key', 'print_locations')->first();
                        if ($productPrintLocationsMeta) {
                            $productPrintLocations = json_decode($productPrintLocationsMeta->value, true);
                        }
                        $product['attributes'] = [];
                        $product['attributes']['print_locations'] = $productPrintLocations;
                        $isDoubleSidedMeta = \DB::table('product_meta')->where('product_id', $product['id'])->where('key', 'is_double_sided')->first(['value']);
                        $isMultipledMeta = \DB::table('product_meta')->where('product_id', $product['id'])->where('key', 'multiple_design')->first(['value']);
                        $product['attributes']['multiple_design'] = isset($isMultipledMeta) ? $isMultipledMeta->value : null;
                        $product['attributes']['is_double_sided'] = isset($isDoubleSidedMeta) ? $isDoubleSidedMeta->value : null;
                        $product['attributes']['is_custom_design'] = $isCustomDesign ? 1 : 0;
                        if (empty($product['attributes']['print_locations']) && !empty($product['variant_default']['template_id'])) {
                            $printLocations = DB::table('product_template')
                                ->join('product_meta', 'product_template.product_id_fake', '=', 'product_meta.product_id')
                                ->where('product_template.id', $product['variant_default']['template_id'])
                                ->where('product_meta.key', 'print_locations')
                                ->value('product_meta.value');

                            if (!is_null($printLocations)) {
                                $product['attributes']['print_locations'] = json_decode($printLocations, true);
                            }
                        }
                        foreach ($product["attributes"] as $key => $item) {
                            if ($key == "multiple_design" && $item == 1) {
                                $isMultipleDesign = true;
                            }
                            if ($key == "is_double_sided" && $item == 1) {
                                $isDoubleSided = true;
                            }
                        }
                        if (!$isShirt || $isMultipleDesign || $isCustomDesign || $isDoubleSided) {
                            $validPrintBack = false;
                        }
                        $product["is_valid_print_back"] = $validPrintBack;
                        if (!isset($product['variant_default'])) {
                            continue;
                        }
                        $variantDefault = null;
                        if (isset($product['variant_default'][0])) {
                            $variantDefault = $product['variant_default'][0];
                        }
                        $pnt = DB::table('product_n_template')->where('product_id', $product['id'])->first();
                        if ($pnt) {
                            $ignoreColors = [];
                            $pnd = DB::table('product_n_design')->where('product_id', $product['id'])->where('is_primary')->first();
                            if ($pnd) {
                                $meta = DB::table('pod_design_meta')->where('design_id', $pnd->design_id)->where('key', 'ignore_colors')->first();
                                if ($meta) {
                                    $ignoreColors = json_decode($meta->value);
                                }
                            }
                            $defaultSku = DB::table('product_template_sku')->where('template_id', $pnt->template_id)->where('is_default', 1)->first();
                            if (!empty($ignoreColors)) {
                                $defaultSku = DB::table('product_template_sku as ps')
                                    ->join('product_template_sku_value as psv', 'psv.sku_id', '=', 'ps.id')
                                    ->where('product_template_sku.template_id', $pnt->template_id)
                                    ->whereNotIn('psv.variant_option_id', $ignoreColors)
                                    ->whereIn('psv.variant_id', 2)
                                    ->orderBy('is_default', 'desc')
                                    ->first(['ps.*']);
                            }
                            if ($defaultSku) {
                                $defaultSku->product_id = $product['id'];
                                $defaultSku->sku = 'P' . $product['id'] . $defaultSku->sku;
                                $optionIds = DB::table('product_template_sku_value')->where('sku_id', $defaultSku->id)->get(['variant_option_id'])->pluck('variant_option_id');
                                $optionNames = DB::table('product_variant_option')->whereIn('id', $optionIds)->get(['name'])->pluck('name')->toArray();
                                $defaultSku->product_name = implode(', ', $optionNames);
                                $variantDefault = (array) $defaultSku;
                            }
                        }
                        $variantDefaultArr = [];
                        if (isset($variantDefault)) {
                            $variantDefaultArr = [$variantDefault];
                            priceChangeDecor($variantDefaultArr, [
                                'pIdCol' => 'product_id'
                            ]);
                        }
                        $product['variant_default'] = $variantDefaultArr;
                    }

                }
            }

        }
        return response()->json(["status" => "successful",
            "result" => $products,
            'count' => $count]);
    }

    protected function isShirt($clothingCategory, $itemCategory) {
        $result = false;
        if (isset($clothingCategory->id)
            && isset($itemCategory->id)
            && $clothingCategory->_lft <= $itemCategory->_lft
            && $clothingCategory->_rgt >= $itemCategory->_rgt
        ) {
            $result = true;
        }
        return $result;
    }

    protected function groupAttrs($items) {
        $categoryIds = [];
        $productIds = [];
        foreach ($items as $item) {
            $categoryIds[$item->category_id] = $item->category_id;
            $productIds[] = $item->id;
        }
        return [
            'category_ids' => array_values($categoryIds),
            'product_ids' => $productIds,
        ];
    }

    protected function getCategories($ids) {
       return DB::table('category')
           ->whereIn('id', $ids)
           ->get(['id', '_lft', '_rgt', 'is_valid_print_back'])
           ->keyBy('id');
    }

    public function getProductVariant(Request $request) {
        $response = [
            "status" => "fail",
        ];
        $id = $request->input('id');
        if ($id) {
            $result = ApiClient::buildCustomRequest('variant/' . $id, 'GET', [
                'has_options' => true
            ], []);
            $sortSize = getOption("size_sort_default");
            if (!$sortSize) {
                $sortSize = config("default.size_sort_default", []);
            }
            $data = $result["result"];
            if (isset($data['productVariants']) && $data['productVariants']) {
                $variantDefault = $data['productVariants'];
                priceChangeDecor($variantDefault, [
                    'pIdCol' => 'product_id'
                ]);
                $data['productVariants'] = $variantDefault;
                $variantsStatistic = $this->statisticProductVariant($data);
                $data["variantsStatistic"] = $variantsStatistic;
            }
            if (!empty($data['variants'])) {
                foreach ($data['variants'] as $key => $variantGroup) {
                    if ($variantGroup["slug"] == "size" && count($sortSize) > 0) {
                        $newGroup = [];
                        foreach($variantGroup['values'] as $variant) {
                            $newGroup[$variant["slug"]] = $variant;
                        }

                        $newSortValues = array_merge(array_flip($sortSize), $newGroup);
                        $newVariantValues = [];
                        foreach ($newSortValues as $value) {
                            if (is_array($value)) {
                                $newVariantValues[] = $value;
                            }
                        }

                        $data['variants'][$key]["values"] = $newVariantValues;
                    }
                }
            }
            $response = [
                "status" => "successful",
                "result" => $data,
            ];
        }
        
        return response()->json($response);
    }

    private function statisticProductVariant($data) {
        $variantsStatistic = [];
        if (isset($data['productVariants'])) {
            foreach($data['productVariants'] as $key => $sku) {
                if (!isset($sku['variants'][0]['variant'])
                    || !isset($sku['variants'][1])
                    || !isset($sku['variants'][1]['variant'])) {
                    continue;
                }
                $vKey = $sku['variants'][0]['id'] . '-' . $sku['variants'][1]['id'];
                if (!isset($variantsStatistic[$vKey])) {
                    $variantsStatistic[$vKey]  = [
                        'count' => 0,
                        'price' => $sku['price'],
                        'display_price' => $sku['display_price'],
                    ];
                }
                $variantsStatistic[$vKey]['count']++;
                if ($variantsStatistic[$vKey]['price'] > $sku['price']) {
                    $variantsStatistic[$vKey]['price'] = $sku['price'];
                    $variantsStatistic[$vKey]['display_price'] = $sku['display_price'];
                }

                if (count($sku['variants']) > 3) {
                    $vKey = $sku['variants'][0]['id'] . '-' . $sku['variants'][1]['id'] . '-' . $sku['variants'][3]['id'];

                    if (!isset($variantsStatistic[$vKey])) {
                        $variantsStatistic[$vKey]  = [
                            'count' => 0,
                            'price' => $sku['price'],
                            'display_price' => $sku['display_price'],
                        ];
                    }
                    $variantsStatistic[$vKey]['count']++;
                    if ($variantsStatistic[$vKey]['price'] > $sku['price']) {
                        $variantsStatistic[$vKey]['price'] = $sku['price'];
                        $variantsStatistic[$vKey]['display_price'] = $sku['display_price'];
                    }
                }
            }
        }
        return $variantsStatistic;

    }

    public function addAllToCart(Request $request) {
        $response = [
            "status" => "fail",
        ];
        $token = $request->has("token") ? $request->input("token") : getFingerprint();
        $items = $request->input('data');
        $user = \Auth::guard("customer")->user();
        if (count($items) > 0) {
            $result = [];
            $notIds = [];
            foreach ($items as $key => $item) {
                $dataAddToCart = [
                    "productId" => $item["productId"],
                    "productSkuId" => $item["productSkuId"],
                    "customerToken" => $token,
                    "quantity" => $item["quantity"],
                    "configurations" => $item["configurations"],
                    "customerId" => isset($user->id) ? $user->id : null,
                    "source" => "bought-together"
                ];
                $notIds[] = $item["productId"];
                $data = ApiClient::buildCustomRequest("cart/add-to-cart", "POST", $dataAddToCart , []);
                if (isset($data["cartItems"])) {
                    foreach ($data["cartItems"] as $index => $value) {
                        if ($value["product_sku_id"] == $item["productSkuId"] && $value["product_id"] == $item["productId"]) {
                            $value["url"] = clroute($value["url"]);
                            $result[] = $value;
                        }
                    }
                }
            }
            $response = [
                "status" => "successful",
                "result" => $result,
            ];
            $id = $request->input("productId", null);
            $resultRecommend = [];
            $recommendationUrl = config("bought-together::sa.recommendation_url", "");
            if ($recommendationUrl) {
                $locale = env('APP_LOCALE');
                $data = $this->triggerAsyncRequest($recommendationUrl . "/recommendation?locale=" . $locale . "&uid=" . $token . "&product_id=" . $id . "&size=10");
                if (isset($data) && isset($data["status"]) && $data["status"] == "successful" && isset($data["result"])) {
                    foreach ($data["result"] as $item) {
                        $notIds[] = $item["id"];
                        $resultRecommend[] = $item;
                    }
                }
            }
            $limit = 10;
            if (count($resultRecommend) > 0) {
                $limit -= count($resultRecommend);
            }
            $relatedProducts = [];
            $tagIds = TagRefer::whereIn('refer_id', $notIds)
                ->where('refer_type', 'PRODUCT')
                ->get(['tag_id'])
                ->pluck('tag_id')
                ->toArray();
            $notIds[] = $id;
            if (count($tagIds) > 0 && false) {
                $relatedProducts = Product::where("status", "ACTIVE")
                    ->join("trending_tag_sale", function($query) use ($tagIds) {
                        $query->on("product_id", "=", "product.id");
                        $query->whereIn("tag_id", $tagIds);
                    })
                    ->whereNotIn("product.id", $notIds)
                    ->groupBy("product.id")
                    ->orderBy("view_count", "DESC")
                    ->orderBy("sold", "DESC")
                    ->limit($limit)
                    ->get(['product.id', 'product.sku', 'product.name', 'product.slug', 'product.image_url', 'product.price', 'product.high_price', 'product.created_at'])
                    ->toArray();
            }
            $response["relatedProducts"] = array_merge($resultRecommend, $relatedProducts);
            priceChangeForceDecor($response["relatedProducts"], 'recommendation');
            priceChangeForceDecor($response["result"], 'recommendation', ["pIdCol" => "product_id", "disableForceDecor" => true]);
        }
        return response()->json($response);

    }

    protected function triggerAsyncRequest($url, $method = "GET", $params = [], $headers = []) {
        $channel = curl_init();
        curl_setopt($channel, CURLOPT_URL, $url);
        // curl_setopt($channel, CURLOPT_NOSIGNAL, 1);
        curl_setopt($channel, CURLOPT_TIMEOUT, 120);
        curl_setopt($channel, CURLOPT_RETURNTRANSFER, 1);
        if($method == "post" || $method == "POST") {
            curl_setopt($channel, CURLOPT_POST, true);
            curl_setopt($channel, CURLOPT_POSTFIELDS, json_encode($params));
        }
        if ($headers) {
            curl_setopt($channel, CURLOPT_HTTPHEADER, $headers);
        }
        curl_setopt($channel, CURLOPT_SSL_VERIFYHOST, 0);
        curl_setopt($channel, CURLOPT_SSL_VERIFYPEER, 0);
        $data = curl_exec($channel);
        curl_close($channel);
        return json_decode($data, true);
    }

    public function getDefaultSku($product) {
        $result = [];
        if ($product->template_id) {
            $result = $this->getDefaultByTemplate($product);
            $ignoreColors = $this->getIgnoreColor($product->id);
            if ($ignoreColors
                && isset($result->variant_color_id)
                && in_array($result->variant_color_id, $ignoreColors)) {
                $result = $this->getVariantFromIgnoreId($product->template_id, $ignoreColors);
            }
            if (isset($result->sku)) {
                $result->product_id = $product->id;
                $result->sku = 'P' . $product->id . $result->sku;
            }
        } else {
            $result = ProductSku::where('product_id', '=', $product->id)
                ->where("is_default", 1)
                ->first();
        }
        $this->decorPrice($result);
        return $result && isset($result->id) ? [$result] : [];
    }

    protected function decorPrice($result) {
        if ($result && isset($result->id)) {
            $result->display_price = formatPrice($result->price);
            $result->display_high_price = formatPrice($result->high_price);
        }
    }

    public function getDefaultByTemplate($product) {
        $key = decorCacheKey('getDefaultByTemplate::' . $product->template_id);
        if (array_key_exists($key, self::$cache)) {
            return self::$cache[$key];
        }
        $result = Cache::get($key);
        if (!$result) {
            $result = DB::table('product_template_sku')
                ->where('template_id', $product->template_id)
                ->where('is_default', 1)
                ->first();
            if ($result) {
                $printLocations = DB::table('product_template')
                    ->join('product_meta', 'product_template.product_id_fake', '=', 'product_meta.product_id')
                    ->where('product_template.id', $product->template_id)
                    ->where('product_meta.key', 'print_locations')
                    ->value('product_meta.value');
                if ($printLocations) {
                    $result->print_locations = json_decode($printLocations, true);
                }
                $result->variant_color_id = DB::table('product_template_sku_value')
                    ->where('variant_id', self::COLOR_VARIANT_ID)
                    ->where('sku_id', $result->id)
                    ->value('variant_option_id');
                $this->decorSkuName($result);
            }
            Cache::put($key, $result, 1440);
            self::$cache[$key] = $result;
        }
        return $result;
    }


    protected function getVariantFromIgnoreId($templateId, $ignoreColors) {
        $result = DB::table('product_template_sku as ps')
            ->join('product_template_sku_value as psv', 'psv.sku_id', '=', 'ps.id')
            ->where('ps.template_id', $templateId)
            ->whereNotIn('psv.variant_option_id', $ignoreColors)
            ->where('psv.variant_id', 2)
            ->first(['ps.*']);
        if (isset($result->id)) {
            $this->decorSkuName($result);
        }
        return $result;

    }


    protected function getIgnoreColor($productId) {
        $result = null;
        $pnd = DB::table('product_n_design')
            ->where('product_id', $productId)
            ->where('is_primary', 1)
            ->first(['design_id']);
        if ($pnd) {
            $meta = DB::table('pod_design_meta')
                ->where('design_id', $pnd->design_id)
                ->where('key', 'ignore_colors')
                ->first(['value']);
            if ($meta) {
                $result = json_decode($meta->value);
            }
        }
        return $result;
    }


    protected function decorSkuName($sku, $table = 'product_template_sku_value') {
        if (!isset($sku->id)) {
            return;
        }
        $optionIds = DB::table($table)->where('sku_id', $sku->id)
            ->get(['variant_option_id'])
            ->pluck('variant_option_id');
        $sku->product_name = '';
        if ($optionIds) {
            $optionNames = DB::table('product_variant_option')->whereIn('id', $optionIds)->get(['name'])->pluck('name')->toArray();
            $sku->product_name = $optionNames ? implode(', ', $optionNames) : '';
        }
    }

    public function getBoughtTogetherProductIds($productId)
    {
        $keys = ['bought_together_products', 'manual_bought_together_products'];
        $result = [];
        $items = ProductMeta::whereIn('key', $keys)
            ->where('product_id', $productId)
            ->get(['value']);
        foreach ($items as $item) {
            $ids = json_decode($item->value, true);
            if ($ids) {
                $result = array_merge($result, $ids);
            }
        }
        return $result;
    }

    public function groupProductCustom($productIds) {
        return \DB::table('product_custom')
            ->whereIn('product_id', $productIds)
            ->pluck('product_id', 'product_id')
            ->toArray();
    }

    protected function groupProductMeta($productIds) {
        $result = [];
        $keys = ['print_locations', 'multiple_design', 'is_double_sided'];
        $items = DB::table('product_meta')
            ->whereIn('product_id', $productIds)
            ->whereIn('key', $keys)
            ->get(['product_id', 'key', 'value']);
        if (count($items) > 0) {
            foreach ($items as $item) {
                $result[$item->product_id][$item->key] = $item->value;
            }
        }
        return $result;
    }
}
