<?php

namespace Modules\ConfigDescription\Controllers;

use App\Helpers\ApiClient;
use Cache;
use Module;
use DB;
use Theme;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Input;
use Modules\ConfigDescription\Controllers\Controller;
use Modules\ConfigDescription\Models\ProductSpecification;
use Modules\ConfigDescription\Models\Option;
use Modules\ConfigDescription\Models\Product;
use Modules\ConfigDescription\Models\Category;
use Modules\ConfigDescription\Models\ProductSku;
use Modules\ConfigDescription\Models\ProductMeta;
use Modules\ConfigDescription\Models\ProductSkuValue;
use Modules\ConfigDescription\Models\ProductNCategory;
use Modules\ConfigDescription\Models\ProductDescription;

class HomeController extends Controller
{
    public function index(Request $request)
    {
        return view('config-description::system.index');
    }

    public function getStyleInfo(Request $request)
    {
        $data = $request->only('product_id', 'type_id', 'style_id');

        $result = $this->getConfiguredDescription($data['product_id'], $data['style_id'] ?? null, $data['type_id'] ?? null);
        $retVal = $result['option_description'];

        if ($retVal) {
            $textBefore = '<strong>' . __('What material is this item made of?') . '</strong></br>';
            $retVal = $textBefore . $retVal;
        }

        return [
            'status' => 'successful',
            'result' => stripSciptTag($retVal)
        ];
    }

    public function getTypeAndColor($productSkuId) {
        $retVal = [
            'type' => '',
            'color' => ''
        ];
        $productSkuValues = ProductSkuValue::where('sku_id', $productSkuId)
            ->join('product_variant_option', 'product_variant_option.id', 'product_sku_value.variant_option_id')
            ->get();
        foreach ($productSkuValues as $item) {
            if ($item->variant_id == 5) {
                $retVal['type'] = $item->name;
            }
            if ($item->variant_id == 2) {
                $retVal['color'] = $item->name;
            }
        }

        return response()->json($retVal);
    }

    private function renderProductProperties($data) 
    {
        $productProperties = [];
        try {
            /* if (isset($data['sku'])) {
                $productProperties[__('Product Name')] = $data['product']['name'];
            } */

            $mainCategory = null;
            if (isset($data['categories'])) {
                $propertiesCategories = [];
                foreach ($data['categories'] as $category) {
                    $propertiesCategories[] = [
                        'name' => $category['name'],
                        'url' => categoryUrl($category['slug']) //route('categoryOrTag', ['slug' => $category['slug']])
                    ];
                    $mainCategory = $category;
                }
                $productProperties[__('Categories')] = $propertiesCategories;
            }

            if (isset($data['sku'])) {
                $productProperties[__('SKU')] = $data['sku'];
            }

            if (isset($data['variant_group'])) 
            {
                $langCode = env('LANGUAGE_CODE');
                $mainCategoryUrl = categoryUrl($mainCategory['slug']);
                foreach ($data['variant_group'] as $key => $variantProperty) 
                {


                    if ($key == __('Color') || $key == 'Color') {
                        $productProperties[__($key)] = [];
                        for ($i=0; $i < count($variantProperty); $i++) {
                            $link = $mainCategoryUrl. '/' . $variantProperty[$i]['slug'];
                            $productProperties[__($key)][] = [
                                'name' => $variantProperty[$i]['name'] . " " . $mainCategory['name'],
                                'url' => $link
                            ];
                        }
                    }

                    if ($key == __('Type') || $key == 'Type') {

                        $productProperties[__($key)] = [];
                        for ($i=0; $i < count($variantProperty); $i++)
                        {
                            $link = $mainCategoryUrl. '/' . $variantProperty[$i]['slug'];
                            $attrName = $variantProperty[$i]['name'] . " " . $mainCategory['name'];
                            if (strpos($langCode, 'en') === 0) {
                                $attrName = $variantProperty[$i]['name'] . "'s " . $mainCategory['name'];
                            }

                            $productProperties[__($key)][] = [
                                'name' => $attrName,
                                'url' => $link
                            ];
                        }
                    }
                }
            }

            return view('config-description::template/product-properties', ['productProperties' => $productProperties])->render();

        } catch (\Exception $ex) {
            $errorMsg = 'Error on line '. $ex->getLine().' in '.$ex->getFile() .': '.$ex->getMessage();
            \Log::error('Error getProductProperties: ' . $errorMsg);
        }
    }

    public static function getConfig() {
        $descriptionConfigs = ProductDescription::orderBy('id', 'desc')->get();
        $options = [];
        foreach ($descriptionConfigs as $item) {
            $option = [
                'id' => $item->id,
                'categories' => !empty($item->categories) ? json_decode($item->categories) : [],
                'types' => !empty($item->types) ? json_decode($item->types) : [],
                'styles' => !empty($item->styles) ? json_decode($item->styles) : [],
                'description' => $item->description
            ];
            $options[] = $option;
        }
        return $options;
    }

    public function asyncGetConfig(Request $request) {
        $retVal = [
            'status' => 'successful',
            'result' => []
        ];
        
        $retVal['result'] = $this->getConfig();

        return $retVal;
    }

    public function saveConfig(Request $request) 
    {
        $data = $request->all();
        $retVal = [
            'status' => 'successful',
            'result' => []
        ];

        $saveData = [
            'categories' => !empty($data['categories']) ? json_encode($data['categories']) : null,
            'types' => !empty($data['types']) ? json_encode($data['types']) : null,
            'styles' => !empty($data['styles']) ? json_encode($data['styles']) : null,
            'description' => $data['description']
        ];

        if (!empty($data['id'])) {

            $config = ProductDescription::find($data['id']);
            if ($config) {

                $config->fill($saveData);
                $config->save();

                // save category
                DB::table('config_description_category')
                    ->where('config_description_id', $data['id'])
                    ->delete();

                DB::table('config_description_category')
                    ->insert(array_map(function($categoryId) use ($data) {
                        return [
                            'config_description_id' => $data['id'],
                            'category_id' => $categoryId
                        ];
                    }, $data['categories']));
            }

        } else {

            $config = ProductDescription::create($saveData);

            DB::table('config_description_category')
                ->insert(array_map(function($categoryId) use ($config) {
                    return [
                        'config_description_id' => $config->id,
                        'category_id' => $categoryId
                    ];
                }, $data['categories']));

        }

        $this->saveSpecification($data, $config);

        $retVal['result'] = $config;

        return $retVal;
    }

    public function saveSpecification($data, $config)
    {
        ProductSpecification::query()->where('config_description_product_description_id', $config->id)->delete();
        if (isset($data['configs']) && count($data['configs']) > 0) {
            foreach ($data['configs'] as $item) {
                $dataSave = [
                    'key' => $item['key'],
                    'value' => $item['value'],
                    'config_description_product_description_id' => $config->id
                ];
                ProductSpecification::query()->create($dataSave);
            }
        }
    }

    public function getDescription(Request $request)
    {
        $retVal = [
            'status' => 'successful',
            'result' => ''
        ];
        $data = $request->only('product_id', 'spid', 'categoryId', 'featureTagId', 'variant_default_sku');

        if (env('APP_NAME') == 'Meear') {
            $retVal['result'] = $this->getDescriptionMeearNew($request);
        } else {
            if (isset($data['product_id'])) {
                try {

                    $data['category'] = DB::table('category')->where('id', $data['categoryId'])->select(['name', 'slug'])->first();
                    $data['featureTag'] = DB::table('tag')->where('id', $data['featureTagId'])->first();

                    if ($data['category']) $data['category'] = get_object_vars($data['category']);
                    if ($data['featureTag']) $data['featureTag'] = get_object_vars($data['featureTag']);

                    $retVal['result'] = $this->getProductDescription($data);

                } catch (\Exception $ex) {}
            }
        }

        return $retVal;
    }

    public function getDescriptionMeearNew($request)
    {
        $result = null;
        if (isset($request->product_id)) {
            try {
                $id = $request->product_id;
                $isBot = $this->checkBot();
                $data = [];
                $filter = ['id' => $id];
                if (Theme::getSetting('show_pending_product', false)) {
                    $filter['statuses'] = 'ACTIVE,PENDING,INACTIVE';
                }
                if ($request->has('spid')) {
                    $filter['default_sku_id'] = $request->get('spid');
                }
                $filter["isBot"] = $isBot;
                $output = ApiClient::buildCustomRequest("product/view/$id", 'GET', $filter, []);
                $data['category'] = [];

                if ($output['status'] == 'successful' && !empty($output['result']['product'])) {
                    $data['featureTag'] = isset($output['result']['feature_tag']) ? $output['result']['feature_tag'] : null;
                    $data['category'] = $output['result']['category'];
                    $data['product_id'] = $id;
                    $data['spid'] = $request->spid;
                    $data['variant_default'] = $output['result']['product']['variant_default'] ?? null;

                    $result = $this->getProductDescriptionNew($data);
                }
            } catch (\Exception $ex) {
            }
        }

        return $result;
    }

    private function checkBot() {
        $isBot = false;
        $PRERENDERS = '/MSNBot|MSNBot-Media|DuckDuckGo|AdIdxBot|BingPreview|TechnicalSEO.com|Majestic|Ahrefs|OnCrawl|Alexa|Mediapartners-Google|Googlebot-News|AdsBot|AdsBot-Google|Googlebot|Bingbot|Slurp|Exabot|Konqueror|DuckDuckBot|Yandex|YandexBot|Baiduspider|Baiduspider-render|facebookexternalhit|Facebot|twitterbot|rogerbot|linkedinbot|embedly|showyoubot|outbrain|pinterest|slackbot|vkShare|W3C_Validator|DotBot|MJ12bot|ia_archiver|Botify|coccocbot|IpriceGroup|SkypeUriPreview|DMCA.com|Applebot|PageSpeed\ Insights|DuplexWeb-Google|Storebot-Google|Google\ Favicon/m';

        if (isset($_SERVER['HTTP_USER_AGENT']) && preg_match($PRERENDERS, $_SERVER['HTTP_USER_AGENT'])) {
            $isBot = true;
        }
        return $isBot;
    }

    /*
        $data[
            product_id
            spid
            featureTag
            category
        ]
    */
    public function getProductDescription($data, $triggerAutoLink = true, $renderPropertiesTable = true)
    {
        $isTemplate = DB::table('product_n_template')->where('product_id', $data['product_id'])->select(['id'])->first();

        $skuTable = 'product_sku_value';
        if ($isTemplate) {
            $skuTable = 'product_template_sku_value';
        }

        $skuValues = DB::table("$skuTable as product_sku_value")
            ->where('product_sku_value.sku_id', $data['spid'])
            ->join('product_variant', 'product_variant.id', '=', 'product_sku_value.variant_id')
            ->join('product_variant_option', 'product_variant_option.id', '=', 'product_sku_value.variant_option_id')
            ->select(['product_sku_value.sku_id', 'product_sku_value.variant_id', 'product_sku_value.variant_option_id', 'product_variant.name', 'product_variant_option.name as option_name', 'product_variant_option.slug'])
            ->get()
            ->toArray();

        $data['sku_values'] = $skuValues; 

        $skuData = [
            'type_id' => null,
            'style_id' => null,
            'size' => null,
            'color' => null,
            'type' => null,
        ];
    
        foreach ($skuValues as $value) {
            if ($value->name == 'Type') {
                $skuData['type_id'] = $value->variant_option_id;
                $skuData['type'] = $value->option_name;
            }
            if ($value->name == 'Style') {
                $skuData['style_id'] = $value->variant_option_id;
                $skuData['style'] = $value->option_name;
            } 
            if ($value->name == 'Size') {
                $skuData['size'] = $value->option_name;
            }
            if ($value->name == 'Color') {
                $skuData['color'] = $value->option_name;
            }
        }
        
        $result = $this->getConfiguredDescription($data['product_id'], $skuData['style_id'] ?? null, $skuData['type_id'] ?? null);

        $retVal = $result['option_description'];

        $isShowContentData = false;
        $product = DB::table('product')->where('id', $data['product_id'])
            ->select(['name', 'description', 'content'])
            ->first();
       
        $isShowContentData = DB::table('product_meta')
            ->where('product_id', $data['product_id'])
            ->where('key', 'is_show_product_content')
            ->first();

        if (!$result['option_description'] && $result['description']) {
            $retVal = $result['description'];
        }

        if (!$result['option_description'] && $product->description) {
            $retVal = $product->description;
        }

        $siteName = config("app.name", "Printerval");

        $seoDescription = $this->getSeoDescription($data['product_id'])->getData();
        
        $textSku = implode(' ', [$skuData['type'] ?? '', $skuData['color'] ?? '']);
        $textSku = preg_replace('/\s+/', ' ', $textSku);

        $featureTag = '';
        if (isset($result['categories']) && count($result['categories']))
        {
            $endCat = end($result['categories']);
            $featureTag = $endCat['name'];
        }
        
        $seoDescriptionStart    = __("%s For %s belong theme %s at %s"); // $product->name / $textSku / $featureTag / $siteName
        $seoDescriptionStart    = $seoDescription->start . ' ' . sprintf($seoDescriptionStart, $product->name, $textSku, $featureTag, $siteName);

        $seoDescriptionEnd      = " " . __("%s or see more %s products") . " "; // $product->name / $featureTag
        $seoDescriptionEnd      = $seoDescription->first_part_of_end . sprintf($seoDescriptionEnd, $product->name, $featureTag) . $seoDescription->last_part_of_end;

        $seoDescriptionStart    = str_replace('{Type}', $skuData['type'] ?? '', $seoDescriptionStart);
        $seoDescriptionEnd      = str_replace('{Type}', $skuData['type'] ?? '', $seoDescriptionEnd);

        $escapedDescription = stripSciptTag($retVal);

        $escapedDescription = '<p>' . $seoDescriptionStart . '</p>' . $escapedDescription . '<p>' . $seoDescriptionEnd . '</p>';

        if ($isShowContentData && $isShowContentData->value == 1) {
            $escapedDescription =  $product->content . $escapedDescription;
        }

        $headingDescription = $this->getHeadingDescription($data['product_id']);
        if ($headingDescription) {
            $escapedDescription = $headingDescription . ' ' . $escapedDescription;
        }

        if (isset($result['categories'])) {
            $data['categories'] = $result['categories'];
        }

        if ($renderPropertiesTable) {
            $productProperties = $this->getProductPropertiesData($data);
            $tableProductProperties = $this->renderProductProperties($productProperties);
            if ($tableProductProperties) {
                $escapedDescription =  $tableProductProperties . $escapedDescription;
            }
        }

        if (class_exists('Modules\Seo\Controllers\AutoLinkController') && $triggerAutoLink)
        {
            $autoLinkController = new \Modules\Seo\Controllers\AutoLinkController();
            $escapedDescription = $autoLinkController->autoLinkProduct($data['product_id'], $escapedDescription);
        }

        return $escapedDescription;
    }

    protected function getHeadingDescription($productId)
    {
        try {
            $headProductContent = DB::table('product_content')->where('product_id', $productId)->first();

            if ($headProductContent && $headProductContent->content) {
                return "<h2 style=\"font-size:16px\">" . $headProductContent->content . "</h2>";
            }
        } catch (\Exception $exception) {
            return null;
        }
    }

    public function getSeoDescription($productId)
    {
        $retVal = '';
        $productMeta = ProductMeta::where('product_id', $productId)
            ->where('key', 'seo_description')
            ->first(['value']);
        if ($productMeta) {
            $retVal = json_decode($productMeta->value, true);
        } else {
            $start = config('config-description::seo_description.start');
            $end = config('config-description::seo_description.end');
            $startKey = array_rand($start);
            $endKey = array_rand($end);
            $seoDescriptionMeta = [
                "start" => $start[$startKey],
                "first_part_of_end" => $end[$endKey]['first_part'],
                "last_part_of_end" => $end[$endKey]['last_part'],
            ];
            try {
                ProductMeta::insert([
                    'product_id' => $productId,
                    'key' => 'seo_description',
                    'value' => json_encode($seoDescriptionMeta)
                ]);
                $retVal = $seoDescriptionMeta;
            } catch (Exception $e) {
                Log::info('getSeoDescription ' . json_encode($e));
            }
        }

        return response()->json($retVal);
    } 

    public function httpGetConfiguredDescription(Request $request) {
        $result = [
            'status' => 'fail'
        ];

        $productId = $request->input('product_id');
        $styleId = $request->input('style_id');
        $typeId = $request->input('type_id');
        $override = $request->input('override', true);

        if ($productId) {
            $result['result'] = $this->getConfiguredDescription($productId, $styleId, $typeId, null, $override);
            $result['status'] = 'successful';
        }

        return $result;
    }

    public function getConfiguredDescription($productId, $styleId = null, $typeId = null) 
    {
        $categories = [];
        $childCategory = DB::table('product_n_category')->where('product_id', $productId)
            ->join('category', 'category.id', 'product_n_category.category_id')
            ->where('is_parent', 0)
            ->select(['category.id', 'category._lft', 'category._rgt'])
            ->first();

        if ($childCategory) 
        {
            $categories = DB::table('category')
                ->orderBy('category._lft')
                ->where('is_hidden', 0)
                ->where('_lft', '<=', $childCategory->_lft)
                ->where('_rgt', '>=', $childCategory->_rgt)
                ->get([\DB::raw('distinct breadcrumb, `sb_category`.`id`, `sb_category`.`slug`, `sb_category`.`name`, `sb_category`.`_lft`')])
                ->toArray();

            $categories = array_map(function($cate) {
                return (array) $cate;
            }, $categories);
        }

        $options = $this->getConfig();
        $result = [
            'depth' => -1,
            'description' => '',
            'option_description' => ''
        ];

        $optionDescription = '';
        if (!count($options)) goto end;

        foreach ($categories as $category) {

            if ($category['breadcrumb']) {
                $category['breadcrumb'] = json_decode($category['breadcrumb'], true);
                if ($category['breadcrumb'] && is_array($category['breadcrumb'])) {
                    $category['breadcrumb'] = array_reverse($category['breadcrumb']);
                    $category['depth'] = count($category['breadcrumb']);
                } else {
                    $category['depth'] = 0;
                }
            } else {
                $category['depth'] = 0;
            }

            try {

                foreach ($options as $option) {

                    $condition = !empty($option['categories']) && in_array($category['id'], $option['categories']);
                    if (!$condition) continue;
 
                    if (
                        (!empty($typeId) && !empty($option['types'])) ||
                        (!empty($styleId) && !empty($option['styles']))
                    ) {

                        if ((!empty($typeId) && !empty($option['types'])) && (!empty($styleId) && !empty($option['styles']))) {
                            $condition = $condition && in_array($typeId, $option['types']) && in_array($styleId, $option['styles']);
                            if ($condition) {
                                $optionDescription = $option['description'];
                                break;
                            }
                        } else if (
                            !empty($typeId) &&
                            !empty($option['types']) &&
                            empty($option['styles'])
                        ) {
                            $condition = $condition && in_array($typeId, $option['types']);
                            if ($condition) {
                                $optionDescription = $option['description'];
                                break;
                            }
                        } else if (
                            !empty($styleId) &&
                            !empty($option['styles']) &&
                            empty($option['types'])
                        ) {
                            $condition = $condition && in_array($styleId, $option['styles']);
                            if ($condition) {
                                $optionDescription = $option['description'];
                                break;
                            }
                        }

                    } else if ($result['depth'] < $category['depth'] && empty($option['types']) && empty($option['styles'])) {
                        $result = [
                            'depth' => $category['depth'],
                            'description' => $option['description']
                        ];
                    }

                    if (!$result['description'] && empty($option['types']) && empty($option['styles'])) {
                        if (is_array($category['breadcrumb']) && count($category['breadcrumb'])) {
                            foreach ($category['breadcrumb'] as $key => $cate) {
                                if (in_array($cate['id'], $option['categories'])) {
                                    $result = [
                                        'depth' => $category['depth'] - $key,
                                        'description' => $option['description']
                                    ];
                                    break;
                                }
                            }
                        }
                    }
                }

            } catch (\Exception $exception) {
                goto end;
            }

            if ($optionDescription) {
                break;
            }
        }

        end:

        $result['option_description'] = $optionDescription;
        $result['option_description'] = $this->nl2p($result['option_description']);
        $result['description'] = $this->nl2p($result['description']);
        $result['categories'] = $categories;

        return $result;
    }

    private function nl2p($text)
    {
        $lines = explode("\n", $text);
        $html = '';
        foreach ($lines as $line) {
            if (trim($line) !== '') {
                $html .= "<p>" . $line . "</p>";
            }
        }
        return $html ?? $text;
    }

    private function getProductPropertiesData($data) 
    {
        $product = DB::table('product')->where('id', $data['product_id'])
            ->select(['id', 'sku'])
            ->first();

        $variantDefault = null;

        if (isset($data['variant_default'])) {
            $variantDefault = $data['variant_default'];
        }

        if (!$variantDefault) 
        {
            $variantDefault = DB::table('product_sku')
                ->where('id', '=', $data['spid'])
                ->select(['sku'])
                ->first();

            if ($variantDefault) {
                $variantDefault = get_object_vars($variantDefault);
            }
        }

        $prodSku = $product->sku;
        if ($variantDefault) {

            if (isset($data['variant_default_sku'])) {
                $variantDefault['sku'] = $data['variant_default_sku'];
            }

            $prodSku = $variantDefault['sku'];
        }

        if (strpos($prodSku, '-') === 0) {
            $prodSku = 'P' . $product->id . $prodSku;
        }

        $groupVariants = [];
        $skuValues = [];

        if (isset($data['sku_values']) && $data['sku_values']) {
            $skuValues = $data['sku_values'];
        } else if (isset($data['product_id'])) {

            $isTemplate = DB::table('product_n_template')->where('product_id', $data['product_id'])->select(['id'])->first();

            $skuTable = 'product_sku_value';
            if ($isTemplate) {
                $skuTable = 'product_template_sku_value';
            }

            $skuValues = DB::table("$skuTable as product_sku_value")
                ->where('product_sku_value.sku_id', $data['spid'])
                ->join('product_variant', 'product_variant.id', '=', 'product_sku_value.variant_id')
                ->join('product_variant_option', 'product_variant_option.id', '=', 'product_sku_value.variant_option_id')
                ->select(['product_sku_value.sku_id', 'product_sku_value.variant_id', 'product_sku_value.variant_option_id', 'product_variant.name', 'product_variant_option.name as option_name', 'product_variant_option.slug'])
                ->get()
                ->toArray();
        }

        foreach ($skuValues as $value) {
            if (!isset($groupVariants[$value->name]) || !in_array($value->option_name, $groupVariants[$value->name])) {
                $groupVariants[$value->name][] = [
                    'name' => $value->option_name,
                    'slug' => $value->slug
                ];
            }
        }

        $variantSorder = [
            'type' => 0,
            'color' => 1,
            'style' => 2,
            'size' => 3,
        ];

        uksort($groupVariants, function ($a, $b) use ($variantSorder) {
            $a = strtolower($a);
            $b = strtolower($b);

            $key_1 = $variantSorder[$a] ?? PHP_INT_MAX;
            $key_2 = $variantSorder[$b] ?? PHP_INT_MAX;

            return $key_1 <=> $key_2;
        });

        return [
            'sku' => $prodSku,
            'variant_group' => $groupVariants,
            'categories' => $data['categories'],
            'product' => $product
        ];
    }

    public function decorCacheKey($key) {
        $locale = env('APP_LOCALE', '');
        return $locale . '::' . $key;
    }

    public function getVariantDescription($input = []) {
        $key = 'CACHE_VARIANT_DESCRIPTION::' . json_encode($input);
        $key = $this->decorCacheKey($key);
        $result = \Cache::get($key);
        if (empty($result)) {

            $productId = $input['product_id'];
            $skuId = $input['sku_id'];
            if (!$productId) {
                $productSku = ProductSku::find($skuId);
                if ($productSku) {
                    $productId = $productSku->product_id;
                }
            }
            $typeVariantId = 5;
            $styleVariantId = 7;
            $typeId = null;
            $styleId = null;
            $skuValues = ProductSkuValue::where('sku_id', $skuId)->get();
            foreach ($skuValues as $item) {
                if ($item->variant_id == $typeVariantId) {
                    $typeId = $item->variant_option_id;
                }
                if ($item->variant_id == $styleVariantId) {
                    $styleId = $item->variant_option_id;
                }
            }

            $result = $this->getProductDescription([
                'product_id' => $productId,
                'type_id' => $typeId,
                'style_id' => $styleId
            ]);

            if ($result) {
                \Cache::put($key, $result, 24 * 60);
            }
        }
        return $result;
    }


    public function getProductDescriptionNew($data, $triggerAutoLink = true, $renderPropertiesTable = true)
    {
        $this->initTimer();
        $isTemplate = DB::table('product_n_template')->where('product_id', $data['product_id'])->exists();
        $skuValues = $this->getSkuValues($data['spid'], $isTemplate);
        $sku = $this->getSku($data['spid'], $isTemplate);
        $product = isset($data['product']) ? $data['product'] : null;
        if (!isset($data['category'])) {
            $category = DB::table('product_n_category')
                ->join('category', 'category.id', 'product_n_category.category_id')
                ->where('product_n_category.product_id', $data['product_id'])
                ->where('product_n_category.is_parent', 0)
                ->first(['category.id', 'category.name', 'category.slug', 'category.breadcrumb']);
            $data['category'] = (array) $category;
        }
        if (!$product) {
            $product = DB::table('product')->where('id', $data['product_id'])
                ->select(['sku', 'name', 'description', 'content', 'id'])
                ->first();
            $product = (array) $product;
            $data['product'] = $product;
        }
        $this->cpTimer('after-$skuValues');
        $data['sku_values'] = $skuValues;
        $data['sku'] = $sku;
        $data['is_template'] = $isTemplate;
        $data['sku_data'] = $this->parseSkuValues($skuValues);
        $skuData = $data['sku_data'];
        $this->cpTimer('after-parseSkuValues');
        $result = $this->getConfiguredDescription(isset($data['category']['id']) ? $data['category']['id'] : '', $skuData['style_id'], $skuData['type_id']);
        $this->cpTimer('after-getConfiguredDescriptionNew');
        $retVal = $result['option_description'];
        if (!$retVal && $product['description']) {
            $retVal = $product['description'];
        }
        $this->cpTimer('before-getSeoDescription');
        $seoDescription = $this->getSeoDescription($data['product_id'])->getData();
        $this->cpTimer('after-getSeoDescription');
        $textSku = preg_replace('/\s+/', ' ', $skuData['type'] . ' ' . $skuData['color']);

        $featureTag = '';
        if (isset($data['category']['name']))
        {
            $featureTag = $data['category']['name'];
        }

        $siteName = config("app.name", "Printerval");
        $seoDescriptionStart    = $seoDescription->start . ' ' . sprintf( __("%s For %s belong theme %s at %s"), $product['name'], $textSku, $featureTag, $siteName);
        $seoDescriptionEnd      = $seoDescription->first_part_of_end . sprintf(" " . __("%s or see more %s products") . " ", $product['name'], $featureTag) . $seoDescription->last_part_of_end;
        $seoDescriptionStart    = str_replace('{Type}', $skuData['type'], $seoDescriptionStart);
        $seoDescriptionEnd      = str_replace('{Type}', $skuData['type'], $seoDescriptionEnd);
        $escapedDescription = stripSciptTag($retVal);
        $escapedDescription = '<p>' . $seoDescriptionStart . '</p>' . $escapedDescription . '<p>' . $seoDescriptionEnd . '</p>';


        $isShowContentData = DB::table('product_meta')
            ->where('product_id', $data['product_id'])
            ->where('key', 'is_show_product_content')
            ->first(['value']);
        if ($isShowContentData && $isShowContentData->value == 1) {
            $escapedDescription =  $product['content'] . $escapedDescription;
        }
        $this->cpTimer('before-getHeadingDescription');
        $headingDescription = $this->getHeadingDescription($data['product_id']);
        $this->cpTimer('after-getHeadingDescription');
        if ($headingDescription) {
            $escapedDescription = $headingDescription . ' ' . $escapedDescription;
        }

        if (env('APP_NAME') == 'Meear') {
            $this->cpTimer('before-renderProductPropertiesMeearV2');
            if ($renderPropertiesTable) {
                $tableProductProperties = $this->renderProductPropertiesMeearV2($data);
                if ($tableProductProperties) {
                    $escapedDescription =  $tableProductProperties . $escapedDescription;
                }
            }
        } else {
            $this->cpTimer('before-renderProductPropertiesV2');
            if ($renderPropertiesTable) {
                $tableProductProperties = $this->renderProductPropertiesV2($data);
                if ($tableProductProperties) {
                    $escapedDescription =  $tableProductProperties . $escapedDescription;
                }
            }
        }

        $this->cpTimer('after-$renderPropertiesTable');
        if (class_exists('Modules\Seo\Controllers\AutoLinkController') && $triggerAutoLink)
        {
            $autoLinkController = new \Modules\Seo\Controllers\AutoLinkController();
            //$escapedDescription = $autoLinkController->autoLinkProduct($data['product_id'], $escapedDescription);
        }
        $this->cpTimer('after-autoLinkProduct');
        $this->persistenceTimer();
        return $escapedDescription;
    }

    private function getSlug($path) {
        $segments = explode('/', $path);

        return end($segments);
    }

    public function renderProductPropertiesMeearV2($data)
    {
        $productProperties = [];

        try {
            $breadcrumb = [];
            $mainCategory = null;
            if (isset($data['category']['breadcrumb'])) {
                $breadcrumb = json_decode($data['category']['breadcrumb'], true);
                if ($breadcrumb) {
                    foreach ($breadcrumb as $key => $item) {
                        $breadcrumb[$key]['url'] = $this->getSlug($item['url']);
                    }
                    $productProperties[__('Categories')] = $breadcrumb;
                    $data['category']['url'] = $breadcrumb[count($breadcrumb)-1]['url'];
                }

            }

            $prodSku = $data['product']['sku'];
            if (isset($data['sku']->id)) {
                $prodSku = $data['sku']->sku;
                if (strpos($prodSku, '-') === 0) {
                    $prodSku = 'P' . $data['product']['id'] . $prodSku;
                }
            }
            $productProperties[__('SKU')] = $prodSku;
            if (isset($data['category']) && isset($data['sku_data']))
            {

                $langCode = env('LANGUAGE_CODE');
                $mainCategoryUrl = isset($data['category']['url']) ? $data['category']['url'] : categoryUrl($data['category']['slug']);
                $keys = [
                    __('Color') => 'color',
                    __('Type') => 'type',
                    __('Size') => 'size',
                    __('Style') => 'style'
                ];
                foreach ($keys as $key => $variantName) {
                    if (isset($data['sku_data'][$variantName]) && isset($data['sku_data'][$variantName . '_slug'])) {
                        $space = strpos($langCode, 'en') === 0 ? "'s " : " ";
                        $productProperties[$key][] = [
                            'name' => $data['sku_data'][$variantName] . $space . $data['category']['name'],
                            'url' => $mainCategoryUrl. '/' . $data['sku_data'][$variantName . '_slug']
                        ];
                    }
                }
            }
            if (env('APP_NAME') == 'Meear') {
                return view('config-description::template/product-properties-meear', ['productProperties' => $productProperties])->render();
            } else {
                return view('config-description::template/product-properties', ['productProperties' => $productProperties])->render();
            }

        } catch (\Exception $ex) {
            $errorMsg = 'Error on line '. $ex->getLine().' in '.$ex->getFile() .': '.$ex->getMessage();
            \Log::error('Error getProductProperties: ' . $errorMsg);
        }
    }

    public function getConfiguredDescriptionNew($categoryId, $styleId = null, $typeId = null)
    {
        $configDescription = null;
        if ($categoryId) {
            $configDescription = $this->getCategoryConfigDescription($categoryId, $styleId, $typeId);
            if (!$configDescription) {
                $configDescription = $this->getCategoryConfigDescription($categoryId, null, null);
            }
        }

        $result['option_description'] = $configDescription;
        $result['categories'] = null;
        return $result;
    }




    protected function getCategoryConfigDescription($categoryId, $styleId = null, $typeId = null) {
        $result = null;
        $query = DB::table('config_description_n_category')
            ->where('category_id', $categoryId);
        if ($styleId) {
            $query->where('style_id', $styleId);
        }
        if ($typeId) {
            $query->where('type_id', $styleId);
        }
        $configDescriptionNCategory = $query->orderBy('level', 'asc')->first(['config_id']);
        if ($configDescriptionNCategory) {
            $configDescription = $this->getConfigDescriptionById($configDescriptionNCategory->config_id);
            $result = isset($configDescription->description) ? $this->nl2p($configDescription->description) : null;
        }
        return $result;
    }

    protected function getConfigDescriptionById($id) {
        return DB::table('config_description_product_description')->where('id', $id)->first();
    }

    protected function initTimer() {
        $this->timer = [
            'start_time' => microtime(true)
        ];
        $this->currentTime = microtime(true);
    }

    protected function cpTimer($name) {
        if (isset($_GET['debug_timer'])) {
            $this->timer[$name] = number_format(microtime(true) -  $this->currentTime, 6);
            $this->currentTime = microtime(true);
        }

    }

    protected function persistenceTimer() {
        if (isset($_GET['debug_timer'])) {
            $this->timer['run_time'] =  number_format(microtime(true) -  $this->timer['start_time'], 6);
            //Log::info('persistence debug Timer', $this->timer);
        }

        if (isset($_GET['show_debug_timerr'])) {
            echo '<pre>';
            echo print_r($this->timer, true);
            echo '</pre>';
            exit;
        }
    }
    private function renderProductPropertiesV2($data)
    {
        $productProperties = [];

        try {
            $breadcrumb = [];
            $mainCategory = null;
            if (isset($data['category']['breadcrumb'])) {
                $breadcrumb = json_decode($data['category']['breadcrumb'], true);
                if ($breadcrumb) {
                    $productProperties[__('Categories')] = $breadcrumb;
                    $data['category']['url'] = $breadcrumb[count($breadcrumb)-1]['url'];
                }

            }

            $prodSku = $data['product']['sku'];
            if (isset($data['sku']->id)) {
                $prodSku = $data['sku']->sku;
                if (strpos($prodSku, '-') === 0) {
                    $prodSku = 'P' . $data['product']['id'] . $prodSku;
                }
            }
            $productProperties[__('SKU')] = $prodSku;
            if (isset($data['category']) && isset($data['sku_data']))
            {

                $langCode = env('LANGUAGE_CODE');
                $mainCategoryUrl = isset($data['category']['url']) ? $data['category']['url'] : categoryUrl($data['category']['slug']);
                $keys = [
                    __('Color') => 'color',
                    __('Type') => 'type',
                ];
                foreach ($keys as $key => $variantName) {
                    if (isset($data['sku_data'][$variantName]) && isset($data['sku_data'][$variantName . '_slug'])) {
                        $space = strpos($langCode, 'en') === 0 ? "'s " : " ";
                        $productProperties[$key][] = [
                            'name' => $data['sku_data'][$variantName] . $space . $data['category']['name'],
                            'url' => $mainCategoryUrl. '/' . $data['sku_data'][$variantName . '_slug']
                        ];
                    }
                }
            }
            return view('config-description::template/product-properties', ['productProperties' => $productProperties])->render();

        } catch (\Exception $ex) {
            $errorMsg = 'Error on line '. $ex->getLine().' in '.$ex->getFile() .': '.$ex->getMessage();
            \Log::error('Error getProductProperties: ' . $errorMsg);
        }
    }

    protected function getSku($skuId, $isTemplate) {
        $skuTable = 'product_sku';
        if ($isTemplate) {
            $skuTable = 'product_template_sku';
        }
        return DB::table($skuTable)->where('id', $skuId)->first(['id', 'sku']);
    }

    protected function getSkuValues($skuId, $isTemplate = false) {
        $skuTable = 'product_sku_value';
        if ($isTemplate) {
            $skuTable = 'product_template_sku_value';
        }
        return DB::table("$skuTable as product_sku_value")
            ->where('product_sku_value.sku_id', $skuId)
            ->join('product_variant', 'product_variant.id', '=', 'product_sku_value.variant_id')
            ->join('product_variant_option', 'product_variant_option.id', '=', 'product_sku_value.variant_option_id')
            ->get(['product_sku_value.sku_id',
                'product_sku_value.variant_id', 'product_sku_value.variant_option_id',
                'product_variant.name', 'product_variant_option.name as option_name',
                'product_variant_option.slug'
            ]);
    }

    protected function parseSkuValues($skuValues) {
        $skuData = [
            'type_id' => null,
            'style_id' => null,
            'size' => null,
            'color' => null,
            'type' => null,
        ];
        if ($skuValues) {
            foreach ($skuValues as $value) {
                $nameSlug = str_slug($value->name);
                $skuData[$nameSlug] = $value->option_name;
                $skuData[$nameSlug . '_id'] = $value->variant_option_id;
                $skuData[$nameSlug . '_slug'] = $value->slug;
            }
        }

        return $skuData;
    }


}
