<?php

namespace Modules\ConfigDescription\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\ConfigDescription\Controllers\Controller;
use App\Helpers\ApiClient;
use EmptyIterator;
use Modules\ConfigDescription\Models\ProductFeatureTag;
use Illuminate\Support\Facades\Cache;
use Modules\ConfigDescription\Models\SlugManager;

class ProductDescriptionController extends Controller
{
    protected $homeController;

    public function __construct()
    {
        $this->homeController = new HomeController();
    }

    public function build(Request $request)
    {
        set_time_limit(3600);

        $startId = $request->input('start_id');
        $stopId = $request->input('stop_id', $startId);

        $insertData = [];
        $insertCount = 0;

        for ($id = $startId; $id <= $stopId; $id++) { 

            $product = DB::table('product')
                ->where('status', 'ACTIVE')
                ->where('id', $id)
                ->first();

            $childCategory = DB::table('product_n_category')->where('product_id', $id)
                ->where('is_parent', 0)
                ->select(['product_n_category.category_id'])
                ->first();

            if (!$product) continue;

            $productInfo = DB::table('product_info')
                ->where('product_id', $product->id)
                ->orderBy('updated_at', 'desc')
                ->first();

            if (!$productInfo) continue;

            $categoryId = $childCategory ? $childCategory->category_id : null;

            $data = [
                'product_id' => $id,
                'spid' => $productInfo->product_sku_id,
                'variant_default_sku' => $productInfo->sku,
                'categoryId' => $categoryId,
                'featureTagId' => $this->getProductFeatureTag($id, $categoryId)
            ];

            $desctiptionData = $this->getDescription($data);

            $insertData[] = [
                'product_id' => $data['product_id'],
                'default_sku' => $data['variant_default_sku'] ?? null,
                'description' => $desctiptionData
            ];

            if (count($insertData) > 500) {
                $insertCount += count($insertData);
                $this->insertDescription($insertData);
                $insertData = [];
            }

        }

        if (count($insertData)) {
            $this->insertDescription($insertData);
            $insertCount += count($insertData);
        }

        return [
            'insert_count' => $insertCount
        ];

    }

    protected function insertDescription($data)
    {
        $values = [];
        foreach ($data as $record) {
            $values[] = '(' . join(', ', array_map(function ($value) {
                return DB::connection()->getPdo()->quote($value);
            }, $record)) . ')';
        }

        $valuesString = join(', ', $values);

        $sql = 'INSERT INTO sb_product_description (product_id, default_sku, description) VALUES';
        $sql .= $valuesString;
        $sql .= " ON DUPLICATE KEY UPDATE default_sku = VALUES(default_sku), description = VALUES(description);";

        DB::statement($sql);
    }

    protected function getDescription($data)
    {
        // $data = $request->only('product_id', 'spid', 'categoryId', 'featureTagId', 'variant_default_sku');
        if (isset($data['product_id'])) {
            try {

                $data['category'] = DB::table('category')->where('id', $data['categoryId'])->select(['name', 'slug'])->first();
                $data['featureTag'] = DB::table('tag')->where('id', $data['featureTagId'])->first();

                if ($data['category']) $data['category'] = get_object_vars($data['category']);
                if ($data['featureTag']) $data['featureTag'] = get_object_vars($data['featureTag']);

                return $this->homeController->getProductDescription($data, false, false);

            } catch (\Exception $ex) {}
        }
    }

    protected function getProductFeatureTag($productId, $categoryId) {
        $result = ProductFeatureTag::getByProduct($productId);
        if (!$result && $categoryId) {
            $result = $this->getProductFeatureTagFromCategory($categoryId);
        }

        if ($result) return $result->id;
    }

    protected function getProductFeatureTagFromCategory($categoryId) {
        $key = 'getProductFeatureTagFromCategory::' . $categoryId;
        $key = $this->decorCacheKey($key);
        $result = Cache::get($key);
        if (!$result) {
            $categoryTags = $this->popularTagFromCategory($categoryId);
            $result = $this->getPriorityTag($categoryTags);
            Cache::put($key, $result, 1440);
        }
        return $result;
    }

    protected function getPriorityTag($tags) {
        $result = null;
        if ($tags) {
            foreach ($tags as $tag) {
                if ($tag && SlugManager::isPriority($tag->slug, 'tag')) {
                    $result = $tag;
                    break;
                }
            }
        }
        return $result;
    }

    function decorCacheKey($key) {
        $locale = env('APP_LOCALE', 'UNKNOWN');
        $host = strtoupper($_SERVER['HTTP_HOST']);
        return $host . '::' . $locale . '::' . $key;
    }

    public function popularTagFromCategory($categoryId) {
        $key = 'popularTagFromCategory' . $categoryId;
        $keyForever = 'popularTagFromCategory-forever' . $categoryId;
        $key = $this->decorCacheKey($key);
        $keyForever = $this->decorCacheKey($keyForever);
        $result = Cache::get($key);
        if (!$result) {
            $foreverData = Cache::get($keyForever);
            if ($foreverData) {
                Cache::put($key, $foreverData, 1440);
            }
            $query = DB::table('product_n_category_top as pnc')
                ->join('tag_refer as tr', 'tr.refer_id', '=', 'pnc.product_id')
                ->join('tag as t', 'tr.tag_id', '=', 't.id')
                ->where('pnc.category_id', $categoryId)
                ->where('tr.refer_type', 'PRODUCT')
                ->groupBy('tr.tag_id')
                ->orderBy('total_product', 'desc')
                ->limit(50);
            $result = $query->get([
                DB::raw('COUNT(*) as total_product'),
                't.id', 't.title', 't.slug'
            ])->toArray();
            Cache::put($key, $result, 1440);
            Cache::put($keyForever, $result, 3 * 1440);
        }
        return $result;
    }

    public function cronBuild()
    {
        $fromId = getOption('cron_build_product_description_from_id', 0);

        $created_at = date('Y-m-d H:i:s', strtotime('-2 hour'));
        $productsNoDescription = DB::table('product')
            ->leftJoin('product_description', 'product_description.product_id', '=', 'product.id')
            ->join('product_info', 'product_info.product_id', '=', 'product.id')
            ->where('product.status', 'ACTIVE')
            ->whereNull('product_description.product_id')
            ->where('product.created_at', '>', $created_at)
            ->select([
                'product.id',
                'product_info.product_sku_id',
                'product_info.sku'
            ])
            ->get();

        $insertData = [];
        $insertCount = 0;

        if (!empty($productsNoDescription)) {
            foreach ($productsNoDescription as $product) {
                $fromId = $product->id;
                if (!$product->product_sku_id) continue;

                $childCategory = DB::table('product_n_category')->where('product_id', $product->id)
                    ->where('is_parent', 0)
                    ->select(['product_n_category.category_id'])
                    ->first();

                $categoryId = $childCategory ? $childCategory->category_id : null;

                $data = [
                    'product_id' => $product->id,
                    'spid' => $product->product_sku_id,
                    'variant_default_sku' => $product->sku,
                    'categoryId' => $categoryId,
                    'featureTagId' => $this->getProductFeatureTag($product->id, $categoryId)
                ];

                $description = $this->getDescription($data);

                $insertData[] = [
                    'product_id' => $data['product_id'],
                    'default_sku' => isset($data['variant_default_sku']) ? $data['variant_default_sku'] : null,
                    'description' => $description
                ];

                $insertCount += count($insertData);
                $this->insertDescription($insertData);
                $insertData = [];
            }
            if (!empty($insertData)) {
                $this->insertDescription($insertData);
            }
        }

        if (!$fromId) {
            $option = DB::table('option')->where('key', 'cron_build_product_description_from_id')->first();
            if (!$option) {
                DB::table('option')->insert([
                    'key' => 'cron_build_product_description_from_id',
                    'value' => $fromId
                ]);
            }
        } else {
            DB::table('option')
                ->where('key', 'cron_build_product_description_from_id')
                ->update(['value' => $fromId]);
        }

        return response()->json([
            'insert_count' => $insertCount
        ]);
    }
}