<?php

namespace Modules\CrawlProduct\Controllers\AutoCrawl;

use DB;
use Illuminate\Http\Request;
use Modules\Pod\Models\User;
use Modules\CrawlProduct\Models\Tag;
use Modules\CrawlProduct\Models\Brand;
use Modules\CrawlProduct\Controllers\Impl\ProductCreation;
use App\Modules\CrawlProduct\Controllers\Services\EtsyService;
use Modules\CrawlProduct\Controllers\Impl\ProductUseTemplateCreation;
use Modules\CrawlProduct\Controllers\AutoCrawl\BaseAutoCrawlController;

class CrawlEtsyController extends BaseAutoCrawlController {
    protected $productCreation;
    protected $productUseTemplateCreation;

    public function __construct(ProductCreation $productCreation, ProductUseTemplateCreation $productUseTemplateCreation) {
        $this->productCreation = $productCreation;
        $this->productUseTemplateCreation = $productUseTemplateCreation;
    }

    public function crawl(Request $request) {
        \Log::useDailyFiles(storage_path() . '/logs/crawl-etsy.log');
        ini_set('memory_limit', '2048M');
        $query = DB::table('crawl_product_etsy');
        
        if (!$request->has('crawl_all')) {
            set_time_limit(10 * 3600);

            if (!$request->get('toId') || !$request->get('fromId')) {
                return [
                    'status' => 'fail',
                    'message' => 'missing fromId or toId'
                ];
            }
            $query->where('id', '>=', $request->get('fromId'));

            $query->where('id', '<=', $request->get('toId'));
        } else {
            set_time_limit(24 * 3600);
        }
        if (!$request->get('is_overwrite_product')) {
            $status = $request->get('status', 'PENDING');
            $query->where('status', $status);
        } else {
            if ($request->has('status')) {
                $query->where('status', $request->get('status'));
            } else {
                $query->where('status', '!=', 'IGNORE');
            }
        }
        if ($request->has('contain')) {
            $query->where('url', 'like', '%' . $request->get('contain') . '%');
        }
        if ($request->has('category')) {
            $query->where('category', $request->get('category'));
        }
        $offset = 0;
        $limit = $request->get('limit', 50);
        $locale = env('APP_LOCALE', 'us');
        if (!$locale) {
            $locale = 'us';
        }
        $localePrefix = $locale == 'us' ? '' : "/$locale";
        $retVal = [
            'status' => 'successful',
            'message' => [],
        ];
        $brand = Brand::take(1)->first();
        while(true) {
            try {
                $cloneQuery = clone $query;
                if ($request->get('is_overwrite_product')) {
                    $cloneQuery->offset($offset);
                    $offset += $limit;
                }
                $products = $cloneQuery->limit($limit)->get(['id']);
                if (!count($products)) {
                    break;
                }
                foreach ($products as $product) {
                    $newRequest = new Request();
                    $newRequest->merge([
                        'craw_data_id' => $product->id,
                        'brand_id' => $brand ? $brand->id : null,
                        'generate_variant' => $request->get('generate_variant', 0),
                        'is_overwrite_product' => $request->get('is_overwrite_product', 0),
                        'is_download_image' => $request->get('is_download_image', 0),
                        'user_id' => $request->get('user_id'),
                    ]);

                    $this->asyncCrawl($newRequest);
                }
            } catch (\Exception $ex) {
                $message = $ex->getMessage() . ' File: ' . $ex->getFile() . ' Line: ' . $ex->getLine();
                $retVal['message'][] = $message;
                \Log::error($message);
            }

        }

        return $retVal;
    }

    public function asyncCrawl(Request $request) {
        \Log::useDailyFiles(storage_path() . '/logs/crawl-etsy.log');
        $retVal = [
            'status' => 'error',
            'message' => [],
        ];
        ini_set('memory_limit', '2048M');
        $dataId = $request->get('craw_data_id');
        $product = DB::table('crawl_product_etsy')->where('id', $dataId)->first();
        if (!$product) {
            $retVal['message'] = 'Product not found: ' . $dataId;
            return $retVal;
        }
        $locale = env('APP_LOCALE', 'us');
        if (!$locale) {
            $locale = 'us';
        }
        try {
            $categoryId = $request->get('category_id');
            if (!$categoryId) {
                $categoryId = $this->getCategory($product->category);
            }
            if (!$categoryId) {
                \Log::error('can not get category : ', [$product->category]);
                $retVal['message'] = 'can not get category';
                return $retVal;
            }

            $request->merge([
                'categories' => $categoryId,
                'url' => $product->url,
            ]);
            $service = new EtsyService($request->all());
            $data = $service->parseHTML($product->data, $product->url);
            $data['direction'] = $this->getDirection($data['variants']);
            $tags = $this->getTags($data);
            if (!empty($data['name'])) {
                $newRequest = new Request();
                $newRequest->merge([
                    'brand_id' => $request->get('brand_id', 1),
                    'country_code' => $locale,
                    'remove_size_chart' => false,
                    'generate_variant' => $request->get('generate_variant', 1),
                    'tags' => $tags,
                ]);

                $newRequest->merge([
                    'categories' => $categoryId
                ]);
                if ($request->has('is_overwrite_product')) {
                    $newRequest->merge(['is_overwrite_product' => 1]);
                }
                if ($request->has('enable_style')) {
                    $newRequest->merge(['enable_style' => explode(",", $request->get('enable_style'))]); 
                }
                $data['force_status'] =  $request->get('force_status', 'ACTIVE');
                $data['is_download_image'] = $request->get('is_download_image', false);
                $data['user_id'] = $request->get('user_id', $this->getRandomSellerId());

                if ($request->get('generate_variant', 1)) {
                    $template = $this->getTemplate([$categoryId], $data['direction']);
                }

                if (!empty($template)) {
                    $data['template'] = $template;
                    $response = $this->productUseTemplateCreation->saveCrawlProduct($newRequest, $data);
                } else {
                    $response = $this->productCreation->saveCrawlProduct($newRequest, $data);
                }

                if (isset($response['status']) && $response['status'] == 'successful') {
                    $retVal['status'] = 'successful';
                    DB::table('crawl_product_etsy')->where('id', $product->id)->update([
                        'status' => 'FINISHED'
                    ]);
                } else if (isset($response['status']) && $response['status'] == 'duplicate') {
                    $retVal['status'] = 'duplicate';
                    DB::table('crawl_product_etsy')->where('id', $product->id)->update([
                        'status' => 'DUPLICATE'
                    ]);
                } else {
                    $retVal['status'] = 'error';
                    DB::table('crawl_product_etsy')->where('id', $product->id)->update([
                        'status' => 'ERROR'
                    ]);
                    \Log::error('saveCrawlProduct', [$response]);
                }
            } else {
                $retVal['status'] = 'error';
                $product->status = 'ERROR';
                $message = 'Data Id: ' . $product->id . ' cannot get product name';
                \Log::error($message);
                $retVal['message'][] = $message;
                DB::table('crawl_product_etsy')->where('id', $product->id)->update([
                    'status' => 'ERROR'
                ]);
            }
        } catch (\Exception $ex) {
            $retVal['status'] = 'error';
            DB::table('crawl_product_etsy')->where('id', $product->id)->update([
                'status' => 'ERROR'
            ]);
            $message = 'Data Id: ' . $product->id . ' ' . $ex->getMessage() . ' File: ' . $ex->getFile() . ' Line: ' . $ex->getLine();
            $retVal['message'][] = $message;
            \Log::error('CRAWL ERROR', [$message]);
        }

        return $retVal;
    }

    public function getRandomSellerId() {
        $user = User::where('email', 'mactrantung@gmail.com')->first();
        if ($user) {
            return $user->id;
        }

        return 1;
    }

    public function getCategory($slug) {
        $category = DB::table('category')->where('slug', $slug)->where('type', 'PRODUCT')->first(['id']);
        if ($category) {
            return $category->id;
        }
        $locale = env('APP_LOCALE');
        if (!$locale) {
            $locale = 'us';
        }
        $categoryId = config('crawl-product::crawl-' . $locale . '.slug-to-category.' . $slug);
        return $categoryId;
    }

    public function getTags($data) {
        $result = [];

        foreach ($data['tags'] as $tagTitle) {
            $tag = Tag::firstOrCreate(
                [
                    'title' => $tagTitle
                ],
                ['slug' => strToSlug($tagTitle)]
            );

            $result[] = $tag->id;
        }

        return implode(',', $result);
    }
}
