<?php

namespace Modules\CrawlProduct\Controllers\AutoCrawl;

use DB;
use Illuminate\Http\Request;
use Modules\Pod\Models\User;
use Modules\CrawlProduct\Models\Tag;
use Illuminate\Support\Facades\Input;
use Modules\CrawlProduct\Models\Brand;
use Modules\CrawlProduct\Models\Category;
use Modules\CrawlProduct\Models\ProductMeta;
use Modules\CrawlProduct\Models\ProductNUser;
use Modules\CrawlProduct\Controllers\Controller;
use Modules\CrawlProduct\Controllers\Impl\ProductCreation;
use App\Modules\CrawlProduct\Controllers\Services\EtsyService;
use Modules\CrawlProduct\Controllers\Impl\ProductUseTemplateCreation;

class CrawlRedbubbleController extends BaseAutoCrawlController {
    protected $productCreation;
    protected $productUseTemplateCreation;

    public function __construct(ProductCreation $productCreation, ProductUseTemplateCreation $productUseTemplateCreation) {
        $this->productCreation = $productCreation;
        $this->productUseTemplateCreation = $productUseTemplateCreation;
    }

    public function crawl(Request $request) {
        set_time_limit(10 * 3600);
        \Log::useDailyFiles(storage_path() . '/logs/crawl-redbubble.log');
        ini_set('memory_limit', '2048M');
        $query = DB::table('crawl_redbubble');
        if (!$request->get('toId') || !$request->get('fromId')) {
            return [
                'status' => 'fail',
                'message' => 'missing fromId or toId'
            ];
        }
        $fromId = $request->get('fromId', 0);
        $toId = $request->get('toId', 0);
        $query->whereNotNull('download_url');
        if (!$request->get('is_overwrite_product')) {
            $status = $request->get('status', 'PENDING');
            $query->where('status', $status);
        } else {
            if ($request->has('status')) {
                $query->where('status', $request->get('status'));
            } else {
                $query->where('status', '!=', 'IGNORE');
            }
        }
        if ($request->has('contain')) {
            $query->where('url', 'like', '%' . $request->get('contain') . '%');
        }
        if ($request->has('category')) {
            $query->where('category', $request->get('category'));
        }
        $step = $request->get('step', 50);
        $locale = env('APP_LOCALE', 'us');
        if (!$locale) {
            $locale = 'us';
        }
        $localePrefix = $locale == 'us' ? '' : "/$locale";
        $retVal = [
            'status' => 'successful',
            'message' => [],
        ];
        $brand = Brand::take(1)->first();
        while($fromId <= $toId) {
            $stepToId = $fromId + $step < $toId ? $fromId + $step : $toId;
            try {
                $cloneQuery = clone $query;
                $cloneQuery->where('id', '>=', $fromId);
                $cloneQuery->where('id', '<=', $stepToId);
                $crawlItems = $cloneQuery->get(['id']);
                foreach ($crawlItems as $crawlItem) {
                    $this->asyncRequest(
                        url($localePrefix . '/crawl-product/async-auto-crawl/crawl-redbubble'), 
                        'POST',
                        [
                            'craw_data_id' => $crawlItem->id,
                            'brand_id' => $brand ? $brand->id : null,
                            'generate_variant' => $request->get('generate_variant', 1),
                            'is_overwrite_product' => $request->get('is_overwrite_product', 0),
                            'is_download_image' => $request->get('is_download_image', 0),
                        ], 
                        [],
                        5
                    );
                }
                if (count($crawlItems) > 0) {
                    sleep(20);
                }
            } catch (\Exception $ex) {
                $message = $ex->getMessage() . ' File: ' . $ex->getFile() . ' Line: ' . $ex->getLine();
                $retVal['message'][] = $message;
                \Log::error($message);
            }

            $fromId += $step + 1;
        }

        return $retVal;
    }

    public function getDefaultProductId($categories) {
        $locale = env('APP_LOCALE', 'us');
        $locale = $locale ? $locale : 'us';
        $defaultProductId = null;
        if (count($categories)) {
            $config = \DB::table('crawl_product_default_product')->whereIn('category_id', $categories)->first();
            if ($config) {
                $defaultProductId = $config->product_id;
            }
            if (!$defaultProductId) {
                foreach ($categories as $categoryId) {
                    $id = config('crawl-product::default.product_template.' . $locale . '.' . $categoryId);
                    if ($id) {
                        $defaultProductId = $id;
                        break;
                    }
                }
            }
        }

        return $defaultProductId;
    }

    public function asyncCrawl(Request $request) {
        \Log::useDailyFiles(storage_path() . '/logs/crawl-redbubble.log');
        $retVal = [
            'status' => 'error2',
            'message' => [],
        ];
        ini_set('memory_limit', '2048M');
        if ($request->has('crawl_item')) {
            $crawlItem = $request->get('crawl_item');
        } else {
            $dataId = $request->get('craw_data_id');
            $crawlItem = DB::table('crawl_redbubble')->where('id', $dataId)->first()->toArray();
            if (!$crawlItem) {
                $retVal['message'] = 'Product not found: ' . $dataId;
                return $retVal;
            }
        }
        $locale = env('APP_LOCALE', 'us');
        if (!$locale) {
            $locale = 'us';
        }
        try {
            $category = $this->getCategory($crawlItem['category']);
            if (!$category) {
                \Log::error('can not get category : ', [$crawlItem]);
                $retVal['message'][] = 'can not get category';
                return $retVal;
            }
            
            $data = $this->buildData($crawlItem, $category);
            $tags = $this->getTags($data);
            if (!empty($data['name'])) {
                $newRequest = new Request();
                $newRequest->merge([
                    'brand_id' => $request->get('brand_id', 1),
                    'country_code' => $locale,
                    'remove_size_chart' => false,
                    'generate_variant' => $request->get('generate_variant', 1),
                    'tags' => $tags,
                    'approve_advertising' => 0
                ]);

                $newRequest->merge([
                    'categories' => $category->id
                ]);
                if ($request->has('is_overwrite_product')) {
                    $newRequest->merge(['is_overwrite_product' => 1]);
                }
                $data['force_status'] =  $request->get('force_status', 'ACTIVE');
                $data['is_download_image'] = $request->get('is_download_image', false);
                $pdc = DB::table('product_design_crawl_code')->where('value', $data['design_crawl_code'])->first();
                if ($pdc) {
                    $newRequest->merge(['is_overwrite_product' => 1]);
                    $pnu = ProductNUser::where('product_id', $pdc->product_id)->first();
                    if ($pnu) {
                        $data['user_id'] = $pnu->user_id;
                    } else {
                        $data['user_id'] = $this->getRandomSellerId();
                    }
                } else {
                    $data['user_id'] = $this->getRandomSellerId();
                }
                $template = null;
                if (!empty($data['direction'])) {
                    $templateId = $this->getTemplateByDirection([$category->id], $data['direction']);
                    if ($templateId) {
                        $template = $this->getTemplate($templateId);
                    }
                }
                if (!$template) {
                    $template = $this->getTemplate([$category->id], $data['direction']);
                }
                if ($template) {
                    $data['template'] = $template;
                    $response = $this->productUseTemplateCreation->saveCrawlProduct($newRequest, $data);
                } else {
                    $response = $this->productCreation->saveCrawlProduct($newRequest, $data);
                }
                if (isset($response['status']) && $response['status'] == 'successful') {
                    // $product = $response['result'];
                    // $this->saveDesign($product, $data['crawl_code'], $crawlItem['design_url']);
                    $retVal['status'] = 'successful';
                    $retVal['result'] = $response['result'];
                } else if (isset($response['status']) && $response['status'] == 'duplicate') {
                    $retVal['status'] = 'duplicate';
                } else {
                    $retVal['status'] = 'error';
                    $retVal['message'][] = $response;
                    \Log::error('saveCrawlProduct', [$response]);
                }
            } else {
                $retVal['status'] = 'error';
                $message = 'cannot get product name';
                \Log::error($message);
                $retVal['message'][] = $message;
            }
        } catch (\Exception $ex) {
            $message = $ex->getMessage() . ' File: ' . $ex->getFile() . ' Line: ' . $ex->getLine();
            $retVal['status'] = 'error';
            $retVal['message'][] = $message;
            \Log::error('CRAWL ERROR', [$message]);
        }

        return $retVal;
    }

    public function getTemplateByDirection($categoryIds, $direction) {
        $locale = env('APP_LOCALE', 'us');
        if (!$locale) {
            $locale = 'us';
        }
        foreach ($categoryIds as $categoryId) {
            if (config('crawl-product::template.template_by_direction.' . $locale . '.' . $categoryId . '.' . $direction, false)) {
                return config('crawl-product::template.template_by_direction.' . $locale . '.' . $categoryId . '.' . $direction, false);
                break;
            }
        }
    }

    public function getRandomSellerId() {
        $user = User::where('email', 'mactrantung@gmail.com')->first();
        if ($user) {
            return $user->id;
        }

        return 1;
    }

    public function getCategory($redbubbleCategorySlug) {
        $category = Category::where('slug', $redbubbleCategorySlug)->first(['id', 'name', 'slug']);
        if ($category) {
            $category->display_name = $category->name;
            return $category;
        }
        $slugs = [
            "kids-pullover-hoodies" => [
                "slug" => "kids-pullover-hoodies",
                "display_name" => "Kid Pullover Hoodie"
            ],
            "baby-onepieces" => [
                "slug" => "onesies",
                "display_name" => "Onesie"
            ],
            "longsleeve-onesies" => [
                "slug" => "onesies",
                "display_name" => "Onesie"
            ],
            "backpacks" => [
                "slug" => "backpacks",
                "display_name" => "Backpack"
            ],
            "pin-buttons" => [
                "slug" => "pin-buttons",
                "display_name" => "Pin Button"
            ],
            "drawstring-bags" => [
                "slug" => "drawstring-bags",
                "display_name" => "Drawstring Bag"
            ],
            "socks" => [
                "slug" => "socks",
                "display_name" => "Socks"
            ],
            "tote-bags" => [
                "slug" => "bags",
                "display_name" => "Bag"
            ],
            "aprons" => [
                "slug" => "kitchen-aprons",
                "display_name" => "Kitchen Apron"
            ],
            "tapestry" => [
                "slug" => "tapestries",
                "display_name" => "Tapestry"
            ],
            "bath-mats" => [
                "slug" => "bath-mat",
                "display_name" => "Bath Mat"
            ],
            "shower-curtains" => [
                "slug" => "shower-curtains",
                "display_name" => "Shower Curtain"
            ],
            "posters" => [
                "slug" => "premium-matte-vertical-posters",
                "display_name" => "Premium Matte Vertical Poster"
            ],
            "iphone-cases" => [
                "slug" => "iphone-cases",
                "display_name" => "Iphone Case"
            ],
            "leggings" => [
                "slug" => "leggings",
                "display_name" => "Leggings",
            ],
            "tank-tops" => [
                "slug" => "tank-tops",
                "display_name" => "Tank Tops",
            ],
            "canvas-prints" => [
                "slug" => "canvas",
                "display_name" => "Canvas",
            ],
            "travel-mugs" => [
                "slug" => "travel-coffee-mugs",
                "display_name" => "Travel Coffee Mugs",
            ],
            "t-shirt" => [
                "slug" => "t-shirts",
                "display_name" => "T-shirts",
            ],
            "hoodies" => [
                "slug" => "hoodies",
                "display_name" => "Hoodies",
            ],
            "lightweight-hoodies" => [
                "slug" => "hoodies",
                "display_name" => "Hoodies",
            ],
            "sweatshirts" => [
                "slug" => "sweatshirts",
                "display_name" => "Sweatshirts",
            ],
            "lightweight-sweatshirts" => [
                "slug" => "sweatshirts",
                "display_name" => "Sweatshirts",
            ],
            "stickers" => [
                "slug" => "stickers",
                "display_name" => "Stickers",
            ],
            "baseball-caps" => [
                "slug" => "baseball-caps",
                "display_name" => "Baseball Caps",
            ],
            "scarves" => [
                "slug" => "scarves",
                "display_name" => "Scarves",
            ],
        ];

        if (!isset($slugs[$redbubbleCategorySlug])) {
            return null;
        }

        $item = $slugs[$redbubbleCategorySlug];

        $category = Category::where('slug', $item['slug'])->first(['id', 'name', 'slug']);
        if ($category) {
            $category->display_name = $item['display_name'];
        }
        return $category;
    }

    public function buildData($crawlItem, $category) {
        $parseUrl = parse_url($crawlItem['url']);
        $crawlCode = substr($parseUrl['path'], strpos($parseUrl['path'], '/i/') + 3);
        $crawlCode = substr($crawlCode, 0, strripos($crawlCode, '.') - 1);

        $retVal = [
            'name'  => $crawlItem['title'] . ' ' . $category->display_name,
            'slug' => strToSlug($crawlItem['title']),
            'description'  => '',
            'content'  => '',
            'image_url'  => $crawlItem['image_url'],
            'galleries'  => [],
            'price'  => '',
            'high_price'  => '',
            'note'  => 'Redbuble url: ' . $crawlItem['url'],
            'crawl_code'  => md5($crawlCode),
            'design_crawl_code' => $crawlItem['design_id'],
            'site'  => 'redbubble',
            'comments'  => '',
            'meta_title'  => $crawlItem['title'],
            'meta_description'  => '',
            'meta_keywords'  => '',
            'is_fast_shipment' => '',
            'related_products'  => '',
            'tags' => json_decode($crawlItem['tags']),
            'store_link' => $crawlItem['store'],
            'variants' => [],
            'product_variants' => []
        ];

        if (isset($crawlItem['attributes'])) {
            $attrbutes = $crawlItem['attributes'];
        } else {
            $attrbutes = DB::table('crawl_redbubble_attribute')
                ->where('design_id', $crawlItem['design_id'])
                ->where('category', $category)
                ->get()
                ->toArray();
        }

        $result = $this->buildVariants($attrbutes);
        if (count($result['product_variants'])) {
            $retVal['image_url'] = $result['product_variants'][0]['image_url'];
            $retVal['galleries'] = $result['product_variants'][0]['galleries'];
        }

        $retVal['variants'] = $result['variants'];
        $retVal['product_variants'] = $result['product_variants'];
        $retVal['direction'] = $this->getDirection($retVal['variants']);

        return $retVal;
    }


    public function getDirection($variants) {
        foreach ($variants as $variant) {
            if ($variant['slug'] == 'size' && count($variant['options'])) {
                foreach ($variant['options'] as $option) {
                    $name = preg_replace("/[^\d\.x\\/\-]/", "", $option['name']);
                    preg_match("/([\d\.]+)\s*x\s*([\d\.]+)/", $name, $matches);
                    if (count($matches) > 2) {
                        $width = doubleval($matches[1]);
                        $height = doubleval($matches[2]);
                        if ($width == $height) {
                            return 'square';
                        }
                        
                        if ($width < $height) {
                            if ($width / $height <= 0.6) {
                                return 'vertical-long';
                            }

                            return 'vertical';
                        } else {
                            if ($height / $width <= 0.6) {
                                return 'horizontal-long';
                            }

                            return 'horizontal';
                        }
                    }
                    break;
                }
            }
        }

        return null;
    }

    public function getTags($data) {
        $result = [];

        foreach ($data['tags'] as $tagTitle) {
            $tag = Tag::firstOrCreate(
                [
                    'title' => $tagTitle
                ],
                ['slug' => strToSlug($tagTitle)]
            );

            $result[] = $tag->id;
        }

        return implode(',', $result);
    }

    public function buildVariants($attrbutes) {
        $variants = [];
        $productVariants = [];
        $ignoreVariants = [
            'configuration', 'printLocation'
        ];
        
        foreach ($attrbutes as $attrbute) {
            $attrbute = (array) $attrbute;
            $productVariant = [
                'galleries' => [],
                'price' => 0,
                'high_price' => 0,
                'image_url' => '',
                'is_default' => 0,
                'variants' => []
            ];
            $items = json_decode($attrbute['attributes'], true);
            foreach ($items as $item) {
                if (!in_array($item['name'], $ignoreVariants)) {
                    $variantSlug = strtolower($item['name']);
                    $varinatName = $this->translateName($variantSlug, ucfirst($variantSlug));
                    $variantSlug = $this->toFriendlyString($varinatName);
                    if (!isset($variants[$variantSlug])) {
                        $variants[$variantSlug] = [
                            'name' => $varinatName,
                            'slug' => $variantSlug,
                            'type' => 'OPTION',
                            'options' => []
                        ];
                    }
                    $optionName = $this->translateName(strtolower($item['value']), ucfirst(strtolower($item['value'])));
                    $option = [
                        'variant_slug' => $variantSlug,
                        'slug' => $this->toFriendlyString($optionName),
                        'name' => $optionName 
                    ];
                    if (!isset($variants[$variantSlug]['options'][$option['slug']])) {
                        $variants[$variantSlug]['options'][$option['slug']] = $option;
                    }
                    $productVariant['variants'][] = $option;
                }
            }

            $imgItems = json_decode($attrbute['preview_set'], true);
            foreach ($imgItems['previews'] as $item) {
                if (!$productVariant['image_url']) {
                    $productVariant['image_url'] = $item['url'];
                } else {
                    $productVariant['galleries'][] = $item['url'];
                }
            }

            $productVariants[] = $productVariant;
        }

        $variantValues = [];
        foreach ($variants as $variant) {
            $variant['options'] = array_values($variant['options']);
            $variantValues[] = $variant;
        }
        return [
            'variants' => $variantValues,
            'product_variants' => $productVariants
        ];
    }

    public function translateName($key, $name) {
        $locale = env('APP_LOCALE', 'us');
        if (!$locale) {
            $locale = 'us';
        }

        $value = config('crawl-product::crawl-' . $locale . '.' . $key);
        if ($value) {
            if (!is_array($value)) {
                return $value;
            }

            return $name;
        }
        $option = \DB::table('option')->where('key', 'crawl-product::translate-name')->first();
        if ($option) {
            $config = json_decode($option->value, true);
            if (!empty($config)) {
                $slugs = explode('.', $key);
                if (count($slugs) == 2 && isset($config[$slugs[0]]) && isset($config[$slugs[0]][$slugs[1]]) && $config[$slugs[0]][$slugs[1]]) {
                    $name = $config[$slugs[0]][$slugs[1]];
                }
            }
        }
        
        return $name;
    }
    
    protected function toFriendlyString ($string) {
        $string = trim($string);
        $countryCode = Input::get('country_code', 'us');
        if (in_array($countryCode, ['jp', 'kr'])) {
            $string = japaneseToEnglish($string);
        }
        $retVal = preg_replace('/\W+/', '-', strtolower(trim($string)));
        return trim($retVal, '-');
    }

    public function saveDesign($product, $crawlCode, $designUrl) {
        $productIds = DB::table('product_design_crawl_code')
            ->where('value', 'redbubble' . $crawlCode)
            ->get()
            ->pluck('product_id');
        $pnd = DB::table('product_n_design')
            ->whereIn('product_id', $productIds)
            ->where('is_primary', 1)
            ->first();
        if ($pnd) {
            $designId = $pnd->design_id;
        } else {
            $designId = DB::table('design')
                ->insertGetId([
                    'name' => $product->name,
                    'image_url' => $designUrl,
                    'source' => 'redbubble',
                    'created_at' => date('Y-m-d H:i:s', time()),
                    'updated_at' => date('Y-m-d H:i:s', time())
                ]);
        }

        DB::table('product_n_design')
            ->insert([
                'product_id' => $product->id,
                'design_id' => $designId,
                'is_primary' => 1
            ]);
    }
}
