<?php

namespace Modules\CrawlProduct\Controllers;

use Illuminate\Http\Request;
use InvalidArgumentException;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Redis;
use Modules\CrawlProduct\Models\Brand;
use Modules\CrawlProduct\Models\Product;
use Modules\CrawlProduct\Models\ProductSku;
use Modules\CrawlProduct\Models\ProductMeta;
use Modules\CrawlProduct\Models\ProductGallery;
use Modules\CrawlProduct\Models\ProductVariant;
use Modules\CrawlProduct\Models\ProductSkuValue;
use Modules\CrawlProduct\Models\ProductNCategory;
use Modules\CrawlProduct\Models\ProductVariantOption;
use Modules\CrawlProduct\Controllers\Impl\ProductCreation;
use App\Modules\CrawlProduct\Controllers\Services\EtsyService;
use App\Modules\CrawlProduct\Controllers\Services\AmazonService;
use App\Modules\CrawlProduct\Controllers\Services\GeckoCustomService;
use App\Modules\CrawlProduct\Controllers\Services\MacornerService;
use App\Modules\CrawlProduct\Controllers\Services\MakeZBrightGiftsService;
use App\Modules\CrawlProduct\Controllers\Services\PawfectHouseService;
use App\Modules\CrawlProduct\Controllers\Services\PawsionateService;
use App\Modules\CrawlProduct\Controllers\Services\SunflowerlyService;
use App\Modules\CrawlProduct\Controllers\Services\TrendingCustomService;
use App\Modules\CrawlProduct\Controllers\Services\WanderPrintService;
use App\Modules\CrawlProduct\Controllers\Services\CadeauPlusService;
use App\Modules\CrawlProduct\Controllers\Services\GeschenkemallService;
use Modules\Trademarks\Controllers\Impl\LanguageChecker;
use Modules\CrawlProduct\Controllers\Impl\TrademarkChecker;

class HomeController extends Controller
{
    protected $trademarkChecker;
    protected $productCreation;
    protected $languageChecker;
    
    public function __construct(ProductCreation $productCreation, TrademarkChecker $trademarkChecker, LanguageChecker $languageChecker) {
        $this->productCreation = $productCreation;
        $this->trademarkChecker = $trademarkChecker;
        $this->languageChecker = $languageChecker;
    }

    public function index(Request $request)
    {
        set_time_limit(0);
        $response = [
            'status' => 'fail'
        ];
        $url = $request->input('url');
        $url = trim($url);
        if ($url) {
            $domain = str_ireplace('www.', '', parse_url($url, PHP_URL_HOST));
            $service = null;
            if (!$request->has('remove_size_chart') && config('crawl-product::sa.remove_size_chart_image.enable')) {
                $request->merge(['remove_size_chart' => true]);
            }
            if (strpos($domain, 'amazon') !== false) {
                $service = new AmazonService($request->all());
            } else if ($domain == 'etsy.com') {
                $service = new EtsyService($request->all());
            }
            if ($service) {
                $data = $service->crawl($url);
                if ($data && !empty($data['name'])) {
                    $response = $this->productCreation->saveCrawlProduct($request, $data);
                } else {
                    $message = 'Crawl url fail';
                    if (!empty($data['message'])) {
                        $message = $data['message'];
                    }
                    $response['message'] = $message;
                }
            }
        }
        if ($request->input('parent_id')) {
            return $response;
        }
        return response()->json($response);
    }

    public function store(Request $request) {
        $data = $request->all();
        return $this->productCreation->storeProduct($data);
    }

    public function management () {
        $configProxy = config('crawl-product::sa.proxy', []);
        $listCountryCode = array_keys($configProxy);
        $countryCodes = [];
        foreach ($listCountryCode as $countryCode) {
            $countryCodes[] = [
                'name' => strtoupper($countryCode),
                'code' => $countryCode
            ];
        }
        return view('crawl-product::management', ['listCountryCode' => json_encode($countryCodes)]);
    }

    public function changePrice () {
        $cacheData = Cache::get('data-update-price::' .env('APP_LOCALE') . '::' . Auth::user()->id);
        return view('crawl-product::change-price', ['cacheData' => $cacheData]);
    }

    public function replaceVariant() {
        return view('crawl-product::replace-variant');
    }

    public function updatePrice (Request $request) {
        set_time_limit(1800);
        ini_set('memory_limit', '2048M');
        $categories = $request->input('categories', []);
        $price = $request->input('price', 0);
        $currentPrice = $request->input('current_price', 0);
        $highPrice = $request->input('high_price', 0);
        $listChangePrice = $request->input('listChangePrice', []);
        $createDateFrom = $request->input('create_date_from', '');
        $createDateTo = $request->input('create_date_to', '');
        Log::info('updatePrice input' , [$request->all()]);
        $categoryIds = [];
        foreach ($categories as $category) {
            $categoryIds[] = $category['id'];
        }
        if ($categoryIds) {
            $price = floatval($price);
            $listProductIds = ProductNCategory::whereIn('category_id', $categoryIds)
                ->get([\DB::raw('distinct product_id')])
                ->pluck('product_id')
                ->toArray();
            if ($listProductIds && ($currentPrice || $createDateFrom || $createDateTo)) {
                $tmpProductIds = [];
                foreach (array_chunk($listProductIds, 1000) as $productIds)
                {
                    $query = Product::whereIn('id', $productIds);
                    if ($currentPrice) {
                        $query->where('price', '=', $currentPrice);
                    }
                    if ($createDateFrom) {
                        $from = \DateTime::createFromFormat('d/m/Y', $createDateFrom);
                        $from->setTime(00, 00, 00);
                        $query->where('created_at', '>=', $from);
                    }
                    if ($createDateTo) {
                        $to = \DateTime::createFromFormat('d/m/Y', $createDateTo);
                        $to->setTime(23, 59, 59);
                        $query->where('created_at', '<=', $to);
                    }
                    $productIds = $query->pluck('id')
                        ->toArray();
                    $tmpProductIds = array_merge($tmpProductIds, $productIds);
                }
                $listProductIds = $tmpProductIds;
            }
            if ($listProductIds) {
                $dataUpdate = [];
                if ($price) {
                    $dataUpdate['price'] = $price;
                }
                if ($highPrice) {
                    $dataUpdate['high_price'] = $highPrice;
                }
                $userId = !empty(Auth::user()->id) ? Auth::user()->id : 1;
                foreach (array_chunk($listProductIds, 1000) as $productIds)
                {
                    if ($dataUpdate) {
                        $dataUpdateProduct = $dataUpdate;

                        foreach (array_chunk($productIds, 1000) as $arr)
                        {
                            Product::whereIn('id', $arr)->update($dataUpdateProduct);
                        }
                        foreach (array_chunk($productIds, 1000) as $arr)
                        {
                            ProductSku::whereIn('product_id', $arr)->update($dataUpdate);
                        }

                    }
                    if ($listChangePrice) {
                        foreach ($listChangePrice as $changePrice) {
                            if ($changePrice['sizes']) {
                                $variantSizeOptionIds = [];
                                foreach ($changePrice['sizes'] as $size) {
                                    $variantSizeOptionIds[] = $size['id'];
                                }

                                $productSkuHasSizeIds = ProductSkuValue::forceIndex('pIdOption')
                                    ->whereIn('variant_option_id', $variantSizeOptionIds)
                                    ->whereIn('product_id', $productIds)
                                    ->get([\DB::raw('distinct sku_id')])
                                    ->pluck('sku_id')
                                    ->toArray();
                                if ($productSkuHasSizeIds) {
                                    $dataVariantUpdate = [];
                                    if (!empty($changePrice['price'])) {
                                        $dataVariantUpdate['price'] = floatval($changePrice['price']);
                                    }
                                    if (!empty($changePrice['high_price'])) {
                                        $dataVariantUpdate['high_price'] = floatval($changePrice['high_price']);
                                    }
                                    if ($dataVariantUpdate) {
                                        foreach (array_chunk($productSkuHasSizeIds, 1000) as $arrSize)
                                        {
                                            ProductSku::whereIn('id', $arrSize)
                                                ->whereIn('product_id', $productIds)
                                                ->update($dataVariantUpdate);

                                            $productSkuDefaultIds = ProductSku::whereIn('product_id', $productIds)
                                                ->whereIn('id', $arrSize)
                                                ->where('is_default', '=', 1)
                                                ->get([\DB::raw('distinct product_id')])
                                                ->pluck('product_id')
                                                ->toArray();
                                            if ($productSkuDefaultIds) {
                                                Product::whereIn('id', $productSkuDefaultIds)->update($dataVariantUpdate);
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
                Log::info('updatePrice items' , [count($listProductIds)]);
            }
        }
        return response()->json([
            'status' => 'successful'
        ]);
    }

    public function saveDataUpdatePrice (Request $request) {
        $data = json_encode($request->all());
        Cache::put('data-update-price::' .env('APP_LOCALE') . '::' . Auth::user()->id, $data, 90 * 86400);
        return response()->json([
            'status' => 'successful'
        ]);
    }

    public function rebuildAmazonAsin () {
        set_time_limit(7200);
        $idsCrawled = ProductMeta::where('key', '=', 'amazon_crawl_code')
            ->whereNotNull('value')
            ->pluck('product_id')->toArray();
        $query = Product::where('note', 'like', '%Amazon url:%');
        if (count($idsCrawled) > 0) {
            $query->whereNotIn('id', $idsCrawled);
        }
        $products = $query->get();
        $service = new AmazonService([]);
        $listAsin = [];
        foreach ($products as $product) {
            $re = '/Amazon url: ([^\n\r ]+)/';
            preg_match_all($re, $product->note, $matches, PREG_SET_ORDER, 0);
            if (isset($matches[0][1])) {
                $asin = $service->crawlAsin($matches[0][1]);
                if ($asin) {
                    ProductMeta::create([
                        'product_id' => $product->id,
                        'key' => 'amazon_crawl_code',
                        'value' => $asin
                    ]);
                    $listAsin[] = $asin;
                }
            }
        }
        return response()->json([
            'status' => 'successful',
            'listAsin' => $listAsin
        ]);
    }

    public function rebuildAmazonDescription () {
        return '';
        set_time_limit(0);
        $query = Product::where('note', 'like', '%Amazon url:%')
            ->where('description', 'not like', '<ul%')
            ->where('status', '=', 'active')
            ->whereNull('deleted_at');
        $products = $query->get(['id', 'note', 'description']);
        $service = new AmazonService([]);
        foreach ($products as $product) {
            $re = '/Amazon url: ([^\n\r ]+)/';
            preg_match_all($re, $product->note, $matches, PREG_SET_ORDER, 0);
            if (isset($matches[0][1])) {
                $description = $service->crawlDescription($matches[0][1]);
                if ($description) {
                    $product->description = $description;
                    $product->save();
                }
            }
        }
        return response()->json([
            'status' => 'successful',
        ]);
    }

    public function rebuildAmazonProductMeta () {
        return '';
        set_time_limit(0);
        $productIdsSeo = ProductMeta::where('key', '=', 'seo')->pluck('product_id')->toArray();
        $query = Product::where('note', 'like', '%Amazon url:%')
            ->where('status', '=', 'active')
            ->whereNotIn('id', $productIdsSeo)
            ->whereNull('deleted_at');
        $products = $query->get(['id', 'note', 'description']);
        $service = new AmazonService([]);
        foreach ($products as $product) {
            $re = '/Amazon url: ([^\n\r ]+)/';
            preg_match_all($re, $product->note, $matches, PREG_SET_ORDER, 0);
            if (isset($matches[0][1])) {
                $productMeta = $service->crawlProductMeta($matches[0][1]);
                if ($productMeta) {
                    ProductMeta::create([
                        'product_id' => $product->id,
                        'key' => 'seo',
                        'value' => json_encode([
                            'meta_title' => $productMeta['meta_title'],
                            'meta_description' => $productMeta['meta_description'],
                            'meta_keywords' => $productMeta['meta_keywords']
                        ])
                    ]);
                }
            }
        }
        return response()->json([
            'status' => 'successful',
        ]);
    }

    public function updateProductName (Request $request) {
        set_time_limit(15 * 60);

        $products = $request->input('products', []);
        $checkProducts = [];
        foreach ($products as $product) {
            $productObj = Product::find($product['id']);
            if ($productObj) {
                if (preg_match("/\b([A-Z])\s([A-Z])\s([A-Z])/", $product['name'])) {
                    $product['name'] = strtolower($product['name']);
                }
                $productObj->name = decoreText($product['name']);
                $productObj->slug = strToSlug($product['name']);
                if (in_array($request->input('country_code'), ['jp', 'kr'])) {
                    $productObj->slug = strToSlug(japaneseToEnglish($product['name']));
                }
                $productObj->save();
                $checkProducts[] = $productObj;
                $this->trademarkChecker->check($productObj);
            }
        }
        if (count($checkProducts)) {
            $this->languageChecker->check($checkProducts);
        }

        return response()->json([
            'status' => 'successful'
        ]);
    }

    public function rebuildAmazonVariantColor (Request $request) {
        return '';
        $time = microtime(true);
        set_time_limit(600);
        ini_set('memory_limit', '2048M');
        $productIds = ProductMeta::where('key', '=', 'amazon_crawl_code')
            ->pluck('product_id')
            ->toArray();
        $productIdsColor = ProductSkuValue::where('variant_id', '=', 2)
            ->groupBy('product_id')
            ->pluck('product_id')
            ->toArray();
        $ids = array_diff($productIds, $productIdsColor);
        $service = new AmazonService([]);
        $ids = array_values($ids);
        $retVal = [];
        $from = intval($request->input('from'));
        $to = intval($request->input('to'));
        foreach ($ids as $key => $id) {
            if ($key < $from || $key > $to) {
                continue;
            }
            $product = Product::find($id);
            if ($product) {
                $re = '/Amazon url: ([^\n\r ]+)/';
                preg_match_all($re, $product->note, $matches, PREG_SET_ORDER, 0);
                if (isset($matches[0][1])) {
                    $crawl = $service->crawlColor($matches[0][1]);
                    if ($crawl) {
                        $retVal[] = $id;
                    }
                }
            }
        }
        return $retVal;
    }

    public function reCrawlColor (Request $request) {
        set_time_limit(7200);
        $ids = [-1];
        $productCategory = ProductNCategory::where('category_id', '=', 7)
            ->groupBy('product_id')
            ->pluck('product_id')
            ->toArray();
        $products = Product::whereIn('id', $ids)
            ->whereIn('id', $productCategory)
            ->orderBy('id', 'asc')
            ->get();
        $service = new AmazonService($request->all());
        $from = intval($request->input('from', 0));
        $to = intval($request->input('to', 19));
        $productIds = [];
        foreach ($products as $key => $product) {
            if ($key < $from || $key > $to) {
                continue;
            }
            $re = '/Amazon url: ([^\n\r ]+)/';
            preg_match_all($re, $product->note, $matches, PREG_SET_ORDER, 0);
            if (isset($matches[0][1])) {
                $data = $service->crawl($matches[0][1]);
                if ($data) {
                    ProductNCategory::where('product_id', $product->id)
                        ->where('category_id', 61)->delete();
                    ProductNCategory::insert([
                        'product_id' => $product->id,
                        'category_id' => 61
                    ]);
                    ProductSkuValue::where('product_id', $product->id)->delete();
                    $skuIds = ProductSku::where('product_id', $product->id)->get(['id'])->pluck('id')->toArray();
                    ProductSku::where('product_id', $product->id)->delete();
                    ProductGallery::where('product_id', $product->id)->where('type', 'PRODUCT')->delete();
                    ProductGallery::whereIn('product_id', $skuIds)->where('type', 'VARIANT')->delete();
                    $galleries = [];
                    if (isset($data['galleries'])) {
                        $galleries = $data['galleries'];
                    }
                    $this->storeProductGalleries($galleries, $product->id);
                    if (isset($data['product_variants'])) {
                        foreach ($data['product_variants'] as $keyVariant => $skuData) {
                            $data['product_variants'][$keyVariant]['price'] = 12.95;
                        }
                    }
                    $this->storeProductVariant($product, $data);
                    $productIds[] = $product->id;
                }
            }
        }
        return response()->json([
            'status' => 'successful',
            'data' => $productIds
        ]);
    }

    public function rebuildVariantOption () {
        $list = [
            "Uma cor	" => "	Uma cor",
            "Preto	" => "	Preto",
            "Azul marinho	" => "	Azul marinho",
            "Azul real	" => "	Azul royal",
            "Cinza grafite	" => "	Cinza grafite",
            "Roxo	" => "	Roxo",
            "Blanco	" => "	Branco",
            "Rojo	" => "	Vermelho",
            "Gris Jaspeado	" => "	Cinza claro",
            "Zafiro	" => "	Safira",
            "Rosado Neón	" => "	Rosa neon",
            "Morado	" => "	Roxo",
            "Tk1484	" => "	Tk1484",
            "Asfalto	" => "	Asfalto",
            "Ardósia	" => "	Ardósia",
            "Branco	" => "	Branco",
            "Vermelho arando	" => "	Vermelho arando",
            "Verde azeitona	" => "	Verde azeitona",
            "Cinza	" => "	Cinzento",
            "Cinza azulado	" => "	Cinzento-azulado",
            "Negro	" => "	Preto",
            "Azul Marino	" => "	Azul marinho",
            "Plata	" => "	Prata",
            "Jaspeado Oscuro	" => "	Jaspeado escuro",
            "Azul Jaspeado	" => "	Azul Jaspeado",
            "Rosado	" => "	Cor-da-rosa",
            "Arándano	" => "	Mirtilo",
            "Marrón	" => "	Castanho",
            "Verde Oliva	" => "	Verde oliva",
            "Azul Bebé	" => "	Azul-bebé",
            "Hierba	" => "	Erva",
            "Verde Kelly	" => "	Verde Kelly",
            "Limón	" => "	Limão",
            "Naranja	" => "	Laranja",
            "1-0	" => "	1-0",
            "1-2	" => "	1-2",
            "1-a2	" => "	1-a2",
            "1-a3	" => "	1-a3",
            "1a-1swirl 1	" => "	1a-1redemoinho 1",
            "1hairy Chest 3	" => "	1baú Peludo 3",
            "1six-pack Abs 2	" => "	1Barriga Tanquinho 2",
            "1titanic Cat	" => "	1gato titânico",
            "Hairy Chest 2	" => "	Baú Peludo 2",
            "Lion 2	" => "	Leão 2",
            "1-1	" => "	1-1",
            "1-1 Geometric Swirl	" => "	1-1 Redemoinho geométrico",
            "1-1b	" => "	1-1b",
            "1-1cat	" => "	1-1cat",
            "1-1colourful	" => "	1-1colorido",
            "1-1colourful 2	" => "	1-1colorido 2",
            "1-1colourful Smog	" => "	1-1Smog colorido",
            "1-1galaxy	" => "	1-1galáxia",
            "1-2 Geometric Swirl	" => "	1-2 Redemoinho geométrico",
            "1-2 Howling Wolf	" => "	1-2 Lobo Uivante",
            "1-2 Skull	" => "	1-2 Crânio",
            "1-2 Skulls	" => "	1-2 Crânios",
            "1-2 Wolf Word	" => "	1-2 Wolf Word",
            "1-5	" => "	1-5",
            "1-a	" => "	1-a",
            "1-flame 1	" => "	1-flama 1",
            "1-lion	" => "	1-leão",
            "1a-wolf 2	" => "	1a-lobo 2",
            "Alpaca 2	" => "	Alpaca 2",
            "Black Plaid	" => "	Xadrez Preto",
            "Colorful Flame	" => "	Flama Colorida",
            "Howling Wolf	" => "	Lobo Uivante",
            "Pizza Cat	" => "	Pizza Cat",
            "Six-pack Abs	" => "	Barriga Tanquinho",
            "Wolf	" => "	Lobo",
            "1-2c	" => "	1-2c",
            "1-4	" => "	1-4",
            "Hairy Chest	" => "	Baú Peludo",
            "Verde Bosque	" => "	Floresta Verde",
            "Rosa- Mujer	" => "	Mulher-rosa",
            "Pizarra	" => "	Borda",
            "Negro (	" => "	Preto",
            "^^	" => "	^^",
            "B1	" => "	B1",
            "Beer4	" => "	Cerveja4",
            "Cat Surf	" => "	Surf de Gato",
            "Colorful Ink 4	" => "	Tinta Colorida 4",
            "Dinosaur	" => "	Dinosaur",
            "Vortex2	" => "	Vórtice2",
            "White Smoke	" => "	Fumaça branca",
            "Beer2	" => "	Cerveja2",
            "Black-wolf	" => "	Lobo-preto",
            "Cat Book	" => "	Livro do gato",
            "Cat-a	" => "	Gato-a",
            "Colorful Ink	" => "	Tinta Colorida",
            "Colorful Ink 5	" => "	Tinta Colorida 5",
            "Colorful Smoke2	" => "	Fumaça Colorida2",
            "Colorful Vortex	" => "	Vórtice Colorido",
            "Diamond	" => "	Diamante",
            "Galaxy 2	" => "	Galáxia 2",
            "Octopus	" => "	Polvo",
            "Skeleton	" => "	Esqueleto",
            "Space Antelope	" => "	Space Antelope",
            "Space Cat	" => "	Gato do Espaço",
            "Tecnología B	" => "	Tecnología B",
            "Vortex	" => "	Vórtice",
            "Cats	" => "	Gatos",
            "Colorful	" => "	Colorida",
            "Colorful Ink 3	" => "	Tinta Colorida 3",
            "Colorful Smoke	" => "	Fumaça Colorida",
            "Zebra Pattern	" => "	Padrão de Zebra",
            "Galaxy 1	" => "	Galáxia 1",
            "Zebra Pattern 2	" => "	Padrão de Zebra 2",
            "Colorful Ink 2	" => "	Tinta Colorida 2",
            "Milk	" => "	Leite",
            "Palm Mark	" => "	Palm Mark",
            "Vortex2.	" => "	Vórtice2.",
            "Azul	" => "	Azul",
            "Blanco Plateado	" => "	Prata Branca",
            "Gris Claro	" => "	Cinzento Claro",
            "Plateado	" => "	Plateado",
            "Dark Marbled	" => "	Marmorizado Escuro",
            "Blanco/negro	" => "	Branco/preto",
            "Jaspeado oscuro/blanco	" => "	Jaspeado escuro/branco",
            "Azul marino/blanco	" => "	Azul marinho/branco",
            "Azul real/blanco	" => "	Azul royal/branco",
            "Rojo/blanco	" => "	Vermelho/branco",
            "Jaspeado negro/atlético	" => "	Jaspeado preto/ atlético",
            "Azul marino/jaspeado atlético	" => "	Azul marinho/jaspeado atlético",
            "Azul Eléctrico	" => "	Azul elétrico",
            "Rosa	" => "	Rosa",
        ];

        foreach ($list as $key => $item) {
            $variantOptions = ProductVariantOption::where('name', '=', trim($key))->get();
            if (count($variantOptions)) {
                foreach ($variantOptions as $variantOption) {
                    $variantOption->name = trim($item);
                    $variantOption->slug = strToSlug(trim($item));
                    $variantOption->save();
                }
            }
        }
        return response()->json(['status' => 'successful']);

    }

    public function crawlSearch (Request $request) {
        $service = new AmazonService($request->all());
        $keyword = $request->input('keyword');
        $countryCode = $request->input('country_code', 'us');
        $domain = config('crawl-product::sa.config_by_country')[$countryCode]['amazon_domain'];
        $urlSearch = 'https://www.'. $domain .'/s?' . http_build_query(['k' => $keyword]);
        $data = $service->crawlCategory($urlSearch, $countryCode);
        return response()->json([
            'data' => $data,
            'status' => 'successful'
        ]);
    }

    public function rebuildSlug (Request $request) {
        set_time_limit(3600);
        $count = 0;
        if (in_array(env('APP_LOCALE'), ['jp', 'kr'])) {
            $minLength = $request->input('min', 0);
            $maxLength = $request->input('max', 0);
            $products = Product::whereRaw('LENGTH(slug) >= ' . $minLength)
                ->whereRaw('LENGTH(slug) <= ' . $maxLength)
                ->get(['id', 'slug', 'name'])->toArray();
            foreach (array_chunk($products , 20) as $arr) {
                $listName = [];
                foreach ($arr as $product) {
                    $listName[] = $product['name'];
                }
                $data = japaneseToEnglish($listName);
                if (count($arr) == count($data)) {
                    foreach ($arr as $key => $item) {
                        if (!empty($data[$key])) {
                            Product::where('id', '=', $item['id'])
                                ->update(['slug' => strToSlug($data[$key])]);
                            $count++;
                        }
                    }
                }
            }
        }
        return response()->json([
            'count' => $count,
            'status' => 'successful'
        ]);
    }

    public function crawlImage(Request $request) {
        set_time_limit(7200);
        ini_set('memory_limit', '2048M');
        $productIds = $request->input('productIds', []);
        if (!is_array($productIds)) {
            $productIds = explode(',', $productIds);
        }
        if (empty($productIds)) {
            return response()->json([
                'status' => 'fail',
                'message' => 'Empty productIds!'
            ]);
        }

        $products = Product::query()
                        ->whereIn('id', $productIds)
                        ->select('id', 'note')
                        ->get();
        $parserInput = ['ignore_customization' => true];
        foreach ($products as $product) {
            $crawlInfo = [];
            preg_match('/https?:\/\/[^\s]+/i', $product->note, $matches);
            if (!empty($matches)) {
                $parseUrl = parse_url($matches[0]);
                switch ($parseUrl['host']) {
                    case 'sunflowerly.com':         $parseService = new SunflowerlyService($parserInput); break;
                    case 'pawsionate.com':          $parseService = new PawsionateService($parserInput); break;
                    case 'makezbrightgifts.com':    $parseService = new MakeZBrightGiftsService($parserInput); break;
                    case 'trendingcustom.com':      $parseService = new TrendingCustomService($parserInput); break;
                    case 'macorner.co':             $parseService = new MacornerService($parserInput); break;
                    case 'wanderprints.com':        $parseService = new WanderPrintService($parserInput); break;
                    case 'pawfecthouse.com':        $parseService = new PawfectHouseService($parserInput); break;
                    case 'geckocustom.com':         $parseService = new GeckoCustomService($parserInput); break;
                    case 'cadeauplus.com':          $parseService = new CadeauPlusService($parserInput); break;
                    case 'geschenkemall.com':       $parseService = new GeschenkemallService($parserInput); break;
                    default: break;
                }
                if (method_exists($parseService, 'parseHTML')) {
                    $ch = curl_init();
                    $timeout = 200;
                    curl_setopt($ch, CURLOPT_URL, $matches[0]);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                    curl_setopt($ch, CURLOPT_HTTPHEADER, [
                        'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/58.0.3029.110 Safari/537.3',
                    ]);
                    curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
                    $htmlStr = curl_exec($ch);
                    curl_close($ch);

                    $crawlInfo = @$parseService->parseHTML($htmlStr, $matches[0]);
                }
            }

            if (!empty($crawlInfo['galleries']) && !empty($crawlInfo['image_url'])) {
                ProductGallery::query()->where('product_id', $product->id)->delete();
                ProductGallery::query()->insert(array_map(function ($gallery) use ($product) {
                    return [
                        'product_id' => $product->id,
                        'image_url' => $gallery,
                        'type' => 'PRODUCT',
                        'created_at' => date('Y-m-d H:i:s'),
                        'updated_at' => date('Y-m-d H:i:s')
                    ];
                }, $crawlInfo['galleries']));
                Product::query()->where('id', $product->id)->update([
                    'image_url' => $crawlInfo['image_url'],
                    'updated_at' => date('Y-m-d H:i:s')
                ]);
            } else {
                ProductMeta::updateOrCreate([
                    'product_id' => $product->id,
                    'key' => 'recrawl_image'
                ], [
                    'value' => 'Empty photos!'
                ]);
            }
        }
        
        return response()->json([
            'status' => 'successful'
        ]);
    }

}
