<?php

namespace Modules\CrawlProduct\Controllers\Impl;

use DateTime;
use GuzzleHttp\Client;
use Modules\Pod\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Input;
use Illuminate\Support\Facades\Redis;
use Modules\CrawlProduct\Models\Brand;
use Modules\CrawlProduct\Models\Product;
use Modules\CrawlProduct\Models\ProductSku;
use Modules\CrawlProduct\Models\ProductMeta;
use Modules\CrawlProduct\Models\ProductNUser;
use Modules\CrawlProduct\Models\ProductGallery;
use Modules\CrawlProduct\Models\ProductVariant;
use Modules\CrawlProduct\Models\ProductSkuValue;
use Modules\CrawlProduct\Models\ProductNCategory;
use Modules\CrawlProduct\Models\ProductVariantOption;
use Modules\CrawlProduct\Controllers\Impl\SkuGenerator;
use Modules\CrawlProduct\Controllers\Impl\ImageDownloader;
use Modules\CrawlProduct\Controllers\Impl\TrademarkChecker;
use Modules\CrawlProduct\Controllers\Impl\VariantGenerator;


class ProductCreation {
    protected $imageDownloader;
    protected $skuGenerator;
    protected $variantGenerator;
    protected $trademarkChecker;
    
    public function __construct(
        ImageDownloader $imageDownloader, 
        SkuGenerator $skuGenerator,
        VariantGenerator $variantGenerator,
        TrademarkChecker $trademarkChecker
    ) {
        $this->imageDownloader = $imageDownloader;
        $this->skuGenerator = $skuGenerator;
        $this->variantGenerator = $variantGenerator;
        $this->trademarkChecker = $trademarkChecker;
    }

    public function saveCrawlProduct($request, $data) {
        if ($request->input('categories') && $request->input('generate_variant')) {
            $result = $this->variantGenerator->buildVariants(explode(',', $request->input('categories')), $data);
            if (!empty($result)) {
                $data['variants'] = $result['variants'];
                $data['product_variants'] = $result['product_variants'];
            }
        }
        if ($request->input('name')) {
            $data['name'] = decoreText($request->input('name'));
        }
        if ($request->input('description')) {
            $data['description'] = $request->input('description');
        }
        if ($request->input('note')) {
            if (isset($data['note'])) {
                $data['note'] = $request->input('note') . " | " . $data['note'];
            } else {
                $data['note'] = $request->input('note');
            }
        }
        if ($request->input('brand_id')) {
            $data['brand_id'] = $request->input('brand_id');
        }
        if ($request->input('enable_style')) {
            $data['enable_style'] = $request->input('enable_style') ;
        }
        $minPrice = floatval($request->input('min_price', 0));
        $discount = floatval($request->input('discount', 0));
        $fastShipmentPrice = floatval($request->input('fast_shipment_price', 0));
        if ($discount || $minPrice || $fastShipmentPrice) {
            $data['price'] = $data['price'] - $discount;
            $data['price'] = round($data['price'], 2);
            if ($data['price'] < $minPrice) {
                $data['price'] = $minPrice;
            }
            if ($data['price'] > 0) {
                $data['status'] = 'ACTIVE';
            }
            if ($fastShipmentPrice && isset($data['is_fast_shipment']) && $data['is_fast_shipment']) {
                $data['price'] = round($fastShipmentPrice, 2);
            }
            foreach ($data['product_variants'] as &$variant) {
                $variant['price'] = $variant['price'] - $discount;
                $variant['price'] = round($variant['price'], 2);
                if ($variant['price'] < $minPrice) {
                    $variant['price'] = $minPrice;
                }
                if ($fastShipmentPrice && isset($data['is_fast_shipment']) && $data['is_fast_shipment']) {
                    $variant['price'] = round($fastShipmentPrice, 2);
                }
            }
        }
        $data['group_id'] = $request->input('group_id');
        if ($request->input('approve_advertising')) {
            $data['approve_advertising'] = 1;
        }
        $response = $this->storeProduct($data);
        if ($response['status'] == 'successful') {

            $product = $response['result'];
            $product->save();

            $tagIds = [];
            if (isset($data['ex_tags']) && count($data['ex_tags'])) {
                $tagIds = $this->makeTags($data['ex_tags']);
            }

            if ($request->input('tags')) {
                $tagIds = array_merge($tagIds, explode(',', $request->input('tags')));
            }

            if (count($tagIds)) {
                DB::table('tag_refer')
                    ->where('refer_id', '=', $product->id)
                    ->where('refer_type', '=', 'PRODUCT')
                    ->delete();

                $tagIds = array_unique($tagIds);
                $tagReferData = [];
                foreach ($tagIds as $tagId) {
                    $tagReferData[] = [
                        'tag_id' => $tagId,
                        'refer_id' => $product->id,
                        'refer_type' => 'PRODUCT'
                    ];
                }

                DB::table('tag_refer')->insert($tagReferData);
            }
            DB::table('product_n_category')->where('product_id', '=', $product->id)->delete();

            if ($request->input('categories')) {
                $categories = explode(',', $request->input('categories'));
                $pncData = [];
                foreach ($categories as $categoryId) {
                    $pncData[] = [
                        'category_id' => $categoryId,
                        'product_id' => $product->id,
                        'created_at' => new \DateTime(),
                        'updated_at' => new \DateTime()
                    ];
                }
                DB::table('product_n_category')->insert($pncData);
            }

            /** save to product_custom table */
            if (!empty($data['custom_design_type'])) {
                if (!DB::table('product_custom')->where('product_id', $product->id)->exists()) {
                    DB::table('product_custom')->insert([
                        'product_id' => $product->id,
                        'type' => $data['custom_design_type']
                    ]);
                }
            }
            if (!empty($data['custom_options'])) {
                $saveOptions = $data['custom_options'];
                foreach ($saveOptions as &$option) {
                    $option['product_id'] = $product->id;
                }
                DB::table('product_custom_option')->insert($saveOptions);
            }

            $locale = env('APP_LOCALE', 'us');
            if (empty($locale)) {
                $locale = 'us';
            }
            $configRemoveSizeChartImage = config('crawl-product::sa.remove_size_chart_image', []);
            if ($configRemoveSizeChartImage && !empty($configRemoveSizeChartImage['enable']) && !empty($configRemoveSizeChartImage['url'][$locale])) {
                $urlRequest = str_replace('[productId]', $product->id, $configRemoveSizeChartImage['url']);
                $urlRequest = str_replace('[locale]', $locale, $urlRequest);
                try {
                    $dataRequest = $this->asyncRequest($urlRequest);
                    Log::info('dataRequestRemoveSizeChartImage::' . $urlRequest . '::' . $dataRequest);
                } catch (\Exception $ex) {

                }
            }
        }

        $checkCategoryResult = [
            'message' => 'Khớp danh mục',
            'suggested_categories' => []
        ];
        $categorySuggestSelected = null;
        $categoryId = $request->input('categories');

        $response['result_check_category'] = $checkCategoryResult;
        $response['category_suggest_selected'] = $categorySuggestSelected;

        return $response;
    }

    protected function getCategorySuggestSelected(int $productId, int $categoryId, array $arrayCategories) {
        $arrayCategorySuggestId = array_column($arrayCategories, 'id');
        $id = null;

        $productNCategory = DB::table('product_n_category')->where('product_id', $productId)->where('is_parent', 0)->first();
        if ($productNCategory) {
            $id = $productNCategory->category_id;
        }

        if (!$id && count(array_column($arrayCategories, 'id')) > 0 && in_array($categoryId, $arrayCategorySuggestId)) {
            $id = $categoryId;
        }

        return $id;
    }

    public function getSuggestedCategoryData(array $data) {
        $retVal = [];

        if ($data['product_image'] && $data['product_image']) {
            $client = new Client();
            
            $url = "https://printerval.com/service/category/detect?image_url={$data['product_image']}&name={$data['product_name']}";
            $response = $client->get($url);
            
            $dataResponse = json_decode($response->getBody(), true);
            
            if (isset($dataResponse['status']) && isset($dataResponse['result']) && $dataResponse['status'] == 'successful') {
                $retVal = $dataResponse['result'];
            }
        }

        return $retVal;
    }


    protected function makeTags($titles)
    {
        $tagIds = [];
        foreach ($titles as $title) {
            $tagIds[] = $this->storeOrRetrieveTag($title);
        }
        return $tagIds;
    }

    protected function storeOrRetrieveTag($tagTitle)
    {
        $tagSlug = sluggify($tagTitle);
        if (DB::table('tag')->where('slug', $tagSlug)->exists()) {
            return DB::table('tag')->where('slug', $tagSlug)->first()->id;
        } else {
            return DB::table('tag')->insertGetId([
                'title' => $tagTitle,
                'slug' => $tagSlug,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s'),
            ]);
        }
    }

    public function storeProduct($data) {
        // \DB::beginTransaction();
        $product = null;
        try {
            $isDownloadImage = !isset($data['is_download_image']) || !empty($data['is_download_image']);
            $productData = [];
            $productKeys = ['name', 'slug', 'image_url', 'price', 'high_price', 'note', 'approve_advertising'];
            if (config('crawl-product::sa.store_description')) {
                $productKeys[] = 'content'; 
                $productKeys[] = 'description';
            }
            $product = null;
            $enableStyle = isset($data['enable_style']) ? $data['enable_style'] : null;

            foreach ($productKeys as $key) {
                if (isset($data[$key])) {
                    $productData[$key] = $data[$key];
                }
            }
            $productData['name'] = decoreText($productData['name']);
            if (empty($productData['slug'])) {
                $productData['slug'] = strToSlug($productData['name']);
            }
            if (strlen($productData['slug']) > 240) {
                $productData['slug'] = substr($productData['slug'], 0 , 240);
            }
            if (Input::get('product_id')) {
                $product = Product::where('id', '=', Input::get('product_id'))->first();
            } else if (!empty($data['crawl_code']) && !empty($data['site'])) {
                    $productIds = DB::table('product_crawl_code')
                        ->where('value', '=', $data['crawl_code'])
                        ->where('site',  $data['site'])
                        ->get(['product_id'])
                        ->pluck('product_id');
                if (!empty($productIds)) {
                    $product = Product::whereIn('id', $productIds)
                        ->whereNull('deleted_at')
                        ->first();
                }
            }
            if (!empty($productData['image_url']) && strpos($productData['image_url'], 'http') !== 0) {
                $productData['image_url'] = 'https://' . $productData['image_url'];
            }
            if (isset($productData['image_url'])) {
                $productData['image_url'] = $this->imageDownloader->downloadImage($productData['image_url']);
            }
            $productData['sku'] = $this->skuGenerator->generateSku();
            $productData['deleted_at'] = null;
            $brand = Brand::take(1)->first();
            if ($brand) {
                $productData['brand_id'] = $brand->id;
            }
            $productData['status'] = 'ACTIVE';
            if (empty($productData['price']) || hasPermission('crawl-product:pending-status')) {
                $productData['status'] = 'PENDING';
            }
            if (!empty($productData['price']) && !empty($data['force_status'])) {
                $productData['status'] = $data['force_status'];
            } else if (Input::get('is_overwrite_product') && isset($data['force_status'])) {
                $productData['status'] = $data['force_status'];
            }
            if (!empty($product->is_trademark)) {
                $productData['status'] = 'PENDING';
            }   
            $userId = null;
            if (Input::get('user_email')) {
                $user = \DB::table('users')->where('email', Input::get('user_email'))->first();
                if ($user) {
                    $userId = $user->id;
                }
            } else if (isset($data['user_id'])) {
                $userId = $data['user_id'];
            }
            if (!$userId) {
                $userId = Input::get('user_id') ? Input::get('user_id') : (!empty(Auth::user()->id) ? Auth::user()->id : 0);
                if (!$userId) {
                    $user = \DB::table('users')->where('email', config('sa.default_seller_email', 'info@megaads.vn'))->first();
                    if ($user) {
                        $userId = $user->id;
                    }
                }
            }

            $data['user_id'] = $userId;

            if ($product) {
                if (Input::get('is_overwrite_product')) {
                    if($product->note) {
                        $re = '/' . ucfirst($data['site']) . ' url: ([^\n\r ]+)/';
                        preg_match_all($re, $product->note, $matches, PREG_SET_ORDER, 0);
                        if ($matches) {
                            foreach ($matches as $item) {
                                $product->note = str_replace($item[0], '', $product->note);
                            }
                        }
                        $productData['note'] = $product->note . ' ' . $productData['note'];
                    }
                    $product->fill($productData);
                    $product->save();
                    $this->trademarkChecker->checkWarning($product);
                    $this->clearBeforeOverwrite($product, $data);
                } else {
                    return [
                        'status' => 'duplicate',
                        'result' => $product,
                    ];
                }

            } else {
                $productData['actor_id'] = $userId;
                $product = Product::create($productData);
                $this->trademarkChecker->checkWarning($product);
                
                $userEmail = '';
                if ($userId) {
                    $user = \DB::table('users')->where('id', '=', $userId)->first();
                    if ($user) {
                        $userEmail = $user->email;
                    }
                }
                \Modules\CrawlProduct\Models\Log::create([
                    'actor_email' => $userEmail,
                    'target_id' => $product->id,
                    'target_type' => 'PRODUCT',
                    'event_type' => 'CREATE',
                    'data' => json_encode($productData)
                ]);

            }
            $galleries = [];

            if (isset($data['galleries'])) {
                $galleries = $data['galleries'];
            }
            $this->storeProductMeta($product->id, $data);
            $this->storeProductGalleries($galleries, $product->id, $isDownloadImage);
            $this->storeProductVideos($data, $product->id, $isDownloadImage);
            $this->storeProductVariant($product, $data);

            // \DB::commit();
        } catch (\Exception $ex) {
            // \DB::rollback();
            return [
                'status' => 'fail',
                'message' => $ex->getMessage() . ' Line: ' . $ex->getLine() . ' File: ' . $ex->getFile(),
            ];
        }
        return [
            'status' => 'successful',
            'result' => $product,
        ];
    }

    public function clearBeforeOverwrite($product, $data) {
        ProductNCategory::where('product_id', $product->id)->delete();
        ProductSkuValue::where('product_id', $product->id)->delete();
        $skuIds = ProductSku::where('product_id', $product->id)->get(['id'])->pluck('id')->toArray();
        ProductSku::where('product_id', $product->id)->delete();
        ProductGallery::where('product_id', $product->id)->where('type', 'PRODUCT')->delete();
        ProductGallery::whereIn('product_id', $skuIds)->where('type', 'VARIANT')->delete();
        if (Schema::hasTable('product_video')) {
            DB::table('product_video')->where('product_id', $product->id)->delete();
        }
        ProductMeta::where('product_id', $product->id)->delete();
        ProductNUser::where('product_id', $product->id)->delete();
        DB::table('product_n_template')->where('product_id', $product->id)->delete();
        DB::table('product_custom')->where('product_id', $product->id)->delete();
        DB::table('product_custom_option')->where('product_id', $product->id)->delete();
    }

    public function storeVariant($data) {
        $variants = [];
        $variantOptions = [];
        if (isset($data['variants'])) {
            foreach ($data['variants'] as $variantData) {
                $options = $variantData['options'];
                unset($variantData['options']);
                if (!isset( $variantData['slug'])) {
                    $variantData['slug'] = strToSlug($variantData['name']);
                }
                $variant = ProductVariant::where('slug', $variantData['slug'])->first();
                if (!$variant) {
                    $variant = ProductVariant::create($variantData);
                }
                $variants[$variant->slug] = $variant;

                foreach ($options as $optionData) {
                    if (!isset( $optionData['slug'])) {
                        $optionData['slug'] = strToSlug($optionData['name']);
                    }
                    $option = ProductVariantOption::where('variant_id', $variant->id)
                        ->where('name', $optionData['name'])
                        ->where('slug', $optionData['slug'])
                        ->first();
                    if (!$option) {
                        $option = ProductVariantOption::where('variant_id', $variant->id)
                        ->where('slug', $optionData['slug'])
                        ->first();
                    }
                    if (!$option) {
                        $optionData['variant_id'] = $variant->id;
                        if (!empty($optionData['image_url']) && strpos($optionData['image_url'], 'http') !== 0) {
                            $optionData['image_url'] = 'https://' . $optionData['image_url'];
                        }
                        if (isset($optionData['image_url'])) {
                            if (!isset($data['is_download_image']) || !empty($data['is_download_image'])) {
                                $optionData['image_url'] = $this->imageDownloader->downloadImage($optionData['image_url']);
                            }
                        }
                        $option = ProductVariantOption::create($optionData);
                        if($option) {
                            $optionCode = $this->skuGenerator->generateOptionCode($option);
                            $option->code = $optionCode;
                        }
                    }

                    $variantOptions[$variant->slug . '+_+_+' . $option->slug] = $option;
                    $variantOptions[$option->slug] = $option;
                }

            }

        }

        return [
            'variants' => $variants,
            'variantOptions' => $variantOptions,
        ];
    }

    public function storeProductVariant($product, $data) {
        $storeVariantResult = $this->storeVariant($data);
        $variants = $storeVariantResult['variants'];
        $variantOptions = $storeVariantResult['variantOptions'];

        if (isset($data['product_variants'])) {
            $hasDefault = false;
            foreach ($data['product_variants'] as $key => $skuData) {
                if (!empty($skuData['is_default'])) {
                    $hasDefault = true;
                    break;
                }
            }
            if (!$hasDefault) {
                $minPrice = 999999999;
                $minPriceIndex = -1;
                foreach ($data['product_variants'] as $key => $item) {
                    if ($item['price'] < $minPrice) {
                        $minPrice = $item['price'];
                        $minPriceIndex = $key;
                    }
                }
                if ($minPriceIndex > 0) {
                    $data['product_variants'][$minPriceIndex]['is_default'] = 1;
                    $hasDefault = true;
                }              
            }
            $isDownloadImage = !isset($data['is_download_image']) || !empty($data['is_download_image']);
            $createSkuData = [];
            $optionsBySku = [];
            $galleriesBySku = [];
            $enableStyle = isset($data['enable_style']) ? $data['enable_style'] : null;

            foreach ($data['product_variants'] as $key => $skuData) {
                $galleries = $skuData['galleries'];
                $skuVariants = $skuData['variants'];
                $styleIsOk = true;
                foreach ($skuVariants as &$skuVariant) {
                    $optionSlug = '';
                    if (isset($skuVariant['variant_slug'])) {
                        $optionSlug = $skuVariant['variant_slug'] . '+_+_+' . $skuVariant['slug'];
                    }
                    if (isset($variantOptions[$optionSlug])) {
                        $skuVariant['code'] = $variantOptions[$optionSlug]->code;
                        $skuVariant['variant_option_id'] = $variantOptions[$optionSlug]->id;
                    } else if (isset($variantOptions[$skuVariant['slug']])) {
                        $skuVariant['code'] = $variantOptions[$skuVariant['slug']]->code;
                        $skuVariant['variant_option_id'] = $variantOptions[$skuVariant['slug']]->id;
                    }
                    if (
                        isset($skuVariant['variant_slug'])
                        && $skuVariant['variant_slug'] == 'style'
                        && $enableStyle
                        && !in_array($skuVariant['variant_option_id'], $enableStyle)
                    ) {
                        $styleIsOk = false;
                    }
                }
                if (!$styleIsOk) {
                    continue;
                }
                unset($skuData['galleries']);
                unset($skuData['variants']);
                if (!empty($skuData['image_url']) && strpos($skuData['image_url'], 'http') !== 0) {
                    $skuData['image_url'] = 'https://' . $skuData['image_url'];
                }
                if (isset($skuData['image_url']) && $isDownloadImage) {
                    $skuData['image_url'] = $this->imageDownloader->downloadImage($skuData['image_url']);
                }
                if (array_key_exists('image_url', $skuData) && $skuData['image_url'] === null) {
                    $skuData['image_url'] = '';
                }
                $skuData['product_id'] = $product->id;
                if (!isset($skuData['is_default'])) {
                    $skuData['is_default'] = 0;
                }
                if (!$hasDefault) {
                    $skuData['is_default'] = $key == 0;
                }
                $skuData['sku'] = $this->initSkuCode($product->id, $skuVariants);
                $skuData['created_at'] = new \DateTime();
                if (!empty($skuData['is_default'])) {
                    $product->price = $skuData['price'];
                    if (!hasPermission('crawl-product:pending-status')) {
                        if (!$product->price && !Input::get('force_status')) {
                            $product->status = 'PENDING';
                        } elseif (DB::table('product')->where('id', $product->id)->where('is_trademark', 0)->exists()) {
                            $product->status = 'ACTIVE';
                        }
                    }
                    $product->save();
                }

                $createSkuData[] = $skuData;
                $optionsBySku[$skuData['sku']] = $skuVariants;
                $galleriesBySku[$skuData['sku']] = $galleries;
            }

            foreach (array_chunk($createSkuData, 50) as $partData) {
                ProductSku::insert($partData);
            }

            $skuIdbySku = ProductSku::where('product_id', $product->id)->get(['id', 'sku'])->pluck('id', 'sku')->toArray();

            $createProductSkuValue = [];
            $createSkuGalleries = [];

            foreach ($skuIdbySku as $sku => $skuId) {
                $skuVariants = $optionsBySku[$sku];
                foreach ($skuVariants as $item) {
                    $slug = $item['slug'];
                    if (isset($item['variant_slug'])) {
                        $slug = $item['variant_slug'] . '+_+_+' . $item['slug'];
                    }
                    if (isset($variantOptions[$slug])) {
                        $skuOption = $variantOptions[$slug];
                        $createProductSkuValue[] = [
                            'sku_id' => $skuId,
                            'product_id' => $product->id,
                            'variant_id' => $skuOption->variant_id,
                            'variant_option_id' => $skuOption->id,
                            'created_at' => date('Y-m-d H:i:s', time()),
                            'updated_at' => date('Y-m-d H:i:s', time()),
                        ];
                    }
                }

                $galleries = $galleriesBySku[$sku];
                foreach ($galleries as $imageUrl) {
                    $createSkuGalleries[] = [
                        'image_url' => $imageUrl,
                        'product_id' => $skuId
                    ];
                }
            }
            $this->storeSkuGalleries($createSkuGalleries, $isDownloadImage);
            if ($createProductSkuValue) {
                foreach (array_chunk($createProductSkuValue, 500) as $partData) {
                    ProductSkuValue::insert($partData);
                }
            }
        }
    }

    public function storeProductMeta($productId, $data) {
        $productMetas = [];

        if (isset($data['metas']) && count($data['metas'])) {
            foreach ($data['metas'] as &$value) {
                if (!isset($value['product_id'])) {
                    $value['product_id'] = $productId;
                    $value['created_at'] = new DateTime();
                    $value['updated_at'] = new DateTime();
                }
            }
            $productMetas = $data['metas'];
        }

        if (!empty($data['store_link'])) {
            $productMetas[] = [
                'key' => 'store_link',
                'value' => $data['store_link'],
                'product_id' => $productId,
                'created_at' => new DateTime(),
                'updated_at' => new DateTime(), 
            ];
        }
        
        if (!empty($data['crawl_code']) && !empty($data['site'])) {
            \DB::table('product_crawl_code')->insert([
                'product_id' => $productId,
                'value' =>  $data['crawl_code'],
                'site' => $data['site'],
                'created_at' => new DateTime(),
                'updated_at' => new DateTime(), 
            ]);
        }
        if (!empty($data['design_crawl_code']) && !empty($data['site'])) {
            \DB::table('product_design_crawl_code')->insert([
                'product_id' => $productId,
                'value' => $data['site'] . '-' . $data['design_crawl_code']
            ]);
        }
        if (count($productMetas)) {
            ProductMeta::insert($productMetas);
        }
        $sellerId = $this->getRandomSellerId();
        ProductNUser::insert([[
            'product_id' => $productId,
            'user_id' => $sellerId,
        ]]);
       
    }

    public function getRandomSellerId() {
        $limit = 500;
        $user = User::where('role', 'SELLER')->where('status', 'ACTIVE')->whereNull('seller_token')->orderBy('id', 'desc')->first();
        if ($user) {
            $count = DB::table('product_n_user')->where('user_id', $user->id)->limit($limit)->count();
            $userId = $user->id;
            if ($count >= $limit) {
                $user = User::where('role', 'SELLER')->where('status', 'PENDING')->where('product_count', 0)->whereNull('seller_token')->orderBy('id', 'desc')->first();
                if ($user) {
                    User::where('id', $user->id)->update([
                        'status' => 'ACTIVE'
                    ]);
                    $userId = $user->id;
                } else {
                    $user = User::where('role', 'SELLER')->where('status', 'ACTIVE')->whereNull('seller_token')->inRandomOrder()->first();
                    if ($user) {
                        return $user->id;
                    }
                }
            }

            return $userId;
        }

        return 1;
    }

    public function storeProductVideos($data, $productId, $isDownloadImage) {
        if (!empty($data['videos'])) {
            foreach ($data['videos'] as $video) {
                $video['product_id'] = $productId;
                $video['created_at'] = date('Y-m-d H:i:s', time());
                $video['updated_at'] = date('Y-m-d H:i:s', time());
                if ($isDownloadImage) {
                    $url = $this->imageDownloader->downloadImage($video['image_url']);
                    if ($url) {
                        $video['image_url'] = $url;
                    }
                }
                DB::table('product_video')
                    ->insert($video);
            }
        }
    }

    public function storeProductGalleries($galleries, $productId, $isDownloadImage) {
        $data = [];
        foreach ($galleries as $url) {
            if ($url && strpos($url, 'http') !== 0) {
                $url = 'https://' . $url;
            }
            if ($isDownloadImage) {
                $url = $this->imageDownloader->downloadImage($url);
            }
            $data[] = [
                'product_id' => $productId,
                'type' => 'PRODUCT',
                'image_url' => $url,
                'created_at' => date('Y-m-d H:i:s', time()),
                'updated_at' => date('Y-m-d H:i:s', time())
            ];
        }
        if ($data) {
            ProductGallery::insert($data);
        }
    }

    public function storeSkuGalleries($galleries, $isDownloadImage) {
        $data = [];
        foreach ($galleries as $item) {
            if ($item['image_url'] && strpos($item['image_url'], 'http') !== 0) {
                $item['image_url'] = 'https://' . $item['image_url'];
            }
            if ($isDownloadImage) {
                $item['image_url'] = $this->imageDownloader->downloadImage($item['image_url']);
            }
            $data[] = [
                'product_id' => $item['product_id'],
                'type' => 'VARIANT',
                'image_url' => $item['image_url'],
                'created_at' => date('Y-m-d H:i:s', time()),
                'updated_at' => date('Y-m-d H:i:s', time())
            ];
        }
        foreach (array_chunk($data, 500) as $partData) {
            ProductGallery::insert($partData);
        }
    }

    public function initSkuCode($productId, $options, $limit = 30)
    {
        $skuCode = ['P' . $productId];
        $check = true;
        // P{PRODUCT_ID}-{COLOR}-{SIZE}-{OTHER_ATTR}(2 ADD ATTR)
        foreach ($options as $option) {
            $skuCode[] = $option['code'];
        }
        $microtime = floor(microtime(true) * 1000);
        $index = str_random(3) . substr($microtime, -2) . str_random(3); // 2 số cuối của microtime
        $subLen = $limit - strlen($index) - 1;
        return substr(implode('-', $skuCode), 0, $subLen) . '-' . $index;
    }

    public function codeGenerator($type = null, $length = 7) {
        if ($type == null)
            throw new InvalidArgumentException('missing argument type');
        $retVal = '';
        $type = strtoupper($type);
        $key = $type;
        if (!Redis::exists($key)) {
            Redis::set($key, 0);
        }
        $nextId = Redis::incr($key);
        $retVal .= str_pad($nextId, $length, '0', STR_PAD_LEFT);
        // append prefix code
        if ($type === 'ORDER') {
            $locale = env("APP_LOCALE");
            if ($locale == "") {
                $locale = "us";
            }
            $retVal  = strtoupper($locale) . $retVal;
        }
        return $retVal;
    }

    public function asyncRequest($url, $method = "GET", $params = [], $headers = [], $timeout = 120) {
        $channel = curl_init();
        curl_setopt($channel, CURLOPT_URL, $url);
        // curl_setopt($channel, CURLOPT_NOSIGNAL, 1);
        curl_setopt($channel, CURLOPT_TIMEOUT, 120);
        curl_setopt($channel, CURLOPT_RETURNTRANSFER, 1);
        if($method == "post" || $method == "POST") {
            curl_setopt($channel, CURLOPT_POST, true);
            curl_setopt($channel, CURLOPT_POSTFIELDS, json_encode($params));
        }
        if ($headers) {
            curl_setopt($channel, CURLOPT_HTTPHEADER, $headers);
        }
        curl_setopt($channel, CURLOPT_SSL_VERIFYHOST, 0);
        curl_setopt($channel, CURLOPT_SSL_VERIFYPEER, 0);
        $data = curl_exec($channel);
        curl_close($channel);
        return json_decode($data, true);
    }
}