<?php

namespace Modules\CrawlProduct\Controllers\Impl;

use DB;
use Modules\CrawlProduct\Models\ProductSku;
use Modules\CrawlProduct\Models\ProductVariant;
use Modules\CrawlProduct\Models\ProductVariantOption;

class VariantGenerator {
    public function getProductVariants($productId = null) {
        $skus = ProductSku::where('product_id', $productId)->with(['skuValues', 'galleries'])->get();
        $productVariantIds = [];
        $productVariantOptionIds = [];
        $productVariantTmps = [];
        foreach($skus as $sku) {
            if (isset($sku->skuValues) && count($sku->skuValues) > 0) {
                foreach($sku->skuValues as $value) {
                    if (!in_array($value->variant_id, $productVariantIds)) {
                        $productVariantIds[] = $value->variant_id;
                    }
                    if (!in_array($value->variant_option_id, $productVariantOptionIds)) {
                        $productVariantOptionIds[] = $value->variant_option_id;
                    }
                    if (!array_key_exists($value->variant_id, $productVariantTmps)) {
                        $productVariantTmps[$value->variant_id] = [];
                    }
                    if (!in_array($value->variant_option_id, $productVariantTmps[$value->variant_id])) {
                        $productVariantTmps[$value->variant_id][] = $value->variant_option_id;
                    }
                }
            }
        }
        $productVariantByIds = [];
        if (count($productVariantIds) > 0) {
            $productVariants = ProductVariant::whereIn('id', $productVariantIds)->get();
            foreach($productVariants as $item) {
                $productVariantByIds[$item->id] = $item;
            }
        }
        $productVariantOptionByIds = [];
        if (count($productVariantOptionIds) > 0) {
            $productVariantOptions = ProductVariantOption::whereIn('id', $productVariantOptionIds)->get();
            foreach($productVariantOptions as $item) {
                $productVariantOptionByIds[$item->id] = $item;
            }
        }

        $variants = [];
        foreach ($productVariantTmps as $key => $values) {
            if (isset($productVariantByIds[$key])) {
                $variant = [
                    'id' => $key,
                    'slug' => $productVariantByIds[$key]->slug,
                    'name' => $productVariantByIds[$key]->name,
                    'type' => $productVariantByIds[$key]->type,
                    'options' => []
                ];
                foreach($values as $item) {
                    if (isset($productVariantOptionByIds[$item])) {
                        $variant['options'][$productVariantOptionByIds[$item]->id] = [
                            'id' => $productVariantOptionByIds[$item]->id,
                            'name' => $productVariantOptionByIds[$item]->name,
                            'slug' => $productVariantOptionByIds[$item]->slug,
                            'image_url' => $productVariantOptionByIds[$item]->image_url,
                            'variant' => $productVariantByIds[$key]->name,
                        ];
                    }
                }
                $variants[$variant['slug']] = $variant;
            }
        }

        $productVariants = [];
        foreach($skus as $sku) {
            $skuImg = $sku->image_url ? $sku->image_url : '';
            $productVariant = [
                'id' => $sku->id,
                'sku' => $sku->sku,
                'barcode' => $sku->barcode,
                'image_url' => $skuImg,
                'price' => $sku->price,
                'high_price' => $sku->high_price,
                'is_default' => $sku->is_default,
                'product_id' => $productId,
                'inventory' => $sku->inventory,
                'status' => $sku->status,
                'galleries' => [],
                'variants' => []
            ];
            if (isset($sku->skuValues) && count($sku->skuValues) > 0) {
                foreach($sku->skuValues as $item) {
                    if (isset($productVariantOptionByIds[$item->variant_option_id])) {
                        $variant = [
                            'id' => $productVariantOptionByIds[$item->variant_option_id]->id,
                            'name' => $productVariantOptionByIds[$item->variant_option_id]->name,
                            'code' => $productVariantOptionByIds[$item->variant_option_id]->code,
                            'slug' => $productVariantOptionByIds[$item->variant_option_id]->slug
                        ];
                        if (isset($productVariantByIds[$item->variant_id])) {
                            $variant['variant_id'] = $item->variant_id;
                            $variant['variant'] = $productVariantByIds[$item->variant_id]->name;
                            $variant['variant_slug'] = $productVariantByIds[$item->variant_id]->slug;
                            $variant['variant_type'] = $productVariantByIds[$item->variant_id]->type;
                            $variant['image_url'] = $productVariantOptionByIds[$item->variant_option_id]->image_url;
                        }
                        $productVariant['variants'][] = $variant;
                        $galleries = $sku->galleries->pluck('image_url')->toArray();
                        if (count($galleries)) {
                            $arrImage = [];
                            foreach($galleries as $item) {
                                if ($item) {
                                    $arrImage[] = $item;
                                }
                            }
                            $productVariant['galleries'] = $arrImage;
                        }
                    }
                }
            }
            $productVariants[] = $productVariant;
        }

        return [
            'variants' => $variants,
            'product_variants' => $productVariants,
        ];
    }

    public function sortVariants($variants, $keepAll = false) {
        $retVal = [];
        $slugs = ['type', 'style', 'color', 'hand', 'size', 'mug-size', 'material-type', 'model'];
        foreach ($slugs as $slug) {
            foreach($variants as $variant) {
                if (
                    (isset($variant['variant_slug']) && $variant['variant_slug'] == $slug) ||
                    (isset($variant['slug']) && $variant['slug'] == $slug)
                ) {
                    $retVal[] = $variant;
                    break;
                }
            }
        }

        if ($keepAll) {
            foreach ($variants as $variant) {
                if (
                    (isset($variant['variant_slug']) && !in_array($variant['variant_slug'], $slugs)) &&
                    (isset($variant['slug']) && !in_array($variant['slug'], $slugs))
                ) {
                    $retVal[] = $variant;
                }
            }
        }

        return $retVal;
    }

    public function variantBySlug($variants) {
        $retVal = [];

        foreach ($variants as $variant) {
            $retVal[$variant['slug']] = $variant;
        }

        return $retVal;
    }

    private function combinations($arrays, $i = 0) {
        if (!isset($arrays[$i])) {
            return array();
        }
        if ($i == count($arrays) - 1) {
            return $arrays[$i];
        }

        // get combinations from subsequent arrays
        $tmp = $this->combinations($arrays, $i + 1);

        $result = array();

        // concat each array from tmp with each element from $arrays[$i]
        foreach ($arrays[$i] as $v) {
            foreach ($tmp as $t) {
                $result[] = is_array($t) ?
                    array_merge(array($v), $t) :
                    array($v, $t);
            }
        }

        return $result;
    }

    private function combinationOneArray($array) {
        // initialize by adding the empty set
        $results = array(array( ));
    
        foreach ($array as $element) {
            foreach ($results as $combination) {
                array_push($results, array_merge($combination, array($element)));
            }
        }

        array_shift($results);
    
        return $results;
    }

    public function buildProductVariants($attributes, $data, $ignoreVariants) {
        $variantsKey = [];
        $variants = [];
        foreach ($attributes as $attribute) {
            if ($attribute['options']) {
                $data['variants'][] = $attribute;
                $variantsKey[] = array_keys($attribute['options']);
                $variants[] = $attribute;
            }
        }
        $productVariants = [];
        $combinations = $this->combinations($variantsKey);
        foreach ($combinations as $listKey) {
            if (!is_array($listKey)) {
                $listKey = [$listKey];
            }
            $keyImage = [];
            $arrVariants = [];
            $variantSlugs = [];
            $variantPrice = $data['price'];
            foreach ($listKey as $key => $value) {
                $variant = $variants[$key]['options'][$value];
                if (!empty($variant['key_image'])) {
                    $keyImage[] = $variant['name'];
                }
                $arrVariants[] = [
                    'variant_slug' => $variants[$key]['slug'],
                    'slug' => $variant['slug'],
                ];
                if (!in_array($variants[$key]['slug'], $ignoreVariants)) {
                    $variantSlugs[] = $variant['slug'];
                }
                if (!empty($variant['price'])) {
                    $variantPrice = $variant['price'];
                }
            }
            $galleries = [];
            
            $itemVariant = [
                'image_url' => array_shift($galleries),
                'galleries' => $galleries,
                'price' => $variantPrice,
                'high_price' => 0,
                'variants' => $arrVariants
            ];
            if (!$itemVariant['image_url']) {
                $itemVariant['image_url'] = $data['image_url'];
            }
            if (!isset($productVariants[implode('_', $variantSlugs)])) {
                $productVariants[implode('_', $variantSlugs)] = [];
            }
            $productVariants[implode('_', $variantSlugs)][] = $itemVariant;
        }
        return $productVariants;
    }

    public function getDefaultProductId($categories) {
        $locale = env('APP_LOCALE', 'us');
        $locale = $locale ? $locale : 'us';
        $templateId = null;
        if (count($categories)) {
            $config = \DB::table('crawl_product_default_product')->whereIn('category_id', $categories)->first();
            if ($config) {
                $templateId = $config->product_id;
            }
            if (!$templateId) {
                foreach ($categories as $categoryId) {
                    $id = config('crawl-product::default.product_template.' . $locale . '.' . $categoryId);
                    if ($id) {
                        $templateId = $id;
                        break;
                    }
                }
            }
        }

        return $templateId;
    }

    public function buildVariants($categories, $data) {
        $locale = env('APP_LOCALE', 'us');
        $locale = $locale ? $locale : 'us';
        $defaultProductId = $this->getDefaultProductId($categories);
        if ($defaultProductId) {
            $result = $this->getProductVariants($defaultProductId);
            $templateVariants = $result['variants'];
            $templateProductVariants = $result['product_variants'];
            return $this->buildVariantsFromTemplate($templateVariants, $templateProductVariants, $data);
        }
        return null;
    }

    public function getProductVariantByKey($productVariants, $ignoreVariants, $getAll = false) {
        $productVariantsByKey = [];
        foreach ($productVariants as $productVariant) {
            $optionSlugs = [];
            if ($getAll) {
                $allSlugs = [];
            }
            $productVariant['variants'] = $this->sortVariants($productVariant['variants']);
            foreach ($productVariant['variants'] as $option) {
                if ($getAll) {
                    $allSlugs[] = $option['slug'];
                }
                if (!in_array($option['variant_slug'], $ignoreVariants)) {
                    $optionSlugs[] = $option['slug'];
                }
            }
            if ($getAll) {
                $productVariantsByKey[implode('_', array_unique($allSlugs))] = $productVariant;
            }
            $variantKey = implode('_', array_unique($optionSlugs));
            if (
                !isset($productVariantsByKey[$variantKey]) || 
                (
                    isset($productVariantsByKey[$variantKey]['is_default']) && 
                    isset($productVariant['is_default']) && 
                    $productVariantsByKey[$variantKey]['is_default'] < $productVariant['is_default']
                )
            ) {
                $productVariantsByKey[$variantKey] = $productVariant;
            }
        }

        return $productVariantsByKey;
    }

    public function buildVariantOptionByVariantSlug($variants, $ignoreVariants) {
        $retVal = [];
        foreach ($variants as $variant) {
            if (
                (!isset($variants[$variant['slug']])) ||
                in_array($variant['slug'], $ignoreVariants)
            ) {
                $variants[$variant['slug']] = $variant;
            }

            if (!isset($retVal[$variant['slug']])) {
                $retVal[$variant['slug']] = [];
            }
            foreach ($variant['options'] as $option) {
                $retVal[$variant['slug']][$option['slug']] = 1;
            }
        }

        return $retVal;
    }

    public function buildVariantKeys($variants) {
        $variantKeys = [];
        foreach ($variants as $variant) {
            $variantKeys[] = array_map(function ($item) use ($variant) {
                $obj = new \stdClass();
                $obj->slug = $item['slug'];
                $obj->variant_slug = $variant['slug'];

                return $obj;
            }, $variant['options']);
        }

        return $variantKeys;
    }

    public function mergeVariantOptions($templateVariants, $variants, $mergeSlug) {
        $templateIndex = null;
        $variantIndex = null;
        foreach ($templateVariants as $key => $variant) {
            if ($variant['slug'] == $mergeSlug) {
                $templateIndex = $key;
                break;
            }
        }

        foreach ($variants as $key => $variant) {
            if ($variant['slug'] == $mergeSlug) {
                $variantIndex = $key;
                break;
            }
        }

        if ($templateIndex && $variantIndex) {
            $templateVariants[$templateIndex]['options'] = $variants[$variantIndex]['options'];
        }

        return $templateVariants;
    }

    public function mergeAllVariantOptions($templateVariants, $variants, $ignoreVariants) {
        foreach ($templateVariants as $tKey => $tVariant) {
            foreach ($variants as $key => $variant) {
                if ($tVariant['slug'] == $variant['slug'] && !in_array($variant['slug'], $ignoreVariants)) {
                    $templateVariants[$tKey]['options'] = $variants[$key]['options'];
                    break;
                }
            }
        }

        return $templateVariants;
    }

    public function buildVariantsFromTemplate($templateVariants, $templateProductVariants, $data) {
        $templateIgnoreVariants = ['color'];
        $ignoreVariants = ['size'];
        $templateVariants = $this->sortVariants($templateVariants, true);

        $variants = $templateVariants;
        $variants = $this->variantBySlug($variants);
        $variants = $this->sortVariants($variants, true);

        $variantKeys = $this->buildVariantKeys($variants);
        if (count($variantKeys) == 1) {
            $variantKeys = [[], $variantKeys];
        }
        $combinations = $this->combinations($variantKeys);

        $productVariants = [];
        foreach ($templateProductVariants as $item) {
            unset($item['id']);
            unset($item['sku']);
            unset($item['product_id']);
            $item['galleries'] = [];
            $item['image_url'] = '';
            $itemVariants = $this->sortVariants($item['variants'], true);
            $keys = array_map(function ($option) {
                return $option['slug'];
            }, $itemVariants);
            $productVariants[implode('_', $keys)] = $item;
        }
        $templateProductVariantByKey = $this->getProductVariantByKey($templateProductVariants, $templateIgnoreVariants, true);
        $productVariantByKey = $this->getProductVariantByKey($data['product_variants'], $ignoreVariants);

        foreach ($combinations as $combination) {
            $slugs = [];
            $validSlugs = [];
            $allSlugs = [];
            foreach ($combination as $item) {
                $allSlugs[] = $item->slug;
                if (!in_array($item->variant_slug, $templateIgnoreVariants)) {
                    $slugs[] = $item->slug;
                }
                if (!in_array($item->variant_slug, $ignoreVariants)) {
                    $validSlugs[] = $item->slug;
                }
            }
            $productVariant = [];
            $key = implode('_', $slugs);
            $allKey = implode('_', $allSlugs);
            if (isset($templateProductVariantByKey[$allKey])) {
                $productVariant = $templateProductVariantByKey[$allKey];
            } else if (isset($templateProductVariantByKey[$key])) {
                $productVariant = $templateProductVariantByKey[$key];
            } 

            if (isset($productVariant['price'])) {
                $productVariant['galleries'] = [];
                $productVariant['image_url'] = '';

                $combinationOneArray = $this->combinationOneArray($validSlugs);

                $combinationOneArray = array_reverse($combinationOneArray);

                foreach ($combinationOneArray as $values) {
                    $vKey = implode('_', $values);
                    if (isset($productVariantByKey[$vKey])) {
                        $productVariant['image_url'] = $productVariantByKey[$vKey]['image_url']; 
                        $productVariant['galleries'] = $productVariantByKey[$vKey]['galleries']; 
                        break;
                    }
                }

                $productVariant['variants'] = json_decode(json_encode($combination), true);
                if ($productVariant['image_url'] && count($productVariant['variants'])) {
                    if (isset($productVariants[implode('_', $allSlugs)])) {
                        $productVariants[implode('_', $allSlugs)]['image_url'] = $productVariant['image_url'];
                        $productVariants[implode('_', $allSlugs)]['galleries'] = $productVariant['galleries'];
                    }
                }
            }
        }

        return [
            'variants' => $variants,
            'product_variants' => array_values($productVariants),
        ];
    }   
}