<?php

namespace App\Modules\CrawlProduct\Controllers\Services;

use Modules\CrawlProduct\Controllers\Impl\ImageDownloader;

include_once '../app/Modules/CrawlProduct/Helpers/simple_html_dom.php';

class DobaService extends BaseService
{
    private $imageDownloader;

    public function __construct($input) {
        parent::__construct($input);
        $this->imageDownloader = new ImageDownloader();
    }

    public function parseHTML($htmlStr, $url) {
        ini_set('memory_limit', '2048M');
        $html = str_get_html($htmlStr);

        $priceEle = $html->find('.price-cnt.price-current', 0);
        $price = 0;
        if ($priceEle) {
            $price = preg_replace("/[^\d\.]+/", '', $priceEle->plaintext);
        }

        $model = $html->find('.ant-modal.variantsDia', 0);
        $variants = $this->getVariants($html);
        if ($model) {
            $productVariants = $this->getProductVariantsUsingModel($model);
        } else {
            $productVariants = $this->getProductVariants($variants, $price);
        }

        $crawlCode = $this->getCrawlCode($html, $url);

        $name = $html->find('h1', 0);
        $gallery = $this->getGallery($html);
        $result = [
            'name'  => $this->replaceDomain($name->plaintext),
            'slug' => $this->toFriendlyString($this->replaceDomain($name->plaintext)),
            'description'  => $this->getDescription($html),
            'content'  => '',
            'image_url'  => array_shift($gallery),
            'galleries'  => $gallery,
            'price'  => $price,
            'high_price'  => '',
            'note'  => 'Doba url: ' . $url,
            'crawl_code'  => $crawlCode,
            'site'  => 'doba',
            'comments'  => '',
            'meta_title'  => $name->plaintext,
            'meta_description'  => $this->getProductMeta($html, 'meta[name="description"]'),
            'meta_keywords'  => $this->getProductMeta($html, 'meta[name="keywords"]'),
            'is_fast_shipment' => '',
            'related_products'  => '',
            'variants' => $variants,
            'product_variants' => $productVariants,
            'tags' => [],
            'store_link' => $this->getStoreLink($html)
        ];

        if (!$this->getGallery) {
            $result['galleries'] = [];
        }

        return $result;
    }

    public function getDescription($html) {
        $description = '';
        $descriptionEle = $html->find('.detail-container', 0);
        if ($descriptionEle) {
            $description = $descriptionEle->outertext;
        }

        $description = $this->replaceDescriptionClass($description);

        if (config('crawl-product::sa.is_download_image', true)) {
            $description = $this->downloadDescriptionImage($description);
        }

        return $description;
    }

    public function getGallery($html) {
        $retVal = [];
        $items = $html->find('.thumb-list .thumb');

        foreach ($items as $item) {
            $img = $item->find('img', 0);
            if ($img) {
                $retVal[] = $this->getImageUrl($img->src);
            }
        }

        return $retVal;
    }

    public function getVariants($html) {
        $variants = [];
        $variantEles = $html->find('.sku-attrs .attr-sku-item.has-sku-item');
        if (count($variantEles)) {
            foreach ($variantEles as $variantEle) {
                $name = $variantEle->find('.attr-sku-name', 0);
                $variant = [
                    'name' => trim($name->plaintext),
                    'slug' => $this->toFriendlyString(trim($name->plaintext)),
                    'type' => 'RADIOBOX',
                    'options' => []
                ];
    
                $optionEles = $variantEle->find('.attr-sku-inner .sku-item');
                foreach ($optionEles as $optionEle) {
                    $nameEle = $optionEle->find('a[role="button"]', 0);
                    $option = [
                        'name' => trim($nameEle->plaintext),
                        'slug' => $this->toFriendlyString(trim($nameEle->plaintext)),
                        'image_url' => null,
                        'variant_slug' => $variant['slug']
                    ];
                    $variant['options'][] = $option;
                }
                $variants[] = $variant;
            }
        }

        return $variants;
    }

    public function getProductVariantsUsingModel($model) {
        $th = $model->find('.table-tr.table-th', 0);
        $fieldMap = [
            'Image' => 'image_url',
            'Price' => 'price',
            'MSRP' => 'high_price'
        ];
        $fields = [];
        $productVariants = [];
        $variantIndexs = [];
        if ($th) {
            $cells = $th->find('.table-td');
            if (count($cells)) {
                $index = 0;
                foreach ($cells as $cell) {
                    $name = $cell->plaintext;
                    if (isset($fieldMap[$name])) {
                        $name = $fieldMap[$name];    
                        $fields[$index] = $name;
                    }
                    if (strpos($cell->class, 'variant-prop') !== false) {
                        $variantIndexs[$index] = $this->toFriendlyString(trim($name));
                    }
                    $index++;
                }
            }
        }
        $rows = $model->find('.table-tr');
        array_shift($rows);
        foreach ($rows as $row) {
            $productVariant = [
                'image_url' => '',
                'price' => 0,
                'high_price' => 0,
                'variants' => [],
                'galleries' => []
            ];
            $cells = $row->find('.table-td');
            if (count($cells)) {
                $index = 0;
                foreach ($cells as $cell) {
                    if(isset($fields[$index])) {
                        if ($fields[$index] == 'image_url') {
                            $img = $cell->find('img', 0);
                            if ($img) {
                                $productVariant['image_url'] = $this->getImageUrl($img->src);
                            }
                        } else {
                            $value = trim(str_replace('US$ ', '', $cell->plaintext));
                            $productVariant[$fields[$index]] = $value;
                        }
                    } else if (isset($variantIndexs[$index])) {
                        $option = [
                            'name' => trim($cell->plaintext),
                            'slug' => $this->toFriendlyString(trim($cell->plaintext)),
                            'image_url' => $variantIndexs[$index] == 'color' ? $productVariant['image_url'] : null,
                            'variant_slug' => $variantIndexs[$index]
                        ];

                        $productVariant['variants'][] = $option;
                    }
                    $index++;
                }
            }
            if (!empty($productVariant['variants'])) {
                $productVariants[] = $productVariant;
            }
        }

        return $productVariants;
    }

    public function getProductVariants($variants, $price) {
        $retVal = [];

        $variantOptionSlugs = [];
        $optionBySlugs = [];
        foreach ($variants as $variant) {
            $slugs = [];
            foreach ($variant['options'] as $option) {
                $slugs[] = $option['slug'];
                $optionBySlugs[$option['slug']] = $option;
            }
            $variantOptionSlugs[] = $slugs;
        }

        $combinations = $this->combinations($variantOptionSlugs);
        foreach ($combinations as $listSlug) {
            if (!is_array($listSlug)) {
                $listSlug = [$listSlug];
            }
            $itemVariant = [
                'image_url' => null,
                'galleries' => [],
                'price' => $price,
                'high_price' => 0,
                'variants' => []
            ];
            $options = [];
            foreach ($listSlug as $slug) {
                $option = $optionBySlugs[$slug];
                if (!empty($option['image_url'])) {
                    $itemVariant['image_url'] = $option['image_url'];
                }
                $options[] = $option;
            }

            $itemVariant['variants'] = $options;

            if (count($options)) {
                $retVal[] = $itemVariant;
            }
        }

        return $retVal;
    }

    public function replaceDescriptionClass($descriptionHtml) {
        $html = str_get_html($descriptionHtml);
        $elements = $html->find('*');
        foreach ($elements as $element) {
            if ($element->getAttribute('class')) {
                $classNames = explode(' ', $element->getAttribute('class'));
                foreach ($classNames as &$className) {
                    $className = 'product_description_prefix_' . $className;
                }
                $element->setAttribute('class', implode(' ', $classNames));
            }
        }

        return $html->save();
    }

    public function downloadDescriptionImage($descriptionHtml) {
        $html = str_get_html($descriptionHtml);
        $elements = $html->find('img');
        foreach ($elements as &$element) {
            if ($element->{'data-src'}) {
                $imageUrl = $this->getImageUrl($element->{'data-src'});
                $imageUrl = $this->imageDownloader->downloadImage($imageUrl);
                if ($imageUrl) {
                    $element->{'data-src'} = $imageUrl;
                }
            } else if ($element->src) {
                $imageUrl = $this->getImageUrl($element->src);
                $imageUrl = $this->imageDownloader->downloadImage($imageUrl);
                if ($imageUrl) {
                    $element->src = $imageUrl;
                }
            }
        }

        return $html->save();
    }

    public function getImageUrl($src) {
        $retVal = $src;
        if (strpos($retVal, '//') === 0) {
            $retVal = 'https:' . $retVal;
        } else if (strpos($retVal, 'http') === false) {
            $retVal = 'https://' . $retVal;
        }

        return $retVal;
    }

    private function getProductMeta ($html, $element) {
        $retVal = '';
        $metaTitleElements = $html->find($element, 0);
        if ($metaTitleElements) {
            $retVal = $metaTitleElements->attr['content'];
            $retVal = $this->replaceDomain($retVal);
            $retVal = trim($retVal);
        }
        return $retVal;
    }

    private function replaceDomain ($content) {
        $content = str_ireplace('product on doba.com', '', $content);
        $content = str_ireplace('doba.com', '', $content);
        $retVal = str_ireplace('doba', '', $content);
        return trim($retVal);
    }

    public function getCrawlCode($html, $url) {
        $crawlCode = '';
        $code = $html->find('.prod-info-text', 0);
        if ($code) {
            $crawlCode = trim(str_replace('Item No.: ', '', $code->plaintext));
        }

        return $crawlCode;
    }

    public function getStoreLink($html) {
        $retVal = '';
        $companyEle = $html->find('.prod-corp-name', 0);
        if ($companyEle) {
            $href = $companyEle->href;
            if (strpos($href, '?') !== false) {
                $href = substr($href, 0, strpos($href, '?'));
            }
            $retVal = $href;
        }

        return $retVal;
    }
}