<?php

namespace App\Modules\CrawlProduct\Controllers\Services;
use Imagick;
use Modules\CrawlProduct\Models\Product;
use Modules\CrawlProduct\Models\ProductMeta;
use Modules\CrawlProduct\Models\ProductVariantOption;
use App\Modules\CrawlProduct\Controllers\Services\BaseService;

include_once '../app/Modules/CrawlProduct/Helpers/simple_html_dom.php';

class TeePublicService extends BaseService
{
    public $config = [];
    static $cache = [];
    public function parseHTML($htmlStr, $url) {
        ini_set('memory_limit', '2048M');
        $this->config = $this->getConfig();
        $html = str_get_html($htmlStr);
        $regex = "/TeePublic\['ProductOptions'\] = (.*);/m";

        preg_match($regex, $htmlStr, $matches);
        // $designOptions = [];
        // $canvasOptions = [];
        if ($matches && count($matches) > 1) {
            $productOptions = json_decode($matches[1], true);
            // $designOptions = $productOptions['DesignOptions'];
            // $canvasOptions = $productOptions['CanvasOptions'];
        }
        $nameEle = $html->find('h1', 0);
        if (!$nameEle) {
            return [
                'name' => ''
            ];
        }
        $name = trim($nameEle->title);
        $galleries = $this->getGalleries($html);
        $imageUrl = array_shift($galleries);
        if (!$this->getGallery) {
            $galleries = [];
        }
        $crawlCode = str_replace('/', '', parse_url($url)['path']);
        preg_match("/([a-z\-]+)([a-z\-\d]+)/", $crawlCode, $matches);
        $designCrawlCode = '';
        if (count($matches) > 2) {
            $designCrawlCode = $matches[2];
        }
        $data = [
            'name' => $name,
            'description' => $this->getProductMeta($html, 'meta[property="og:description"]'),
            'content' => '',
            'image_url' => $imageUrl,
            'galleries' => $galleries,
            'price' => $this->getProductMeta($html, 'meta[property="og:price:amount"]'),
            'high_price' => 0,
            'note' => 'Teepublic url: ' . $url,
            'crawl_code' => str_replace('/', '-', parse_url($url)['path']),
            'site' => 'teepublic',
            'comments' => [],
            'meta_title' => $this->getProductMeta($html, 'meta[property="og:title"]'),
            'meta_description' => $this->getProductMeta($html, 'meta[property="og:description"]'),
            'meta_keywords' => '',
            'is_fast_shipment' => 0,
            'related_products' => [],
            'tags' => $this->getTags($html),
            'design_crawl_code' => $designCrawlCode,
            'store_link' => $this->getStoreLink($html),
            'variants' => [],
            'product_variants' => []
        ];

        return $data;
    }

    public function getGalleries($html) {
        $retVal = [];
        $elements = $html->find('.glide__slides .glide__slide');
        foreach ($elements as $ele) {
            $img = $ele->find('img', 0);
            if ($img) {
                $src = str_replace(' ', '%20', $img->src);
                if (strpos($src,'http') === 0) {
                    $retVal[] = $src;
                }
            }
        }

        return $retVal;
    }

    public function getProductVariants($designOptions, $canvasOptions) {
        $variants = [];
        $productVariants = [];
        if (isset($canvasOptions['hierarchy'])) {
            foreach ($canvasOptions['hierarchy'] as $item) {
                $variant = [
                    'name' => $item['name'],
                    'slug' => $item['slug'],
                    'options' => []
                ];
                $name = $this->translateName($variant['slug'], $variant['name']);
                if ($variant['name'] != $name) {
                    $variant['name'] = $name;
                    $variant['slug'] = $this->toFriendlyString(trim($name));
                }

                $variants[$variant['slug']] = $variant;

            }
        }

        $productGalleries = [];
        if (isset($designOptions['images'])) {
            foreach ($designOptions['images'] as $image) {
                $imageUrls = [];
                foreach ($image['images'] as $img) {
                    $src = str_replace(' ', '%20', $img['url']);
                    if (strpos($src,'http') === 0) {
                        $imageUrls[] = $src;
                    }
                }
                foreach ($image['product_ids'] as $id) {
                    $productGalleries[$id] = $imageUrls;
                }
            }
        }

        if (isset($canvasOptions['products'])) {
            $locale = env('APP_LOCALE') ? env('APP_LOCALE') : 'us';
            $ignoreOptionSlugs = config('crawl-product::crawl-' . $locale . '.ignore_option_slugs', []);
            foreach ($canvasOptions['products'] as $item) {
                if (isset($productGalleries[$item['product_id']])) {
                    $galleries = $productGalleries[$item['product_id']];
                    $options = [];
                    foreach ($item['attributes'] as $variantSlug => $attr) {
                        $variantName = $this->translateName($variantSlug, $variantSlug);
                        if ($variantName != $variantSlug) {
                            $variantSlug = $this->toFriendlyString($variantName);
                        }
                        $name = $this->translateName($attr['name'], $attr['name']);;
                        
                        $slug = $this->toFriendlyString(trim($name));

                        if (in_array($slug, $ignoreOptionSlugs)) {
                            continue;
                        }

                        $name = $this->translateName($variantSlug . '.' . $slug, $name);
                        $slug = $this->toFriendlyString(trim($name));
                        $option = [
                            'name' => $name,
                            'slug' => $slug,
                            'variant_slug' => $variantSlug,
                        ];
                        $options[] = $option;

                        $variants[$variantSlug]['options'][$option['slug']] = $option;
                    }
                    $imageUrl = array_shift($galleries);
                    if (!$this->getGallery) {
                        $galleries = [];
                    }
                    $productVariant = [
                        'image_url' => $imageUrl,
                        'galleries' => $galleries,
                        'price' => $item['sale_price'],
                        'high_price' => $item['retail_price'],
                        'variants' => $options
                    ];
                    $productVariants[] = $productVariant;
                }
            }
        }

        $this->removeKeys($variants);

        return [
            'variants' => $variants,
            'product_variants' => $productVariants
        ];
    }

    public function translateName($key, $name) {
        $locale = env('APP_LOCALE', 'us');
        if (!$locale) {
            $locale = 'us';
        }

        $value = config('crawl-product::crawl-' . $locale . '.' . $key);
        if ($value) {
            if (!is_array($value)) {
                return $value;
            }

            return $name;
        }
        $config = $this->getConfigTranslate();
        if (!empty($config)) {
            $slugs = explode('.', $key);
            if (count($slugs) == 2 && isset($config[$slugs[0]]) && isset($config[$slugs[0]][$slugs[1]]) && $config[$slugs[0]][$slugs[1]]) {
                $name = $config[$slugs[0]][$slugs[1]];
            }
        }

        return $name;
    }

    protected function getConfigTranslate() {
        $key = 'crawl-product::translate-name';
        if (array_key_exists($key, self::$cache)) {
            return self::$cache[$key];
        }
        $option = \DB::table('option')->where('key', $key)->first();
        $result = null;
        if ($option) {
            $result = json_decode($option->value, true);
        }
        self::$cache[$key] = $result;
        return $result;

    }

    private function getProductMeta ($html, $element) {
        $retVal = '';
        $metaTitleElements = $html->find($element, 0);
        if ($metaTitleElements) {
            $retVal = $metaTitleElements->attr['content'];
            $retVal = $this->replaceDomain($retVal);
            $retVal = trim($retVal);
        }
        return $retVal;
    }

    private function replaceDomain ($content) {
        return str_replace('Teepublic', 'Printerval', $content);
    }

    private function buildColors(&$variants, $canvasOptions) {
        $colors = [];
        foreach ($canvasOptions['colors'] as $color) {
            $slug = $this->toFriendlyString(trim($color['name']));
            $name = $this->translateName('color' . '.' . $slug, $color['name']);
            $slug = $this->toFriendlyString(trim($name));
            $colors[$slug] = $color['hex'];
        }
        $colorImg = [];
        foreach ($colors as $slug => $hex) {
            $option = ProductVariantOption::where('variant_id', 2)->where('slug', $slug)->first();
            if ($option && $option->image_url) {
                $colorImg[$slug] = $option->image_url;
            } else {
                $imagick = new Imagick();
                $imagick->newImage(900, 900, new \ImagickPixel($hex));
                $newFilename = uniqid() . '-' . time();
                $path = $this->getPath('variant');
                $filePath = $path . '/' . $newFilename . '.jpg';
                $imagick->setImageCompression(Imagick::COMPRESSION_JPEG);
                $imagick->setImageCompressionQuality(85);
                $imagick->writeImage(public_path($filePath));
                $this->optimizeImage(public_path($filePath));
                $colorImg[$slug] = config('sa.file_manager_url') . $filePath;
            }
        }

        foreach ($variants as $slug => &$variant) {
            if ($slug == 'color') {
                foreach ($variant['options'] as $optionSlug => &$option) {
                    if (isset($colorImg[$optionSlug])) {
                        $option['image_url'] = $colorImg[$optionSlug];
                    }
                }
            }
        }
    }

    public function optimizeImage($pathToImage) {
        try {
            exec("jpegoptim --all-progressive --strip-com --strip-xmp --strip-exif --strip-iptc " . $pathToImage);
        } catch (\Exception $ex) {
            \Log::info('optimizeImage', [$ex->getMessage() . ' File: ' . $ex->getFile() . ' Line: ' . $ex->getLine()]);
        }
    }

    private function removeKeys(&$variants) {
        $variants = array_values($variants);
        foreach ($variants as &$variant) {
            $variant['options'] = array_values($variant['options']);
        }
    }

    public function getPath($type)
    {
        $locale = '';
        if (env('APP_LOCALE') && config('app.upload_by_locale', false)) {
            $locale = env('APP_LOCALE');
            if ($locale == '' || empty($locale)) {
                $locale = '/us';
            } else {
                $locale = '/' . $locale;
            }
        } else if (!env('APP_LOCALE') && config('app.upload_by_locale', false)) {
            $locale = '/us';
        }
        $date = date('d-m-Y', time());
        $basePath = "/files/$locale/$type/$date";
        $path = public_path() . $basePath;
        if (!file_exists($path)) {
            mkdir($path, 0777, true);
        }
        return $basePath;
    }

    public function getTags($html) {
        $retVal = [];
        $elements = $html->find('.m-design__additional-info .m-design__additional-info-list .link.c-link__pill');
        if ($elements && count($elements)) {
            foreach ($elements as $element) {
                $retVal[] = strtolower(trim($element->plaintext));
            }
        }
        return array_unique($retVal);
    }

    public function deleteProduct($input) {
        if (!empty($input['design_ids'])) {
            return $this->deleteBySourceIds($input['design_ids']);
        }
    }

    private function deleteBySourceIds($designIds) {
        $result = [];
        foreach ($designIds as $designId) {
            $productIds = DB::table('product_design_crawl_code')
                ->where('value', 'like', "%teepublic-$designId-%")
                ->get(['product_id'])
                ->pluck('product_id')
                ->toArray();
            if (!empty($productIds)) {
                $result = array_merge($result, $productIds);
                Product::whereIn('id', $productIds)->update([
                    'status' => 'PENDING'
                ]);
            }
        }

        return $result;
    }

    public function getStoreLink($html) {
        $retVal = '';

        $element = $html->find('.m-design__by-name', 0);
        if ($element) {
            $retVal = trim($element->href);
            if ($retVal && $retVal[0] == '/') {
                $retVal = 'https://www.teepublic.com' . $retVal;
            }
        }

        return $retVal;
    }
}
