<?php

namespace Modules\CrawlProduct\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\CrawlProduct\Models\Product;
use Modules\CrawlProduct\Models\ProductSku;
use Modules\CrawlProduct\Models\ProductGallery;
use Modules\CrawlProduct\Controllers\Controller;
use Modules\CrawlProduct\Models\ProductSkuValue;
use Modules\CrawlProduct\Models\ProductNCategory;
use Modules\CrawlProduct\Controllers\Impl\ProductCreation;
use Modules\CrawlProduct\Controllers\Impl\VariantGenerator;

class VariantGeneratorController extends Controller {
    protected $variantGenerator;
    protected $productCreation;

    public function __construct(VariantGenerator $variantGenerator, ProductCreation $productCreation) {
        $this->variantGenerator = $variantGenerator;
        $this->productCreation = $productCreation;
    }

    public function generateVariantByCategory(Request $request) {
        set_time_limit(10 * 3600);
        $retVal = [];
        $locale = env('APP_LOCALE', 'us');
        $locale = $locale ? $locale : 'us';
        if ($request->get('categories', '')) {
            $categories = explode(',', $request->get('categories', ''));
        } else {
            $categories = array_keys(config('crawl-product::default.product_template.' . $locale));
        }
        foreach ($categories as $categoryId) {
            $templateId = config('crawl-product::default.product_template.' . $locale . '.' . $categoryId);
            $result = $this->variantGenerator->getProductVariants($templateId);
            $templateVariants = $result['variants'];
            $templateProductVariants = $result['product_variants'];
            $variantIds = [];
            foreach ($templateVariants as $variant) {
                $variantIds[] = $variant['id'];
            }
            $query = ProductNCategory::where('category_id', $categoryId);
            if ($request->has('product_ids')) {
                $query->whereIn('product_id', explode(',', $request->get('product_ids')));
            }
            $productIds = $query->get([DB::raw('distinct product_id')])->pluck('product_id');
            $productIds = ProductSkuValue::select(DB::raw('sku_id, product_id, count(*)'))
                ->whereIn('variant_id', $variantIds)
                ->whereIn('product_id', $productIds)
                ->groupBy('sku_id')
                ->groupBy('product_id')
                ->having(DB::raw('count(*)'), '<', count($variantIds))
                ->orderBy('product_id')
                ->get()
                ->pluck('product_id')
                ->toArray();
            $productIds = array_unique($productIds);

            foreach($productIds as $productId) {
                $product = Product::where('id', $productId)
                    ->select(['id', 'image_url', 'price', 'high_price'])
                    ->whereNull('deleted_at')
                    ->first();
                if ($product) {
                    $data = $this->variantGenerator->getProductVariants($productId);
                    $data['image_url'] = $product->image_url;
                    $data['price'] = $product->price;
                    $data['high_price'] = $product->high_price;
                    $data['is_download_image'] = false;
                    
                    $result = $this->variantGenerator->buildVariantsFromTemplate($templateVariants, $templateProductVariants, $data);
                    if (!empty($result)) {
                        $data['product_variants'] = $result['product_variants'];
                        $data['variants'] = $result['variants'];
                        ProductSkuValue::where('product_id', $product->id)->delete();
                        $skuIds = ProductSku::where('product_id', $product->id)->get(['id'])->pluck('id')->toArray();
                        ProductSku::where('product_id', $product->id)->delete();
                        ProductGallery::whereIn('product_id', $skuIds)->where('type', 'VARIANT')->delete();
                        $this->productCreation->storeProductVariant($product, $data);
                    }
                    $retVal[] = $product->id;
                } 

            }
        }

        return $retVal;
    }

    public function generateProductVariant(Request $request) {
        $retVal = [
            'status' => 'successful',
            'message' => ''
        ];
        if (!$request->get('categories') || !$request->get('product_id')) {
            return [
                'status' => 'fail',
                'message' => 'categories + product_id is required'
            ];
        }
        $categoryIds = explode(',', $request->get('categories'));
        $productId = $request->get('product_id');
        $product = Product::where('id', $productId)
            ->select(['id', 'image_url', 'price', 'high_price'])
            ->whereNull('deleted_at')
            ->first();
        $data = $this->variantGenerator->getProductVariants($productId);
        $data['image_url'] = $product->image_url;
        $data['price'] = $product->price;
        $data['high_price'] = $product->high_price;
        $data['is_download_image'] = false;
        $defaultProductId = $this->getDefaultProductId($categoryIds);
        if ($defaultProductId) {
            $result = $this->variantGenerator->getProductVariants($defaultProductId);
            $templateVariants = $result['variants'];
            $templateProductVariants = $result['product_variants'];
            $result = $this->variantGenerator->buildVariantsFromTemplate($templateVariants, $templateProductVariants, $data);
            if (!empty($result)) {
                $data['product_variants'] = $result['product_variants'];
                $data['variants'] = $result['variants'];
                ProductSkuValue::where('product_id', $product->id)->delete();
                $skuIds = ProductSku::where('product_id', $product->id)->get(['id'])->pluck('id')->toArray();
                ProductSku::where('product_id', $product->id)->delete();
                ProductGallery::whereIn('product_id', $skuIds)->where('type', 'VARIANT')->delete();
                $this->productCreation->storeProductVariant($product, $data);
            } else {
                return [
                    'status' => 'fail',
                    'message' => 'buildVariantsFromTemplate is empty'
                ];
            }
        } else {
            return [
                'status' => 'fail',
                'message' => 'defaultProductId not found'
            ];
        }
        
        return $retVal;
    }

    public function getDefaultProductId($categoryIds) {
        $childCategory = DB::table('category')
            ->whereIn('id', $categoryIds)
            ->whereNotNull('parent_id')
            ->orderBy('_lft', 'desc')
            ->first();
        $defaultProductId = $this->variantGenerator->getDefaultProductId([$childCategory->id]);

        $podTemplates = DB::table('pod_product_template')->where('status', 'ACTIVE')->get(['product_id', 'product_template_id']);
        $categoryIdsByProductId = [];
        foreach ($podTemplates as $item) {
            $productId = $item->product_id;
            if ($item->product_template_id) {
                $template = DB::table('product_template')->where('id', $item->product_template_id)->first();
                if ($template) {
                    $productId = $template->product_id_fake;
                }
            }
            $cateIds = DB::table('product_n_category')->where('product_id', $productId)->where('is_parent', 0)->get(['category_id'])->pluck('category_id');
            $templateChildCategory = DB::table('category')
                ->whereIn('id', $cateIds)
                ->whereNotNull('parent_id')
                ->orderBy('_lft', 'desc')
                ->first();
            if ($templateChildCategory) {
                $categoryIdsByProductId[$productId] = $templateChildCategory->id;
            }
        }
        foreach ($categoryIdsByProductId as $productId => $cateId) {
            if ($childCategory->id == $cateId) {
                return $productId;
            }
        }

        return $defaultProductId;
    }
}