<?php

namespace Modules\CrawlProduct\Models;

use Illuminate\Database\Eloquent\Model;

class ProductSku extends Model
{
    protected $table = 'product_sku';

    protected $fillable = [
        'sku',
        'barcode',
        'price',
        'high_price',
        'image_url',
        'product_id',
        'is_default',
        'status',
        'inventory'
    ];
    protected $appends = ['product_name'];

    protected static function boot() {
        parent::boot();
        
        static::deleted(function($productSku) {
            $productSku->galleries()->delete();
        });
    }

    public function skuValues()
    {
        return $this->hasMany(ProductSkuValue::class, 'sku_id', 'id');
    }

    public function getProductNameAttribute()
    {
        $name = '';
        $productVariantName = [];
        if ($this->product_id && $this->id) {
            $productOrigin = Product::find($this->product_id);
            $name = $productOrigin->name;
            $skuValues = ProductSkuValue::where('sku_id', '=', $this->id)->get();
            $listProductSku = ProductSkuValue::where('product_id', $this->product_id)->get();

            $groupProductsSku = [];
            foreach ($listProductSku as $sku) {
                $groupProductsSku[$sku->sku_id][] = $sku;
            }

            $variantOptionIds = [];
            foreach ($skuValues as $skuValue) {
                if ($skuValue->variant_id && $skuValue->variant_option_id) {
                    $variant = ProductVariant::find($skuValue->variant_id);
                    $variantOption = ProductVariantOption::find($skuValue->variant_option_id);
                    
                    $variantOptionIds[] = $variantOption->id;
                    
                    if ($variant->slug == 'size') {
                        $attrName = $variant->name . ': ' . $variantOption->name;
                    } else {
                        $attrName = $variantOption->name;
                    }
                    
                    $productVariantName[] = [
                        "variant" => $variant,
                        "variantOption" => $variantOption,
                        "name" => $attrName
                    ];
                }
            }

            foreach ($productVariantName as $groupVariantName) {
                $cpVariantOptionIds = $variantOptionIds;
                $pos = array_search($groupVariantName['variantOption']['id'], $cpVariantOptionIds);
                unset($cpVariantOptionIds[$pos]);

                $foundSku = array_filter($groupProductsSku, function($skus) use($cpVariantOptionIds) {
                    $matchCount = 0;
                    foreach ($skus as $sku) {
                        if(in_array($sku->variant_option_id, $cpVariantOptionIds))
                            $matchCount ++;
                    }
    
                    if($matchCount == count($cpVariantOptionIds)) {
                        return true;
                    }
    
                });

                if(count($foundSku) > 1) {
                    $name .= ', ' . $groupVariantName['name'];
                }
            }
        }

        return $name;
    }

    public function getSalePercentAttribute () {
        if ($this->high_price && $this->price && $this->high_price > 0) {
            $salePercent = round(($this->high_price - $this->price) / $this->high_price * 100);
            if ($salePercent > 0) {
                return "-$salePercent%";
            }
        }

        return null;
    }

    public function getDisplayDropPriceAttribute () {
        return formatPrice($this->high_price - $this->price);
    }

    public function galleries()
    {
        return $this->hasMany(ProductGallery::class, 'product_id', 'id')->where('type', 'VARIANT');
    }

    public function getSkuNameAttribute() {
        $result = null;
        $skuValues = ProductSkuValue::where('sku_id', $this->id)->get(['variant_id', 'variant_option_id']);
        if ($skuValues) {
            $tmp = [];
            foreach ($skuValues as $skuValue) {
                $variant = ProductVariant::find($skuValue->variant_id, ['name']);
                $variantOption = ProductVariantOption::find($skuValue->variant_option_id, ['name']);
                if ($variant && $variantOption) {
                    $tmp[] = $variant->name . ': ' . $variantOption->name;
                }
            }
            $result = implode(', ', $tmp);
        }
        return $result;
    }

    public function getSpecificInfoAttribute() {
        $result = [];
        $skuValues = ProductSkuValue::where('sku_id', $this->id)->get(['variant_id', 'variant_option_id']);
        if ($skuValues) {
            foreach ($skuValues as $skuValue) {
                $variant = ProductVariant::find($skuValue->variant_id, ['name']);
                $variantOption = ProductVariantOption::find($skuValue->variant_option_id, ['name']);
                if ($variant && $variantOption) {
                    $result[$variant->name] =  $variantOption->name;
                }
            }
        }
        return $result;
    }
}
