const Utils = require(__dir + '/src/helper/utils');
const TaskQueue = require(__dir + '/src/helper/queue');
const Service = require(__dir + '/src/helper/service');
const etsyParser = require(__dir + '/src/parser/etsy');
const CrawlEtsyData = require(__dir + '/src/models/crawl_product_etsy');
const CrawlEtsyUrl = require(__dir + '/src/models/crawl_product_etsy_url');
const CrawlEtsyUrlObj = new CrawlEtsyUrl();
const CrawlEtsyDataObj = new CrawlEtsyData();
const EtsyParser = new etsyParser();
const maxDepth = 15;
const enableHostName = [
    'www.etsy.com',
    'etsy.com',
];
const UrlLog = new Set();
class Etsy {

    constructor() {
        this.handleProductQueue = new TaskQueue(2, 'handle etsy');
        this.spiderQueue = new TaskQueue(2, 'spider product');
        this.insertProductUrlQueue = new TaskQueue(5, 'queue etsy insert');
        this.category = null;
    }

    async spider(url, nesting, history , category = null) {
        let response = await Service.sendRequestEtsy(url);
        if (!this.validateSpider(url, nesting, history)) {
            this.log('validateSpider ', url);
            return Promise.resolve();
        }
        history.add(url);
        if (response) {
            if (this.hasProduct(url)) {
                await CrawlEtsyDataObj.InsertOrUpdate({
                    url: url,
                    category: category,
                    data: response
                });
            } else {
                let parseData = EtsyParser.parse(url, response);
                if (parseData && parseData.links) {
                    for (let itemUrl of parseData.links) {
                        itemUrl = Utils.buildUrl(itemUrl);
                        this.pushInsertUrl(itemUrl);
                        // if (this.hasProduct(itemUrl)) {
                        //     this.pushSpider(itemUrl, 1, history, category);
                        // }
                    }
                }
            }
        } else {
            this.log('response is empty ', url);
        }
    }

    async crawlProducts(from, to) {
        let items = await CrawlEtsyUrlObj.getById(from, to);
        if (items && items.length > 0) {
            for (let item of items) {
                this.pushProduct(item);
            }
        }
    }

    log() {
        Utils.log(...arguments);
    }

    hasProduct(url) {
        return url.includes('/listing/');
    }

    validateSpider(url, nesting, history) {
        let result = true;
        let hostname = Utils.getHostnameFromUrl(url);
        if (!url
            || nesting > maxDepth
            || history.has(url)
            || !enableHostName.includes(hostname)) {
            result = false;
        }
        return result;
    }

    pushInsertUrl(url, category = '', fromUrl = '') {
        let task = () => {
            let item = {
                url: url
            }
            if (fromUrl) {
                item.from_url = fromUrl
            }
            if (category) {
                item.category = category
            }
            return CrawlEtsyUrlObj.InsertOrUpdate(item).catch(function (e) {
                console.log('err pushInsertUrl:' + url, e);
                return Promise.resolve();
            });
        };

        this.insertProductUrlQueue.pushTask(task);
    }

    pushSpider(link, nesting, history, category) {
        let self = this;
        let task = () => {
            return self.spider(link, nesting, history, category).catch(function (e) {
                console.log('err request:' + link, e);
                return Promise.resolve();
            });
        };
        self.spiderQueue.pushTask(task);
    }

    async crawlProduct(item) {
        let response = await Service.sendRequestEtsy(item.url);
        let isCrawl = 2;
        if (response) {
            await CrawlEtsyDataObj.InsertOrUpdate({
                url: item.url,
                category: item.category,
                data: response
            });
            isCrawl = 1;

        }
        await CrawlEtsyUrlObj.updateById({
            id: item.id,
            is_crawl: isCrawl
        });
    }

    pushProduct(item) {
        let self = this;
        let task = () => {
            return self.crawlProduct(item).catch(function (e) {
                CrawlEtsyUrlObj.updateById({
                    id: item.id,
                    is_crawl: 2
                });
                console.log('err pushProduct request:' + item.url, e);
                return Promise.resolve();
            });
        };
        self.handleProductQueue.pushTask(task);
    }

    async crawlCategories() {
        let categories = this.getCategories();
        const history = new Set();
        for (let cate of categories) {
            this.crawlCategory(cate, history)
        }
    }

    crawlCategory(cate, history) {
        for (let i = 1; i <= cate.max_page; i++) {
            let url = cate.url + '&page=' + i;
            if (cate.url.includes('?')) {
                url = cate.url + '?page=' + i;
            }
            this.pushSpider(url, 1, history, cate.name)
        }
    }


    getCategories() {
        return [
            {
                url: 'https://www.etsy.com/search?explicit=1&q=poster&order=highest_reviews',
                name: 'posters',
                max_page: 250
            }, {
                url: 'https://www.etsy.com/search/art-and-collectibles/prints?q=Painting&ref=cat_hobby_119_0',
                name: 'posters',
                max_page: 250
            },
            {
                url: 'https://www.etsy.com/search/art-and-collectibles/prints?q=Traveling&ref=cat_hobby_119_1',
                name: 'posters',
                max_page: 250
            },  {
                url: 'https://www.etsy.com/search/art-and-collectibles/prints?q=traveling+poster&ref=s2qit&explicit=1&s2qii=2&s2qit=as&prq=Traveling',
                name: 'posters',
                max_page: 250
            },  {
                url: 'https://www.etsy.com/search/art-and-collectibles/prints?q=Dogs&ref=cat_hobby_119_2',
                name: 'posters',
                max_page: 250
            },  {
                url: 'https://www.etsy.com/search/art-and-collectibles/prints?q=Photography&ref=cat_hobby_119_3',
                name: 'posters',
                max_page: 250
            },  {
                url: 'https://www.etsy.com/search/art-and-collectibles/prints?q=photography+prints+women&ref=s2qit&explicit=1&s2qii=5&s2qit=as&prq=Photography',
                name: 'posters',
                max_page: 250
            }, {
                url: 'https://www.etsy.com/search/art-and-collectibles/prints?q=Cats&ref=cat_hobby_119_4',
                name: 'posters',
                max_page: 250
            }

        ];
    }


}


module.exports = Etsy;