<?php

namespace Modules\Dashboard\Controllers\Report;

use Cache;
use Illuminate\Http\Request;
use Modules\Ticket\Models\Ticket;
use Illuminate\Support\Facades\DB;
use Modules\Dashboard\Controllers\Controller;
use Modules\Dashboard\Models\DesignJob;

class StatisticController extends Controller 
{
    public function statisticAvg(Request $request) {
        set_time_limit(5 * 60);
        header('Access-Control-Allow-Origin: *', true);
        
        $cacheKey = decorCacheKey('Dashboard::StatisticAvg::V3::' . date('Y-m-d', time()));
        if (Cache::has($cacheKey) && !$request->has('clear_cache')) {
            return $retVal = [
                'status' => 'successful',
                'result' => Cache::get($cacheKey)
            ];
        }
        $retVal = [
            'status' => 'successful',
            'result' => [
                'order' => 0,
                'design' => 0,
                'ticket' => 0,
                'product' => 0,
                'product_template' => 0,
                'auto_fullfill' => 0
            ]
        ];
        $orderIdsByLocale = $this->getOrderIdsByLocale();
        $orderCount = 0;
        $autoFullfillCount = 0;
        $orderProcessTime = 0;
        foreach ($orderIdsByLocale as $locale => $orderIds) {
            $orderCount += count($orderIds);
            $autoFullfillCount += $this->getAutoFullfillCount($orderIds, $locale);
            $orderProcessTime += $this->getOrderProcessTime($orderIds, $locale);
        }
        $retVal['result']['auto_fullfill'] = $orderCount > 0 ? round($autoFullfillCount / $orderCount * 100, 2, PHP_ROUND_HALF_UP) : 0;
        $retVal['result']['order'] = $orderCount > 0 ? round($orderProcessTime / $orderCount, 0, PHP_ROUND_HALF_UP) : 0;
        $retVal['result']['design'] = $this->getDesignAvgTime();
        $retVal['result']['ticket'] = $this->getTicketAvgTime();
        $retVal['result']['product'] = $this->getProductGenCount();
        $retVal['result']['product_template'] = $this->getProductTemplateGenCount();

        Cache::put($cacheKey, $retVal['result'], 15);

        return $retVal;
    }

    private function getOrderProcessTime($orderIds, $locale) {
        if (class_exists(`\App\Helpers\DBConnect`)) {
            $connection = \App\Helpers\DBConnect::connect($locale);
        } else {
            $connection = DB::connection();
        }
        return $connection->table('order_status_duration')
            ->whereIn('order_id', $orderIds)
            ->whereIn('type', ['processing_time', 'fulfill_time'])
            ->sum('duration');
    }

    private function getTicketAvgTime() {
        return Ticket::where('status', 'close')
            ->where('closed_at', '>=', date('Y-m-d 00:00:00', time()))
            ->where('closed_at', '<=', date('Y-m-d 23:59:59', time()))
            ->where('is_spam', 0)
            ->avg(DB::raw('TIMESTAMPDIFF(SECOND, created_at, closed_at)'));
    }

    private function getDesignAvgTime() {
        return DesignJob::whereNotNull('finished_at')
            ->where('status', 'done')
            ->where('finished_at', '>=', date('Y-m-d 00:00:00', time()))
            ->where('finished_at', '<=', date('Y-m-d 23:59:59', time()))
            ->avg(DB::raw('TIMESTAMPDIFF(SECOND, assigned_at, finished_at)'));
    }

    private function getLocales() {
        return array_map(function ($item) {
            return $item['locale'];
        }, array_values(array_filter(getModuleLocale(), function($config) {
            return $config['enable'] && $config['locale'] != 'glob';
        })));

    }

    private function getProductGenCount() {
        $locales = $this->getLocales();
        $productCount = 0;
        foreach ($locales as $locale) {
            if (class_exists(`\App\Helpers\DBConnect`)) {
                $connection = \App\Helpers\DBConnect::connect($locale);
            } else {
                $connection = DB::connection();
            }
            $productCount += $connection->table('pod_generate_image_log')
                ->where('created_at', '>=', date('Y-m-d 00:00:00', time()))
                ->where('created_at', '<=', date('Y-m-d 23:59:59', time()))
                ->count();
        }

        return $productCount;
    }

    private function getProductTemplateGenCount() {
        $locales = $this->getLocales();
        $productCount = 0;
        foreach ($locales as $locale) {
            if (class_exists(`\App\Helpers\DBConnect`)) {
                $connection = \App\Helpers\DBConnect::connect($locale);
            } else {
                $connection = DB::connection();
            }

            $productCount += $connection->table('product_template_design_code')
                ->join('product', 'product_template_design_code.product_id', 'product.id')
                ->where('product.created_at', '>=', date('Y-m-d 00:00:00', time()))
                ->where('product.created_at', '<=', date('Y-m-d 23:59:59', time()))
                ->count();
        }

        return $productCount;
    }

    private function getOrderIdsByLocale() {
        $locales = $this->getLocales();
        $retVal = [];
        foreach ($locales as $locale) {
            if (class_exists(`\App\Helpers\DBConnect`)) {
                $connection = \App\Helpers\DBConnect::connect($locale);
            } else {
                $connection = DB::connection();
            }
            $orders = $connection->table('order')
                ->leftJoin('order_meta as om', function($leftJoin)
                {
                    $leftJoin->on('om.order_id', '=', 'order.id')
                        ->where('om.key', '=', 'remake_origin_code' );
                })
                ->where('order.payment_status', 'PAID')
                ->where('order.created_at', '>=', date('Y-m-d 00:00:00', time()))
                ->where('order.created_at', '<=', date('Y-m-d 23:59:59', time()))
                ->whereNull('om.value')
                ->whereNotIn('order.customer_id', config('dashboard::report-order.ignore_customers', [5031, 14529, 366023]))
                ->whereNull('order.parent_id')
                ->get(['order.id']);
            $retVal[$locale] = $orders->pluck('id')->toArray();
        }

        return $retVal;
    }

    private function getAutoFullfillCount($orderIds, $locale) {
        if (class_exists(`\App\Helpers\DBConnect`)) {
            $connection = \App\Helpers\DBConnect::connect($locale);
        } else {
            $connection = DB::connection();
        }
        return $connection->table('order_meta')
            ->where('key', 'is_auto_fulfill')
            ->where('value', 4)
            ->whereIn('order_id', $orderIds)
            ->count();
    }

  public function fetchCloudflareData(Request $request)
  {
    $response = [
      'status' => 'fail'
    ];
    $previousDay =$request->get('date', '-7 days');
    $dateReport = date('Y-m-d', strtotime($previousDay));
    $apiUser = env('CLF_API_USER', 'bachnx2303@gmail.com');
    $apiToken = env('CLF_API_TOKEN', '2f9fb4f1a0fa82f9f9d1bda1143148e6a4676');
    $zoneTags = env('CLF_API_ZONE', '57efed19098aa289ecf2ada753ebd894');
    $apiURL = env('CLF_API_URL', 'https://api.cloudflare.com/client/v4/graphql');
    if (function_exists('triggerSyncRequest')) {
      $headers = [
        "X-Auth-Email: {$apiUser}",
        "X-Auth-Key: {$apiToken}"
      ];
      $viewerQuery = "{\n";
      $viewerQuery .= "viewer {\n zones(filter: { zoneTag: \"$zoneTags\"}) {\n";
      $viewerQuery .= " httpRequestsAdaptiveGroups(\n orderBy: [date_ASC] \n limit: 10000 \n filter: {\ndate_gt: \"$dateReport\"\n clientAsn:\"15169\"} \n) {\n";
      $viewerQuery .= "dimensions {\n date }\n";
      $viewerQuery .= "sum { \n visits }\n";
      $viewerQuery .= "}\n"; //end httpRequests1dGroups
      $viewerQuery .= "}\n"; //end viewer internal
      $viewerQuery .= "}\n"; //end viewer
      $viewerQuery .= "}\n"; //end root

      $params = [
        "query" => $viewerQuery
      ];
      $result = triggerSyncRequest($apiURL, 'POST', $params, $headers);
      if (isset($result['data']) && !empty($result['data'])) {
        $data = $result['data']['viewer']['zones'][0]['httpRequestsAdaptiveGroups'];
        $date = [];
        $total = [];
        foreach ($data as $item) {
          $date[] = (date("Y-m-d 00:00:00", strtotime($item['dimensions']['date'])));
          $total[] = $item['sum']['visits'];
        }
        $response = [
          'status' => 'successful',
          'data' => [
            'date' => $date,
            'total' =>$total
          ]
        ];
      }
    }

    return response()->json($response);
  }
}
