<?php

namespace Modules\Dashboard\Controllers;

use DateTime;
use DateInterval;
use Illuminate\Http\Request;
use Modules\Dashboard\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Modules\Dashboard\Controllers\Report\CustomerReportController;
use Modules\Ticket\Models\Ticket;

class ReportController extends Controller {

    private function _getSeparateOrder($dateFrom, $dateTo) {
        $retVal = [-1];
        $separates = DB::table('order')
                            ->where('order.payment_status', '=', 'PAID')
                            ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                            ->join('order_meta', function ($join) {
                                $join->on( 'order.id', '=', 'order_meta.order_id')->where('order_meta.key', '=', 'separate_origin_code');
                            })
                            ->select('order.id')
                            ->get();
        foreach ($separates as $order) {
            array_push($retVal, $order->id);
        }
        return $retVal;
    }

    public function reportByCostEstimate(Request $request) {
        $retVal = ['status' => 'error'];

        $sellerToken = $request->input('sellerToken');
        if($sellerToken) {
            $sellerId = DB::table('users')->where('seller_token', $sellerToken)->first(['id'])->id;
        }

        if ($request->has('dateFrom') && $request->has('dateTo')) {
            $currencyRatio = 1;
            if (function_exists('getModuleLocale')) {
                $locales = getModuleLocale();
                $locale = current(array_filter($locales, function($config) {
                    return $config['locale'] == strtolower(env('APP_LOCALE'));
                }));
                if (!empty($locale['currencyRatio'])) {
                    $currencyRatio = $locale['currencyRatio'];
                }
            }
            
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => [], 'totalCost' => 0];

            $period = new \DatePeriod(
                new \DateTime($dateFrom),
                new \DateInterval('P1D'),
                new \DateTime($dateTo)
            );

            foreach ($period as $key => $value) {
                $retVal['data'][$value->format('Y-m-d')] = [
                    'totalCost' => 0,
                ];
            }

            $queryOrders = DB::table('order')
                    ->join('order_meta', function ($join) {
                        $join->on( 'order.id', '=', 'order_meta.order_id')->where('order_meta.key', '=', 'estimate_costs');
                    })
                    ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                    ->where('order.payment_status', '=', 'PAID');
            if(isset($sellerId) && $sellerId) {
                $queryOrders = $this->queryJoinFromOrderToProduct($queryOrders, $sellerId);
            }
            $orders = $queryOrders->select(DB::raw('IFNULL(SUM(sb_order_meta.value), 0) AS totalCost'), DB::raw('DATE(sb_order.created_at) AS createdAt'))
                    ->groupBy('createdAt')
                    ->get();

            foreach ($orders as $order) {
                if (!empty($retVal['data'][$order->createdAt])) {
                    $retVal['data'][$order->createdAt]['totalCost'] = ($request->has('convert_currency')) ? round($order->totalCost / $currencyRatio, 2) : $order->totalCost;
                    
                    $retVal['totalCost'] += $retVal['data'][$order->createdAt]['totalCost'];
                }
            }
            
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportByCost(Request $request) {
        $retVal = ['status' => 'error'];

        if ($request->has('dateFrom') && $request->has('dateTo')) {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => [], 'totalCost' => 0];

            $period = new \DatePeriod(
                new \DateTime($dateFrom),
                new \DateInterval('P1D'),
                new \DateTime($dateTo)
            );

            foreach ($period as $key => $value) {
                $retVal['data'][$value->format('Y-m-d')] = [
                    'totalCost' => 0,
                ];
            }

            $orders = DB::table('order')
                ->whereBetween('created_at', [$dateFrom, $dateTo])
                ->where('payment_status', '=', 'PAID')
                ->select(DB::raw('IFNULL(SUM(cost + shipping_cost + tax_cost), 0) AS totalCost'), DB::raw('DATE(created_at) AS createdAt'), DB::raw('SUM(transaction_fee) as totalTransationFee'))
                ->groupBy('createdAt')
                ->get();
            $currencyRatio = 1;
            if (function_exists('getModuleLocale')) {
                $locales = getModuleLocale();
                $locale = current(array_filter($locales, function($config) {
                    return $config['locale'] == strtolower(env('APP_LOCALE'));
                }));
                if (!empty($locale['currencyRatio'])) {
                    $currencyRatio = $locale['currencyRatio'];
                }
            }
            foreach ($orders as $order) {
                if (!empty($retVal['data'][$order->createdAt])) {
                    $retVal['data'][$order->createdAt]['totalCost'] = $order->totalCost + round($order->totalTransationFee / $currencyRatio, 2);
                    $retVal['totalCost'] += $retVal['data'][$order->createdAt]['totalCost'];
                }
            }
            
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportByOrder(Request $request) {
        $currencyRatio = 1;
        if (function_exists('getModuleLocale')) {
            $locales = getModuleLocale();
            $locale = current(array_filter($locales, function($config) {
                return $config['locale'] == strtolower(env('APP_LOCALE'));
            }));
            if (!empty($locale['currencyRatio'])) {
                $currencyRatio = $locale['currencyRatio'];
            }
        }

        $retVal = ['status' => 'error'];
        if ($request->get('date')) {
            $date = DateTime::createFromFormat('Y-m-d', $request->get('date'));
            if ($date) {
                $request->merge([
                    'dateFrom' => $date->getTimestamp(),
                    'dateTo' => $date->getTimestamp(),
                ]);
            }
        }

        if ($request->has('dateFrom') && $request->has('dateTo')) {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => [], 'totalQuantity' => 0, 'totalAmount' => 0];
            $reportBy = $request->get('report_by', 'date');

            $period = new \DatePeriod(
                new \DateTime($dateFrom),
                \DateInterval::createFromDateString($reportBy == 'hour' ? '1 hour' : '1 day'),
                new \DateTime($dateTo)
            );

            foreach ($period as $key => $value) {
                $retVal['data'][$reportBy == 'hour' ? $value->format('Y-m-d H') : $value->format('Y-m-d')] = [
                    'totalQuantity' => 0,
                    'totalAmount' => 0
                ];
            }

            $groupByStr = $reportBy == 'hour' ? 'DATE_FORMAT(`created_at`, "%Y-%m-%d %H") AS createdAt' : 'DATE(created_at) AS createdAt';

            $quantityOrders = DB::table('order')
                                ->whereNull('parent_id')
                                ->whereBetween('created_at', [$dateFrom, $dateTo])
                                ->where('payment_status', '=', 'PAID')
                                ->whereNotIn('id', $this->_getSeparateOrder($dateFrom, $dateTo))
                                ->select(DB::raw('COUNT(*) AS totalQuantity'), DB::raw($groupByStr))
                                ->groupBy('createdAt')
                                ->get();

            foreach ($quantityOrders as $order) {
                if (!empty($retVal['data'][$order->createdAt])) {
                    $retVal['data'][$order->createdAt]['totalQuantity'] = $order->totalQuantity;
                    $retVal['totalQuantity'] += $retVal['data'][$order->createdAt]['totalQuantity'];
                }
            }

            $amountOrders = DB::table('order')
                            ->whereBetween('created_at', [$dateFrom, $dateTo])
                            ->where('payment_status', '=', 'PAID')
                            ->select(DB::raw('SUM(amount) AS totalAmount'), DB::raw($groupByStr))
                            ->groupBy('createdAt')
                            ->get();

            foreach ($amountOrders as $order) {
                if (!empty($retVal['data'][$order->createdAt])) {
                    $retVal['data'][$order->createdAt]['totalAmount'] = ($request->has('convert_currency')) ? round($order->totalAmount / $currencyRatio, 2) : $order->totalAmount;

                    $retVal['totalAmount'] += $retVal['data'][$order->createdAt]['totalAmount'];
                }
            }

            $refundTimes = DB::table('order_meta')
                            ->where('key', '=', 'refund_time')
                            ->whereBetween('value', [ $dateFrom, $dateTo ])
                            ->select('order_id')
                            ->get();

            $returnIds = [-1];
            foreach ($refundTimes as $refund) {
                $returnIds[] = $refund->order_id;
            }

            $ordersReturn = DB::table('order')->whereIn('id', $returnIds)
                            ->select(DB::raw('SUM(return_fee) AS returnAmount'), DB::raw($groupByStr))
                            ->groupBy('createdAt')
                            ->get();

            foreach ($ordersReturn as $orderReturn) {
                if (!empty($retVal['data'][$orderReturn->createdAt])) {
                    $returnFee = ($request->has('convert_currency')) ? round($orderReturn->returnAmount / $currencyRatio, 2) : $orderReturn->returnAmount;
                    $retVal['data'][$orderReturn->createdAt]['totalAmount'] -= $returnFee;
                    $retVal['totalAmount'] -= $returnFee;
                }
            }
            
        }

        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    private function queryJoinFromOrderToProduct($query, $sellerId) {
        return $query->join('order_item', 'order_item.order_id', '=', 'order.id')
        ->join('product', 'product.id', '=', 'order_item.product_id')
        ->where('product.actor_id', $sellerId);
    }

    public function reportBySource(Request $request) {
        $retVal = ['status' => 'error'];
        if ($request->has('dateFrom') && $request->has('dateTo')) {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => []];

            $retVal['data'] = DB::table('order')
                        ->whereNull('order.parent_id')
                        ->where('order.payment_status', '=', 'PAID')
                        ->whereNotIn('order.id', $this->_getSeparateOrder($dateFrom, $dateTo))
                        ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                        ->leftJoin('order_meta', function ($join) {
                            $join->on( 'order.id', '=', 'order_meta.order_id')->where('order_meta.key', '=', 'from');
                        })
                        ->select(DB::raw("IFNULL(sb_order_meta.value, 'other') AS source"), DB::raw('COUNT(*) AS quantity'))
                        ->groupBy('order_meta.value')
                        ->orderBy('quantity', 'DESC')
                        ->get();
            
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportByCategory(Request $request) {
        $retVal = ['status' => 'error'];
        if ($request->has('dateFrom') && $request->has('dateTo')) {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => []];

            $categories = DB::table('order_item')
                                ->join('order', 'order_item.order_id', '=', 'order.id')
                                ->join('product_n_category', 'order_item.product_id', '=', 'product_n_category.product_id')
                                ->join('category', 'product_n_category.category_id', '=', 'category.id')
                                ->where('product_n_category.is_parent', '=', 0)
                                ->where('order.payment_status', '=', 'PAID')
                                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                                ->select([
                                    'product_n_category.category_id AS categoryId', 
                                    'category.name AS name', 
                                    DB::raw('COUNT(*) AS quantity')
                                ])
                                ->groupBy('categoryId')
                                ->groupBy('name')
                                ->orderBy('quantity', 'DESC')
                                ->get();
            
            $retVal['data'] = $categories;
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportByProduct(Request $request) {
        $retVal = ['status' => 'error'];
        if ($request->has('dateFrom') && $request->has('dateTo')) {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => []];

            $retVal['data'] = DB::table('order_item')
                                ->join('order', 'order_item.order_id', '=', 'order.id')
                                ->join('product', 'order_item.product_id', '=', 'product.id')
                                ->where('order.payment_status', '=', 'PAID')
                                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                                ->select('order_item.product_id AS productId', DB::raw('COUNT(*) AS quantity'), 'product.name')
                                ->groupBy('order_item.product_id')
                                ->orderBy('quantity', 'DESC')
                                ->limit(15)
                                ->get();

        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportByCategorySource(Request $request) {
        $retVal = ['status' => 'error'];
        if ($request->has('dateFrom') && $request->has('dateTo')) {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => []];

            $period = new \DatePeriod(
                new \DateTime($dateFrom),
                new \DateInterval('P1D'),
                new \DateTime($dateTo)
            );

            foreach ($period as $key => $value) {
                $retVal['data'][$value->format('Y-m-d')] = [
                    'categories' => [],
                    'sources' => []
                ];
            }

            $categories = DB::table('order_item')
                                ->join('order', 'order_item.order_id', '=', 'order.id')
                                ->join('product_n_category', 'order_item.product_id', '=', 'product_n_category.product_id')
                                ->join('category', 'product_n_category.category_id', '=', 'category.id')
                                ->where('product_n_category.is_parent', '=', 0)
                                ->where('order.payment_status', '=', 'PAID')
                                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                                ->select([
                                    'product_n_category.category_id AS categoryId', 
                                    'category.name AS name',
                                    DB::raw('DATE(sb_order.created_at) AS createdAt'),
                                    DB::raw('COUNT(*) AS quantity')
                                ])
                                ->groupBy('categoryId', 'createdAt', 'name')
                                ->get();

            $sources =  DB::table('order')
                                ->whereNull('order.parent_id')
                                ->where('order.payment_status', '=', 'PAID')
                                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                                ->whereNotIn('order.id', $this->_getSeparateOrder($dateFrom, $dateTo))
                                ->leftJoin('order_meta', function ($join) {
                                    $join->on( 'order.id', '=', 'order_meta.order_id')->where('order_meta.key', '=', 'from');
                                })
                                ->select([
                                    DB::raw("IFNULL(sb_order_meta.value, 'other') AS source"), 
                                    DB::raw('DATE(sb_order.created_at) AS createdAt'),
                                    DB::raw('COUNT(*) AS quantity')
                                ])
                                ->groupBy('order_meta.value', 'createdAt')
                                ->get();

            foreach ($categories as $category) {
                if (key_exists($category->createdAt, $retVal['data'])) {
                    array_push($retVal['data'][$category->createdAt]['categories'], [
                        'name' => $category->name,
                        'quantity' => $category->quantity
                    ]);
                }
            }

            foreach ($sources as $item) {
                if (key_exists($item->createdAt, $retVal['data'])) {
                    array_push($retVal['data'][$item->createdAt]['sources'], [
                        'name' => $item->source,
                        'quantity' => $item->quantity
                    ]);
                }
            }
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function getCustomerReport(Request $request)
    {
        $reportController = new CustomerReportController();

        $response = [
            'status' => 'fail'
        ];

        if (!$request->has('dateFrom') || !$request->has('dateTo')) return $response;

        $response['status'] = 'successful';

        $locate = $request->input('locale', 'us');

        if (class_exists(`\App\Helpers\DBConnect`)) {
            $connection = \App\Helpers\DBConnect::connect($locate);
        } else {
            $connection = DB::connection();
        }

        $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
        $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));

        $period = new \DatePeriod(
            new \DateTime($dateFrom),
            new \DateInterval('P1D'),
            new \DateTime($dateTo)
        );

        $groupUserType = [
           /*  'order/new customer' => 0,
            'order/returning customer' => 0, */
            0 => [],
            1 => []
        ];

        $orders = $connection->table('order')
            ->whereNull('order.parent_id')
            ->where('order.payment_status', '=', 'PAID')
            ->select(['order.id', 'order.created_at', 'order.customer_id'])
            ->whereBetween('order.created_at', [$dateFrom, $dateTo])
            ->get();
            
        $reportController->buildCustomerReturning($orders, $dateFrom);
        $orders = $reportController->mapOrderCustomer($orders);

        $xAxis = [];
        foreach ($period as $value) {
            $xAxis[] = $value->format('Y-m-d');
        }

        foreach ($orders as $order) 
        {
            $dateKey = substr($order->created_at, 0, 10);
            if (!isset($groupUserType[0][$dateKey])) $groupUserType[0][$dateKey] = 0;
            if (!isset($groupUserType[1][$dateKey])) $groupUserType[1][$dateKey] = 0;

            if ($order->is_returning) {
                $groupUserType[1][$dateKey] ++;
            } else {
                $groupUserType[0][$dateKey] ++;
            }
        }

        $response['customer_registed'] = $reportController->getRegistedCustomerData($dateFrom, $dateTo);
        $response['group_by_date'] = [
            'order/new customer' => $groupUserType[0],
            'order/returning customer' => $groupUserType[1],
        ];
        $response['xAxis'] = $xAxis;

        return $response;
    }

    public function getTicketReport(Request $request)
    {
        $response = [
            'status' => 'fail'
        ];

        if (!$request->has('dateFrom') || !$request->has('dateTo')) return $response;

        $response['status'] = 'successful';
        $locate = $request->input('locale', 'us');

        $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
        $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));

        $period = new \DatePeriod(
            new \DateTime($dateFrom),
            new \DateInterval('P1D'),
            new \DateTime($dateTo)
        );

        $tickets = Ticket::where('locate', $locate)
            ->whereBetween('created_at', [$dateFrom, $dateTo])
            ->select(['type', 'created_at'])
            ->get();

        $xAxis = [];
        foreach ($period as $value) {
            $xAxis[] = $value->format('Y-m-d');
        }

        $groupTicket = [];
        $groupTicketDates = [];

        foreach ($tickets as $ticket) 
        {
            $dateKey = substr($ticket->created_at, 0, 10);

            if (!isset($groupTicket[$ticket->type])) $groupTicket[$ticket->type] = 0;
            $groupTicket[$ticket->type] ++;

            if (!isset($groupTicketDates[$ticket->type])) $groupTicketDates[$ticket->type] = [];
            if (!isset($groupTicketDates[$ticket->type][$dateKey])) $groupTicketDates[$ticket->type][$dateKey] = 0;
            $groupTicketDates[$ticket->type][$dateKey] ++;
        }

        $response['group_by_type'] = $groupTicket;
        $response['group_by_date'] = $groupTicketDates;
        $response['xAxis'] = $xAxis;

        return $response;
    }

    public function getOrderWithDevices(Request $request)
    {
        $response = [
            'status' => 'fail'
        ];

        if (!$request->has('dateFrom') || !$request->has('dateTo')) return $response;

        $response['status'] = 'successful';

        $locate = $request->input('locale', 'us');

        if (class_exists(`\App\Helpers\DBConnect`)) {
            $connection = \App\Helpers\DBConnect::connect($locate);
        } else {
            $connection = DB::connection();
        }

        $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
        $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));

        $period = new \DatePeriod(
            new \DateTime($dateFrom),
            new \DateInterval('P1D'),
            new \DateTime($dateTo)
        );

        $orders = $connection->table('order')
            ->whereNull('order.parent_id')
            ->where('order.payment_status', '=', 'PAID')
            ->select(['order.id', 'order.created_at', 'order_meta.value as device'])
            ->leftJoin('order_meta', function ($join) {
                $join->on( 'order.id', '=', 'order_meta.order_id')->where('order_meta.key', '=', 'device');
            })
            ->whereBetween('order.created_at', [$dateFrom, $dateTo])
            ->get();

        $groupDevices = [
            'desktop' => 0,
            'mobile' => 0,
            'tablet' => 0,
            'other' => 0,
        ];

        $groupDates = [
            'desktop' => [],
            'mobile' => [],
            'tablet' => [],
            'other' => [],
        ];

        foreach ($orders as $order) 
        {
            $dateKey = substr($order->created_at, 0, 10);
            if ($order->device) {

                $order->device = strtolower($order->device);

                if (!isset($groupDevices[$order->device])) $groupDevices[$order->device] = 0;
                $groupDevices[$order->device] ++;

                // check date
                if (!isset($groupDates[$order->device])) $groupDates[$order->device] = [];
                if (!isset($groupDates[$order->device][$dateKey])) $groupDates[$order->device][$dateKey] = 0;

                $groupDates[$order->device][$dateKey] ++;

            } else {

                $groupDevices['other'] ++;
                if (!isset($groupDates['other'][$dateKey])) $groupDates['other'][$dateKey] = 0;
                $groupDates['other'][$dateKey] ++;

            }
        }

        $formatedGroupDates = [];
        foreach ($groupDates as $device => $dateData) {
            $formatedGroupDates[$device] = [];
            foreach ($period as $value) {
                $dateValue = $value->format('Y-m-d');
                $formatedGroupDates[$device][$dateValue] = 0;
                if (isset($dateData[$dateValue]) && $dateData[$dateValue] > 0) {
                    $formatedGroupDates[$device][$dateValue] = $dateData[$dateValue];
                }
            }
        }

        $xAxis = [];
        foreach ($period as $value) {
            $xAxis[] = $value->format('Y-m-d');
        }

        $response['group_by_device'] = $groupDevices;
        $response['group_by_date'] = $formatedGroupDates;
        $response['xAxis'] = $xAxis;

        return $response;

    }

    public function reportByCategoryForSeller(Request $request) {
        $retVal = ['status' => 'error'];

        $sellerToken = $request->input('sellerToken');
        if ($sellerToken) {
            $sellerId = DB::table('users')->where('seller_token', $sellerToken)->first(['id'])->id;
        }

        if ($request->has('dateFrom') && $request->has('dateTo') && isset($sellerId) && $sellerId) {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => []];

            $queryCategories = DB::table('order_item')
                                ->join('order', 'order_item.order_id', '=', 'order.id')
                                ->join('product_n_category', 'order_item.product_id', '=', 'product_n_category.product_id')
                                ->join('category', 'product_n_category.category_id', '=', 'category.id')
                                ->join('product', 'product.id', '=', 'order_item.product_id')
                                ->where('product.actor_id', $sellerId)
                                ->where('product_n_category.is_parent', '=', 0)
                                ->where('order.payment_status', '=', 'PAID')
                                ->whereBetween('order.created_at', [$dateFrom, $dateTo]);

            $categories = $queryCategories->select([
                'product_n_category.category_id AS categoryId', 
                'category.name AS name', 
                DB::raw('COUNT(*) AS quantity')
            ])
            ->groupBy('categoryId')
            ->groupBy('name')
            ->orderBy('quantity', 'DESC')
            ->get();

            $retVal['data'] = $categories;
        }
        
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportByOrderForSeller(Request $request) {
        $retVal = ['status' => 'error'];

        $sellerToken = $request->input('sellerToken');
        if($sellerToken) {
            $sellerId = DB::table('users')->where('seller_token', $sellerToken)->first(['id'])->id;
        }

        $currencyRatio = 1;
        if (function_exists('getModuleLocale')) {
            $locales = getModuleLocale();
            $locale = current(array_filter($locales, function($config) {
                return $config['locale'] == strtolower(env('APP_LOCALE'));
            }));
            if (!empty($locale['currencyRatio'])) {
                $currencyRatio = $locale['currencyRatio'];
            }
        }

        if ($request->has('dateFrom') && $request->has('dateTo') && isset($sellerId) && $sellerId) {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => [], 'totalQuantity' => 0, 'totalAmount' => 0];

            $period = new \DatePeriod(
                new \DateTime($dateFrom),
                new \DateInterval('P1D'),
                new \DateTime($dateTo)
            );

            foreach ($period as $key => $value) {
                $retVal['data'][$value->format('Y-m-d')] = [
                    'totalQuantity' => 0,
                    'totalAmount' => 0
                ];
            }

            $queryQuantityOrders = DB::table('order')
            ->whereNull('order.parent_id')
            ->whereBetween('order.created_at', [$dateFrom, $dateTo])
            ->where('order.payment_status', '=', 'PAID')
            ->whereNotIn('order.id', $this->_getSeparateOrder($dateFrom, $dateTo));
            $queryQuantityOrders = $this->queryJoinFromOrderToProduct($queryQuantityOrders, $sellerId);
            $quantityOrders = $queryQuantityOrders->select(DB::raw('COUNT(*) AS totalQuantity'), DB::raw('DATE(sb_order.created_at) AS createdAt'))
                                ->groupBy('createdAt')
                                ->get();

            foreach ($quantityOrders as $order) {
                if (!empty($retVal['data'][$order->createdAt])) {
                    $retVal['data'][$order->createdAt]['totalQuantity'] = $order->totalQuantity;
                    $retVal['totalQuantity'] += $retVal['data'][$order->createdAt]['totalQuantity'];
                }
            }

            $queryAmountOrders = DB::table('order')
                                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                                ->where('order.payment_status', '=', 'PAID');
            $queryAmountOrders = $this->queryJoinFromOrderToProduct($queryAmountOrders, $sellerId);

            $amountOrders = $queryAmountOrders->select(DB::raw('SUM(sb_order_item.quantity * sb_order_item.price) AS totalAmount'), DB::raw('DATE(sb_order.created_at) AS createdAt'))
                            ->groupBy('createdAt')
                            ->get();

            foreach ($amountOrders as $order) {
                if (!empty($retVal['data'][$order->createdAt])) {
                    $retVal['data'][$order->createdAt]['totalAmount'] = ($request->has('convert_currency')) ? round($order->totalAmount / $currencyRatio, 2) : $order->totalAmount;

                    $retVal['totalAmount'] += $retVal['data'][$order->createdAt]['totalAmount'];
                }
            }

            $refundTimes = DB::table('order_meta')
                            ->where('key', '=', 'refund_time')
                            ->whereBetween('value', [ $dateFrom, $dateTo ])
                            ->select('order_id')
                            ->get();

            $returnIds = [-1];
            foreach ($refundTimes as $refund) {
                $returnIds[] = $refund->order_id;
            }

            $queryOrdersReturn = DB::table('order')->whereIn('order.id', $returnIds);
            $queryOrdersReturn = $this->queryJoinFromOrderToProduct($queryOrdersReturn, $sellerId);
            $ordersReturn = $queryOrdersReturn->select(DB::raw('SUM(sb_order.return_fee) AS returnAmount'), DB::raw('DATE(sb_order.created_at) AS createdAt'))
            ->groupBy('createdAt')
            ->get();
            
            foreach ($ordersReturn as $orderReturn) {
                if (!empty($retVal['data'][$orderReturn->createdAt])) {
                    $returnFee = ($request->has('convert_currency')) ? round($orderReturn->returnAmount / $currencyRatio, 2) : $orderReturn->returnAmount;
                    $retVal['data'][$orderReturn->createdAt]['totalAmount'] -= $returnFee;
                    $retVal['totalAmount'] -= $returnFee;
                }
            }
            
        }

        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportByCostForSeller(Request $request) {
        $retVal = ['status' => 'error'];

        $sellerToken = $request->input('sellerToken');
        if($sellerToken) {
            $sellerId = DB::table('users')->where('seller_token', $sellerToken)->first(['id'])->id;
        }

        if ($request->has('dateFrom') && $request->has('dateTo') && isset($sellerId) && $sellerId) {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => [], 'totalCost' => 0];

            $period = new \DatePeriod(
                new \DateTime($dateFrom),
                new \DateInterval('P1D'),
                new \DateTime($dateTo)
            );

            foreach ($period as $key => $value) {
                $retVal['data'][$value->format('Y-m-d')] = [
                    'totalCost' => 0,
                ];
            }

            $orders = DB::table('order')
                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                ->where('payment_status', '=', 'PAID')
                ->join('order_item', 'order_item.order_id', '=', 'order.id')
                ->join('product', 'product.id', '=', 'order_item.product_id')
                ->where('product.actor_id', $sellerId)
                ->select(DB::raw('IFNULL(SUM(sb_order.cost + sb_order.shipping_cost + sb_order.tax_cost), 0) AS totalCost'), DB::raw('DATE(sb_order.created_at) AS createdAt'), DB::raw('SUM(sb_order.transaction_fee) as totalTransationFee'))
                ->groupBy('createdAt')
                ->get();
            $currencyRatio = 1;
            if (function_exists('getModuleLocale')) {
                $locales = getModuleLocale();
                $locale = current(array_filter($locales, function($config) {
                    return $config['locale'] == strtolower(env('APP_LOCALE'));
                }));
                if (!empty($locale['currencyRatio'])) {
                    $currencyRatio = $locale['currencyRatio'];
                }
            }
            foreach ($orders as $order) {
                if (!empty($retVal['data'][$order->createdAt])) {
                    $retVal['data'][$order->createdAt]['totalCost'] = $order->totalCost + round($order->totalTransationFee / $currencyRatio, 2);
                    $retVal['totalCost'] += $retVal['data'][$order->createdAt]['totalCost'];
                }
            }
            
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportByCostEstimateForSeller(Request $request) {
        $retVal = ['status' => 'error'];

        $sellerToken = $request->input('sellerToken');
        if($sellerToken) {
            $sellerId = DB::table('users')->where('seller_token', $sellerToken)->first(['id'])->id;
        }

        if ($request->has('dateFrom') && $request->has('dateTo') && isset($sellerId) && $sellerId) {
            $currencyRatio = 1;
            if (function_exists('getModuleLocale')) {
                $locales = getModuleLocale();
                $locale = current(array_filter($locales, function($config) {
                    return $config['locale'] == strtolower(env('APP_LOCALE'));
                }));
                if (!empty($locale['currencyRatio'])) {
                    $currencyRatio = $locale['currencyRatio'];
                }
            }
            
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => [], 'totalCost' => 0];

            $period = new \DatePeriod(
                new \DateTime($dateFrom),
                new \DateInterval('P1D'),
                new \DateTime($dateTo)
            );

            foreach ($period as $key => $value) {
                $retVal['data'][$value->format('Y-m-d')] = [
                    'totalCost' => 0,
                ];
            }

            $orders = DB::table('order')
                    ->join('order_meta', function ($join) {
                        $join->on( 'order.id', '=', 'order_meta.order_id')->where('order_meta.key', '=', 'estimate_costs');
                    })
                    ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                    ->where('order.payment_status', '=', 'PAID')
                    ->join('order_item', 'order_item.order_id', '=', 'order.id')
                    ->join('product', 'product.id', '=', 'order_item.product_id')
                    ->where('product.actor_id', $sellerId)
                    ->select(DB::raw('IFNULL(SUM(sb_order_meta.value), 0) AS totalCost'), DB::raw('DATE(sb_order.created_at) AS createdAt'))
                    ->groupBy('createdAt')
                    ->get();

            foreach ($orders as $order) {
                if (!empty($retVal['data'][$order->createdAt])) {
                    $retVal['data'][$order->createdAt]['totalCost'] = ($request->has('convert_currency')) ? round($order->totalCost / $currencyRatio, 2) : $order->totalCost;
                    
                    $retVal['totalCost'] += $retVal['data'][$order->createdAt]['totalCost'];
                }
            }
            
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportByProductForSeller(Request $request) {
        $retVal = ['status' => 'error'];
        $sellerToken = $request->input('sellerToken');
        if ($sellerToken) {
            $sellerId = DB::table('users')->where('seller_token', $sellerToken)->first(['id'])->id;
        }

        if ($request->has('dateFrom') && $request->has('dateTo') && isset($sellerId) && $sellerId) {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => []];

            $retVal['data'] = DB::table('order_item')
                                ->join('order', 'order_item.order_id', '=', 'order.id')
                                ->join('product', 'order_item.product_id', '=', 'product.id')
                                ->where('order.payment_status', '=', 'PAID')
                                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                                ->where('product.actor_id', $sellerId)
                                ->select('order_item.product_id AS productId', DB::raw('COUNT(*) AS quantity'), 'product.name')
                                ->groupBy('order_item.product_id')
                                ->orderBy('quantity', 'DESC')
                                ->limit(15)
                                ->get();
        }

        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportBySourceForSeller(Request $request) {
        $retVal = ['status' => 'error'];
        $sellerToken = $request->input('sellerToken');
        if ($sellerToken) {
            $sellerId = DB::table('users')->where('seller_token', $sellerToken)->first(['id'])->id;
        }

        if ($request->has('dateFrom') && $request->has('dateTo') && isset($sellerId) && $sellerId) {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => []];

            $retVal['data'] = DB::table('order')
                        ->join('order_item', 'order_item.order_id', '=', 'order.id')
                        ->join('product', 'product.id', '=', 'order_item.product_id')
                        ->where('product.actor_id', $sellerId)
                        ->whereNull('order.parent_id')
                        ->where('order.payment_status', '=', 'PAID')
                        ->whereNotIn('order.id', $this->_getSeparateOrder($dateFrom, $dateTo))
                        ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                        ->leftJoin('order_meta', function ($join) {
                            $join->on( 'order.id', '=', 'order_meta.order_id')->where('order_meta.key', '=', 'from');
                        })
                        ->select(DB::raw("IFNULL(sb_order_meta.value, 'other') AS source"), DB::raw('COUNT(*) AS quantity'))
                        ->groupBy('order_meta.value')
                        ->orderBy('quantity', 'DESC')
                        ->get();
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }
}
