<?php

namespace Modules\Dashboard\Controllers\ReportEvent;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Dashboard\Controllers\Controller;
use Cache;

class ReportEventController extends Controller
{
    use EventReportTrait;

    public function reportOrderByLocale(Request $request) 
    {
        $currencyRatio = $this->getLocaleCurrencyRatio();

        $retVal = ['status' => 'error'];

        if ($request->has('dateFrom') && $request->has('dateTo')) 
        {
            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => [], 'totalQuantity' => 0, 'totalAmount' => 0];
            $eventId = $request->input('event_id');

            $period = new \DatePeriod(
                new \DateTime($dateFrom),
                new \DateInterval('P1D'),
                new \DateTime($dateTo)
            );

            foreach ($period as $value) {
                $retVal['data'][$value->format('Y-m-d')] = [
                    'totalQuantity' => 0,
                    'totalAmount' => 0
                ];
            }

            $separateOrderIds = $this->getSeparateOrderIds($dateFrom, $dateTo);
            $eventOrderIds = $this->getEventOrderIds($dateFrom, $dateTo, $eventId);

            $availOrderIds = array_diff($eventOrderIds, $separateOrderIds);

            $quantityOrders = DB::table('order')
                ->whereNull('parent_id')
                ->whereBetween('created_at', [$dateFrom, $dateTo])
                ->where('payment_status', '=', 'PAID')
                ->whereIn('id', $availOrderIds)
                ->select(DB::raw('COUNT(*) AS totalQuantity'), DB::raw('DATE(created_at) AS createdAt'))
                ->groupBy('createdAt')
                ->get();

            foreach ($quantityOrders as $order) {
                if (!empty($retVal['data'][$order->createdAt])) {
                    $retVal['data'][$order->createdAt]['totalQuantity'] = $order->totalQuantity;
                    $retVal['totalQuantity'] += $retVal['data'][$order->createdAt]['totalQuantity'];
                }
            }

            $amountEventOrderItems = DB::table('order_item')
                ->join('order', 'order.id', 'order_item.order_id')
                ->join('season_event_n_product', 'season_event_n_product.product_id', 'order_item.product_id')
                ->where('order.payment_status', '=', 'PAID')
                ->whereBetween('order_item.created_at', [$dateFrom, $dateTo])
                ->whereIn('order_item.order_id', $availOrderIds)
                ->select([
                    DB::raw('DATE(sb_order_item.created_at) AS createdAt'), 
                    'order_item.quantity', 
                    'order_item.price'
                ])
                ->get();

            foreach ($amountEventOrderItems as $item) {
                if (!empty($retVal['data'][$item->createdAt])) {

                    if (!isset($retVal['data'][$item->createdAt]['totalAmount'])) $retVal['data'][$item->createdAt]['totalAmount']  = 0;
                    
                    $amount = $item->price * $item->quantity;
                    $addingValue = ($request->has('convert_currency')) ? round($amount / $currencyRatio, 2) : $amount;

                    $retVal['data'][$item->createdAt]['totalAmount'] += $addingValue;
                    $retVal['totalAmount'] += $retVal['data'][$order->createdAt]['totalAmount'];
                }
            }

            $refundTimes = DB::table('order_meta')
                ->where('key', '=', 'refund_time')
                ->whereBetween('value', [ $dateFrom, $dateTo ])
                ->select('order_id')
                ->get();

            $returnIds = [-1];
            foreach ($refundTimes as $refund) {
                $returnIds[] = $refund->order_id;
            }

            $ordersReturn = DB::table('order')->whereIn('id', $returnIds)
                ->select(DB::raw('SUM(return_fee) AS returnAmount'), DB::raw('DATE(created_at) AS createdAt'))
                ->groupBy('createdAt')
                ->get();

            foreach ($ordersReturn as $orderReturn) {
                if (!empty($retVal['data'][$orderReturn->createdAt])) {
                    $returnFee = ($request->has('convert_currency')) ? round($orderReturn->returnAmount / $currencyRatio, 2) : $orderReturn->returnAmount;
                    $retVal['data'][$orderReturn->createdAt]['totalAmount'] -= $returnFee;
                    $retVal['totalAmount'] -= $returnFee;
                }
            }
        }

        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportOrderByCategory(Request $request) 
    {
        $retVal = ['status' => 'error'];
        if ($request->has('dateFrom') && $request->has('dateTo')) {

            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => []];
            $eventId = $request->input('event_id');

            $query = DB::table('order_item')
                ->join('order', 'order_item.order_id', '=', 'order.id')
                ->join('product_n_category', 'order_item.product_id', '=', 'product_n_category.product_id')
                ->join('category', 'product_n_category.category_id', '=', 'category.id')
                ->join('season_event_n_product', 'season_event_n_product.product_id', 'order_item.product_id')
                ->where('product_n_category.is_parent', '=', 0)
                ->where('order.payment_status', '=', 'PAID')
                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                ->select([
                    'product_n_category.category_id AS categoryId', 
                    'category.name AS name', 
                    DB::raw('COUNT(*) AS quantity')
                ])
                ->groupBy('categoryId')
                ->groupBy('name')
                ->orderBy('quantity', 'DESC');

            if ($eventId) {
                $query->where('season_event_n_product.season_event_id', $eventId);
            }
            
            $retVal['data'] = $query->get();
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportOrderBySource(Request $request) 
    {
        $retVal = ['status' => 'error'];
        if ($request->has('dateFrom') && $request->has('dateTo')) {

            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => []];
            $eventId = $request->input('event_id');

            $separateOrderIds = $this->getSeparateOrderIds($dateFrom, $dateTo);
            $eventOrderIds = $this->getEventOrderIds($dateFrom, $dateTo, $eventId);

            $availOrderIds = array_diff($eventOrderIds, $separateOrderIds);

            $retVal['data'] = DB::table('order')
                ->whereNull('order.parent_id')
                ->where('order.payment_status', '=', 'PAID')
                ->whereIn('order.id', $availOrderIds)
                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                ->leftJoin('order_meta', function ($join) {
                    $join->on( 'order.id', '=', 'order_meta.order_id')->where('order_meta.key', '=', 'from');
                })
                ->select(DB::raw("IFNULL(sb_order_meta.value, 'other') AS source"), DB::raw('COUNT(*) AS quantity'))
                ->groupBy('order_meta.value')
                ->orderBy('quantity', 'DESC')
                ->get();
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportByProduct(Request $request) 
    {
        $retVal = ['status' => 'error'];
        if ($request->has('dateFrom') && $request->has('dateTo')) {

            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => []];
            $eventId = $request->input('event_id');

            $separateOrderIds = $this->getSeparateOrderIds($dateFrom, $dateTo);
            $eventOrderIds = $this->getEventOrderIds($dateFrom, $dateTo, $eventId);

            $availOrderIds = array_diff($eventOrderIds, $separateOrderIds);

            $retVal['data'] = DB::table('order_item')
                ->join('order', 'order_item.order_id', '=', 'order.id')
                ->join('product', 'order_item.product_id', '=', 'product.id')
                ->join('season_event_n_product', 'season_event_n_product.product_id', 'order_item.product_id')
                ->where('order.payment_status', '=', 'PAID')
                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                ->whereIn('order.id', $availOrderIds)
                ->select('order_item.product_id AS productId', DB::raw('COUNT(*) AS quantity'), 'product.name')
                ->groupBy('order_item.product_id')
                ->orderBy('quantity', 'DESC')
                ->limit(15)
                ->get();
        }
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function reportByCategorySource(Request $request) 
    {
        $retVal = ['status' => 'error'];
        if ($request->has('dateFrom') && $request->has('dateTo')) {

            $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
            $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
            $retVal = ['status' => 'successful', 'data' => []];
            $eventId = $request->input('event_id');

            $period = new \DatePeriod(
                new \DateTime($dateFrom),
                new \DateInterval('P1D'),
                new \DateTime($dateTo)
            );

            foreach ($period as $value) {
                $retVal['data'][$value->format('Y-m-d')] = [
                    'categories' => [],
                    'sources' => []
                ];
            }

            $categories = DB::table('order_item')
                ->join('order', 'order_item.order_id', '=', 'order.id')
                ->join('product_n_category', 'order_item.product_id', '=', 'product_n_category.product_id')
                ->join('category', 'product_n_category.category_id', '=', 'category.id')
                ->join('season_event_n_product', 'season_event_n_product.product_id', 'order_item.product_id')
                ->where('product_n_category.is_parent', '=', 0)
                ->where('order.payment_status', '=', 'PAID')
                ->where('season_event_n_product.season_event_id', $eventId)
                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                ->select([
                    'product_n_category.category_id AS categoryId', 
                    'category.name AS name',
                    DB::raw('DATE(sb_order.created_at) AS createdAt'),
                    DB::raw('COUNT(*) AS quantity')
                ])
                ->groupBy('categoryId', 'createdAt', 'name')
                ->get();

            $separateOrderIds = $this->getSeparateOrderIds($dateFrom, $dateTo);
            $eventOrderIds = $this->getEventOrderIds($dateFrom, $dateTo, $eventId);

            $availOrderIds = array_diff($eventOrderIds, $separateOrderIds);

            $sources = DB::table('order')
                ->whereNull('order.parent_id')
                ->where('order.payment_status', '=', 'PAID')
                ->whereBetween('order.created_at', [$dateFrom, $dateTo])
                ->whereIn('order.id', $availOrderIds)
                ->leftJoin('order_meta', function ($join) {
                    $join->on( 'order.id', '=', 'order_meta.order_id')->where('order_meta.key', '=', 'from');
                })
                ->select([
                    DB::raw("IFNULL(sb_order_meta.value, 'other') AS source"), 
                    DB::raw('DATE(sb_order.created_at) AS createdAt'),
                    DB::raw('COUNT(*) AS quantity')
                ])
                ->groupBy('order_meta.value', 'createdAt')
                ->get();

            foreach ($categories as $category) {
                if (key_exists($category->createdAt, $retVal['data'])) {
                    array_push($retVal['data'][$category->createdAt]['categories'], [
                        'name' => $category->name,
                        'quantity' => $category->quantity
                    ]);
                }
            }

            foreach ($sources as $item) {
                if (key_exists($item->createdAt, $retVal['data'])) {
                    array_push($retVal['data'][$item->createdAt]['sources'], [
                        'name' => $item->source,
                        'quantity' => $item->quantity
                    ]);
                }
            }
        }
        
        return response()->json($retVal, 200, [], JSON_NUMERIC_CHECK);
    }

    public function getSeasonEvents(Request $request)
    {
        // $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
        // $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
        $retVal = ['status' => 'successful', 'data' => []];
        
        $seasonEvents = DB::table('season_event')
            // ->whereBetween('season_event.', [$dateFrom, $dateTo])
            ->where('status', 'ACTIVE')
            ->orderBy('day', 'desc')
            ->select(['name', 'id'])
            ->get();

        if ($seasonEvents) {
            $retVal['data'] = $seasonEvents;
        }

        return $retVal;
    }

    public function getOrderByConversion(Request $request)
    {
        $response = [
            'status' => 'fail'
        ];

        if (!$request->has('dateFrom') || !$request->has('dateTo')) return $response;

        $response['status'] = 'successful';

        $locate = $request->input('locale', 'us');

        if (class_exists(`\App\Helpers\DBConnect`)) {
            $connection = \App\Helpers\DBConnect::connect($locate);
        } else {
            $connection = DB::connection();
        }

        $dateFrom = date('Y-m-d 00:00:00', $request->input('dateFrom'));
        $dateTo = date('Y-m-d 23:59:59', $request->input('dateTo'));
        $eventId = $request->input('event_id');

        $period = new \DatePeriod(
            new \DateTime($dateFrom),
            new \DateInterval('P1D'),
            new \DateTime($dateTo)
        );

        $separateOrderIds = $this->getSeparateOrderIds($dateFrom, $dateTo);
        $eventOrderIds = $this->getEventOrderIds($dateFrom, $dateTo, $eventId);

        $availOrderIds = array_diff($eventOrderIds, $separateOrderIds);

        $sumAmountOrderByDate = $connection->table('order')
            ->select(DB::raw("DATE_FORMAT(sb_order.created_at, '%Y-%m-%d') as date"), DB::raw('sum(amount) as sum'))
            ->where('order.payment_status', 'PAID')
            ->where('order.created_at', '>=', $dateFrom)
            ->where('order.created_at', '<=', $dateTo)
            ->whereIn('order.id', $availOrderIds)
            ->whereNotIn('order.customer_id', config('dashboard::report-order.ignore_customers', [5031, 14529, 366023]))
            ->groupBy('date')
            ->get()
            ->keyBy('date');

        $orderCountByDate = $connection->table('order')
            ->select(DB::raw("DATE_FORMAT(sb_order.created_at, '%Y-%m-%d') as date"), DB::raw('count(*) as count'), DB::raw('sum(amount) as sum'))
            ->leftJoin('order_meta as om', function($leftJoin) {
                $leftJoin->on('om.order_id', '=', 'order.id')->where('om.key', '=', 'remake_origin_code' );
            })
            ->where('order.payment_status', 'PAID')
            ->where('order.created_at', '>=', $dateFrom)
            ->where('order.created_at', '<=', $dateTo)
            ->whereNull('om.value')
            ->whereNotIn('order.customer_id', config('dashboard::report-order.ignore_customers', [5031, 14529, 366023]))
            ->whereNull('order.parent_id')
            ->whereIn('order.id', $availOrderIds)
            ->groupBy('date')
            ->get()
            ->keyBy('date');

        $cartCountByDate = $connection->table('cart_item')
            ->join('cart', 'cart.id', '=', 'cart_item.cart_id')
            ->join('season_event_n_product', 'season_event_n_product.product_id', 'cart_item.product_id')
            ->select(DB::raw("DATE_FORMAT(sb_cart.created_at, '%Y-%m-%d') as date"), DB::raw('count(*) as count'))
            ->where('cart_item.created_at', '>=', $dateFrom)
            ->where('cart_item.created_at', '<=', $dateTo)
            ->where(function ($query) {
                $query->whereNull('customer_id')->orWhereNotIn('customer_id', config('dashboard::report-order.ignore_customers', [5031, 14529, 366023]));
            });

        if ($eventId) {
            $cartCountByDate->where('season_event_n_product.season_event_id', $eventId);
        }

        $cartCountByDate = $cartCountByDate->groupBy('date')
            ->get()
            ->keyBy('date');

        $viewCountByDate = [];
        $today = new \DateTime(date('Y-m-d 00:00:00', time()));
        foreach ($period as $value) {
            $date = $value->format('Y-m-d');
            $cacheKey = 'OrderByConversion::product_view_' . $locate . '_V3_' . $date;
            if ($value < $today && Cache::has($cacheKey)) {
                $viewCountByDate[$date] = Cache::get($cacheKey);
            } else {
                $viewCountByDate[$date] = $connection->table('user_viewed')
                    ->select(DB::raw("distinct imei, target"))
                    ->where('type', 'product')
                    ->where('created_at', '>=', $value->format('Y-m-d 00:00:00'))
                    ->where('created_at', '<=', $value->format('Y-m-d 23:59:59'))
                    ->count();
                if ($viewCountByDate[$date]) {
                    Cache::put($cacheKey, $viewCountByDate[$date], 15 * 86400);
                }
            }
        }

        $groupDates = [
            'order_cart' => [],
            'order_view' => [],
            'amount_avg' => [],
        ];

        $xAxis = [];
        foreach ($period as $value) {
            $date = $value->format('Y-m-d');
            $xAxis[] = $date;
            if (isset($orderCountByDate[$date]) && isset($cartCountByDate[$date]) && $cartCountByDate[$date]->count) {
                $groupDates['order_cart'][$date] = round($orderCountByDate[$date]->count * 100 / $cartCountByDate[$date]->count, 2, PHP_ROUND_HALF_UP);
            } else {
                $groupDates['order_cart'][$date] = 0;
            }

            if (isset($orderCountByDate[$date]) && isset($viewCountByDate[$date]) && $viewCountByDate[$date]) {
                $groupDates['order_view'][$date] = round($orderCountByDate[$date]->count * 100 / $viewCountByDate[$date], 2, PHP_ROUND_HALF_UP);
            } else {
                $groupDates['order_view'][$date] = 0;
            }

            if (isset($orderCountByDate[$date]) && $orderCountByDate[$date]->count) {
                $groupDates['amount_avg'][$date] = round($sumAmountOrderByDate[$date]->sum / $orderCountByDate[$date]->count, 2, PHP_ROUND_HALF_UP);
            } else {
                $groupDates['amount_avg'][$date] = 0;
            }
        }

        $response['group_by_date'] = $groupDates;
        $response['xAxis'] = $xAxis;

        return $response;

    }
}
