<?php

namespace Modules\Discount\Controllers;

use GuzzleHttp\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\URL;
use Modules\Discount\Controllers\Controller;
use Module;
use DB;
use Modules\Discount\Models\Discount;
use Modules\Discount\Models\DiscountItem;
use Modules\Discount\Models\Product;
use Modules\Discount\Models\Category;

class DiscountController extends Controller
{
    const BATCH_SIZE = 100;
    public function __construct()
    {        
    }

    public function find(Request $request) {
        $filter = $this->buildFilter($request);
        $result = $this->getResult($filter);
        $productIds = [];
        $categoryIds = [];
        foreach ($result as $item) {
            foreach ($item->items as $it) {
                if ($item->apply_type == 'product') {
                    $productIds[] = $it->reference_id; 
                }
                if ($item->apply_type == 'category') {
                    $categoryIds[] = $it->reference_id;
                }
            }
        }
        $productByIds = [];
        $categoryByIds = [];
        if (!empty($productIds)) {
            $productByIds = $this->getProduct($productIds);
        }
        if (!empty($categoryIds)) {
            $categoryByIds = $this->getCategory($categoryIds);
        }
        $this->decorResult($result, $productByIds, $categoryByIds);
        $pageId = $filter['page_id'];
        $pageSize = $filter['page_size'];
        unset($filter['page_id']);
        unset($filter['page_size']);
        $filter['metric'] = 'count';
        $total = $this->getResult($filter);
        $pagesCount = $this->recordsCountToPagesCount($total, $pageSize);
        $response = array(
            "status" => 'successful',
            "result" => $result,
            'pagesCount' => $pagesCount,
            'pageId' => $pageId
        );
        return response()->json($response);
    }

    private function decorResult(&$items, $productByIds, $categoryByIds) {
        foreach ($items as $item) {
            foreach ($item->items as $it) {
                if ($item->apply_type == 'category') {
                    if (isset($categoryByIds[$it->reference_id])) {
                        $it->category_id = $categoryByIds[$it->reference_id]['id'];
                        $it->name = $categoryByIds[$it->reference_id]['name'];
                        $it->tree_name = $categoryByIds[$it->reference_id]['tree_name'];
                    }
                }
                if ($item->apply_type == 'product') {
                    if (isset($productByIds[$it->reference_id])) {
                        $it->product_id = $productByIds[$it->reference_id]->id;
                        $it->name = $productByIds[$it->reference_id]->name;
                        $it->image_url = $productByIds[$it->reference_id]->image_url;
                    }
                }
            }
        }
    }

    private function getProduct($ids) {
        $items = Product::whereIn('id', $ids)
                        ->get(['id', 'name', 'image_url']);
        $retVal = [];
        foreach ($items as $item) {
            $retVal[$item->id] = $item;
        }
        return $retVal;
    }

    private function getCategory($ids) {
        $items = Category::whereIn('id', $ids)
                        ->get(['id', 'name', 'breadcrumb']);
        $retVal = [];
        foreach ($items as $item) {
            $breadcrumb = json_decode($item->breadcrumb);
            $treeTitle = $item->name;
            $treeTitles = [];
            foreach ($breadcrumb as $ctg) {
                $treeTitles[] = $ctg->name;
            }
            if (!empty($treeTitles)) {
                $treeTitle = implode(' / ', $treeTitles);
            }
            $newItem = [
                'id' => $item->id, 
                'name' => $item->name,
                'tree_name' => $treeTitle
            ];
            $retVal[$item->id] = $newItem;
        }
        return $retVal;
    }

    private function buildFilter($request) {
        $retVal = [];
        if ($request->has('keyword')) {
            $retVal['keyword'] = $request->input('keyword');
        }
        if ($request->has('keyword1')) {
            $retVal['keyword1'] = $request->input('keyword1');
        }
        if ($request->has('type')) {
            $retVal['type'] = $request->input('type');
        }
        if ($request->has('ignore')) {
            $retVal['ignore'] = $request->input('ignore');
        }
        $retVal['page_size'] = 20;
        $retVal['page_id'] = $request->has('page_id') ? $request->input('page_id') : 0;
        return $retVal;
    }

    private function getResult($filter) {
        $query = Discount::query();
        if (isset($filter['keyword']) && $filter['keyword'] != '') {
            $query->where('title', 'LIKE', '%' . $filter['keyword'] . '%');
            $query->orWhere('code', 'LIKE', '%' . $filter['keyword'] . '%');
        }
        if (isset($filter['keyword1']) && $filter['keyword1'] != '') {
            $query->where('title', 'LIKE', '%' . $filter['keyword1'] . '%');
            $query->orWhere('code', 'LIKE', '%' . $filter['keyword1'] . '%');
        }
        if (isset($filter['ignore']) && $filter['ignore'] != '') {
            $query->where('title', 'NOT LIKE', '%' . $filter['ignore'] . '%');
        }
        if (isset($filter['type']) && $filter['type'] != '') {
            $query->where('type', '=', $filter['type']);
        }
        $query->orderBy('id', 'DESC');
        $query->with(['items']);
        if (isset($filter['metric']) && $filter['metric'] == 'count') {
            return $query->count();
        } else {
            if (isset($filter['page_size']) && isset($filter['page_id'])) {
                $query->forPage($filter['page_id'] + 1, $filter['page_size']);
            }
            return $query->get();
        }
    }

    private function recordsCountToPagesCount($recordsCount, $pageSize)
    {
        $retVal = (int)($recordsCount / $pageSize);
        if ($recordsCount % $pageSize > 0) {
            $retVal++;
        }
        //return
        return $retVal;
    }

    public function store(Request $request, $id = null) {
        $response = [
            'status' => 'fail'
        ];
        try {
            $data = $this->buildData($request);
            $email = $request->has('email') ? $request->get('email') : null;
            $log = [
                'actor_type' => null,
                'actor_email' => $email,
                'target_type' => 'DISCOUNT',
                'created_at' => new \DateTime()
            ];
            if (isset($data['code']) && $data['code'] != '') {
                $notValids = $this->checkDuplicateCode($data);
                if ($notValids) {
                    $response['message'] = 'Mã ' . $data['code'] . ' đã tồn tại trên hệ thống, vui lòng nhập một mã khác';
                    return response()->json($response);
                }
            }
            if ($request->has('apply_items')) {
                $applyType = 'sản phẩm';
                if ($request->input('apply_type') == 'category') {
                    $applyType = 'danh mục';
                }
                $notValids = $this->checkItemUsage($data['type'], $request);
                if (!empty($notValids['items'])) {
                    $response['message'] = 'Các ' . $applyType . ': ' . implode(', ', $notValids['items']) . ' đã được sử dụng ở loại giảm giá: ' . implode(', ', $notValids['discounts']) . '. Vui lòng chọn lại.';
                    return response()->json($response);
                }
                $minimumRequireNotValids = $this->checkItemMinimumRequire($data['type'], $request);
                if (!empty($minimumRequireNotValids['items'])) {
                    $response['message'] = 'Các ' . $applyType . ': ' . implode(', ', $minimumRequireNotValids['items']) . ' đã được sử dụng ở chương trình có yêu cầu tối thiểu: ' . implode(', ', $minimumRequireNotValids['discounts']) . '. Vui lòng chọn lại.';
                    return response()->json($response);
                }
                
            }
            if (isset($data['id']) && !empty($data['id'])) {
                $discount = Discount::find($data['id']);
                $log['event_type'] = 'UPDATE';
            } else {
                $discount = Discount::create();
                $log['event_type'] = 'CREATE';
            }
            
            foreach($data as $key => $value) {
                $discount->{$key} = $value;
            }
            $discount->save();
            if ($discount && $request->has('apply_items')) {
                $this->saveItem($discount, $request->input('apply_items'));
            }

            $log['target_id'] = $discount->id;
            $log['data'] = json_encode($request->all());
            \DB::table("log")->insert($log);
            $response['status'] = 'successful';
            $response['result'] = $discount;
        } catch (\Exception $ex) {
            $response['message'] = 'Error: ' . $ex->getMessage() . '. Line: ' . $ex->getLine();
        }
        return response()->json($response);
    }

    private function checkDuplicateCode($data) {
        $query = Discount::where('code', '=', $data['code']);
        if (isset($data['id']) && $data['id'] != '' && $data['id'] != null) {
            $query->where('id', '!=', $data['id']);
        }
        return $query->first();
    }

    private function checkItemMinimumRequire($type, $request) {
        $items = $request->input('apply_items');
        $applyType = $request->input('apply_type');
        $minimumRequireType = $request->input('minimum_require_type');
        $retVal = [];
        $ids = [];
        foreach($items as $item) {
            $ids[] = $item['id'];
        }
        $query = \DB::table('discount as d')
                            ->leftJoin('discount_item as d_item', 'd_item.discount_id', '=', 'd.id')
                            ->where('d.status', '=', 'active')
                            ->where('d.type', '=', $type)
                            ->where('d.apply_type', '=', $applyType)
                            ->where('d.minimum_require_type', '!=', $minimumRequireType)
                            ->whereIn('d_item.reference_id', $ids);
        if ($request->has('id') && $request->input('id') != null && $request->input('id') != '') {
            $query->where('d.id', '!=', $request->input('id'));
        }
        $items = $query->get([ 'd.title', 'd_item.reference_id']);
        $retVal = $this->getReference($items, $applyType);
        return $retVal;
    }

    private function checkItemUsage($type, $request) {
        $items = $request->input('apply_items');
        $applyType = $request->input('apply_type');
        $retVal = [];
        $ids = [];
        foreach($items as $item) {
            $ids[] = $item['id'];
        }
        $query = \DB::table('discount as d')
                            ->leftJoin('discount_item as d_item', 'd_item.discount_id', '=', 'd.id')
                            ->where('d.status', '=', 'active')
                            ->where('d.type', '!=', $type)
                            ->where('d.apply_type', '=', $applyType)
                            ->whereIn('d_item.reference_id', $ids);
        if ($request->has('id') && $request->input('id') != null && $request->input('id') != '') {
            $query->where('d.id', '!=', $request->input('id'));
        }
        $items = $query->get(['d.title', 'd_item.reference_id']);
        $retVal = $this->getReference($items, $applyType);
        return $retVal;
    }

    private function getReference($items, $applyType) {
        $retVal = [];
        $idExists = [];
        $discountExists = [];
        if (!empty($items)) {
            foreach($items as $item) {
                $idExists[] = $item->reference_id;
                if (!in_array($item->title, $discountExists)) {
                    $discountExists[] = $item->title;
                }
            }
        }
        $itemExists = [];
        if (!empty($idExists)) {
            if ($applyType == 'product') {
                $itemExists = Product::whereIn('id', $idExists)
                                    ->pluck('name')->toArray();
            }
            if ($applyType == 'category') {
                $itemExists = Category::whereIn('id', $idExists)
                                    ->pluck('name')->toArray();
            }
            
        }
        $retVal = [
            'discounts' => $discountExists,
            'items' => $itemExists
        ];
        return $retVal;
    }

    private function saveItem($discount, $items) {
        DiscountItem::where('discount_id', '=', $discount->id)->delete();
        foreach($items as $item) {
            $itemData = [
                'discount_id' => $discount->id,
                'reference_id' => $item['id']
            ];
            if (isset($item['get_id']) && $item['get_id'] != '') {
                $itemData['get_id'] = $item['get_id'];
            }
            $discountItem = DiscountItem::create();
            foreach($itemData as $key => $value) {
                $discountItem->{$key} = $value;
            }
            $discountItem->save();
        }
    }

    private function buildData($request) {
        $columns = ['title', 'code', 'status', 'is_show_on_site', 'type', 'value', 'minimum_require_type', 'minimum_require_value', 
                    'apply_type', 'limited_number_uses', 'max_discount_value', "is_limit_by_user", "is_show_promo_code"];

        $retVal = []; 
        foreach($columns as $column) {
            $retVal[$column] = ($request->has($column) && $request->input($column) != '') ? $request->input($column) : '';
        }
        if ($request->has('id') && $request->input('id') != '') {
            $retVal['id'] = $request->input('id');
        }
        if ($request->has('end_time') && $request->input('end_time') != '') {
            $endTime = \DateTime::createFromFormat('d/m/Y', $request->input('end_time'));
            if ($endTime) {
                $endTime->setTime(23,59,59);
                $retVal['end_time'] = $endTime;
            }
        } else {
            $retVal['end_time'] = null;
        }
        if (isset($retVal['code']) && $retVal['code'] != '') {
            $retVal['code'] = strtoupper($retVal['code']);
        }
        return $retVal;
    }

    public function classifyData(Request $request)
    {
        $retVal = ['affectedRow' => 0];
        set_time_limit(3600);
        $file = $request->file('csv_file');
        $inputFileType = \PHPExcel_IOFactory::identify($file);
        $objReader = \PHPExcel_IOFactory::createReader($inputFileType);
        $objPHPExcel = $objReader->load($file);

        // choose page
        $sheet = $objPHPExcel->setActiveSheetIndex(0);
        // get amount of rows
        $totalRow = 0;
        foreach ($sheet->getRowIterator() as $row) {
            $cellIterator = $row->getCellIterator();
            $cellIterator->setIterateOnlyExistingCells(true); // Loop only non-null cells
            $cells = [];
            foreach ($cellIterator as $cell) {
                $cells[] = $cell->getValue();
            }
            if (array_filter($cells)) { // Check if there are non-empty cells in the row
                $totalRow++;
            }
        }

        $data = [];

        // xlsx file type
        for ($i = 2; $i <= $totalRow; $i++) {
            $startTimeFormat = str_replace('/', '-', $sheet->getCellByColumnAndRow(3, $i)->getValue());
            $endTimeFormat = str_replace('/', '-', $sheet->getCellByColumnAndRow(4, $i)->getValue());
            $data[] = [
                'name' => $sheet->getCellByColumnAndRow(0, $i)->getValue(),
                'title' => $sheet->getCellByColumnAndRow(1, $i)->getValue(),
                'sale_percent' => $sheet->getCellByColumnAndRow(2, $i)->getValue(),
                'start_at' => date('Y-m-d H:i:s', strtotime($startTimeFormat)),
                'end_at' => date('Y-m-d H:i:s', strtotime($endTimeFormat)),
                'status' => 'ACTIVE',
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];
        }

        if (!empty($data)) {
            $chunks = array_chunk($data, self::BATCH_SIZE);
            foreach ($chunks as $value) {
                \DB::table('flash_sale')->insert($value);
            }
            $retVal = [
                'affectedRow' => count($data),
                'status' => 'successful'
            ];
        }
        return response()->json($retVal);
    }

    public function storeDataImported(Request $request)
    {
        $data = $request->get('data');

        if (!empty($data)) {
            \DB::table('flash_sale')->insert($data);
            return response()->json(['status' => 'successful']);
        }
        return $data;
    }
    public function flashSale() {
        return view('discount::flash-sale.index');
    }
}
