<?php

namespace Modules\Discount\Controllers;

use Illuminate\Http\Request;
use Modules\Discount\Controllers\Controller;
use Module;
use DB;
use Modules\Discount\Models\DiscountItem;
use Modules\Discount\Models\Product;
use Modules\Discount\Models\Category;
use App\Helpers\ApiClient;
use Modules\Discount\Models\Customer;
use Modules\Discount\Models\Order;
use Modules\Discount\Models\Discount;
use Modules\Discount\Models\ProductNCategory;

class HomeController extends Controller
{
    const BUY_OVER_AMOUNT = "Buy over #amount to get #off off";
    const BUY_OVER_AMOUNT_WITH_CODE = "Buy over #amount to get #off off with code #code";
    const BUY_OVER_QUANTITY = "Buy #quantity to get #off off";
    const BUY_OVER_QUANTITY_WITH_CODE = "Buy #quantity to get #off off with code #code";
    const SALE_OFF = "Sale #off off";
    const SALE_OFF_WITH_CODE = "Sale #off off with code #code";
    const APPLY_WITH_CODE = "Apply #off OFF with Code #code";
    const ADD_MORE = "Add #amount more to get #off off";
    const ADD_MORE_WITH_CODE = "Add #amount more to get #off off with code #code";

    public function __construct()
    {
    }
    public function index(Request $request)
    {
        return view('discount::discount.index');
    }

    public function category(Request $request)
    {
        $keyword = $request->has('keyword') ? $request->input('keyword') : null;
        $response = [
            'status' => 'fail'
        ];
        if ($keyword != null) {
            $items = Category::where('type', '=', 'PRODUCT')
                ->where('is_hidden', '!=', 1)
                ->where('name', 'LIKE', '%' . $keyword . '%')
                ->get(['id', 'name', 'breadcrumb']);
            $results = [];
            foreach ($items as $item) {
                $breadcrumb = json_decode($item->breadcrumb);
                $treeTitle = $item->name;
                $treeTitles = [];
                if (!empty($breadcrumb)) {
                    foreach ($breadcrumb as $ctg) {
                        $treeTitles[] = $ctg->name;
                    }
                    if (!empty($treeTitles)) {
                        $treeTitle = implode(' / ', $treeTitles);
                    }
                }
                $newItem = [
                    'id' => $item->id,
                    'name' => $item->name,
                    'tree_name' => $treeTitle
                ];
                $results[] = $newItem;
            }
            $response['status'] = 'successful';
            $response['result'] = $results;
        }
        return response()->json($response);
    }

    public function product(Request $request)
    {
        $keyword = $request->has('keyword') ? $request->input('keyword') : null;
        $response = [
            'status' => 'fail'
        ];
        if ($keyword != null) {
            $items = Product::where('name', 'LIKE', '%' . $keyword . '%')
                        ->where('status', '!=', 'PENDING')
                        ->limit(10)
                        ->get(['id', 'name', 'image_url']);
            $response['status'] = 'successful';
            $response['result'] = $items;
        }
        return response()->json($response);
    }

    public function getProductDiscountInfo(Request $request)
    {
        $response = [
            "status" => "fail"
        ];
        if ($request->has('product_id')) {
            $productId = $request->input('product_id');
            $isCustomize = \DB::table('product_meta')->where('product_id', '=', $productId)
                ->where('key', '=', 'is_custom_design')
                ->where('value', '=', '1')
                ->exists();
            if ($isCustomize) {
                return $response;
            }
            $productDiscountByProduct = $this->getDiscountByProduct($productId);
            $productDiscountByCategory = $this->getDiscountByCategory($productId);
            $result = [];
            $type = "";
            if (count($productDiscountByProduct) > 0) {
                list($result, $type) = $this->buildDiscountText($productDiscountByProduct);
            } else if (count($productDiscountByCategory) > 0) {
                list($result, $type) = $this->buildDiscountText($productDiscountByCategory);
            } else {
                $getProductDiscountByAll = $this->getDiscountByAll();
                if (count($getProductDiscountByAll)) {
                    list($result, $type) = $this->buildDiscountText($getProductDiscountByAll);
                }
            }
            $response = [
                "status" => "successful",
                "result" => $result,
                "type" => $type,
            ];
        }
        return response()->json($response);
    }

    private function getDiscountByProduct($productId)
    {
        $productDiscount = DiscountItem::where("reference_id", $productId)
            ->join("discount", "discount.id", "=", "discount_id")
            ->where("discount.status", "active")
            ->where("discount.apply_type", "product")
            ->where("is_show_on_site", 1)
            ->where(function ($q) {
                $q->where("end_time", ">", date('Y-m-d h:i:s'));
                $q->orWhereNull("end_time");
            })
            ->orderBy("minimum_require_value", "ASC")
            ->get(['discount.*', 'discount_item.reference_id', 'discount_item.discount_id', 'discount_item.get_id']);
        return $productDiscount;
    }

    private function getDiscountByCategory($productId)
    {
        $product = Product::with("categories")->where('id', $productId)->first();
        $discount = [];
        foreach ($product->categories as $key => $category) {
            $productDiscount = DiscountItem::where("reference_id", $category->id)
                ->join("discount", "discount.id", "=", "discount_id")
                ->where("discount.status", "active")
                ->where("discount.apply_type", "category")
                ->where("is_show_on_site", 1)
                ->where(function ($q) {
                    $q->where("end_time", ">", date('Y-m-d h:i:s'));
                    $q->orWhereNull("end_time");
                })
                ->orderBy("minimum_require_value", "ASC")
                ->get(['discount.*', 'discount_item.reference_id', 'discount_item.discount_id', 'discount_item.get_id']);
            if (count($productDiscount) > 0) {
                $discount = $productDiscount;
            }
        }

        return $discount;
    }

    private function getDiscountByAll() {
        $productDiscount = Discount::where("status", "active")
            ->where("is_show_on_site", 1)
            ->where("apply_type", "all")
            ->where(function ($q) {
                $q->where("end_time", ">", date('Y-m-d h:i:s'));
                $q->orWhereNull("end_time");
            })
            ->orderBy("minimum_require_value", "ASC")
            ->get();
        return $productDiscount;
    }

    private function buildDiscountText($productDiscount)
    {
        $type = "";
        $typeRequire = "";
        if (count($productDiscount) > 0) {
            $type = $productDiscount[0]->type;
            $typeRequire = $productDiscount[0]->minimum_require_type;
        }
        $discountItem = [];
        if ($type == 'percent') {
            foreach ($productDiscount as $key => $item) {
                $value = 0;
                if ($item->minimum_require_type == 'amount') {
                    if ($item->code && $item->code != "") {
                        $discountMessage = str_replace("#amount", formatPrice($item->minimum_require_value), __(self::BUY_OVER_AMOUNT_WITH_CODE));
                        $discountMessage = str_replace("#off", round($item->value) . '%', $discountMessage);
                        $discountMessage = str_replace("#code", $item->code, $discountMessage);
                    } else {
                        $discountMessage = str_replace("#amount", formatPrice($item->minimum_require_value), __(self::BUY_OVER_AMOUNT));
                        $discountMessage = str_replace("#off", round($item->value) . '%', $discountMessage);
                    }
                    $value = formatPrice($item->minimum_require_value);
                } else if ($item->minimum_require_type == 'quantity') {
                    if ($item->code && $item->code != "") {
                        $discountMessage = str_replace("#quantity", round($item->minimum_require_value), __(self::BUY_OVER_QUANTITY_WITH_CODE));
                        $discountMessage = str_replace("#off", round($item->value) . '%', $discountMessage);
                        $discountMessage = str_replace("#code", $item->code, $discountMessage);
                    } else {
                        $discountMessage = str_replace("#quantity", round($item->minimum_require_value), __(self::BUY_OVER_QUANTITY));
                        $discountMessage = str_replace("#off", round($item->value) . '%', $discountMessage);
                    }
                    $value = round($item->minimum_require_value);
                } else if ($item->minimum_require_type == 'none') {
                    if ($item->code && $item->code != "") {
                        $discountMessage = str_replace("#off", round($item->value) . '%', __(self::SALE_OFF_WITH_CODE));
                        $discountMessage = str_replace("#code", $item->code, $discountMessage);
                    } else {
                        $discountMessage = str_replace("#off", round($item->value) . '%', __(self::SALE_OFF));
                    }
                }
                $discountItem[] = [
                    'value' => $value,
                    'get' => $item->value,
                    'message' => $discountMessage,
                    'code' => $item->code,
                ];
            }
        } else if ($type == 'fixed_amount') {
            foreach ($productDiscount as $key => $item) {
                $value = 0;
                if ($item->minimum_require_type == 'amount') {
                    if ($item->code && $item->code != "") {
                        $discountMessage = str_replace("#amount", formatPrice($item->minimum_require_value), __(self::BUY_OVER_AMOUNT_WITH_CODE));
                        $discountMessage = str_replace("#off", formatPrice($item->value), $discountMessage);
                        $discountMessage = str_replace("#code", $item->code, $discountMessage);
                    } else {
                        $discountMessage = str_replace("#amount", formatPrice($item->minimum_require_value), __(self::BUY_OVER_AMOUNT));
                        $discountMessage = str_replace("#off", formatPrice($item->value), $discountMessage);
                    }
                    $value = formatPrice($item->minimum_require_value);
                } else if ($item->minimum_require_type == 'quantity') {
                    $value = round($item->minimum_require_value);
                    if ($item->code && $item->code != "") {
                        $discountMessage = str_replace("#quantity", round($item->minimum_require_value), __(self::BUY_OVER_QUANTITY_WITH_CODE));
                        $discountMessage = str_replace("#off", formatPrice($item->value), $discountMessage);
                        $discountMessage = str_replace("#code", $item->code, $discountMessage);
                    } else {
                        $discountMessage = str_replace("#quantity", round($item->minimum_require_value), __(self::BUY_OVER_QUANTITY));
                        $discountMessage = str_replace("#off", formatPrice($item->value), $discountMessage);
                    }
                } else if ($item->minimum_require_type == 'none') {
                    if ($item->code && $item->code != "") {
                        $discountMessage = str_replace("#off", formatPrice($item->value), __(self::SALE_OFF_WITH_CODE));
                        $discountMessage = str_replace("#code", $item->code, $discountMessage);
                    } else {
                        $discountMessage = str_replace("#off", formatPrice($item->value), __(self::SALE_OFF));
                    }
                }
                $discountItem[] = [
                    'value' => $value,
                    'message' => $discountMessage,
                    'code' => $item->code,
                ];
            }
        }
        return [$discountItem, $typeRequire];
    }

    public function getDiscountByCartItem(Request $request)
    {
        $cartItems = $this->getCartItems();
        $listProduct = [];
        foreach ($cartItems as $key => $value) {
            if (!array_key_exists($value['product_id'], $listProduct)) {
                $listProduct[$value['product_id']] = [
                    "product_id" => $value['product_id'],
                    "itemAmount" => round($value['price'] * $value['quantity'], 2),
                    "quantity" => $value['quantity']
                ];
            } else {
                $listProduct[$value['product_id']]["itemAmount"] = round($listProduct[$value['product_id']]["itemAmount"] + ($value['price'] * $value['quantity']), 2);
                $listProduct[$value['product_id']]["quantity"] += $value['quantity'];
            }
        }
        foreach ($listProduct as $key => $item) {
            $listDiscount = [];
            $validDiscount = [];
            $listDiscount = $this->getDiscountByProduct($item['product_id']);
            if (count($listDiscount) > 0) {
                list ($message, $title, $validDiscount) = $this->getBuyMoreText($listDiscount, $item, $listProduct);
            } else {
                $listDiscount = $this->getDiscountByCategory($item['product_id']);
                if (count($listDiscount) > 0) {
                    list ($message, $title, $validDiscount) = $this->getBuyMoreText($listDiscount, $item, $listProduct);
                } else {
                    $listDiscount = $this->getDiscountByAll();
                    if (count($listDiscount) > 0) {
                        list ($message, $title, $validDiscount) = $this->getBuyMoreText($listDiscount, $item, $listProduct);
                    }
                }
            }
            $discount = 0;

            if (!empty($validDiscount) && ($validDiscount->code == null || $validDiscount->code == "")) {
                if ($validDiscount->type == 'fixed_amount') {
                    $discount = $validDiscount->value;
                } else {
                    $discount = round(($item['itemAmount'] * $validDiscount->value) / 100, 2);
                    if ($validDiscount->max_discount_value && $validDiscount->max_discount_value != 0 && $discount > $validDiscount->max_discount_value) {
                        $discount = $validDiscount->max_discount_value;
                    }
                }
                $listProduct[$key]['discount_type'] = $validDiscount->type;
                $listProduct[$key]['discount_id'] = $validDiscount->id;
                $listProduct[$key]['discount_code'] = $validDiscount->code;
            }
            $listProduct[$key]['total_discount'] = $discount;
        }
        $listUseDiscount = [];
        $result = [];
        foreach ($listProduct as $key => $value) {
            if (isset($value['discount_type']) && $value['discount_type'] == "percent") {
                $result[$key] = $value;
            }
            if (isset($value['discount_id']) && !in_array($value['discount_id'], $listUseDiscount) && $value['discount_type'] == "fixed_amount") {
                $result[$key] = $value;
                $listUseDiscount[] = $value['discount_id'];
            }
        }
        $response = [
            "status" => "successful",
            "result" => $result,
        ];
        return response()->json($response);
    }

    private function getCartItems()
    {
        $cartItems = [];
        $query = ApiClient::buildClient("cart/get-cart-item");
        $token = getFingerprint();
        $query->addField("token", $token);
        $result = $query->get();
        if (isset($result['status']) && $result['status'] == 'successful') {
            $cartItems = $result['items'];
        }
        return $cartItems;
    }

    function getDiscountInfoCartItem() {
        $response = [
            'status' => 'fail'
        ];
        $cartItems = $this->getCartItems();
        $listProduct = [];
        foreach ($cartItems as $key => $value) {
            if (!array_key_exists($value['product_id'], $listProduct)) {
                $listProduct[$value["product_id"]] = [
                    "product_id" => $value['product_id'],
                    "quantity" => $value['quantity'],
                    "itemAmount" => round($value['price'] * $value['quantity'], 2),
                ];
            } else {
                $listProduct[$value["product_id"]]["quantity"] += $value['quantity'];
                $listProduct[$value['product_id']]["itemAmount"] = round($listProduct[$value['product_id']]["itemAmount"] + ($value['price'] * $value['quantity']), 2);
            }
        }
        $dataResponse = [];
        foreach ($listProduct as $item) {
            $listDiscount = [];
            $validDiscount = [];
            $listDiscount = $this->getDiscountByProduct($item['product_id']);
            $message = "";
            $title = "";
            if (count($listDiscount) > 0) {
                list ($message, $title, $validDiscount) = $this->getBuyMoreText($listDiscount, $item, $listProduct);
            } else {
                $listDiscount = $this->getDiscountByCategory($item['product_id']);
                if (count($listDiscount) > 0) {
                    list ($message, $title, $validDiscount) = $this->getBuyMoreText($listDiscount, $item, $listProduct);
                } else {
                    $listDiscount = $this->getDiscountByAll();
                    if (count($listDiscount) > 0) {
                        list ($message, $title, $validDiscount) = $this->getBuyMoreText($listDiscount, $item, $listProduct);
                    }
                }
            }
            $result = [
                "message" => $message,
                "discount_title" => $title,
                "product_id" => $item['product_id']
            ];
            if ($validDiscount && $validDiscount->id) {
                if ($validDiscount->code && $validDiscount->code != "") {
                    if ($validDiscount->type == "percent") {
                        $result["messageApply"] = str_replace("#off", round($validDiscount->value) . '%', __(__(self::APPLY_WITH_CODE)));
                        $result["messageApply"] = str_replace("#code", $validDiscount->code, $result["messageApply"]);
                    } else if ($validDiscount->type == "fixed_amount") {
                        $result["messageApply"] = str_replace("#off", formatprice($validDiscount->value), __(self::APPLY_WITH_CODE));
                        $result["messageApply"] = str_replace("#code", $validDiscount->code, $result["messageApply"]);
                    }
                    $result["code"] = $validDiscount->code;
                }
            }
            $dataResponse[] = $result;
        }
        $response = [
            'status' => "successful",
            "result" => $dataResponse
        ];
        return response()->json($response);
    }

    private function getBuyMoreText($discounts, $item, $listProduct) {
        $type = "";
        if (count($discounts) > 0) {
            $type = $discounts[0]->type;
        }
        $message = "";
        $title = "";
        $validDiscount = [];
        for ($i = 0; $i < count($discounts); $i++) {
            $quantity = $item['quantity'];
            $totalAmount = $item["itemAmount"];
            $value = $discounts[$i];
            foreach ($listProduct as $key => $product) {
                if ($product['product_id'] != $item['product_id']) {
                    if ($value->apply_type == "category") {
                        $isExistsCustomDiscount = DiscountItem::join("discount", "discount.id", "=", "discount_id")
                            ->where("discount.status", "active")
                            ->where("reference_id", $product['product_id'])
                            ->where("discount.apply_type", "product")
                            ->exists();
                        if (!$isExistsCustomDiscount) {
                            $isExists = ProductNCategory::where("product_id", $product['product_id'])
                                ->where("category_id", $value->reference_id)
                                ->exists();
                            if ($isExists) {
                                $totalAmount += $product['itemAmount'];
                                $quantity += $product['quantity'];
                            }
                        }
                    } else if ($value->apply_type == "product") {
                        $isExists = DiscountItem::join("discount", "discount.id", "=", "discount_id")
                            ->where("discount.status", "active")
                            ->where("discount_id", $value->id)
                            ->where("reference_id", $product['product_id'])
                            ->exists();
                        if ($isExists) {
                            $totalAmount += $product['itemAmount'];
                            $quantity += $product['quantity'];
                        }
                    } else if ($value->apply_type == "all") {
                        $totalAmount += $product['itemAmount'];
                        $quantity += $product['quantity'];
                    }
                }
            }
            $totalAmount = round($totalAmount, 2);
            if ($value->minimum_require_type == 'amount' && $value->minimum_require_value <= $totalAmount) {
                $validDiscount = $value;
            } else if ($value->minimum_require_type == 'quantity' && $value->minimum_require_value <= $quantity) {
                $validDiscount = $value;
            } else if ($value->minimum_require_type == 'none') {
                $validDiscount = $value;
            }

            if ($type == "fixed_amount") {
                if ($value->minimum_require_type == 'amount' && $value->minimum_require_value > $totalAmount) {
                    if ($value->code && $value->code != "") {                 
                        $message = str_replace("#amount", formatPrice($value->minimum_require_value - $totalAmount), __(self::ADD_MORE_WITH_CODE));
                        $message = str_replace("#off", formatPrice($value->value), $message);
                        $message = str_replace("#code", $value->code, $message);
                    } else {
                        $message = str_replace("#amount", formatPrice($value->minimum_require_value - $totalAmount), __(self::ADD_MORE));
                        $message = str_replace("#off", formatPrice($value->value), $message);
                    }
                    $title = $value->title;
                    break;
                } else if ($value->minimum_require_type == 'quantity' && $value->minimum_require_value > $quantity) {
                    if ($value->code && $value->code != "") {
                        $message = str_replace("#amount", round($value->minimum_require_value - $quantity), __(self::ADD_MORE_WITH_CODE));
                        $message = str_replace("#off", formatPrice($value->value), $message);
                        $message = str_replace("#code", $value->code, $message);
                    } else {
                        $message = str_replace("#amount", formatPrice($value->minimum_require_value - $quantity), __(self::ADD_MORE));
                        $message = str_replace("#off", formatPrice($value->value), $message);
                    }
                    $title = $value->title;
                    break;
                }
            } else if ($type == "percent") {
                if ($value->minimum_require_type == 'amount' && $value->minimum_require_value > $totalAmount) {
                    if ($value->code && $value->code != "") {
                        $message = str_replace("#amount", formatPrice($value->minimum_require_value - $totalAmount), __(self::ADD_MORE_WITH_CODE));
                        $message = str_replace("#off", round($value->value) . "%", $message);
                        $message = str_replace("#code", $value->code, $message);
                    } else {
                        $message = str_replace("#amount", formatPrice($value->minimum_require_value - $totalAmount), __(self::ADD_MORE));
                        $message = str_replace("#off", round($value->value) . '%', $message);
                    }
                    $title = $value->title;
                    break;
                } else if ($value->minimum_require_type == 'quantity' && $value->minimum_require_value > $quantity) {
                    if ($value->code && $value->code != "") {
                        $message = str_replace("#amount", round($value->minimum_require_value - $quantity), __(self::ADD_MORE_WITH_CODE));
                        $message = str_replace("#off", round($value->value) . '%', $message);
                        $message = str_replace("#code", $value->code, $message);
                    } else {
                        $message = str_replace("#amount", formatPrice($value->minimum_require_value - $quantity), __(self::ADD_MORE));
                        $message = str_replace("#off", round($value->value) . '%', $message);
                    }
                    $title = $value->title;
                    break;
                }
            }
        }
        
        return [$message, $title, $validDiscount];
    }

    public function generateDiscount(Request $request) {
        $response = [
            "status" => "fail"
        ];
        $today = new \DateTime();
        $promotions = \DB::table("promotion_code")->where(function ($query) use ($today) {
            $query->where("expired_at", ">", $today);
            $query->orWhereNull("expired_at");
        })->where("status", "enable")->get();

        foreach ($promotions as $key => $value) {
            $isExists = Discount::where("code", $value->code)->exists();
            if (!$isExists) {
                $dataUpdate = [
                    'title' => 'discount',
                    'status' => "active",
                    'code' => $value->code,
                    'limited_number_uses' => $value->limit_number,
                    'number_uses' => $value->used_number,
                    'apply_type' => 'all',
                    'start_time' => $value->created_at,
                    'end_time' => $value->expired_at,
                    'max_discount_value' => $value->max_discount,
                    'created_at' => $value->created_at,
                    'updated_at' => $value->updated_at,
                ];
                if ($value->discount && $value->discount > 0) {
                    $dataUpdate["type"] = 'fixed_amount';
                    $dataUpdate["value"] = $value->discount;
                } else {
                    $dataUpdate["type"] = 'percent';
                    $dataUpdate["value"] = $value->discount_percent;
                }
                if ($value->min_amount && $value->min_amount > 0) {
                    $dataUpdate["minimum_require_type"] = 'amount';
                    $dataUpdate["minimum_require_value"] = $value->min_amount;
                } else {
                    $dataUpdate["minimum_require_type"] = 'none';
                }
                Discount::insert($dataUpdate);
            }
        }
        $response = [
            "status" => "successful",
        ];
        return response()->json($response);
    }

    public function checkPromotionCode(Request $request) {
        $today = new \DateTime();
        $todayDate = $today->format("Y-m-d H:i:s");
        $response = [
            "status" => "fail",
            "message" => __('Enter a valid promotion code!'),
        ];
        $code = $request->input("code");
        if ($code) {
            $amount = $request->input("amount", 0);
            $quantity = $request->input("quantity", 0);
            $phone = preg_replace('/[^0-9]/', '', $request->input('phone'));
            $products = $request->input("products", []);
            $discountPromotion = 0;
            $discount = Discount::where("code", $code)
                            ->where("status", "active")
                            ->where(function ($q) {
                                $q->where("end_time", ">", date('Y-m-d h:i:s'));
                                $q->orWhereNull("end_time");
                            })
                            ->first();
            if ($discount &&
            (!$discount->start_time || $discount->start_time <= $todayDate) &&
            (!$discount->end_time || $discount->end_time > $todayDate)
            ) {
                $checkByProduct = false;
                if ($discount->apply_type == 'all') {
                    $checkByProduct = true;
                } else {
                    $amount = 0;
                    $quantity = 0;
                    $validProducts = [];
                    foreach ($products as $product) {
                        $isExists = false;
                        if ($discount->apply_type == 'product') {
                            $isExists = DiscountItem::where("discount_id", $discount->id)->where("reference_id", $product['id'])->exists();
                        } else if ($discount->apply_type == 'category') {
                            $isExists = DiscountItem::where("discount_id", $discount->id)
                                ->join("product_n_category", "product_n_category.category_id", "=", "discount_item.reference_id")
                                ->where("product_id", $product['id'])
                                ->exists();
                        }
                        if ($isExists) {
                            $validProducts[] = $product;
                        }
                    }

                    if (count($validProducts) > 0) {
                        $checkByProduct = true;
                        foreach ($validProducts as $product) {
                            $amount += $product['price'];
                            $quantity += $product['quantity'];
                        }
                    }
                }
                if ($checkByProduct) {
                    $checkByAmount = false;
                    if (($discount->minimum_require_type == 'amount' && $amount >= $discount->minimum_require_value) ||
                        ($discount->minimum_require_type == 'quantity' && $quantity >= $discount->minimum_require_value) ||
                        ($discount->minimum_require_type == 'none')
                    ) {
                        $checkByAmount = true;
                    }
                    if ($checkByAmount) {
                        $checkByLimit = true;
                        if ($phone && $discount->is_limit_by_user == 1) {
                            $customer = Customer::where('phone', $phone)->orderBy("id", "DESC")->first();
                            if ($customer) {
                                $count = Order::where("promotion_code", $discount->code)->where("payment_status", "PAID")->where("customer_id", $customer->id)->count();
                                if ($count >= $discount->limited_number_uses && $discount->limited_number_uses != 0) {
                                    $checkByLimit = false;
                                }
                            }
                        }
                        if ($checkByLimit) {
                            $discountPercent = 0;
                            $maxDiscount = 0;
                            $discountType = "normal";
                            $isValid = false;
                            if ($discount->type == 'fixed_amount') {
                                $discountPromotion = $discount->value;
                                $isValid = true;
                            } else if ($discount->type == 'percent') {
                                $discountPromotion = round(($amount * $discount->value) / 100, 2);
                                $discountPercent = $discount->value;
                                $maxDiscount = $discount->max_discount_value;
                                if ($discount->max_discount_value && $discount->max_discount_value != 0 && $discountPromotion > $discount->max_discount_value) {
                                    $discountPromotion = $discount->max_discount_value;
                                }
                                $isValid = true;
                            } else if ($discount->type == 'freeship') {
                                if ($request->has("shipping_fee") && $request->has("shipping_type") && $request->input("shipping_type") == "standard") {
                                    $discountPromotion = $request->input("shipping_fee");
                                    $discountType = "freeship";
                                    $isValid = true;
                                }
                            }
                            if ($isValid) {
                                $response = [
                                    "status" => "successful",
                                    "result" => $discountPromotion,
                                    "discountPercent" => $discountPercent,
                                    "maxDiscount" => $maxDiscount,
                                    "discountType" => $discountType,
                                ];
                            }
                        }
                        
                    }
                }
            }
        }
        return response()->json($response);
    }

    public function incrementDiscount(Request $request) {
        $orderId = $request->input("order_id");
        $order = $this->getOrder($orderId);
        if (isset($order)) {
            $items = isset($order['items']) ? $order['items'] : [];
        }

        $listProduct = [];
        foreach ($items as $key => $value) {
            if (!array_key_exists($value['product_id'], $listProduct)) {
                $listProduct[$value['product_id']] = [
                    "product_id" => $value['product_id'],
                    "itemAmount" => round($value['price'] * $value['quantity'], 2),
                    "quantity" => $value['quantity']
                ];
            } else {
                $listProduct[$value['product_id']]["itemAmount"] = round($listProduct[$value['product_id']]["itemAmount"] + ($value['price'] * $value['quantity']), 2);
                $listProduct[$value['product_id']]["quantity"] += $value['quantity'];
            }
        }
        $listValidDiscount = [];
        foreach ($listProduct as $key => $item) {
            $listDiscount = [];
            $validDiscount = [];
            $listDiscount = $this->getDiscountByProduct($item['product_id']);
            if (count($listDiscount) > 0) {
                list ($message, $title, $validDiscount) = $this->getBuyMoreText($listDiscount, $item, $listProduct);
            } else {
                $listDiscount = $this->getDiscountByCategory($item['product_id']);
                if (count($listDiscount) > 0) {
                    list ($message, $title, $validDiscount) = $this->getBuyMoreText($listDiscount, $item, $listProduct);
                }
            }

            if (!empty($validDiscount)) {
                if (!array_key_exists($validDiscount->id, $listValidDiscount)) {
                    $listValidDiscount[$validDiscount->id] = $validDiscount;
                }
            }
        }
        if (isset($order['promotion_code']) && $order['promotion_code'] != "") {
            $discount = Discount::where("code", $order['promotion_code'])->first();
            if ($discount) {
                $listValidDiscount[$discount->id] = $discount;
            }
        }
        foreach ($listValidDiscount as $item) {
            $dataUpdate = [
                "number_uses" => $item->number_uses + 1,
            ];
            if ($item->limited_number_uses && $item->limited_number_uses > 0 && $item->limited_number_uses <= $item->number_uses + 1 && $item->is_limit_by_user != 1) {
                $dataUpdate["status"] = "inactive";
            }
            $discount = Discount::where("id", $item->id)->update($dataUpdate);
        }
        return response()->json(["status" => "successful"]);
    }

    public function getOrder($orderId) {
        $result = false;
        $data = ApiClient::buildCustomRequest("/order/retrieve/$orderId", 'GET', [], ["api-token: megaads@123"]);
        if (array_key_exists("status", $data) && $data['status'] == 'successful') {
            $result = (array) $data["result"];
        }
        return $result;
    }

    public function createCustomDiscount(Request $request) {
        $response = [
            "status" => "fail"
        ];
        try {
            $code = $request->input('code', null);
            if (!$code) {
                $isExists = true;
                while ($isExists) {
                    $code = $this->randString(10);
                    $isExists = Discount::where("code", $code)->exists();
                }
            }
            $value = $request->input('value', null);
            if (!$value || $value == 0) {
                $response = [
                    "status" => "fail",
                    "status" => "Invalid value discount!"
                ];
            } else {
                $dataUpdate = [
                    'title' => $request->input("title", 'Discount'),
                    'status' => "active",
                    'code' => $code,
                    'limited_number_uses' => $request->input("limitNumber", null),
                    'number_uses' => null,
                    'apply_type' => $request->input("applyType", "all"),
                    'start_time' => $request->input("createdTime", null),
                    'end_time' => $request->input("endTime", null),
                    'max_discount_value' => $request->input("maxDiscount", null),
                    'type' => $request->input("type", "fixed_amount"),
                    'value' => $value,
                    'created_at' => new \DateTime(),
                    'updated_at' => new \DateTime(),
                ];
                $minAmount = $request->input("minAmount", 0);
                if ($minAmount && $minAmount > 0) {
                    $dataUpdate["minimum_require_type"] = 'amount';
                    $dataUpdate["minimum_require_value"] = $minAmount;
                } else {
                    $dataUpdate["minimum_require_type"] = 'none';
                }
                $discount = Discount::create($dataUpdate);
                $response = [
                    "status" => "successful",
                    "result" => $discount
                ];
            }
        } catch (\Exception $ex) {
            \Log::error('createDiscount: ' . json_encode($ex));
        }
        
        return response()->json($response);
    }

    function randString($length) {
        $char = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
        $char = str_shuffle($char);
        for($i = 0, $rand = '', $l = strlen($char) - 1; $i < $length; $i ++) {
            $rand .= $char{mt_rand(0, $l)};
        }
        return $rand;
    }

    public function getDiscountInfoByCategory(Request $request) {
        $response = [
            "status" => 'fail'
        ];
        $cateId = $request->input("category_id");
        if ($cateId) {
            $result = [];
            $category = Category::find($cateId);
            $productDiscount = DiscountItem::where("reference_id", $category->id)
                ->join("discount", "discount.id", "=", "discount_id")
                ->where("discount.status", "active")
                ->where("discount.apply_type", "category")
                ->where("is_show_on_site", 1)
                ->orderBy("minimum_require_value", "ASC")
                ->get(['discount.*', 'discount_item.reference_id', 'discount_item.discount_id', 'discount_item.get_id']);
            if (count($productDiscount) == 0) {
                $productDiscount = Discount::where("status", "active")
                    ->where("is_show_on_site", 1)
                    ->where("apply_type", "all")
                    ->orderBy("minimum_require_value", "ASC")
                    ->get();
            }
            if ($productDiscount) {
                list($result, $type) = $this->buildDiscountText($productDiscount);
            }
            $response = [
                "status" => "successful",
                "result" => $result
            ];
        }
        return response()->json($response);
    }

    public function getFreeShippingInfo(Request $request) {
        $response = [
            "status" => 'fail'
        ];
        $freeship = Discount::where("status", "active")
            ->where("type", "freeship")
            ->where(function ($q) {
                $q->where("end_time", ">", date('Y-m-d h:i:s'));
                $q->orWhereNull("end_time");
            })
            ->orderBy("id", "DESC")
            ->first();
        if ($freeship) {
            $response = [
                "status" => "successful",
                "result" => [
                    "code" => $freeship->code,
                    "minimum_require_value" => $freeship->minimum_require_value,
                    "minimum_require_type" => $freeship->minimum_require_type,
                ]
            ];
        }
        return response()->json($response);
    }
}
