<?php
namespace Modules\EmailMarketing\Controllers;

use Illuminate\Http\Request;
use Modules\EmailMarketing\Models\EmailBlacklist;

class BlacklistController extends BaseController
{

    /**
     * Third party email marketing api key.
     */
    protected $apiKey = '';
    
    /**
     * Using param for check current locale is valid
     */
    protected $locale = '';


    /**
     * 
     */
    public function __construct()
    {
        $this->locale = env('APP_LOCALE');
        $this->apiKey = config('email-marketing::sa.api.token');
    }

    /**
     * 
     */
    public function showBlacklist()
    {
        if ($this->locale !== 'central' || $this->locale == '') {
            return response()->json(['status' => 'fail', 'message' => 'Access denied on this locale!'], 403);
        } else {
            return view('email-marketing::blacklist.index');
        }
    }

    /**
     * @param Illuminate\Http\Request request
     * 
     * @return array response
     */
    public function lists(Request $request)
    {   
        if ($this->locale !== 'central' || $this->locale == '') {
            return response()->json(['status' => 'fail', 'message' => 'Access denied on this locale!'], 403);
        } else {
            $response = $this->getSuccessStatus([]);
            $pageSize = $request->get('page_size', 10);
            $pageId = $request->get('page_id', 0);
            $offset = $pageSize * $pageId;
            
            $query = $this->buildFilters($request->all());
            $total = $query->count();
            if ($total > 0) {
                $pageCount = ceil($total/$pageSize);
                $query->limit($pageSize);
                $query->offset($offset);
                $result = $query->get();
                $response = $this->getSuccessStatus($result);
                $response['paginate'] = [
                    'pageId' => (int) $pageId,
                    'pagesCount'  => (int) $pageCount,
                    'total' => $total
                ];
            }
            return response()->json($response);
        }
    }

    /**
     * 
     * @param Illuminate\Http\Request request
     * 
     * @return array response
     */
    public function addEmailToBlacklist(Request $request)
    {
        if ($this->locale !== 'central' || $this->locale == '') {
            return response()->json(['status' => 'fail', 'message' => 'Access denied on this locale!'], 403);
        } else {
            $response = $this->getDefaultStatus();
    
            if ($request->has('email')) {
                $isExists = EmailBlacklist::where('email', $request->get('email'))->first(['id']);
                if (empty($isExists)) {
                    $blacklist = new EmailBlacklist;
                    $blacklist->fill($request->all());
                    if ($blacklist->save()) {
                        $response = $this->getSuccessStatus($blacklist);
                        // $this->addMemberToBlackList($request->get('email'));
                    }
                }
            } else {
                $response['message'] = 'Invalid parameter(s)';
            }
    
            return response()->json($response);
        }
    }

     /**
     * 
     * @param Illuminate\Http\Request request
     * 
     * @return array response
     */
    public function removeEmailFromBlacklist(Request $request)
    {
        if ($this->locale !== 'central' || $this->locale == '') {
            return response()->json(['status' => 'fail', 'message' => 'Access denied on this locale!'], 403);
        } else {
            $response = $this->getDefaultStatus();
            if ($request->has('email')) {
                $isExists = EmailBlacklist::where('email', $request->get('email'))->first(['id']);
                if (!empty($isExists)) {
                    $isExists->delete();
                    $emails = explode(',', $request->get('email'));
                    // $this->removeMemberFromBlacklist($emails);
                    $response = $this->getSuccessStatus([]);
                }
            } else {
                $response['message'] = 'Invalid parameter(s)';
            }
            return response()->json($response);
        }
    }

    /**
     * @param array filters
     * 
     * @return Illuminate\Database\Eloquent query
     */
    protected function buildFilters($filters) 
    {
        $query = EmailBlacklist::query();
        
        if (array_key_exists('email', $filters)) {
            $query->where('email', $filters['email']);
        }

        return $query;
    }

    /**
     * 
     * @param string email
     * 
     * @return null
     */
    protected function addMemberToBlackList($email)
    {
        $findList =  $this->checkListIsExist('Black list');
        if ($findList['listId']) {
            $listId = $findList['listId'];
            $customers = [
                'email' => $email
            ];
            $endpoint = '/list/' . $listId . '/members?api_key=' . $this->apiKey;
            $result = NULL;
            if (count($customers) > 0) {
                $params = [
                    'profiles' => $customers
                ];
                $result = $this->sendHttpRequest($endpoint, 'POST', json_encode($params), ["Content-Type: application/json"]);
            }
        }
    }

    /**
     * @param array email
     * 
     * @return null
     */
    protected function removeMemberFromBlacklist($emails)
    {
        $findList =  $this->checkListIsExist('Black list');
        if ($findList['listId']) {
            $listId = $findList['listId'];
            $customers = [
                'emails' => $emails
            ];
            $endpoint = '/list/' . $listId . '/members?api_key=' . $this->apiKey;
            $result = NULL;
            if (count($customers) > 0) {
                $result = $this->sendHttpRequest($endpoint, "DELETE", json_encode($customers), ["Content-Type: application/json"]);
            }
        }
    }


    /**
     * 
     * @param string listName
     * @param boolean autoCreate
     * 
     * @return array retval
     */
    private function checkListIsExist($listName, $autoCreate = false) {
        $retval = [
            'listId' => false,
            'residual' => -1,
            'listName' => ''
        ];
        $result = $this->sendHttpRequest('/lists', 'GET', http_build_query(['api_key' => $this->apiKey]));
        if (isset($result['status']) && $result['status']) {
            $data = $result['result'];
            $lists = [];
            foreach ($data as $item) {
                if (preg_match("/$listName/m", $item->list_name, $matched)) {
                    $listId = $item->list_id;
                    $nameArray = explode('-', $item->list_name);
                    $index = isset($nameArray[1]) ? (int) $nameArray[1] : 0;
                    $lists[$index] = [$item->list_id, $item->list_name];
                }
            }
            if (count($lists) > 0) {
                ksort($lists);
                $lastOfList = end($lists);
                $retval['listId'] = $lastOfList[0];
                $retval['listName'] = $lastOfList[1];
            }
        }
        return $retval;
    }
}