<?php

namespace Modules\EmailMarketing\Controllers;

use App\Customer;
use DateTime;
use Illuminate\Support\Facades\DB;
use App\Helpers\ApiClient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Modules\EmailMarketing\Controllers\Controller;
use Module;
use Modules\EmailMarketing\Models\Email;
use Modules\EmailMarketing\Models\Order;
use Modules\EmailMarketing\Models\OrderMeta;
use Modules\EmailMarketing\Models\Product;
use Modules\EmailMarketing\Models\Unsubscribe;
use App\Utils\Email as UtilsEmail;

class HomeController extends Controller
{
    public function __construct()
    {
    }
    public function cronSendEmailForgetItemInCart(Request $request)
    {
        $timeFromConfig = config("email-marketing::sa.time_from", 10);
        $timeToConfig = config("email-marketing::sa.time_to", 10);
        $timeFrom = date('Y-m-d H:i:s', strtotime("-$timeFromConfig minutes"));
        $timeTo = date('Y-m-d H:i:s', strtotime("-$timeToConfig minutes"));
        $listOrder = [];
        if ($request->has("id")) {
            $orders = Order::where("id", $request->input("id"))
                        ->get(["id", "customer_id", "created_at", "payment_status", "payment_type"]);
        } else {
            $orders = Order::where("payment_status", "PENDING")
                        ->where("created_at", ">=", $timeFrom)
                        ->where("created_at", "<=", $timeTo)
                        ->whereRaw('created_at in (select max(created_at) from sb_order where created_at >= \'' . $timeFrom . '\' and created_at <= \'' . $timeTo . '\' group by (customer_id))')
                        ->groupBy("customer_id")
                        ->orderBy("created_at", "DESC")
                        ->get(["id", "customer_id", "created_at", "payment_status", "payment_type"]);
        }
        foreach ($orders as $key => $value) {
            $paidOrder = Order::where("created_at", ">", $value->created_at)
                            ->where("customer_id", $value->customer_id)
                            ->exists();
            $customer = Customer::where("id", $value->customer_id)->first();
            if (!$paidOrder && $this->checkUnsubscribe($customer->email, "FORGET_ITEM")) {
                $isExists = Email::where("email", $customer->email)->where("status", "UNPAID")->where("order_id", $value->id)->exists();
                if (!$isExists) {
                    Email::insert([
                        "email" => $customer->email,
                        "status" => "UNPAID",
                        "order_id" => $value->id,
                        "created_at" => new \DateTime(),
                        "updated_at" => new \DateTime(),
                    ]);
                    $this->handleSendEmail($customer->email, $value, $customer);
                    $listOrder[] = $value->id;
                }
            }
        }
        $locale = env('APP_LOCALE') ? env('APP_LOCALE') : 'us';
        \Log::info("Send email forget item - " . $locale . " : " . json_encode($listOrder));
        return response()->json([
            "status" => "successful",
            "result" => $listOrder
        ]);
    }

    private function handleSendEmail ($customerEmail, $order, $customer) {
        $apiUrl = config('sa.api_url');
        $paymentLink = null;
        if ($order->payment_type == "PAYPAL") {
            $this->triggerAsyncRequest($apiUrl . "/paypal/invoice?order_id=" . $order->id);
            $invoiceLink = OrderMeta::where("order_id", $order->id)->where("key", "paypal_invoice_link")->first();
            if ($invoiceLink) {
                $paymentLink = $invoiceLink->value;
            }
        } else {
            $this->triggerAsyncRequest($apiUrl . "/stripe/invoice?order_id=" . $order->id);
            $invoiceLink = OrderMeta::where("order_id", $order->id)->where("key", "stripe_invoice_link")->first();
            if ($invoiceLink) {
                $paymentLink = $invoiceLink->value;
            }
        }

        $siteName = config('app.name');
        $email = config('email-marketing::sa.support_email');
        $siteUrl = config('app.url');
        $params = [
            'driver' => config('sa.email_driver', 'ticket'),
            'subject' => translate('Did you forget somethings?'),
            'to' => $customerEmail,
            'from' => config('email-marketing::sa.support_name', '"Printerval Support" <support@printerval.com>'),
            'html' => utf8_encode(view('email-marketing::home.email-item-in-cart', [
                'siteName' => $siteName,
                'siteUrl' => $siteUrl,
                'email' => $email,
                'order' => $order,
                'customer' => $customer,
                'paymentLink' => $paymentLink,
            ])),
        ];

        UtilsEmail::sendToGearman($params);
    }

    protected function triggerAsyncRequest($url, $method = "GET", $params = [], $headers = []) {
        $channel = curl_init();
        curl_setopt($channel, CURLOPT_URL, $url);
        // curl_setopt($channel, CURLOPT_NOSIGNAL, 1);
        curl_setopt($channel, CURLOPT_TIMEOUT, 120);
        curl_setopt($channel, CURLOPT_RETURNTRANSFER, 1);
        if($method == "post" || $method == "POST") {
            curl_setopt($channel, CURLOPT_POST, true);
            curl_setopt($channel, CURLOPT_POSTFIELDS, json_encode($params));
        }
        if ($headers) {
            curl_setopt($channel, CURLOPT_HTTPHEADER, $headers);
        }
        curl_setopt($channel, CURLOPT_SSL_VERIFYHOST, 0);
        curl_setopt($channel, CURLOPT_SSL_VERIFYPEER, 0);
        $data = curl_exec($channel);
        curl_close($channel);
        return json_decode($data, true);
    }

    public function buildCartItem(Request $request) {
        $orderId = $request->input("reference_id");
        $order = Order::with("orderItems")->where("id", $orderId)->first()->toArray();
        if (isset($order["order_items"])) {
            $token = getFingerprint();
            $cartItems = $this->getCartItems();
            foreach ($order["order_items"] as $item) {
                $isExists = false;
                foreach ($cartItems as $key => $value) {
                    if ($value["product_id"] == $item["product_id"] && $value["product_sku_id"] == $item["product_sku_id"]) {
                        $isExists = true;
                    }
                }
                if (!$isExists) {
                    $data = [
                        "productId" => $item["product_id"],
                        "productSkuId" => $item["product_sku_id"],
                        "customerToken" => $token,
                        "quantity" => $item["quantity"],
                        "buynow" => null,
                        'configurations' => $request->input("configurations", null),
                        "canvas" => $item["canvas"],
                        "designUrl" => $item["design_url"],
                    ];
            
                    Module::action('addToCart', [
                        'product_id' => $data['productId'],
                        'product_sku_id' => $data['productSkuId'],
                        'quantity' => $data['quantity'],
                    ]);
                    $response = ApiClient::buildCustomRequest("cart/add-to-cart", "POST", $data, []);
                }
            }
            return redirect()->route('cart', ['internal_source=email-forget-item']);
        }
        return redirect()->route('home');
    }

    private function getCartItems() {
        $cartItems = [];
        $query = ApiClient::buildClient("cart/get-cart-item");
        $token = getFingerprint();
        $query->addField("token", $token);
        $result = $query->get();
        if (isset($result['status']) && $result['status'] == 'successful') {
            $cartItems = $result['items'];
        }
        return $cartItems;
    }

    private function getSkuName($productId, $productSkuId) {
        $origin_name = '';
        $name = [];
        $productVariantName = [];
        $productOrigin = Product::find($productId);
            if (!isset($productOrigin->name)) {
                return $name;
            }
            $origin_name = $productOrigin->name;
            $skuValues = DB::table('product_sku_value')->where('sku_id', '=', $productSkuId)->get();
            $listProductSku = DB::table('product_sku_value')->where('product_id', $productId)->get();

            $groupProductsSku = [];
            foreach ($listProductSku as $sku) {
                $groupProductsSku[$sku->sku_id][] = $sku;
            }

            $variantOptionIds = [];
            foreach ($skuValues as $skuValue) {
                if ($skuValue->variant_id && $skuValue->variant_option_id) {
                    $variant = DB::table('product_variant')->find($skuValue->variant_id);
                    $variantOption = DB::table('product_variant_option')->find($skuValue->variant_option_id);
                    
                    $variantOptionIds[] = $variantOption->id;
                    
                    if ($variant->slug == 'size') {
                        $attrName = $variant->name . ': ' . $variantOption->name;
                    } else {
                        $attrName = $variantOption->name;
                    }
                    
                    $productVariantName[] = [
                        "variant" => $variant,
                        "variantOption" => $variantOption,
                        "name" => $attrName
                    ];
                }
            }

            foreach ($productVariantName as $groupVariantName) {
                $name[] = $groupVariantName['name'];
            }

        return [
            'sku' => implode(', ', $name),
            'origin' => $origin_name,
        ];
    }

    public function cronSendEmailAfterSale(Request $request)
    {
        set_time_limit(3600);

        $ordersDelivered = $this->sendAfterSaleEmail('AFTER_SALE_DELIVERED');
        $ordersExperience = $this->sendAfterSaleEmail('AFTER_SALE_EXPERIENCE');

        return response()->json([
            "status" => "successful",
            "result" => [
                'delivered' => $ordersDelivered,
                'experience' => $ordersExperience,
            ]
        ]);
    }

    private function sendAfterSaleEmail($type = 'AFTER_SALE_DELIVERED') {
        // Lọc log
        $timeLog = [];
        if($type == 'AFTER_SALE_DELIVERED') {
            $days = new DateTime('yesterday');
        } else {
            $days = new DateTime('-4 days');
        }

        $timeLog = [$days->format('Y-m-d') . ' 00:00:00', $days->format('Y-m-d') . ' 23:59:59'];

        $ordersDeliveredQuery = DB::table('order_status_time')
            ->where('order_status_time.status', 'FINISHED')
            ->whereBetween('order_status_time.created_at', $timeLog);

        $ordersDelivered = $ordersDeliveredQuery
            ->join('order', 'order.id' , '=', 'order_status_time.order_id')
            ->select(['order.id', 'order.customer_id'])
            ->get();

        $customerIds = [];
        foreach($ordersDelivered as $order) {
            $customerIds[] = $order->customer_id;
        }
        $customers = Customer::query()
            ->whereIn('id', $customerIds)
            ->select(['id', 'email'])
            ->pluck('email', 'id')
            ->toArray();

        foreach($ordersDelivered as &$order) {
            if (isset($customers[$order->customer_id])) {
                $order->email = $customers[$order->customer_id];
            }
        }

        $ordersDeliveredItems = $ordersDeliveredQuery->join('order_item', 'order_item.order_id' , '=', 'order.id')
            ->join('order_item_info', 'order_item_info.order_item_id' , '=', 'order_item.id')
            ->select([
                'order_item.id', 
                'order_item.order_id', 
                'order_item.product_id', 
                'order_item.product_sku_id',
                'order_item.quantity',
                'order_item.image_url',
                'order_item.price',
                'order_item_info.name',
                'order_item_info.sku',
            ])
            ->get();

        $groupOrderItems = [];
        foreach ($ordersDeliveredItems as &$value) {
            $value->sku_name = $value->sku;
            $value->display_price = formatPrice($value->price);

            $groupOrderItems[$value->order_id][] = $value;
        }

        $listOrder = [];
        $logData = [];

        foreach ($ordersDelivered as &$order) 
        {
            $order->items = [];
            if(isset($groupOrderItems[$order->id])) {
                $order->items = $groupOrderItems[$order->id];
            }

            $uniqueItems = [];
            foreach ($order->items as $item) {
                $uniqueItems[$item->id] = $item;
            }
            $order->items = array_values($uniqueItems);

            $isExists = Email::where("email", $order->email)->where("status", $type)->where("order_id", $order->id)->exists();
            if (!$isExists && $this->checkUnsubscribe($order->email, $type)) {
                Email::insert([
                    "email" => $order->email,
                    "status" => $type,
                    "order_id" => $order->id,
                    "created_at" => new \DateTime(),
                    "updated_at" => new \DateTime(),
                ]);

                $logData = [
                    'target_type' => "EMAIL",
                    'target_id' => $order->id,
                    'event_type' => $type,
                    'data' => json_encode([
                        'order_id' => $order->id,
                        'email' => $order->email,
                    ]),
                    'created_at' => date('Y-m-d H:i:s', time())
                ];

                $listOrder[] = $order->id;
                $siteName = config('app.name');
                $email = config('email-marketing::sa.support_email');
                $siteUrl = config('app.url');
                $params = null;
                if ($type == 'AFTER_SALE_DELIVERED') {
                    $params = [
                        'driver' => config('sa.email_driver', 'ticket'),
                        'subject' => translate("PLEASE READ!"),
                        'to' => $order->email,
                        'from' => getOption('email.support_name', '"Printerval Support" <support@printerval.com>'),
                        'html' => utf8_encode(view('email-marketing::mail-templates.after-sale-delivered', [
                            'siteName' => $siteName,
                            'siteUrl' => $siteUrl,
                            'email' => $email,
                            'order' => $order
                        ])),
                    ];

                } else {
                    $params = [
                        'driver' => config('sa.email_driver', 'ticket'),
                        'subject' => translate("HELP!"),
                        'to' => $order->email,
                        'from' => getOption('email.support_name', '"Printerval Support" <support@printerval.com>'),
                        'html' => utf8_encode(view('email-marketing::mail-templates.after-sale-experience', [
                            'siteName' => $siteName,
                            'siteUrl' => $siteUrl,
                            'email' => $email,
                            'order' => $order
                        ])),
                    ];
                }

                UtilsEmail::sendToGearman($params);
            }
        }

        $locale = env('APP_LOCALE') ? env('APP_LOCALE') : 'us';

        \Log::info("Send email $type - " . $locale . " : " . json_encode($listOrder));
        DB::table('log')->insert($logData);

        return $listOrder;
    }

    private function checkUnsubscribe($email, $type) {
        $isExists = Unsubscribe::where("email", $email)->whereIn("type", [$type, "ALL"])->exists();
        return !$isExists;
    }
}
