<?php

namespace Modules\FinancialStatement\Controllers;

use Illuminate\Http\Request;
use Modules\FinancialStatement\Controllers\Controller;
use Module;
use Modules\FinancialStatement\Models\FinancialStatement;
use Modules\FinancialStatement\Models\Order;
use Modules\Ticket\Helpers\DBConnect;
use DateTime;
use Illuminate\Support\Facades\Redis;

class FinancialStatementController extends Controller
{
    public function getDataStatistic(Request $request) {
        $response = [
            "status" => "fail"
        ];
        try {
            $month = $request->input("month");
            $year = $request->input("year");
            $dateReport = $request->input("date", 1);
            $clearCache = $request->input("clear_cache", 0);
            $isClearCache = $clearCache == 1 ? true : false;

            $result = $this->getDataByMonthYear($month, $year, $dateReport, false, !$isClearCache, $isClearCache);
            $data = $result["data"];
            $lastMonthData = $result["lastMonthData"];
            
            $response = [
                'status' => 'successful',
                'data' => $data,
                'last_month' => $lastMonthData,
            ];
        } catch (\Exception $ex) {
            $response = [
                "status" => "fail",
                "message" => $ex->getMessage() . " - File: " . $ex->getFile() . ' - Line: ' . $ex->getLine(),
            ];
            \Log::info("Get Data Statistic: " . $ex->getMessage() . " - File: " . $ex->getFile()  . ' - Line: ' . $ex->getLine());
        }
        return response()->json($response);
    }

    private function getDataByMonthYear($month, $year, $dateReport, $getLastMonthCentral = false, $getCache = true, $isClearCache = false) {
        $result = [];
        $currentLocale = env("APP_LOCALE", "us");
        if ($currentLocale == "") {
            $currentLocale = "us";
        }
        $percent = round($dateReport / cal_days_in_month(CAL_GREGORIAN, $month, $year), 4, PHP_ROUND_HALF_UP);
        $lastMonth = new DateTime("$year-$month-01");
        $lastMonth->modify( 'first day of last month' );
        $lastMonthData = null;
        $dataByCalculate = false;
        $subSalary = 0;
        if ($currentLocale == "central" || $getLastMonthCentral) {
            if ($lastMonth->format("m") == 1 && $lastMonth->format("Y") == "2022") {
                $subSalary = 3500;
            }
            $lastMonthData = $this->getDataByMonth($lastMonth->format("m"), $lastMonth->format("Y"));
        } else {
            if ($percent == 1) {
                $dateReport = cal_days_in_month(CAL_GREGORIAN, $lastMonth->format("m"), $lastMonth->format("Y"));
            }
            $resultLastMonth = $this->getDataByMonthYear($lastMonth->format("m"), $lastMonth->format("Y"), $dateReport, $getCache);
            if (isset($resultLastMonth["data"])) {
                $dataByCalculate = true;
                $lastMonthData = $resultLastMonth["data"];
            }
        }
        $date = clone $lastMonth;
        if (!$lastMonthData) {
            if ($percent == 1) {
                $dateReport = cal_days_in_month(CAL_GREGORIAN, $date->format("m"), $date->format("Y"));
            }
            $resultLastMonth = $this->getDataByMonthYear($date->format("m"), $date->format("Y"), $dateReport, $getCache);
            if (isset($resultLastMonth["data"])) {
                $lastMonthData = $resultLastMonth["data"];
                $dataByCalculate = true;
            }
        }

        $startMonth = new DateTime("$year-$month-01");
        $endMonth = clone $startMonth;
        $endMonth->setDate($endMonth->format("Y"), $endMonth->format("m"), $dateReport);
        $data = [];
        if ($getCache) {
            $data = $this->getCache($endMonth->format("Y"), $endMonth->format("m"), $dateReport);
        }
        if (count($data) == 0) {
            $data = $this->getDataFinancialStatement($startMonth, $endMonth);
            $newDateReport = $dateReport;
            $now = new DateTime();
            if ($now < $endMonth) {
                $endMonth = $now;
                $newDateReport = $now->format("d");
            }
            $percentByLastMonth = round($newDateReport / cal_days_in_month(CAL_GREGORIAN, $month, $year), 4, PHP_ROUND_HALF_UP);
            $this->buildDataFromLastMonth($data, $lastMonthData, $percentByLastMonth, true, $subSalary, $newDateReport);
        }

        if (!$dataByCalculate) {
            foreach ($lastMonthData as $key => $value) {
                $lastMonthData[$key] = $lastMonthData[$key] * $percent;
            }
        }
        $this->saveCache($endMonth->format("Y"), $endMonth->format("m"), $dateReport, $data, $isClearCache);
        $result["data"] = $data;
        $result["lastMonthData"] = $lastMonthData;
        return $result;
    }

    private function getDataFinancialStatement($startMonth, $endMonth) {
        $locales = [];
        if (function_exists('getModuleLocale')) {
            $locales = getModuleLocale();
        }
        
        $currentLocale = env("APP_LOCALE", "us");
        if ($currentLocale == "") {
            $currentLocale = "us";
        }

        $data["revenue"] = 0;
        $data["return_amount"] = 0;
        $data["cost_price"] = 0;
        $data["other_cost"] = 0;
        $data["print_refund"] = 0;
        $data["dispute_fee"] = 0;
        $data["paymentRefundDispute"] = 0;
        if ($currentLocale == "central") {
            foreach ($locales as $key => $locale) {
                $connectionLocale = DBConnect::connect($locale["locale"]);
                $this->getFinancialData($startMonth, $endMonth, $data, $connectionLocale, $locale);
            }
        } else {
            $locale = [];
            foreach ($locales as $key => $item) {
                if ($currentLocale == $item["locale"]) {
                    $locale = $item;
                }
            }
            $connectionLocale = DBConnect::connect($locale["locale"]);
            $this->getFinancialData($startMonth, $endMonth, $data, $connectionLocale, $locale);
        }
        $adsCost = $this->getAdsCost($startMonth, $endMonth, $locale);
        $data["ads_cost"] = $adsCost;
        return $data;
    }

    private function getFinancialData($startMonth, $endMonth, &$data, $connectionLocale, $locale) {
        if ($locale["locale"] != 'central') {
            $result = $this->getAvailableData($startMonth, $endMonth, $connectionLocale, $locale);
            $data["revenue"] += round($result["revenue"], 2);
            $data["return_amount"] += round($result["return_amount"], 2);
            $data["cost_price"] += round($result["cost_price"], 2);
            $data["other_cost"] += round($result["other_cost"], 2);
            $data["print_refund"] += round($result["print_refund"], 2);
            $data["dispute_fee"] += round($result["dispute_fee"], 2);
            $data["paymentRefundDispute"] += round($result["paymentRefundDispute"], 2);
        }
    }

    private function getAvailableData($start, $end, $connectionLocale, $locale) {
        $result = [];
        $totalOrder = $connectionLocale->table("order_meta")
                                    ->where("order_meta.key", "=", "in_production_time")
                                    ->where("order_meta.value", ">=", $start->format('Y-m-d 00:00:00'))
                                    ->where("order_meta.value", "<=", $end->format('Y-m-d 23:59:59'))
                                    ->join("order", function($q) {
                                        $q->on("order.id", "=", "order_meta.order_id");
                                        $q->where("status", "<>", "PENDING");
                                        $q->where("status", "<>", "PROCESSING");
                                    })
                                    ->leftJoin("order_meta as om1", function($q) {
                                        $q->on("om1.order_id", "=", "order_meta.order_id");
                                        $q->where("om1.key", "no_cost_of_capital");
                                    })
                                    ->where("payment_status", "PAID")
                                    ->orderBy("order_meta.value")
                                    ->get(["order.id", "code", "amount", "status", "cost", "shipping_cost", "tax_cost", "payment_type", "created_at", "order_meta.value", "om1.value as cost_status"]);
        $refundOrder = $connectionLocale->table("order_meta")
                                    ->where("order_meta.key", "refund_time")
                                    ->where("order_meta.value", ">=", $start->format('Y-m-d 00:00:00'))
                                    ->where("order_meta.value", "<=", $end->format('Y-m-d 23:59:59'))
                                    ->join("order", function($q) {
                                        $q->on("order.id", "=", "order_meta.order_id");
                                        $q->where("order.return_fee", ">", 0);
                                    })
                                    ->leftJoin("order_meta as om1", function($q) {
                                        $q->on("om1.order_id", "=", "order_meta.order_id");
                                        $q->where("om1.key", "print_refund_data");
                                    })
                                    ->get(["order.id", "order.code", "amount", "status", "return_fee", "cost", "shipping_cost", "tax_cost",
                                        "payment_type", "created_at", "return_reason", "om1.value as print_refund"
                                    ]);
        $disputeOrder = $connectionLocale->table("order_meta")
                            ->where("order_meta.key", "dispute_time")
                            ->where("order_meta.value", ">=", $start->format('Y-m-d 00:00:00'))
                            ->where("order_meta.value", "<=", $end->format('Y-m-d 23:59:59'))
                            ->leftJoin("order_meta as om1", function($q) {
                                $q->on("om1.order_id", "=", "order_meta.order_id");
                                $q->where("om1.key", "dispute_fee");
                            })
                            ->get(["order_meta.order_id", "om1.value as dispute_fee"]);
        $paymentGatewayRefundDispute = $connectionLocale->table("order_meta")
                            ->where("order_meta.key", "payment_dispute_refund_time")
                            ->where("order_meta.value", ">=", $start->format('Y-m-d 00:00:00'))
                            ->where("order_meta.value", "<=", $end->format('Y-m-d 23:59:59'))
                            ->leftJoin("order_meta as om1", function($q) {
                                $q->on("om1.order_id", "=", "order_meta.order_id");
                                $q->where("om1.key", "payment_dispute_refund_fee");
                            })
                            ->get(["order_meta.order_id", "om1.value as payment_dispute_refund_fee"]);
        $currencyRatio = 1;
        if (!empty($locale['currencyRatio'])) {
            $currencyRatio = $locale['currencyRatio'];
        }
        $revenue = 0;
        $returnAmount = 0;
        $costPrice = 0;
        $otherCost = 0;
        foreach ($totalOrder as $key => $order) {
            $paymentGateFee = 0;
            $paymentConfig = config('financial-statement::payment', []);
            $type = $order->payment_type;
            $d1 = new DateTime($order->created_at);
            $d2 = new DateTime('2022-03-30 00:00:00');
            if ($d1 > $d2 && $order->payment_type == "STRIPE") {
                $type = "STRIPE_NEW";
            }
            if (isset($paymentConfig[$type])){
                $paymentGateConfig = $paymentConfig[$type];
                $rate = $paymentGateConfig['rate'];
                $currencyUnit = $locale["currencyUnit"];
                if (isset($paymentGateConfig['extraRate'][$currencyUnit])){
                    $rate = $paymentGateConfig['rate'] + $paymentGateConfig['extraRate'][$currencyUnit];
                }
                $paymentGateFee = $order->amount * $rate;
                if (isset($paymentGateConfig['fixedFee'][$currencyUnit])){
                    $paymentGateFee += $paymentGateConfig['fixedFee'][$currencyUnit];
                }
                if (isset($paymentGateConfig['ratio'][$currencyUnit])) {
                    $currencyRatio = $paymentGateConfig['ratio'][$currencyUnit];
                }
            }
            $paymentGateFee = round($paymentGateFee / $currencyRatio, 2);
            $order->paymentGateFee = $paymentGateFee;
            $orderCost = 0;
            if ($order->cost) {
                $orderCost += $order->cost;
            }
            if ($order->shipping_cost) {
                $orderCost += $order->shipping_cost;
            }
            if ($order->tax_cost) {
                $orderCost += $order->tax_cost;
            }
            if ($orderCost == 0) {
                $orderMeta = $connectionLocale->table("order_meta")->where("order_id", $order->id)->where("key", "estimate_costs")->first();
                if ($orderMeta) {
                    $orderCost += round($orderMeta->value / $currencyRatio, 2);
                    $d1 = new DateTime($order->created_at);
                    $d2 = new DateTime('2021-12-10 14:30:00');
                    if ($d1 > $d2) {
                        $orderCost = round($orderCost - $paymentGateFee, 2);
                    }
                }
            }
            $revenue += round($order->amount / $currencyRatio, 2);
            if ($order->status != "CANCELED" && !$order->cost_status) {
                $costPrice += $orderCost;
            }
            $otherCost += $paymentGateFee;
        }
        $printRefund = 0;
        $reasonLostAllMonney = config("financial-statement::app.full_refund_reason", []);
        foreach ($refundOrder as $key => $order) {
            $paymentGateFee = 0;
            $paymentConfig = config('financial-statement::payment', []);
            $type = $order->payment_type;
            $d1 = new DateTime($order->created_at);
            $d2 = new DateTime('2022-03-30 00:00:00');
            if ($d1 > $d2 && $order->payment_type == "STRIPE") {
                $type = "STRIPE_NEW";
            }
            if (isset($paymentConfig[$type])){
                $paymentGateConfig = $paymentConfig[$type];
                $paymentGateFee = $order->amount * $paymentGateConfig['rate'];
                $currencyUnit = $locale["currencyUnit"];
                if (isset($paymentGateConfig['fixedFee'][$currencyUnit])){
                    $paymentGateFee += $paymentGateConfig['fixedFee'][$currencyUnit];
                }
                if (isset($paymentGateConfig['ratio'][$currencyUnit])) {
                    $currencyRatio = $paymentGateConfig['ratio'][$currencyUnit];
                }
            }
            $orderRefundCost = 0;
            if ($order->cost) {
                $orderRefundCost += $order->cost;
            }
            if ($order->shipping_cost) {
                $orderRefundCost += $order->shipping_cost;
            }
            if ($order->tax_cost) {
                $orderRefundCost += $order->tax_cost;
            }
            if ($orderRefundCost == 0) {
                $orderMeta = $connectionLocale->table("order_meta")->where("order_id", $order->id)->where("key", "estimate_costs")->first();
                if ($orderMeta) {
                    $orderRefundCost += round($orderMeta->value / $currencyRatio, 2);
                    $d1 = new DateTime($order->created_at);
                    $d2 = new DateTime('2021-12-10 14:30:00');
                    if ($d1 > $d2) {
                        $orderRefundCost = round($orderRefundCost - $paymentGateFee, 2);
                    }
                }
            }
            $returnFee = 0;
            if ($order->print_refund) {
                $dataRefund = json_decode($order->print_refund);
                if ($dataRefund->print_return_status == "full") {
                    $returnFee = round($order->return_fee / $currencyRatio, 2) - $orderRefundCost;
                } else if ($dataRefund->print_return_status == "partial") {
                    $returnFee = round($order->return_fee - $dataRefund->print_return_amount / $currencyRatio, 2);
                } else if ($dataRefund->print_return_status == "none") {
                    $returnFee = round($order->return_fee / $currencyRatio, 2);
                }
                $printRefund += $returnFee;
            } else {
                $returnFee = round($order->return_fee / $currencyRatio, 2);
                if (!in_array($order->return_reason, $reasonLostAllMonney)) {
                    $returnFee = $returnFee - $orderRefundCost;
                    $printRefund += $orderRefundCost;
                }    
            }
            $returnAmount += $returnFee;
        }
        $disputeFee = 0;
        foreach ($disputeOrder as $key => $order) {
            $disputeFee += round($order->dispute_fee / $currencyRatio, 2);
        }
        $paymentRefundDispute = 0;
        foreach ($paymentGatewayRefundDispute as $key => $order) {
            $paymentRefundDispute += round($order->payment_dispute_refund_fee / $currencyRatio, 2);
        }
        $otherCost = $otherCost + $disputeFee - $paymentRefundDispute;
        $result["revenue"] = round($revenue, 2);
        $result["return_amount"] = round($returnAmount, 2);
        $result["cost_price"] = round($costPrice, 2);
        $result["other_cost"] = round($otherCost, 2);
        $result["print_refund"] = round($printRefund, 2);
        $result["dispute_fee"] = round($disputeFee, 2);
        $result["paymentRefundDispute"] = round($paymentRefundDispute, 2);
        return $result;
    }

    private function getDataByMonth($month, $year) {
        $result = [];
        $connection = DBConnect::connect();
        $financialStatement = $connection->table("financial_statement")
                                                ->where("type", "total")
                                                ->where("month", $month)
                                                ->where("year", $year)
                                                ->first();
        if ($financialStatement) {
            $result = json_decode($financialStatement->data, true);
            $cost = 0;
            foreach ($result as $key => $value) {
                if (strpos($key, "cost") !== false && $key != "cost_price") {
                    $cost += floatval($value);
                }
            }
            $result["cost"] = round($cost, 2);
        }
        return $result;
    }

    private function getFinancialStatementFromUrl($date, $dateReport) {
        $try = 2;
        $index = 0;
        $currentLocale = env("APP_LOCALE");
        $lastMonthData = null;
        $dataByCalculate = false;

        //Lấy dữ liệu thị trường hiện tại
        while (!$lastMonthData && $try > $index) {
            $url = "/financial-statement/find?month={$date->format("m")}" . "&year={$date->format("Y")}" . "&date=$dateReport";
            if ($currentLocale != "us") {
                $url = "/$currentLocale" . $url;
            }
            $dataLastMonth = $this->triggerSyncRequest(env("APP_URL") . $url, 'GET', []);
            if (isset($dataLastMonth["status"]) && $dataLastMonth["status"] == "successful") {
                $lastMonthData = $dataLastMonth["data"];
                $dataByCalculate = true;
            }
            $date->modify("-1 month");
            $index ++;
        }
            
        //Lấy dữ liệu thị trường central
        if (!$dataLastMonth) {
            $try = 2;
            $index = 0;

            while (!$lastMonthData && $try > $index) {
                $url = "/central/financial-statement/find?month={$date->format("m")}" . "&year={$date->format("Y")}" . "&date=$dateReport";
                $dataLastMonth = $this->triggerSyncRequest(env("APP_URL") . $url, 'GET', []);
                if (isset($dataLastMonth["status"]) && $dataLastMonth["status"] == "successful") {
                    $lastMonthData = $dataLastMonth["data"];
                    $dataByCalculate = true;
                }
                $date->modify("-1 month");
                $index ++;
            }
        }
        return [
            "dataByCalculate" => $dataByCalculate,
            "lastMonthData" => $lastMonthData,
        ];
    }

    private function getAdsCost($startMonth, $endMonth, $locale) {
        $campaign = "Printerval";
        if (isset($locale["locale"]) && $locale["locale"] != "central") {
            $campaign .= strtoupper($locale["locale"]);
        }
        $start = clone $startMonth;
        $end = clone $endMonth;
        $start->setTime(0, 0, 0);
        $end->setTime(23, 59, 59);
        $result = 0;
        $adsUrl = config("financial-statement::app.ads_url");
        $accountKey = config("financial-statement::app.account_key", "printerval");
        $url = $adsUrl . "ads/get-summary-data?dateFrom=" . $start->getTimestamp() ."&dateTo=" . $end->getTimestamp();
        $adsResponse = $this->triggerSyncRequest($url . "&searchStartCampaignName=$campaign&type=bing", 'GET', []);
        $adwordsResponse = $this->triggerSyncRequest($url . "&searchStartCampaignName=$campaign&type=adwords&searchAccountKey=$accountKey", 'GET', []);
        if (isset($adsResponse["status"]) && $adsResponse["status"] == "successful") {
            foreach ($adsResponse["data"] as $key => $value) {
                $d = DateTime::createFromFormat('Y-m-d', $value["date_report"]);
                if ($start <= $d && $end >= $d) {
                    $result += $value["cost"];
                }
            }
        }
        if (isset($adwordsResponse["status"]) && $adwordsResponse["status"] == "successful") {
            foreach ($adwordsResponse["data"] as $key => $value) {
                $d = DateTime::createFromFormat('Y-m-d', $value["date_report"]);
                if ($start <= $d && $end >= $d) {
                    $result += $value["cost"];
                }
            }
        }
        return $result;
    }

    private function buildDataFromLastMonth(&$data, $lastMonthData, $percentByDate, $withSalary = true, $subSalary = 0, $dayGetData = 1) {
        $locales = [];
        if (function_exists('getModuleLocale')) {
            $locales = getModuleLocale();
        } 
        $currentLocale = env("APP_LOCALE", "us");
        if ($currentLocale == "") {
            $currentLocale = "us";
        }
        $sameData = [
            "bonus_cost", "management_cost", "tool_cost", "organization_cost",
            "office_cost", "sale_cost", "other_ads_cost", "design_cost"
        ];
        foreach ($sameData as $key => $value) {
            if (!isset($data[$value]) && isset($lastMonthData[$value])) {
                $data[$value] = round(($lastMonthData[$value] / $lastMonthData["revenue"]) * $data["revenue"], 2);
            }
        }
        $data["salary_cost"] = $percentByDate * ($lastMonthData["salary_cost"] - $subSalary);
        if ($currentLocale != "central") {
            $data["salary_cost"] = round((($lastMonthData["salary_cost"] - $subSalary)/ $lastMonthData["revenue"]) * $data["revenue"], 2);
        }
        $division = 1;
        if ($dayGetData == 1) {
            $division = count($locales);
        }
        $data["insurance_cost"] = $lastMonthData["insurance_cost"] * ($data["salary_cost"] / $lastMonthData["salary_cost"]) / $division;
        $data["rent_cost"] = ($lastMonthData["rent_cost"] * $percentByDate) / $division;
        $cost = 0;
        foreach ($data as $key => $value) {
            if (strpos($key, "cost") !== false && $key != "cost_price") {
                if ($key == "salary_cost") {
                    if ($withSalary) {
                        $cost += floatval($value);
                    }
                } else {
                    $cost += floatval($value);
                }
            }
        }
        $data["cost"] = round($cost, 2);
    }

    protected function triggerSyncRequest($url, $method = 'GET', $params = [], $headers = []) {
        $ch = curl_init();
        $timeout = 10;
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        if ($headers) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }
        if ($method != 'GET') {
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($params));
        }

        curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
        $data = curl_exec($ch);
        curl_close($ch);
        return json_decode($data, true);
    }

    public function getDataProfit(Request $request) {
        $response = [
            "status" => "fail"
        ];
        $currentLocale = env("APP_LOCALE", "us");
        if ($currentLocale == "") {
            $currentLocale = "us";
        }
        $startDate = DateTime::createFromFormat('d/m/Y', $request->input("start"));
        $endDate = DateTime::createFromFormat('d/m/Y', $request->input("end"));
        $connection = DBConnect::connect($currentLocale);
        $financialStatement = $connection->table("financial_statement")
                                                ->where("type", "date")
                                                ->where("date", ">=", $startDate->format("Y-m-d"))
                                                ->where("date", "<=", $endDate->format("Y-m-d"))
                                                ->orderBy("date")
                                                ->get();
        foreach ($financialStatement as $index => $data) {
            $data->data = json_decode($data->data);
        }
        $response = [
            "status" => "successful",
            "result" => $financialStatement
        ];
        return response()->json($response);
    }

    public function buildNewDataFinancial(Request $request) {
        $response = [
            "status" => "fail",
        ];
        $fromDay = $request->input("from_day", 7);
        $date = new DateTime();
        $locales = [];
        if (function_exists('getModuleLocale')) {
            $locales = getModuleLocale();
        }
        $dataByDay = [];
        $adsCost = $this->getAdsCostByDayToNow($fromDay);
        $currentLocale = env("APP_LOCALE", "us");
        if ($currentLocale == "") {
            $currentLocale = "us";
        }
        for ($i = $fromDay; $i >= 0; $i--) {
            $dayGetData = clone $date;
            $dayGetData = $dayGetData->modify("-$i day");
            $dateFormat = $dayGetData->format('Y-m-d');
            $dataByDay[$dateFormat] = [];
            $dataByDay[$dateFormat]["ads_cost"] = 0;
            if (isset($adsCost[$dateFormat])) {
                $dataByDay[$dateFormat]["ads_cost"] = $adsCost[$dateFormat];
            }
            $salary = 0;
            if ($currentLocale == "central") {
                foreach ($locales as $index => $locale) {
                    $connectionLocale = DBConnect::connect($locale["locale"]);
                    $result = $this->getAvailableData($dayGetData, $dayGetData, $connectionLocale, $locale);
                    $this->buildDataFromLastMonthPercent($result, $dayGetData);
                    foreach ($result as $key => $value) {
                        if (!array_key_exists($key, $dataByDay[$dateFormat])) {
                            $dataByDay[$dateFormat][$key] = 0;
                        }
                        if ($key == "salary_cost") {
                            $salary = $result[$key];
                        } else {
                            $dataByDay[$dateFormat][$key] += $value;
                        }
                    }
                }
            } else {
                $locale = [];
                foreach ($locales as $key => $item) {
                    if ($currentLocale == $item["locale"]) {
                        $locale = $item;
                    }
                }
                $connectionLocale = DBConnect::connect($locale["locale"]);
                $result = $this->getAvailableData($dayGetData, $dayGetData, $connectionLocale, $locale);
                $this->buildDataFromLastMonthPercent($result, $dayGetData);
                foreach ($result as $key => $value) {
                    if (!array_key_exists($key, $dataByDay[$dateFormat])) {
                        $dataByDay[$dateFormat][$key] = 0;
                    }
                    if ($key == "salary_cost") {
                        $salary = $result[$key];
                    } else {
                        $dataByDay[$dateFormat][$key] += $value;
                    }
                }
            }
            $dataByDay[$dateFormat]["cost"] += $dataByDay[$dateFormat]["ads_cost"];
            $dataByDay[$dateFormat]["cost"] += $salary;
            $dataByDay[$dateFormat]["salary_cost"] = $salary;
        }
        
        foreach ($dataByDay as $key => $data) {
            $dayReport = DateTime::createFromFormat('Y-m-d', $key);
            $newData = [
                "day" => $dayReport->format("d"),
                "month" => $dayReport->format("m"),
                "year" => $dayReport->format("Y"),
                "data" => json_encode($data),
                "created_at" => new Datetime(),
                "updated_at" => new Datetime(),
                "type" => "date",
                "date" => $dayReport->format("Y-m-d")
            ];
            $connection = DBConnect::connect($currentLocale);
            $financialStatement = $connection->table("financial_statement")
                                                ->where("day", $dayReport->format("d"))
                                                ->where("month", $dayReport->format("m"))
                                                ->where("year", $dayReport->format("Y"))
                                                ->first();
            if ($financialStatement) {
                unset($newData["created_at"]);
                $connection->table("financial_statement")
                    ->where("day", $dayReport->format("d"))
                    ->where("month", $dayReport->format("m"))
                    ->where("year", $dayReport->format("Y"))
                    ->update($newData);
            } else {
                $financialStatement = $connection->table("financial_statement")->insert($newData);
            }
        }
        $response = [
            "status" => "successful",
            "result" => $dataByDay
        ];
        return response()->json($response);
    }

    private function getAdsCostByDayToNow($fromDay, $locale = 'central') {
        $locale = env("APP_LOCALE");
        $campaign = "Printerval";
        if ($locale != "central") {
            $campaign .= strtoupper($locale);
        }
        $now = new DateTime();
        $date = new DateTime();
        $date = $date->modify("-$fromDay day");
        $result = [];
        $adsUrl = config("financial-statement::app.ads_url");
        $url = $adsUrl . "ads/get-summary-data?dateFrom=" . $date->getTimestamp() ."&dateTo=" . $now->getTimestamp();
        $accountKey = config("financial-statement::app.account_key", "printerval");
        $adsResponse = $this->triggerSyncRequest($url . "&searchStartCampaignName=$campaign&type=bing", 'GET', []);
        $adwordsResponse = $this->triggerSyncRequest($url . "&searchStartCampaignName=$campaign&type=adwords&searchAccountKey=$accountKey", 'GET', []);
        if (isset($adsResponse["status"]) && $adsResponse["status"] == "successful") {
            foreach ($adsResponse["data"] as $key => $value) {
                if (!array_key_exists($value["date_report"], $result)) {
                    $result[$value["date_report"]] = 0;
                }
                $result[$value["date_report"]] += $value["cost"];
            }
        }
        if (isset($adwordsResponse["status"]) && $adwordsResponse["status"] == "successful") {
            foreach ($adwordsResponse["data"] as $key => $value) {
                if (!array_key_exists($value["date_report"], $result)) {
                    $result[$value["date_report"]] = 0;
                }
                $result[$value["date_report"]] += $value["cost"];
            }
        }
        return $result;
    }


    private function buildDataFromLastMonthPercent(&$result, $dayGetData) {
        $date = clone $dayGetData;
        $locale = env("APP_LOCALE");
        $financialStatement = $this->getDataByMonth($date->format("m"), $date->format("Y"), $locale);
        $subSalary = 0;
        while (!$financialStatement) {
            $date->modify("-1 month");
            $financialStatement = $this->getDataByMonth($date->format("m"), $date->format("Y"), $locale);
        }
        if (!$financialStatement) {
            $newCloneDate = clone $dayGetData;
            $financialStatement = $this->getDataByMonth($date->format("m"), $date->format("Y"), "central");
            while (!$financialStatement) {
                $newCloneDate->modify("-1 month");
                $financialStatement = $this->getDataByMonth($date->format("m"), $date->format("Y"), "central");
            }
            if ($financialStatement && $newCloneDate->format("m") == 1 && $newCloneDate->format("Y") == 2022) {
                $subSalary = 3500;
            }
        } else {
            if ($date->format("m") == 1 && $date->format("Y") == 2022) {
                $subSalary = 3500;
            }
        }
        $percent = round(1 / cal_days_in_month(CAL_GREGORIAN, $date->format("m"), $date->format("Y")), 4, PHP_ROUND_HALF_UP);
        if ($financialStatement) {            
            $this->buildDataFromLastMonth($result, $financialStatement, $percent, false, $subSalary);
        }
    }

    public function saveDataFinancialStatement(Request $request) {
        $response = [
            "status" => "fail",
        ];
        $data = $request->all();
        try {
            if (isset($data["month"]) && isset($data["year"])) {
                $month = new DateTime("{$data["year"]}-{$data["month"]}-01");
                unset($data["month"]);
                unset($data["year"]);
                $data = json_encode($data);
                $newData = [
                    "data" => $data,
                    "type" => "total",
                    "month" => $month->format("m"),
                    "year" => $month->format("Y"),
                    "created_at" => new DateTime(),
                    "updated_at" => new DateTime(),
                ];
                $connection = DBConnect::connect();
                $financialStatement = $connection->table("financial_statement")
                                            ->where("month", $month->format("m"))
                                            ->where("year", $month->format("Y"))
                                            ->where("type", "total")
                                            ->first();
                if ($financialStatement) {
                    $financialStatement = $connection->table("financial_statement")->where("id", $financialStatement->id)
                                            ->update($newData);
                } else {
                    $financialStatement = $connection->table("financial_statement")->insert([$newData]);
                }
                $response = [
                    "status" => "successful",
                    "result" => $financialStatement
                ];
            }
        } catch (\Exception $e) {
            $response["message"] = $e->getMessage() . " - Line: " . $e->getLine() . " - File: " . $e->getFile();
        }
        return response()->json($response);
    }

    private function getCache($year, $month, $date) {
        $locale = env("APP_LOCALE");
        $key = "$locale-financial-statement-$year-$month-$date";
        $data = [];
        if (Redis::exists($key)) {
            $data = json_decode(Redis::get($key), true);
        }
        return $data;
    }

    private function saveCache($year, $month, $date, $data, $overwrite = false) {
        $locale = env("APP_LOCALE");
        $key = "$locale-financial-statement-$year-$month-$date";
        if (!Redis::exists($key) || $overwrite) {
            Redis::set($key, json_encode($data), 'EX', 60 * 30);
        }
    }

    public function cronFinancialData() {
        $response = [
            "status" => "fail",
        ];
        $date = new DateTime();
        $endMonthDate = cal_days_in_month(CAL_GREGORIAN, $date->format("m"), $date->format("Y"));
        try {
            $result = $this->getDataByMonthYear($date->format("m"), $date->format("Y"), $date->format("d"), false, false);
            $data = $result["data"];
            if ($data) {
                $this->saveCache($date->format("Y"), $date->format("m"), $endMonthDate, $data, true);
                $this->saveCache($date->format("Y"), $date->format("m"), $date->format("d"), $data, true);
            }
            $response = [
                "status" => "successful",
                "result" => $data,
            ];
        } catch (\Exception $e) {
            $response["message"] = $e->getMessage() . " - Line: " . $e->getLine() . " - File: " . $e->getFile();
        }
        return response()->json($response);
    }
}
