<?php

namespace Modules\FinancialStatement\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Modules\FinancialStatement\Controllers\Controller;
use Modules\FinancialStatement\Services\PaypalService;

class PaypalController extends Controller {


    public function getReport(Request $request) {
        if (!hasPermission('report/business/index')) {
            exit;
        }
        $filter = $this->buildFilterReport($request);

        $revenuePaymentGateway = PaypalService::getRevenue(array_merge($filter, [
            'event_code_in' => ['T0006'],
            'invoice_id_like' => 'PRINT',
        ]));
        $revenue = 0;
        $revenueData = [];
        $revenuePart = [];
        $pendingRevenue = [];
        $lastMonthRevenue = [];
        if (PaypalService::isFrom2024()) {
            $pendingRevenue = PaypalService::getPendingRevenue([
                'created_time_from' => date('Y-m-d 00:00:00', strtotime($filter['transaction_to']) - 86400 * 90),
                'created_time_to' => $filter['transaction_to'],
                'payment_type' => 'PAYPAL',
            ]);
            $lastMonthRevenue = PaypalService::getLastMonthRevenue($filter);
            $revenue =  PaypalService::sumListObject($lastMonthRevenue)
                + PaypalService::sumListObject($revenuePaymentGateway)
                - PaypalService::sumListObject($pendingRevenue);
        } else {
            $revenueData = PaypalService::getRevenue([
                'date_from' => $filter['transaction_from'],
                'date_to' => $filter['transaction_to'],
                'event_code_in' => ['T0006'],
                'join_order_in_production' => 1,
            ]);
            $revenuePart = PaypalService::getRevenuePart([
                'date_from' => $filter['transaction_from'],
                'date_to' => $filter['transaction_to'],
                'payment_type' => 'PAYPAL',
            ]);
            $revenue =  PaypalService::sumListObject($revenueData) + PaypalService::sumListObject($revenuePart);
        }
        $paymentFee = PaypalService::getPaymentFee(array_merge($filter, [
            'event_code_in' => ['T0006'],
            'invoice_id_like' => 'PRINTERVAL',
        ]));
        $refund = PaypalService::getRefund(array_merge($filter, [
            'event_code_in' => ['T1107', 'T0114', 'T1110'],
            'invoice_id_like' => 'PRINTERVAL',
        ]));
        $result = [
            'revenue' => $revenue,
            'payment_fee' => PaypalService::sumListObject($paymentFee),
            'refund' => PaypalService::sumListObject($refund),
            'revenue_payment_gateway' => PaypalService::sumListObject($revenuePaymentGateway),
            'currency_config' => PaypalService::getCurrencyConfig(),
            'list_main' => [
                'revenue' => $revenue,
                'revenue_payment_gateway' => PaypalService::sumListObject($revenuePaymentGateway),
                'payment_fee' => $paymentFee,
                'refund' => $refund,
                'revenueFromPaypal' => $revenuePaymentGateway,
                'revenue_full' => PaypalService::sumListObject($revenueData),
                'revenue_part' => PaypalService::sumListObject($revenuePart),
                'revenue_pending' => PaypalService::sumListObject($pendingRevenue),
                'revenue_last_month' => PaypalService::sumListObject($lastMonthRevenue),
            ],
            'list' => [],
            'site' => env('APP_URL')
        ];
        if (str_contains(env('APP_URL'), 'printerval')) {
            $printblurData = $this->getReportFromPrintblur( date('Y-m-d', strtotime($filter['transaction_from'])),  date('Y-m-d', strtotime($filter['transaction_to'])));
            $this->mergeReport($result, $printblurData);
        }
        return [
           'status' => 'successful',
           'result' => $result,

        ];
    }


    public function getReportFromPrintblur($dateFrom, $dateTo)
    {
        $url = "https://printblur.com/financial-statement/payment/paypal/get-report?date_from=$dateFrom&date_to=$dateTo";
        $response = $this->request($url);
        return $this->parseOtherSiteData($response);

    }

    protected function buildFilterReport($request)
    {
        $dateFrom = $request->input('date_from', date('Y-m-d', time() - 30 * 86400));
        $dateTo = $request->input('date_to', date('Y-m-d', time()));
        $filter = [
            'transaction_from' => date('Y-m-d 00:00:00', strtotime($dateFrom)),
            'transaction_to' => date('Y-m-d 23:59:59', strtotime($dateTo)),
        ];
        return $filter;
    }


    //T1110 : giữ điều tra tranh chấp
    //T2103: giữ tạm thời
    //T0006: user thanh toán
    public function historyTransaction(Request $request) {
        set_time_limit(7200);
        $paypalConfigs = Config::get('financial-statement::paypal');
        $from = $request->input('from', 1);
        $to = $request->input('to', 0);
        $error = [];

        foreach ($paypalConfigs as $config) {
            for($i = $from; $i > $to; $i--) {
                $filters = $this->buildFilters($i);
                try {
                    foreach ($filters as $filter) {
                        $this->storeTransactions($config['clientId'], $config['secret'], $filter);
                    }
                } catch (\Exception $e) {
                    $error[] = [
                        'message' => $e->getMessage() . ' - file ' . $e->getFile()  . ' line ' . $e->getLine(),
                        'filter' => $filter
                    ];
                }
            }

        }

        return response()->json([
            'status' => 'successful',
            'error' => $error,
        ]);
    }

    protected function storeTransactions($clientID, $secret, $filter) {
        $page = 1;
        $maxPage = 1;
        do {
            $filter['page'] = $page;
            $data = PaypalService::getTransactions($clientID, $secret, $filter);
            if (isset($data->transaction_details)) {
                PaypalService::storeItems($data->transaction_details);
            }
            if (isset($data->total_pages)) {
                $maxPage = $data->total_pages;
            }
            $page++;
        } while($page <= $maxPage);
    }

    protected function buildFilters($day) {
        $startTime = time() - $day * 86400;
        $filters =  [
            [
                'start_date' => date('Y-m-d', $startTime) . 'T00:00:00Z',
                'end_date' => date('Y-m-d', $startTime). 'T08:00:00Z',
            ],
            [
                'start_date' => date('Y-m-d', $startTime) . 'T08:00:00Z',
                'end_date' => date('Y-m-d', $startTime). 'T16:00:00Z',
            ],
            [
                'start_date' => date('Y-m-d', $startTime) . 'T16:00:00Z',
                'end_date' => date('Y-m-d', $startTime). 'T23:59:59Z',
            ],
        ];
        return $filters;
    }

    protected function buildFilter($request) {
        //date('Y-m-d\TH:i:sP')
        $startTime = time() - $request->input('from', 2) * 86400;
        $toTime = time() - $request->input('to', 1) * 86400;
        return [
            'start_date' => date('Y-m-d\T00:00:00-0000', $startTime),
            'end_date' => date('Y-m-d\T23:59:59-0000', $toTime),
        ];
    }
}