<?php

namespace Modules\FinancialStatement\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Modules\FinancialStatement\Controllers\Controller;
use Modules\FinancialStatement\Services\PaypalService;
use Modules\FinancialStatement\Services\StripeService;

class StripeController extends Controller {
    const CACHE_LAST_MONTH_PENDING_REVENUE_KEY = 'stripe:last_month:pending_revenue';
    const CACHE_LAST_MONTH_REVENUE_KEY = 'stripe:last_month:revenue';
    public function getReport(Request $request) {
        if (!hasPermission('report/business/index')) {
            exit;
        }
        $filter = $this->buildFilterReport($request);
        $paymentGatewayFilter = array_merge($filter, [
            'types' => ['payment', 'charge'],
            'groupBy' => 'st.currency',
            'join_charges' => 1,
            'invoice_id_like' => 'PRINT',
        ]);
        $revenuePaymentGateway = StripeService::getAmount($paymentGatewayFilter);
        $revenuePaymentGatewayAmount = isset($revenuePaymentGateway->amount) ? StripeService::convertVND($revenuePaymentGateway->amount, 'VND') : 0;
        $revenueOrderFull = 0;
        $revenuePart = [];
        $pendingRevenue = 0;
        $lastMonthRevenue = 0;
        if (PaypalService::isFrom2024()) {
            $pendingRevenue =  Cache::get(self::CACHE_LAST_MONTH_PENDING_REVENUE_KEY . $paymentGatewayFilter['created_from']);
            if (!$pendingRevenue) {
                $pendingRevenue = StripeService::buildPendingRevenue($paymentGatewayFilter);
            }
            $lastMonthRevenue =  Cache::get(self::CACHE_LAST_MONTH_REVENUE_KEY . $paymentGatewayFilter['created_from']);
            if (!$lastMonthRevenue) {
                $lastMonthRevenue = StripeService::getLastMonthRevenue($paymentGatewayFilter);
            }
            $revenue =  $lastMonthRevenue
                + $revenuePaymentGatewayAmount
                - $pendingRevenue;
        } else {
            $revenueData = StripeService::getAmount([
                'date_from' => $filter['date_from'],
                'date_to' => $filter['date_to'],
                'groupBy' => 'st.currency',
                'join_charges' => 1,
                'join_order_in_production' => 1,
            ]);
            $revenuePart = PaypalService::getRevenuePart([
                'date_from' => $filter['date_from'],
                'date_to' => $filter['date_to'],
                'payment_type_not_in' => ['PAYPAL'],
            ]);
            $revenueOrderFull = isset($revenueData->amount) ? StripeService::convertVND($revenueData->amount, 'VND') : 0;
            $revenue =  $revenueOrderFull + PaypalService::sumListObject($revenuePart);
        }
        $paymentFee = StripeService::getFee(array_merge($filter, [
            'types' => ['payment', 'charge'],
            'groupBy' => 'st.currency',
            'join_charges' => 1,
            'invoice_id_like' => 'PRINTER',
        ]));
        $refund = StripeService::getAmount(array_merge($filter, [
            'types' => ['refund', 'payment_refund', 'adjustment'],
            'groupBy' => 'st.currency',
        ]));
        $result = [
            'revenue' => $revenue,
            'revenue_payment_gateway' => $revenuePaymentGatewayAmount,
            'payment_fee' => isset($paymentFee->amount) ? StripeService::convertVND($paymentFee->amount, 'VND') : 0,
            'refund' => isset($refund->amount) ? StripeService::convertVND($refund->amount, 'VND') : 0,
            'main' => [
                'revenue_full' => $revenueOrderFull,
                'revenue_part' => PaypalService::sumListObject($revenuePart),
                'revenue_pending' => $pendingRevenue,
                'revenue_last_month' => $lastMonthRevenue,
            ],
            'list' => [],
            'usd' => [
                'revenue' => $revenue,
                'revenue_part' => $revenuePart,
                'revenue_payment_gateway' => $revenuePaymentGateway,
                'payment_fee' => $paymentFee,
                'refund' => $refund,
            ],
            'site' => env('APP_URL')
        ];

        if (str_contains(env('APP_URL'), 'printerval')) {
            $printblurData = $this->getReportFromPrintblur($filter['date_from'],  $filter['date_to']);
            $this->mergeReport($result, $printblurData);
        }
        return [
            'status' => 'successful',
            'result' => $result,
        ];
    }

    public function getReportFromPrintblur($dateFrom, $dateTo)
    {
        $url = "https://printblur.com/financial-statement/payment/stripe/get-report?date_from=$dateFrom&date_to=$dateTo";
        $response = $this->request($url);
        return $this->parseOtherSiteData($response);

    }

    public function cacheRevenue(Request $request)
    {
        $cacheTime = $request->input('cache_time', 'last_month');
        $firstDayOfMonth = date('Y-m-1 00:00:00', strtotime('last month'));
        $lastDayOfMonth = date('Y-m-t 23:59:59', strtotime('last month'));
        $paymentGatewayFilter = [
            'created_from' => strtotime($firstDayOfMonth),
            'created_to' => strtotime($lastDayOfMonth),
            'date_from' => $firstDayOfMonth,
            'date_to' => $lastDayOfMonth,
            'types' => ['payment', 'charge'],
            'groupBy' => 'st.currency',
            'join_charges' => 1,
            'invoice_id_like' => 'PRINTER',
        ];
        $pendingRevenue = StripeService::buildPendingRevenue($paymentGatewayFilter);
        $lastMonthRevenue = StripeService::getLastMonthRevenue($paymentGatewayFilter);
        Cache::put(self::CACHE_LAST_MONTH_PENDING_REVENUE_KEY . $paymentGatewayFilter['created_from'], $pendingRevenue, 1440 * 7);
        Cache::put(self::CACHE_LAST_MONTH_REVENUE_KEY . $paymentGatewayFilter['created_from'], $lastMonthRevenue, 1440 * 7);
        return [
            'status' => 'successful'
        ];
    }

    protected function buildFilterReport($request)
    {
        $dateFrom = $request->input('date_from', date('Y-m-d', time() - 30 * 86400));
        $dateTo = $request->input('date_to', date('Y-m-d', time()));
        $filter = [
            'created_from' => strtotime(date('Y-m-d 00:00:00', strtotime($dateFrom))),
            'created_to' => strtotime(date('Y-m-d 23:59:59', strtotime($dateTo))),
            'date_from' => $dateFrom,
            'date_to' => $dateTo,
        ];
        return $filter;
    }

    public function historyTransaction(Request $request) {
        set_time_limit(3600);
        $secret = $this->getSecret();
        $filter = $this->getFilter($request);
        $filter['type'] = 'transaction';
        $this->storeTransactions($secret, $filter);
        return response()->json([
            'status' => 'successful'
        ]);
    }

    public function historyCharge(Request $request) {
        set_time_limit(3600);
        $secret = $this->getSecret();
        $filter = $this->getFilter($request);
        $filter['type'] = 'charges';
        $this->storeTransactions($secret, $filter);
        return response()->json([
            'status' => 'successful'
        ]);
    }

    protected function getSecret() {
        return Config::get('financial-statement::stripe.secret');
    }

    protected function storeTransactions($secret, $filter) {
        $count = 0;
        $params = $this->buildParams($filter);
        $type= isset($filter['type']) ? $filter['type'] : '';
        do {
            $response = StripeService::getTransactions($secret, $params, $type);
            if (!$response->data) {
                break;
            }
            $hasMore = $response->has_more;
            $count += count($response->data);
            StripeService::storeItems($response->data, $type);
            $lastObjectId = StripeService::getLastObjectId($response->data);
            if ($count >= $filter['max_item'] ) {
                break;
            }
            if ($lastObjectId) {
                $params['starting_after'] = $lastObjectId;
            }
        } while($hasMore);
    }

    protected function buildParams($filter) {
        $result = [
            'limit' => 100
        ];
        $keys = ['created.gt','created.gte', 'created.lt', 'created.lte', 'starting_after'];
        foreach ($keys as $key) {
            if (isset($filter[$key])) {
                $result[$key] = $filter[$key];
            }
        }
        return $result;

    }

    protected function getFilter($request, $day = 1) {
        $fromTime = time() - ($day + 1) * 86400;
        $toTime = time() - $day * 86400;
        $result = [
            'max_item' => $request->input('max_item', 5000),
//            'created.gte' => $fromTime,
        ];
        if ($request->input('created-gte') && $request->input('created-lte')) {
            $result['created.gte'] = $request->input('created-gte');
            $result['created.lte'] = $request->input('created-lte');
        }
        return $result;
    }



}