<?php

namespace Modules\FinancialStatement\Services;

use Carbon\Carbon;
use Elasticsearch\ClientBuilder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\View;
use Modules\ZSearch\Models\Product;

class PaypalService
{
    static $domain = 'https://api-m.paypal.com';
    static $config = [];
    static $KEY_CURRENCY_RATIO = 'currency_ratio_parse';

    static $localeToCurrency = [
        'us' => 'USD',
        'uk' => 'USD',
    ];

    public static function getRevenue($filter) {
        $result = self::buildQuery($filter)
            ->groupBy('transaction_amount_currency')
        ->get([
            DB::raw('SUM(transaction_amount) as amount'), 'transaction_amount_currency as currency'
        ]);
        self::decorFullCurrency($result);
        return $result;
    }

    public static function getRevenuePart($filter) {
        $filter['production_status'] = 'part';
        $query = self::buildOrderInProductionQuery($filter);
        $query->where(function($where) {
            $where->whereIn('status', ['DELIVERING', 'FINISHED', 'ISSUED', 'PENDING', 'PROCESSING', 'READY_TO_SHIP'])
                ->orWhere('return_fee', '>', 0);
        });
        $query->whereNotNull('currency');
        $query->groupBy('currency');
        $result = $query->get([
            DB::raw('SUM(amount) as amount'), 'currency'
        ]);
        self::decorFullCurrency($result);
        return $result;
    }
    public static function getPendingRevenue($filter) {
        $query = self::buildOrderInProductionQuery($filter);
        $query->where(function($where) use ($filter) {
            $where->whereNull('in_production_time')
                ->orWhere('in_production_time', '>', $filter['created_time_to']);

        });
        $query->whereNotNull('currency');
        $query->groupBy('currency');
        $result = $query->get([
            DB::raw('SUM(amount) as amount'), 'currency'
        ]);
        self::decorFullCurrency($result);
        return $result;
    }

    public static function buildOrderInProductionQuery($filter) {
        $query = DB::table('order_in_production');
        if (array_key_exists('production_status', $filter)) {
            $query->where('production_status', $filter['production_status']);
        }
        if (array_key_exists('date_from', $filter)) {
            $query->where('in_production_time', '>=', $filter['date_from']);
        }
        if (array_key_exists('date_to', $filter)) {
            $query->where('in_production_time', '<=', $filter['date_to']);
        }
        if (array_key_exists('created_time_from', $filter)) {
            $query->where('created_time', '>=', $filter['created_time_from']);
        }
        if (array_key_exists('created_time_to', $filter)) {
            $query->where('created_time', '<=', $filter['created_time_to']);
        }
        if (array_key_exists('payment_type', $filter)) {
            $query->where('payment_type', '=', $filter['payment_type']);
        }
        if (array_key_exists('payment_type_not_in', $filter)) {
            $query->whereNotIn('payment_type', $filter['payment_type_not_in']);
        }
        if (array_key_exists('production_status', $filter)) {
            $query->where('production_status', $filter['production_status']);
        }

        return $query;
    }

    public static function getPaymentFee($filter) {
        unset($filter['event_code_in']);
        unset($filter['invoice_id_like']);
        $result = self::buildQuery($filter)
            ->where('fee_amount', '<', 0)
            ->groupBy('fee_amount_currency')
            ->get([
                DB::raw('SUM(fee_amount * -1) as amount'), 'fee_amount_currency as currency'
            ]);
        self::decorFullCurrency($result);
        return $result;
    }
    public static function getRefund($filter) {
        $result = self::buildQuery($filter)
            ->groupBy('transaction_amount_currency')
            ->get([
                DB::raw('SUM(transaction_amount) as amount'), 'transaction_amount_currency as currency'
            ]);
        self::decorFullCurrency($result);
        return $result;
    }

    public static function decorFullCurrency(&$result) {
        PaypalService::decorCurrency($result);
        PaypalService::decorCurrency($result, 'VND');
    }


    public static function buildQuery($filter) {
        $query = DB::table('payment_paypal_transaction');

        if (array_key_exists('join_order_in_production', $filter)) {
            $query->join('order_in_production as op', 'op.invoice_id', '=', 'payment_paypal_transaction.invoice_id');
            $query->where('op.production_status', 'full')
                ->where(function($where) {
                    $where->whereIn('op.status', ['DELIVERING', 'FINISHED', 'ISSUED', 'PENDING', 'PROCESSING', 'READY_TO_SHIP'])
                        ->orWhere('op.return_fee', '>', 0);
                });

            if (array_key_exists('date_from', $filter)) {
                $query->where('op.in_production_time', '>=', $filter['date_from']);
            }
            if (array_key_exists('date_to', $filter)) {
                $query->where('op.in_production_time', '<=', $filter['date_to']);
            }
        }

        if (array_key_exists('transaction_from', $filter)) {
            $query->where('transaction_initiation_date', '>=', $filter['transaction_from']);
        }
        if (array_key_exists('transaction_to', $filter)) {
            $query->where('transaction_initiation_date', '<=', $filter['transaction_to']);
        }
        if (array_key_exists('event_code_not_in', $filter)) {
            $query->whereNotIn('transaction_event_code', $filter['event_code_not_in']);
        }
        if (array_key_exists('event_code_in', $filter)) {
            $query->whereIn('transaction_event_code', $filter['event_code_in']);
        }
        if (array_key_exists('invoice_id_like', $filter)) {
            $query->where('invoice_id', 'like', '%' . $filter['invoice_id_like'] . '%');
        }
        return $query;
    }

    public static function getTransactions($clientID, $secret, $params) {
        $url = self::buildTransactionUrl($params);
        $accessToken = self::getToken($clientID, $secret);
        return self::requestApi($url, $accessToken);
    }

    public static function buildTransactionUrl($params) {
        $transApiEndpoint = self::getUrl('/v1/reporting/transactions');
        if ($params) {
            //$transApiEndpoint .= '?start_date=' . $params['start_date'] . '&end_date='  . $params['end_date'];
            $transApiEndpoint .= '?' . http_build_query($params);
        }
        return $transApiEndpoint;
    }

    public static function requestApi($url, $accessToken)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'GET');
        $headers = [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $accessToken
        ];
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        $result = curl_exec($ch);
        if (curl_errno($ch)) {
            return null;
        }
        curl_close($ch);
        return json_decode($result);
    }

    public static function getToken($clientID, $secret)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, self::getUrl('/v1/oauth2/token'));
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, 'grant_type=client_credentials');
        curl_setopt($ch, CURLOPT_USERPWD, "$clientID:$secret");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        $headers = array();
        $headers[] = 'Accept: application/json';
        $headers[] = 'Accept-Language: en_US';
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        $response = curl_exec($ch);
        if (curl_errno($ch)) {
            return null;
        }
        curl_close($ch);
        $data = json_decode($response);
        return isset($data->access_token) ? $data->access_token : null;
    }
    
    public static function getUrl($path) {
        return self::$domain . $path;
    }

    public static function storeItems($items) {
        foreach ($items as $item) {
            $data = self::buildTransactionInfo($item->transaction_info);
            if ($data) {
                self::store($data);
            }
        }
    }

    public static function buildTransactionInfo($transactionInfo) {
        $result = [];
        try {
            $result = [
                'account_id' => isset($transactionInfo->paypal_account_id) ? $transactionInfo->paypal_account_id : null,
                'transaction_id' => $transactionInfo->transaction_id,
                'transaction_event_code' => $transactionInfo->transaction_event_code,
                'transaction_amount' => $transactionInfo->transaction_amount->value,
                'transaction_amount_currency' => $transactionInfo->transaction_amount->currency_code,
                'reference_id' => isset($transactionInfo->paypal_reference_id) ? $transactionInfo->paypal_reference_id : null,
                'reference_id_type' => isset($transactionInfo->paypal_reference_id_type) ? $transactionInfo->paypal_reference_id_type : null,
                'fee_amount' => isset($transactionInfo->fee_amount) ? $transactionInfo->fee_amount->value : null,
                'fee_amount_currency' =>  isset($transactionInfo->fee_amount) ? $transactionInfo->fee_amount->currency_code : null,
                'shipping_amount' => isset($transactionInfo->shipping_amount) ? $transactionInfo->shipping_amount->value : null,
                'shipping_amount_currency' => isset($transactionInfo->shipping_amount) ? $transactionInfo->shipping_amount->currency_code : null,
                'transaction_status' => $transactionInfo->transaction_status,
                'ending_balance' => $transactionInfo->ending_balance->value,
                'ending_balance_currency' => $transactionInfo->ending_balance->currency_code,
                'available_balance' => $transactionInfo->available_balance->value,
                'available_balance_currency' => $transactionInfo->available_balance->currency_code,
                'invoice_id' => isset($transactionInfo->invoice_id) ? $transactionInfo->invoice_id : null,
                'protection_eligibility' => $transactionInfo->protection_eligibility,
                'transaction_initiation_date' => new \DateTime($transactionInfo->transaction_initiation_date),
                'transaction_updated_date' => new \DateTime($transactionInfo->transaction_updated_date),
            ];
        } catch (\Exception $e) {
            echo $e->getMessage();
            echo '<pre>';
            echo print_r($transactionInfo);
            echo '</pre>';
            exit;
        }
        return $result;
    }

    public static function store($item) {
        if (self::exists($item['transaction_id'])) {
            $item['updated_at'] = new \DateTime();
            DB::table('payment_paypal_transaction')
                ->where('transaction_id', $item['transaction_id'])
                ->update($item);
        } else {
            $item['created_at'] = new \DateTime();
            DB::table('payment_paypal_transaction')->insert($item);
        }
    }

    public static function exists($transactionId) {
        return DB::table('payment_paypal_transaction')
            ->where('transaction_id', $transactionId)
            ->exists();
    }


    public static function sumListObject($items, $column = 'VND') {
        $result = 0;
        if ($items && count($items) > 0) {
            foreach ($items as $item) {
                $result += is_array($item) ? $item[$column]: $item->{$column};
            }
        }
        return $result;
    }

    public static function decorCurrency(&$items, $toCountry = 'USD', $countryKey = 'currency', $valueKey = 'amount')
    {
        if (!$items) {
            return;
        }
        $rateCountry = self::getRatio($toCountry);
        if ($toCountry == 'VND') {
            $rateCountry = self::getVNDRate();
        }

        foreach ($items as $key => &$item) {
            if (is_array($item)) {
                $item[$toCountry] = self::convertToUsd($item[$valueKey], $item[$countryKey]) * $rateCountry;
            } else {
                $item->{$toCountry} =  self::convertToUsd($item->{$valueKey}, $item->{$countryKey}) * $rateCountry;
            }
        }
    }

    public static function getCurrencyConfig() {
        $countries = ['USD', 'AUD', 'CAD', 'EUR', 'GBP', 'JPY', 'USD', 'VND'];
        $result = [];
        foreach ($countries as $country) {
            $result[$country] = self::getRatio($country);
        }
        return $result;
    }


    public static function convertToUsd($value, $countryCode)
    {
        $countryCode = strtoupper($countryCode);
        return ceil($value/self::getRatio($countryCode));
    }

    public static function convert($value, $countryCode)
    {
        return ceil($value * self::getRatio($countryCode));
    }

    public static function convertVND($value, $countryCode)
    {
        return ceil($value * self::getVNDRate());
    }

    public static function getVNDRate() {
        $result = 25100;
        if (request()->input('date_from') == '2023-10-01') {
            $result = 23883;
        }  else if (request()->input('date_from') == '2023-11-01') {
            $result = 23454;
        }  else if (request()->input('date_from') == '2023-12-01') {
            $result = 24000;
        }  else if (request()->input('date_from') == '2024-09-01' || request()->input('date_from') == '2024-9-1') {
            $result = 24400;
        }
        return $result;
    }

    public static function getRatio($countryCode)
    {
        $config = self::getConfig();
        return isset($config[$countryCode]['ratio']) ? $config[$countryCode]['ratio'] : -1;
    }


    protected static function getConfig()
    {
        if (!self::$config) {


            $option = DB::table('option')->where('key', self::$KEY_CURRENCY_RATIO)->first(['value', 'type']);
            if ($option) {
                if (request()->input('date_to') == '2023-10-31') {
                    $option->value = '{"AED":{"ratio":3.673004},"AFN":{"ratio":74.497601},"ALL":{"ratio":98.120988},"AMD":{"ratio":402.886463},"ANG":{"ratio":1.803383},"AOA":{"ratio":827.070202},"ARS":{"ratio":350.018398},"AUD":{"ratio":1.557755},"AWG":{"ratio":1.8025},"AZN":{"ratio":1.707781},"BAM":{"ratio":1.832912},"BBD":{"ratio":2.020318},"BDT":{"ratio":110.820069},"BGN":{"ratio":1.828559},"BHD":{"ratio":0.377036},"BIF":{"ratio":2848.5},"BMD":{"ratio":1},"BND":{"ratio":1.357413},"BOB":{"ratio":6.91439},"BRL":{"ratio":4.891294},"BSD":{"ratio":1.0006},"BTC":{"ratio":2.8260815e-5},"BTN":{"ratio":83.321869},"BWP":{"ratio":13.540197},"BYN":{"ratio":3.296158},"BYR":{"ratio":19600},"BZD":{"ratio":2.017019},"CAD":{"ratio":1.37998},"CDF":{"ratio":2619.999818},"CHF":{"ratio":0.89803},"CLF":{"ratio":0.032544},"CLP":{"ratio":898.000252},"CNY":{"ratio":7.280405},"COP":{"ratio":4084.25},"CRC":{"ratio":533.901879},"CUC":{"ratio":1},"CUP":{"ratio":26.5},"CVE":{"ratio":103.303759},"CZK":{"ratio":22.97203},"DJF":{"ratio":178.158065},"DKK":{"ratio":6.96047},"DOP":{"ratio":56.849716},"DZD":{"ratio":135.085995},"EGP":{"ratio":30.894801},"ERN":{"ratio":15},"ETB":{"ratio":55.936179},"EUR":{"ratio":0.933335},"FJD":{"ratio":2.26815},"FKP":{"ratio":0.813322},"GBP":{"ratio":0.8131},"GEL":{"ratio":2.704977},"GGP":{"ratio":0.813322},"GHS":{"ratio":11.94029},"GIP":{"ratio":0.813322},"GMD":{"ratio":67.250384},"GNF":{"ratio":8655.000176},"GTQ":{"ratio":7.834684},"GYD":{"ratio":209.512207},"HKD":{"ratio":7.81595},"HNL":{"ratio":24.715563},"HRK":{"ratio":7.168957},"HTG":{"ratio":132.818504},"HUF":{"ratio":353.523979},"IDR":{"ratio":15620.95},"ILS":{"ratio":3.83673},"IMP":{"ratio":0.813322},"INR":{"ratio":83.230101},"IQD":{"ratio":1309.816123},"IRR":{"ratio":42250.000363},"ISK":{"ratio":140.649962},"JEP":{"ratio":0.813322},"JMD":{"ratio":156.604375},"JOD":{"ratio":0.709303},"JPY":{"ratio":150.772966},"KES":{"ratio":151.549755},"KGS":{"ratio":89.319614},"KHR":{"ratio":4125.000082},"KMF":{"ratio":459.500392},"KPW":{"ratio":900.005192},"KRW":{"ratio":1308.770255},"KWD":{"ratio":0.30888},"KYD":{"ratio":0.833892},"KZT":{"ratio":468.864949},"LAK":{"ratio":20749.999652},"LBP":{"ratio":15032.497632},"LKR":{"ratio":328.96457},"LRD":{"ratio":187.626387},"LSL":{"ratio":19.619684},"LTL":{"ratio":2.95274},"LVL":{"ratio":0.60489},"LYD":{"ratio":4.854984},"MAD":{"ratio":10.2295},"MDL":{"ratio":17.911063},"MGA":{"ratio":4477.765803},"MKD":{"ratio":57.521027},"MMK":{"ratio":2101.267958},"MNT":{"ratio":3455.032959},"MOP":{"ratio":8.055574},"MRO":{"ratio":356.999828},"MUR":{"ratio":44.201319},"MVR":{"ratio":15.37978},"MWK":{"ratio":1122.502219},"MXN":{"ratio":17.484601},"MYR":{"ratio":4.683497},"MZN":{"ratio":63.24995},"NAD":{"ratio":19.620191},"NGN":{"ratio":803.150139},"NIO":{"ratio":36.625368},"NOK":{"ratio":11.17752},"NPR":{"ratio":133.315215},"NZD":{"ratio":1.687095},"OMR":{"ratio":0.384967},"PAB":{"ratio":1.000731},"PEN":{"ratio":3.782882},"PGK":{"ratio":3.71587},"PHP":{"ratio":55.895502},"PKR":{"ratio":287.426303},"PLN":{"ratio":4.143576},"PYG":{"ratio":7446.908212},"QAR":{"ratio":3.640749},"RON":{"ratio":4.634986},"RSD":{"ratio":109.822922},"RUB":{"ratio":91.979009},"RWF":{"ratio":1229.189022},"SAR":{"ratio":3.751302},"SBD":{"ratio":8.412809},"SCR":{"ratio":13.082018},"SDG":{"ratio":599.496685},"SEK":{"ratio":10.879425},"SGD":{"ratio":1.355435},"SHP":{"ratio":1.21675},"SLL":{"ratio":19750.000018},"SOS":{"ratio":570.999923},"SRD":{"ratio":37.892495},"STD":{"ratio":20697.981008},"SYP":{"ratio":13001.744643},"SZL":{"ratio":18.46076},"THB":{"ratio":35.519874},"TJS":{"ratio":10.981884},"TMT":{"ratio":3.51},"TND":{"ratio":3.146499},"TOP":{"ratio":2.388503},"TRY":{"ratio":28.48746},"TTD":{"ratio":6.791622},"TWD":{"ratio":32.228495},"TZS":{"ratio":2513.000152},"UAH":{"ratio":36.056118},"UGX":{"ratio":3767.413745},"USD":{"ratio":1},"UYU":{"ratio":39.970198},"UZS":{"ratio":12305.29887},"VEF":{"ratio":3518149.774272},"VND":{"ratio":24300},"VUV":{"ratio":121.084254},"WST":{"ratio":2.772088},"XAF":{"ratio":614.750288},"XAG":{"ratio":0.043958},"XAU":{"ratio":0.000512},"XCD":{"ratio":2.70255},"XDR":{"ratio":0.761178},"XOF":{"ratio":614.741646},"XPF":{"ratio":111.750169},"YER":{"ratio":250.350028},"ZAR":{"ratio":18.45115},"ZMK":{"ratio":9001.203335},"ZMW":{"ratio":22.538775},"ZWL":{"ratio":321.999592}}';
                }
                self::$config = json_decode($option->value, true);
            }
        }
        return self::$config;
    }

    public static function isFrom2024() {
        return strtotime(self::getDateFrom()) >= strtotime('2024-01-01');
    }

    public static function getDateFrom() {
        return request()->input('date_from');
    }

    public static function getLastMonthRevenue($filter) {
        $dateFrom = self::getDateFrom();
        $result = [];
        if (strtotime($dateFrom) == strtotime('2024-1-1')) {
            $result = [
                [
                    'amount' => 77132,
                    'currency' => 'USD',
                ],
                [
                    'amount' => 8883,
                    'currency' => 'AUD',
                ],
                [
                    'amount' => 2834,
                    'currency' => 'CAD',
                ],
                [
                    'amount' => 50700,
                    'currency' => 'EUR',
                ],
                [
                    'amount' => 19225,
                    'currency' => 'GBP',
                ],
                [
                    //@todo error
                    'amount' => 24814,
                    'currency' => 'GBP',
                ],
            ];
        } else if (strtotime($dateFrom) == strtotime('2024-2-1')) {
            $result = [
                [
                    'amount' => 55847,
                    'currency' => 'USD',
                ],
                [
                    'amount' => 8237,
                    'currency' => 'AUD',
                ],
                [
                    'amount' => 4465,
                    'currency' => 'CAD',
                ],
                [
                    'amount' => 52778,
                    'currency' => 'EUR',
                ],
                [
                    'amount' => 15245,
                    'currency' => 'GBP',
                ],
                [
                    'amount' => 58851,
                    'currency' => 'JPY',
                ],
            ];
        } else if (strtotime($dateFrom) == strtotime('2023-3-1')) {
            $result = [
                [
                    'amount' => 49765,
                    'currency' => 'USD',
                ],
                [
                    'amount' =>  9011,
                    'currency' => 'AUD',
                ],
                [
                    'amount' => 2870,
                    'currency' => 'CAD',
                ],
                [
                    'amount' => 41077,
                    'currency' => 'EUR',
                ],
                [
                    'amount' => 12840,
                    'currency' => 'GBP',
                ],
                [
                    'amount' => 31617,
                    'currency' => 'JPY',
                ],
            ];
        } else {
            $startOfMonth = Carbon::parse($filter['transaction_to'])->subMonth();
            $filter = [
                'payment_type' => 'PAYPAL',
                'created_time_from' => date('Y-m-d 00:00:00', strtotime($filter['transaction_to']) - 86400 * 90),
                'created_time_to' => $startOfMonth->endOfMonth()->format('Y-m-d 23:59:59'),
            ];
            $result = PaypalService::getPendingRevenue($filter);
        }
        self::decorFullCurrency($result);
        return $result;
    }



}