<?php
namespace Modules\FinancialStatement\Services;
use Carbon\Carbon;
use Elasticsearch\ClientBuilder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\View;
use Modules\ZSearch\Models\Product;

class StripeService {

    const TYPE_CHARGE = 'charges';
    static $domain = 'https://api.stripe.com';

    public static function getAmount($filter) {
        $result = self::buildQuery($filter)
            ->first([
                DB::raw('SUM(sb_st.amount)/100 as amount'), 'st.currency'
            ]);
        return $result;
    }

    public static function getPendingRevenue($filter) {

        $query = self::buildQuery($filter);
        $query->where(function($where) use ($filter) {
            $where->whereNull('in_production_time')
                ->orWhere('in_production_time', '>', $filter['created_time_to']);

        });
        return $query->first([
            DB::raw('SUM(sb_st.amount)/100 as amount'), 'st.currency'
        ]);
    }

    public static function getFee($filter) {
        unset($filter['invoice_id_like']);
        $result = self::buildQuery($filter)
            ->first([
                DB::raw('SUM(sb_st.fee)/100 as amount'), 'st.currency'
            ]);
        return $result;
    }


    public static function buildQuery($filter) {
        $query = DB::table('payment_stripe_transaction as st');

        if (array_key_exists('join_charges', $filter)) {
            $query->join('payment_stripe_charges as sc', 'sc.transaction_id', '=', 'st.transaction_id');
        }
        if (array_key_exists('join_order_in_production', $filter)) {
            $query->join('order_in_production as op', 'op.invoice_id', '=', 'sc.invoice_id')
                    ->where('op.production_status', 'full')
                  ->where(function($where) {
                        $where->whereIn('op.status', ['DELIVERING', 'FINISHED', 'ISSUED', 'PENDING', 'PROCESSING', 'READY_TO_SHIP'])
                        ->orWhere('op.return_fee', '>', 0);
                  });
            if (array_key_exists('date_from', $filter)) {
                $query->where('op.in_production_time', '>=', $filter['date_from']);
            }
            if (array_key_exists('date_to', $filter)) {
                $query->where('op.in_production_time', '<=', $filter['date_to']);
            }

            if (array_key_exists('production_status', $filter)) {
                $query->where('op.production_status', $filter['production_status']);
            }
            if (array_key_exists('created_time_to', $filter)) {
                $query->where('op.created_time', '<=', $filter['created_time_to']);
            }
        }

        if (array_key_exists('created_from', $filter)) {
            $query->where('st.created', '>=', $filter['created_from']);
        }
        if (array_key_exists('created_to', $filter)) {
            $query->where('st.created', '<=', $filter['created_to']);
        }
        if (array_key_exists('invoice_id_like', $filter)) {
            $query->where(function ($where) use ($filter) {
                $where->where('sc.invoice_id', 'like', '%' . $filter['invoice_id_like'] . '%');
                $where->orWhere('sc.calculated_statement_descriptor', 'like', '%' . $filter['invoice_id_like'] . '%');
            });
             $query->where(function ($where) use ($filter) {
                 $where->where('sc.invoice_id', 'not like', '%BETIMATE%');
                 $where->orWhere('sc.calculated_statement_descriptor', 'not like', '%BETIMATE%');
             });


        }
        if (array_key_exists('types', $filter)) {
            $query->whereIn('st.type', $filter['types']);
        }
        if (array_key_exists('groupBy', $filter)) {
            $query->groupBy($filter['groupBy']);
        }
        return $query;
    }

    public static function getTransactions($secret, $params, $type = '') {
        $url = self::buildTransactionUrl($type, $params);
        return self::requestApi($url, $secret);
    }

    public static function buildTransactionUrl($type, $params) {
        $path = '/v1/balance_transactions';
        if ($type ==  self::TYPE_CHARGE) {
            $path = '/v1/charges';
        }
        $url = self::$domain . $path;
        if ($params) {
            $url .= '?' .  http_build_query($params);
        }
        return $url;
    }

    public static function storeItems($items, $type) {
        $column = self::getKeyColumn($type);
        $table = self::getTable($type);
        foreach ($items as $item) {
            $data = self::buildItem($item, $type);
            if ($data) {
                self::store($data, $column, $table);
            }
        }
    }

    public static function getTable($type) {
        $result = 'payment_stripe_transaction';
        if ($type == self::TYPE_CHARGE) {
            $result = 'payment_stripe_charges';
        }
        return $result;
    }

    public static function getKeyColumn($type) {
        $result = 'transaction_id';
        if ($type == self::TYPE_CHARGE) {
            $result = 'object_id';
        }
        return $result;
    }

    public static function buildItem($item, $type) {
        if ($type == self::TYPE_CHARGE) {
            $result = self::buildChargeItem($item);
        } else {
            $result = self::buildTransactionItem($item);
        }
        return $result;
    }

    public static function buildChargeItem($item) {
        return [
            'object_id' => $item->id,
            'object' => $item->object,
            'transaction_id' => $item->balance_transaction,
            'amount' => $item->amount,
            'amount_captured' => $item->amount_captured,
            'amount_refunded' => $item->amount_refunded,
            'invoice_id' => isset($item->metadata->invoiceId) ? $item->metadata->invoiceId : null,
            'currency' => $item->currency,
            'calculated_statement_descriptor' => $item->calculated_statement_descriptor,
            'created' => $item->created,
        ];
    }

    public static function buildTransactionItem($item) {
        $result = [];
        try {

            $result = [
                'transaction_id' => $item->id,
                'object' => $item->object,
                'amount' => $item->amount,
                'available_on' => $item->available_on,
                'created' => $item->created,
                'currency' => $item->currency,
                'description' => $item->description,
                'exchange_rate' => $item->exchange_rate,
                'fee' => $item->fee,
                'net' => $item->net,
                'reporting_category' => $item->reporting_category,
                'source' => $item->source,
                'status' => $item->status,
                'type' => $item->type,
            ];
        } catch (\Exception $e) {
            echo $e->getMessage();
            echo '<pre>';
            echo print_r($item);
            echo '</pre>';
            exit;
        }
        return $result;
    }

    public static function store($item, $column = 'transaction_id', $table = 'payment_stripe_transaction') {
        if (self::exists($item[$column], $column, $table)) {
            $item['updated_at'] = new \DateTime();
            DB::table($table)
                ->where($column, $item[$column])
                ->update($item);
        } else {
            $item['created_at'] = new \DateTime();
            DB::table($table)->insert($item);
        }
    }

    public static function exists($transactionId, $column, $table) {
        return DB::table($table)
            ->where($column, $transactionId)
            ->exists();
    }

    public static function getLastObjectId($items) {
        $result = null;
        if ($items && isset($items[count($items) - 1])) {
            $result = $items[count($items) - 1]->id;
        }
        return $result;
    }

    public static function requestApi($url, $secret) {
        $curl = curl_init();

        curl_setopt_array($curl, array(
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 0,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'GET',
            CURLOPT_HTTPHEADER => array(
                'Content-Type: application/x-www-form-urlencoded',
                'Authorization: Basic ' . base64_encode($secret)
            ),
        ));

        $response = curl_exec($curl);

        curl_close($curl);
        return json_decode($response);
    }

    public static function convertVND($value, $countryCode)
    {
        return ceil($value * self::getVNDRate());
    }

    public static function getVNDRate() {
        return PaypalService::getVNDRate();
    }


    public static function buildPendingRevenue($filter) {
        $fullPendingRevenue = StripeService::getPendingRevenue([
            'created_to' => $filter['created_to'],
            'created_time_from' => date('Y-m-d 00:00:00', $filter['created_from'] - 86400 * 60),
            'created_time_to' => date('Y-m-d 23:59:59', $filter['created_to']),
            'join_charges' => 1,
            'join_order_in_production' => 1,
            'types' => $filter['types'],
            'invoice_id_like' => $filter['invoice_id_like'],
        ]);

        $fullPendingRevenueAmount = isset($fullPendingRevenue->amount) ? StripeService::convertVND($fullPendingRevenue->amount, 'VND') : 0;
        $partPendingRevenue = PaypalService::getPendingRevenue([
            'created_time_from' => date('Y-m-d 00:00:00', $filter['created_from'] - 86400 * 60),
            'created_time_to' => date('Y-m-d 23:59:59', $filter['created_to']),
            'payment_type_not_in' => ['PAYPAL'],
            'production_status' => 'part'
        ]);
        return $fullPendingRevenueAmount + PaypalService::sumListObject($partPendingRevenue);
    }

    public static function getLastMonthRevenue($filter) {
        $dateFrom = PaypalService::getDateFrom();
        $result = 0;
        if (strtotime($dateFrom) == strtotime('2024-1-1')) {
            $result = StripeService::convertVND(434781, 'VND');
        } else if (strtotime($dateFrom) == strtotime('2024-2-1')) {
            $result = StripeService::convertVND(346760, 'VND');
        } else if (strtotime($dateFrom) == strtotime('2023-3-1')) {
            $result = StripeService::convertVND(354397, 'VND');
        }  else {
            $startOfMonth = Carbon::parse($filter['date_from'])->subMonth();
            $filter['date_from'] = $startOfMonth->startOfMonth()->format('Y-m-d');
            $filter['date_to'] = $startOfMonth->endOfMonth()->format('Y-m-d');
            $filter['created_from'] = strtotime($startOfMonth->startOfMonth()->format('Y-m-d 00:00:00'));
            $filter['created_to'] = strtotime($startOfMonth->endOfMonth()->format('Y-m-d 23:59:59'));
            $result = StripeService::buildPendingRevenue($filter);
        }
        return $result;
    }


}