<?php

namespace Modules\PaymentStats\Controllers;

use Illuminate\Http\Request;
use Modules\PaymentStats\Controllers\Controller;
use Modules\PaymentStats\Models\ProductInfo;
use Modules\PaymentStats\Models\PaymentStats;
use Modules\PaymentStats\Models\Category;
use Modules\PaymentStats\Models\Product;
use Modules\PaymentStats\Models\ProductNCategory;
use Module;
use Maatwebsite\Excel\Facades\Excel;

class PaymentStatsController extends Controller
{
    protected $pageSize = 20;
    public function __construct()
    {        

    }
    
    public function index(Request $request)
    {
        $categories = Category::where('type', '=', 'PRODUCT')
                            ->get(['name as title', 'id']);
        return view('payment-stats::payment-stats.index', [
            'categories' => $categories
        ]);
    }

    public function find(Request $request) {
        $filter = $this->buildFilter($request);
        $response = $this->getData($filter);
        return response()->json($response);
    }

    private function getData($filter) {
        $productIds = [];
        if (isset($filter['orders']) && !empty($filter['orders'])) {
            if (isset($filter['keyword']) && !empty($filter['keyword'])) {
                $products = Product::where(function ($query) use ($filter){
                    $query->orWhere('sku', 'LIKE', '%'.$filter['keyword'].'%')
                        ->orWhere('name', 'LIKE', '%' . $filter['keyword']. '%');
                })->get();
                $productIds = [-1];
                if (!empty($products)) {
                    foreach ($products as $item) {
                        $productIds[] = $item->id;
                    }
                }
                $filter['productIds'] = $productIds;
            }
            $result = $this->getStats($filter);
            if (count($result) > 0) {
                foreach ($result as $item) {
                    $productIds[] = $item->product_id;
                }
                $productByIds = $this->buildProductByIds($productIds);
                
                $this->decorItemStats($result, $productByIds);
            } else {
                unset($filter['orders']);
                $result = $this->getResult($filter);
            }
        } else {
            $result = $this->getResult($filter);
        }
        $productIds = [];
        foreach ($result as $item) {
            $productIds[] = $item->id;
        }
        $filterStast = $filter;
        $filterStast['productIds'] = $productIds;
        unset($filterStast['page_size']);
        unset($filterStast['page_id']);
        $statsByProductIds = $this->getStatsByProductIds($filterStast);
        $this->decorItems($result, $statsByProductIds);
        $pageId = $filter['page_id'];
        unset($filter['page_size']);
        unset($filter['page_id']);
        $filter['metric'] = 'count';
        $itemCount = $this->getResult($filter);
        $pagesCount = $this->recordsCountToPagesCount($itemCount, $this->pageSize);
        if (isset($filter['is_export']) && $filter['is_export']) {
            return $result;
        } else {
            $response = array(
                "status" => 'successful',
                "result" => $result,
                'pagesCount' => $pagesCount,
                'pageId' => $pageId
            );
            return $response;
        }
    }

    private function decorItemStats(&$items, $productByIds) {
        foreach ($items as $item) {
            if (isset($productByIds[$item->product_id])) {
                $item->id = $item->product_id;
                $item->sku = $productByIds[$item->product_id]->sku;
                $item->name = $productByIds[$item->product_id]->name;
                $item->slug = $productByIds[$item->product_id]->slug;
                $item->image_url = $productByIds[$item->product_id]->image_url;
            }
        }
    }

    private function buildProductByIds($productIds) {
        $query = Product::query();
        if (!empty($productIds)) {
            $query->whereIn('id', $productIds);
        }
        $items = $query->get();
        $retVal = [];
        foreach ($items as $item) {
            $retVal[$item->id] = $item;
        }
        return $retVal;
    }

    private function decorItems(&$items, $statsByProductIds) {
        foreach ($items as $item) {
            $item->click = 0;
            $item->payment = 0;
            $item->checkout = 0;
            $item->sale = 0; 
            $item->earnings = 0;
            $item->earnings_per_click = 0;
            $item->clicks_per_sale = 0;
            $item->clicks_per_checkout = 0;
            $item->ratio_sale_on_checkout = 0;
            if (isset($statsByProductIds[$item->id])) {
                $item->click = (int) $statsByProductIds[$item->id]->click;
                $item->payment = (int) $statsByProductIds[$item->id]->payment;
                $item->checkout = (int) $statsByProductIds[$item->id]->checkout;
                $item->sale = (int) $statsByProductIds[$item->id]->sale;
                $item->earnings = (float) $statsByProductIds[$item->id]->earnings;
                $item->earnings_per_click = round($statsByProductIds[$item->id]->earnings_per_click, 2);
                $item->clicks_per_sale = round($statsByProductIds[$item->id]->clicks_per_sale, 2);
                $item->clicks_per_checkout = round($statsByProductIds[$item->id]->clicks_per_checkout, 2);
                $item->ratio_sale_on_checkout = round($statsByProductIds[$item->id]->ratio_sale_on_checkout, 2);
            }
            $item->url = route('home') . '/' . $item->slug . '-p' . $item->id . '.html'; 
        }
    }

    private function getResult($filter) {
        $query = Product::query();
        if (array_key_exists('skus', $filter) && is_array($filter["skus"])) {
            $query->whereIn("sku", $filter['skus']);
        }
        if (array_key_exists('productIds', $filter) && !empty($filter['productIds'])) {
            $query->whereIn('id', $filter['productIds']);
        }

        if (isset($filter['metric']) && $filter['metric'] == 'count') {
            return $query->count();
        } else {
            if (isset($filter['orders']) && count($filter['orders']) > 0) {
                foreach ($filter['orders'] as $key => $value) {
                    $query->orderBy($key, $value);
                }
            } else {
                $query->orderBy('created_at', 'ASC');
            }
            if (isset($filter['page_size']) && isset($filter['page_id'])) {
                $query->forPage($filter['page_id'] + 1, $filter['page_size']);
            }
            return $query->get();
        }
    }

    private function getStatsByProductIds($filter) {
        $stasts = $this->getStats($filter);
        $retVal = [];
        foreach ($stasts as $item) {
            $retVal[$item->product_id] = $item;
        }
        return $retVal;
    }

    private function getStats($filter = []) {
        $query = PaymentStats::query();
        if (isset($filter['productIds']) && !empty($filter['productIds'])) {
            $query->whereIn('product_id', $filter['productIds']);
        }
        if (isset($filter['from'])) {
            $query->where('date', '>=', $filter['from']);
        }
        if (isset($filter['to'])) {
            $query->where('date', '<=', $filter['to']);
        }
        $query->groupBy('product_id')
                ->select("product_id", 
                        \DB::raw("sum(click) as click"), 
                        \DB::raw("sum(payment) as payment"), 
                        \DB::raw("sum(checkout) as checkout"), 
                        \DB::raw("sum(sale) as sale"), 
                        \DB::raw("sum(earnings) as earnings"),
                        \DB::raw("(sum(earnings) / sum(click)) as earnings_per_click"),
                        \DB::raw("(sum(click) / sum(sale)) as clicks_per_sale"),
                        \DB::raw("(sum(click) / sum(checkout)) as clicks_per_checkout"),
                        \DB::raw("(sum(sale) / sum(checkout)) as ratio_sale_on_checkout")
                    );
        if (isset($filter['orders']) && count($filter['orders']) > 0) {
            foreach ($filter['orders'] as $key => $value) {
                $query->orderBy($key, $value);
            }
        }
        if (isset($filter['page_size']) && isset($filter['page_id'])) {
            $query->forPage($filter['page_id'] + 1, $filter['page_size']);
        }
        return $query->get();
    }

    private function buildFilter($request) {
        $params = ['keyword', 'category_id', 'is_export'];
        $retVal = [];
        foreach ($params as $param) {
            if ($request->has($param)) {
                $retVal[$param] = $request->input($param);
            }
        }
        if (isset($retVal['keyword'])) {
            $items = ProductInfo::leftJoin('product', 'product.id', '=', 'product_info.product_id')
                                ->where(function ($query) use ($retVal){
                                $query->where('product_info.name', 'LIKE', '%' . $retVal["keyword"].'%')
                                    ->orWhere('product_info.sku', 'LIKE', '%' . $retVal["keyword"].'%')
                                    ->where('product.name', 'LIKE', '%' . $retVal["keyword"].'%')
                                    ->orWhere('product.sku', 'LIKE', '%' . $retVal["keyword"].'%');
                            })->distinct('product_info.product_id')
                            ->get(['product_info.product_id']);
            $items = $items->toArray();
            $productIds = [];
            if (!empty($items)) {
                foreach($items as $item) {
                    if (!in_array($item['product_id'], $productIds)) {
                        $productIds[] = $item['product_id'];
                    }
                }
            } else {
                $productIds[] = -1;
            }
            $retVal['productIds'] = $productIds;
        }
        if (isset($retVal['category_id'])) {
            $query = ProductNCategory::where('category_id', $retVal['category_id']);
            if (isset($retVal['productIds']) && !empty($retVal['productIds'])) {
                $query->whereIn('product_id', $retVal['productIds']);
            }
            $productIds = $query->pluck('product_id')->toArray();
            if (!empty($productIds)) {
                $retVal['productIds'] = $productIds;
            } else {
                $retVal['productIds'] = [-1];
            }
        }
        $currentDate = new \DateTime();
        $fromDate = $request->has('dateFrom') ? \DateTime::createFromFormat('d/m/Y', $request->input('dateFrom')) : $currentDate;
        $toDate = $request->has('dateTo') ? \DateTime::createFromFormat('d/m/Y', $request->input('dateTo')) : $currentDate;

        $retVal['from'] = $fromDate->setTime(0, 0, 0);
        $retVal['to'] = $toDate->setTime(23,59,59);
        $sort = $request->has('sort') ? $request->input('sort') : '';
        if ($sort != '') {
            $retVal['sort'] = $sort;
            $arrOrders = explode('-', $retVal['sort']);
            $retVal['orders'] = [$arrOrders[0] => $arrOrders[1]];
        }
        $isExport = (isset($retVal['is_export']) &&  $retVal['is_export']) ? $retVal['is_export'] : null;
        if ($isExport != null && $isExport) {
            $retVal['page_size'] = 40000;
        } else {
            $retVal['page_size'] = $this->pageSize;
        }
        $retVal['page_id'] = $request->has('page_id') ? $request->input('page_id') : 0;
        return $retVal;
    }

    private function recordsCountToPagesCount($recordsCount, $pageSize)
    {
        $retVal = (int)($recordsCount / $pageSize);
        if ($recordsCount % $pageSize > 0) {
            $retVal++;
        }
        //return
        return $retVal;
    }

    public function exportExcel(Request $request) {
        set_time_limit(60*60);
        ini_set('memory_limit', '2048M');
        $filters = $this->buildFilter($request);
        $products = $this->getData($filters, $request);
        $items = $products->toArray();
        $url = $this->exportToExcel($items);
        return ['message' => 'Successful', 'Download' => $url];
    }

    protected function exportToExcel($items) {
        /** Build excel file to export */
        /** Setup properties for file excel */
        $titles = ['STT', 'Mã sản phẩm', 'Tên sản phẩm', 'Link', 'Click', 'Trang đặt hàng', 'Bấm thanh toán', 'Sale', 'Thu nhập'];
        $dataExport = [$titles, $items];
        ob_end_clean();
        $retVal = Excel::create("export-payment-stats-" . time(), function ($excel) use ($dataExport) {
            $excel->setTitle('export_orders')->setCreator('Megaads')->setCompany('Megaads');
            $excel->sheet('sản phẩm', function ($sheet) use ($dataExport) {
                $sheet->row(1, $dataExport[0]);

                foreach ($dataExport[1] as $key=> $value) {
                    $insertData = [
                        $key + 1, $value['sku'], $value['name'], $value['url'], $value['click'], $value['checkout'], 
                        $value['payment'], $value['sale'], $value['earnings']
                    ];
                    $sheet->appendRow($insertData);
                }
            });
        })->store('xlsx', false, true);

        $urlDownload = route('system::download', ['fileName' => $retVal['file'], 'type' => 'exports', 'time' => time()]);

        $this->triggerSyncRequest(env('API_URL') . '/api/send-email-manual', 'POST', [
            'email' => auth()->user()->email,
            'subject' => 'Export Payment Stats Successful',
            'content' => "Download: <a href=\"$urlDownload\" target=\"_blank\">" . $urlDownload . "</a>"
        ], [
            'Content-Type: application/json'
        ]);

        return $urlDownload;
    }

    public function fake() {
        $items = ProductInfo::get(['product_id', 'product_sku_id', 'sku'])->toArray();
        $products = [];
        $skuExists = [];
        for($i = 0; $i < 100; $i++) {
            shuffle($items);
            if (!in_array($items[0]['sku'], $skuExists)) {
                $skuExists[] = $items[0]['sku'];
                $products[] = $items[0];
            }
        }
        $date = new \DateTime('01-04-2021');
        $dates[] = $date;
        for($i = 1; $i < 60; $i++) {
            $newDate = clone $date;
            $nextDate = $newDate->modify('+ ' . $i . ' day');
            $dates[] = $nextDate;
        }
        $types = ['click', 'checkout', 'payment', 'sale', 'earnings'];

        for ($i = 0; $i < 10000; $i++) {
            $dataSave = [];
            shuffle($products);
            $dataSave['product_id'] = $products[0]['product_id'];
            $dataSave['product_sku_id'] = $products[0]['product_sku_id'];
            $dataSave['sku'] = $products[0]['sku'];
            shuffle($dates);
            $dataSave['date'] = $dates[0];
            shuffle($types);
            $dataSave['type'] = $types[0];
            if ($types[0] == 'click') {
                $dataSave['click'] = rand(1, 15);
            } else if ($types[0] == 'payment') {
                $dataSave['payment'] = rand(0, 5);
            } else if ($types[0] == 'checkout') {
                $dataSave['checkout'] = rand(0, 20);
            } else if ($types[0] == 'sale') {
                $dataSave['sale'] = rand(0, 3);
            } else if ($types[0] == 'earnings') {
                $dataSave['earnings'] = rand(20, 10000);
            }
            PaymentStats::create($dataSave);
        }

        return response()->json(['status' => 'successful']);
    }


    public function action() {
        Module::action('purchase', [
            'items' => [
                [
                    'product_id' => 27,
                    'quantity' => 1,
                    'price' => 23.5
                ],
                [
                    'product_id' => 28,
                    'quantity' => 1,
                    'price' => 35
                ]
                
            ]
        
        ]);
    }
}
