<?php

namespace Modules\ProductControl\Controllers;

use Module;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Modules\ProductControl\Models\Click;
use Modules\ProductControl\Models\CategoryAverageConversion;

class BuildCategoryAverageConversionController extends Controller
{
    public function build(Request $request)
    {
        set_time_limit(3 * 3600);
        $days = $request->get("days", 30);
        $clickByProductId = [];

        for ($i = 0; $i < $days; $i++) {
            $date = date("Y-m-d", strtotime("-$i days"));
            $clickEachDate = $this->sumClickEachDate($date);
            foreach ($clickEachDate as $productId => $click) {
                if (!isset($clickByProductId[$productId])) {
                    $clickByProductId[$productId] = 0;
                }
 
                $clickByProductId[$productId] += $click;
            }
        }
        $productIds = array_keys($clickByProductId);
        $categoryIdByProductId = $this->getCategoryIdByProductId($productIds);
        $clickByCategoryId = [];
        $orderCountByCategoryId = [];
        $orderCountByProductId = [];
        foreach ($categoryIdByProductId as $productId => $categoryId) {
            $orderCountByProductId[$productId] = $this->getOrderCountByProductId($productId, $days);
        }
        foreach ($categoryIdByProductId as $productId => $categoryId) {
            if (!isset($clickByCategoryId[$categoryId])) {
                $clickByCategoryId[$categoryId] = 0;
            }
            if (isset($clickByProductId[$productId])) {
                $clickByCategoryId[$categoryId] += $clickByProductId[$productId];
            }

            if (!isset($orderCountByCategoryId[$categoryId])) {
                $orderCountByCategoryId[$categoryId] = 0;
            }
            if (isset($orderCountByProductId[$productId])) {
                $orderCountByCategoryId[$categoryId] += $orderCountByProductId[$productId];
            }
        }
        $saveData = [];
        foreach ($clickByCategoryId as $categoryId => $click) {
            if ($click == 0) {
                continue;
            }
            $orderCount = isset($orderCountByCategoryId[$categoryId]) ? $orderCountByCategoryId[$categoryId] : 0;
            $saveData[] = [
                'category_id' => $categoryId,
                'value' => round($orderCount / $click, 2),
                'updated_at' => date('Y-m-d H:i:s')
            ];
        }

        foreach ($saveData as $item) {
            CategoryAverageConversion::updateOrCreate([
                'category_id' => $item['category_id']
            ], $item);
        }

        return [
            'status' => 'successful',
            'result' => $saveData
        ];
    }

    public function getOrderCountByProductId($productId, $days = 30) {
        $startTime = date('Y-m-d 00:00:00', strtotime('-' . $days . ' days'));
        $endTime = date('Y-m-d 23:59:59', strtotime('-1 day'));

        return DB::table('order')
            ->join('order_item', 'order.id', '=', 'order_item.order_id')
            ->where('order.payment_status', 'PAID')
            ->where("order.created_at", ">=", $startTime)
            ->where("order.created_at", "<=", $endTime)
            ->where('order.status', '!=', 'CANCELLED')
            ->where('order_item.product_id', $productId)
            ->get([DB::raw('distinct sb_order.id')])
            ->count();
    }

    public function getCategoryIdByProductId($productIds) {
        $result = [];
        foreach (array_chunk($productIds, 500) as $productIdsChunk) {
            $categoryIdByProductId = DB::table("product_n_category")->whereIn("product_id", $productIdsChunk)->where('is_parent', 0)->get(["product_id", "category_id"])->pluck('category_id', 'product_id')->toArray();
            foreach ($categoryIdByProductId as $productId => $categoryId) {
                $result[$productId] = $categoryId;
            }
        }
        return $result;
    }

    public function sumClickEachDate($date) {
        $startTime = \DateTime::createFromFormat("Y-m-d", $date)->setTime(0, 0, 0);
        $endTime = \DateTime::createFromFormat("Y-m-d", $date)->setTime(23, 59, 59);
        $productIds = Click::where("date", ">=", $startTime)
            ->where('date', "<=", $endTime)
            ->get([DB::raw('distinct product_id')])
            ->pluck('product_id')
            ->toArray();
        $result = [];

        foreach (array_chunk($productIds, 500) as $chunk) {
            $clickEachDate = $this->sumClickEachProductIdAndDate($chunk, $date);
            foreach ($clickEachDate as $productId => $click) {
                $result[$productId] = $click;
            }
        }

        return $result;
    }

    public function sumClickEachProductIdAndDate($productIds, $date) {
        $startTime = \DateTime::createFromFormat("Y-m-d", $date)->setTime(0, 0, 0);
        $endTime = \DateTime::createFromFormat("Y-m-d", $date)->setTime(23, 59, 59);

        return Click::whereIn("product_id", $productIds)
            ->where("date", ">=", $startTime)
            ->where("date", "<=", $endTime)
            ->groupBy("product_id")
            ->get([DB::raw('product_id, sum(clicks) as clicks')])
            ->pluck('clicks', 'product_id')
            ->toArray();
    }
}